#include "include.h"
#include "rfm12b.h"

void RFM12BInit(char frequency, char mode)
{
	int i,j;
	
	// Set RB5 as input for nIRQ
	TRISBbits.TRISB5 = 1;
	ODCBbits.ODB5 = 0;
	
	SPIInit();								// Initialize SPI
	SS_HI;
	DummyDelay(100, 100);					// Dummy Delay
	
	RFM12BModuleInit(frequency, mode);		// Initialize RFM12B Module
	DummyDelay(100, 100);					// Dummy Delay
	
	RFM12BFIFOReset();						// FIFO Reset
	DummyDelay(100, 100);					// Dummy Delay
}	

/* SPI initialization */
/* 	The SPI uses the internal SPI1
	The following remapable pheripheral pins are used
		RP9		- SPI1 Clock
		RP8		- SPI1 Serial Data Out
		RP7 	- SPI1 Serial Data In
		RP10 	- SPI1 Slave Select
	!! These pins have to be changed according the actual configuration !! 
*/
void SPIInit()
{
	// Configure Pins
	TRISBbits.TRISB9 = 0;			// RB9 Output
	RPOR4bits.RP9R = 8;				// Set RP9 as SPI1 Clock
	
	TRISBbits.TRISB8 = 0;			// RB8 Output
	RPOR4bits.RP8R = 7;				// Set RP8 as SPI1 Serial Data Out
	
	TRISBbits.TRISB7 = 1;			// RB7 Input
	RPINR20bits.SDI1R = 7;			// Set RP7 as SPI1 Serial Data In
	
	TRISBbits.TRISB10 = 0;			// RB10 Output for SS
	ODCBbits.ODB10 = 0;
	
	// Set Config Registers
	SPI1CON1bits.DISSCK = 0;		// Internal SPI clock is enabled
	SPI1CON1bits.DISSDO = 0;		// SDO1 pin is controlled by the module
	SPI1CON1bits.MODE16 = 1;		// Communication is word-wide (16 bits)
	SPI1CON1bits.SMP = 1;			// Input data sampled at end of data output time
	SPI1CON1bits.CKE = 1;			// Serial output data changes on transition from active clock state to Idle clock state
	SPI1CON1bits.SSEN = 0;			// SS1 pin not used by module; pin controlled by port function
	SPI1CON1bits.CKP = 0;			// Idle state for clock is a low level; active state is a high level
	SPI1CON1bits.MSTEN = 1;			// Master mode
	SPI1CON1bits.SPRE = 0b111;		// Secondary prescale is 1:1
	SPI1CON1bits.PPRE = 0b01;		// Primary prescale is 16:1
	
	SPI1CON2 = 0x0000;				// Framed support disabled
	
	SPI1STATbits.SPIEN = 1;			// Enable SPI1
}

void RFM12BModuleInit(int frequency, int mode)
{
	switch (frequency)
	{
		case FREQ_434_MHZ:
			RFM12BWriteCommand(0x80D7); 		// 434 MHz, 12.0 pf, En Tx Register, En Rx FIFO Buffer
			break;
		case FREQ_868_MHZ:
			RFM12BWriteCommand(0x80E7); 		// 868 MHz, 12.0 pf, En Tx Register, En Rx FIFO Buffer
			break;
		case FREQ_915_MHZ:
			RFM12BWriteCommand(0x80F7); 		// 915 MHz, 12.0 pf, En Tx Register, En Rx FIFO Buffer
			break;
	}
	switch (mode)
	{
		case TX_MODE:
			RFM12BTXMode();
			break;
		case RX_MODE:
			RFM12BRXMode();
			break;
	}
	
	RFM12BWriteCommand(0xA640); 		// Center Frequency Select 868,000 MHz
	RFM12BWriteCommand(0xC647); 		// Data Rate: 4,8 kbps
	RFM12BWriteCommand(0x94A0); 		// VDI, FAST, 134kHz, LNA Max, DRSSI-103dBm
	RFM12BWriteCommand(0xC2AC); 		// Auto, Slow, Digital, 4
	RFM12BWriteCommand(0xCA81); 		// FIFO IT 8, 2 bytes Synchron, FIFO fill start synchron, low reset sens.
	RFM12BWriteCommand(0xCED4); 		// Sync Pattern 2DD4
	
	RFM12BWriteCommand(0xC483); 		// Keep the F offset only during VDI = high, no rest., EN AFC, EN freq. offset reg.	
	RFM12BWriteCommand(0x9850); 		// Pos freq shift, 90 KHz, Max
	RFM12BWriteCommand(0xCC17); 		// Fast, 256 kbps, disable dither in PLL
	RFM12BWriteCommand(0xE000); 		// NOT USED
	RFM12BWriteCommand(0xC800); 		// NOT USED
	RFM12BWriteCommand(0xC040); 		// 1.6 MHz,2.2V
}

/* Set mode to transmit */
void RFM12BTXMode()
{
	RFM12BWriteCommand(0x8239); 			// En Recieve, En Synthesizer, En Oscillator, Dis ClockOut
}

/* Set mode to recieve */
void RFM12BRXMode()
{
	RFM12BWriteCommand(0x8299); 			// En Recieve, En Synthesizer, En Oscillator, Dis ClockOut
}


/* Reset RFM12B RX FIFO */
void RFM12BFIFOReset()
{
	RFM12BWriteCommand(0xCA81);
	RFM12BWriteCommand(0xCA83);
}

/* Send a command to the RFM12B module */
unsigned int RFM12BWriteCommand(unsigned int command)
{
	unsigned int data_in;
	
	// SS low, start transmit
	SS_LO;
	
	// send out data
	SPI1BUF = command;
	
	// wait for transmit to end
	// if the receiver buffer is full, the message has been transmitted
	while (!SPI1STATbits.SPIRBF);
	
	// avoiding overflow and read the received message
	data_in = SPI1BUF;
	
	// SS high, end transmit	
	SS_HI;
	
	return data_in;
}

/* Send a transmit command to the RFM12B module */
void RFM12BSendMessage(unsigned char message)
{
	while (nIRQ);								// wait until nIRQ goes low
	RFM12BWriteCommand(0xB800 + message);		// send the message
}

/* Transmit a whole packet */
void RFM12BSendPacket(unsigned char packet[], unsigned int length)
{
	unsigned int i;
	
	RFM12BWriteCommand(0x0000);
		
	RFM12BSendMessage(0xAA);		// Preamble
	RFM12BSendMessage(0xAA);
	RFM12BSendMessage(0xAA);
	
	RFM12BSendMessage(0x2D);		// Sync Pattern
	RFM12BSendMessage(0xD4);
	
	for (i = 0; i < length; i++)
	{
		RFM12BSendMessage(packet[i]);
	}
							
	RFM12BSendMessage(0xAA);		// Dummy bytes
	RFM12BSendMessage(0xAA);
	RFM12BSendMessage(0xAA);
	RFM12BSendMessage(0xAA);
}

/* Receive one byte */
unsigned int RFM12BReceiveByte()
{
	unsigned int data_in;
	
	while(1) 
	{
		data_in = RFM12BWriteCommand(0x0000);
		
		if ((data_in & 0x8000))
		{
			data_in = RFM12BWriteCommand(0xB000);
			
			return (data_in & 0x00FF);
		}		
	}
}
