/*
             LUFA Library
     Copyright (C) Dean Camera, 2009.
              
  dean [at] fourwalledcubicle [dot] com
      www.fourwalledcubicle.com
*/

/*
  Copyright 2009  Dean Camera (dean [at] fourwalledcubicle [dot] com)

  Permission to use, copy, modify, and distribute this software
  and its documentation for any purpose and without fee is hereby
  granted, provided that the above copyright notice appear in all
  copies and that both that the copyright notice and this
  permission notice and warranty disclaimer appear in supporting
  documentation, and that the name of the author not be used in
  advertising or publicity pertaining to distribution of the
  software without specific, written prior permission.

  The author disclaim all warranties with regard to this
  software, including all implied warranties of merchantability
  and fitness.  In no event shall the author be liable for any
  special, indirect or consequential damages or any damages
  whatsoever resulting from loss of use, data or profits, whether
  in an action of contract, negligence or other tortious action,
  arising out of or in connection with the use or performance of
  this software.
*/

/** \file
 *
 *  USB Device Descriptors, for library use when in USB device mode. Descriptors are special 
 *  computer-readable structures which the host requests upon device enumeration, to determine
 *  the device's capabilities and functions.  
 */

#include "Mouse_Descriptors.h"

/** HID class report descriptor. This is a special descriptor constructed with values from the
 *  USBIF HID class specification to describe the reports and capabilities of the HID device. This
 *  descriptor is parsed by the host and its contents used to determine what data (and in what encoding)
 *  the device will send, and what it may be sent back from the host. Refer to the HID specification for
 *  more details on HID report descriptors.
 */
USB_Descriptor_HIDReport_Datatype_t_Mouse PROGMEM MouseReport[] =
{
	0x05, 0x01,          /* Usage Page (Generic Desktop)             */
	0x09, 0x02,          /* Usage (Mouse)                            */
	0xA1, 0x01,          /* Collection (Application)                 */
	0x09, 0x01,          /*   Usage (Pointer)                        */
	0xA1, 0x00,          /*   Collection (Application)               */
	0x95, 0x03,          /*     Report Count (3)                     */
	0x75, 0x01,          /*     Report Size (1)                      */
	0x05, 0x09,          /*     Usage Page (Button)                  */
	0x19, 0x01,          /*     Usage Minimum (Button 1)             */
	0x29, 0x03,          /*     Usage Maximum (Button 3)             */
	0x15, 0x00,          /*     Logical Minimum (0)                  */
	0x25, 0x01,          /*     Logical Maximum (1)                  */
	0x81, 0x02,          /*     Input (Data, Variable, Absolute)     */
	0x95, 0x01,          /*     Report Count (1)                     */
	0x75, 0x05,          /*     Report Size (5)                      */
	0x81, 0x01,          /*     Input (Constant)                     */
	0x75, 0x08,          /*     Report Size (8)                      */
	0x95, 0x02,          /*     Report Count (2)                     */
	0x05, 0x01,          /*     Usage Page (Generic Desktop Control) */
	0x09, 0x30,          /*     Usage X                              */
	0x09, 0x31,          /*     Usage Y                              */
	0x15, 0x81,          /*     Logical Minimum (-127)               */
	0x25, 0x7F,          /*     Logical Maximum (127)                */
	0x81, 0x06,          /*     Input (Data, Variable, Relative)     */
	0xC0,                /*   End Collection                         */
	0xC0                 /* End Collection                           */
};

/** Device descriptor structure. This descriptor, located in FLASH memory, describes the overall
 *  device characteristics, including the supported USB version, control endpoint size and the
 *  number of device configurations. The descriptor is read out by the USB host when the enumeration
 *  process begins.
 */
USB_Descriptor_Device_t PROGMEM DeviceDescriptor_Mouse =
{
	.Header                 = {.Size = sizeof(USB_Descriptor_Device_t), .Type = DTYPE_Device},
		
	.USBSpecification       = VERSION_BCD(02.00),
	.Class                  = 0x00,
	.SubClass               = 0x00,
	.Protocol               = 0x00,
				
	.Endpoint0Size          = FIXED_CONTROL_ENDPOINT_SIZE,
		
	.VendorID               = 0x03EB,
	.ProductID              = 0x2041,
	.ReleaseNumber          = 0x0000,
		
	.ManufacturerStrIndex   = 0x01,
	.ProductStrIndex        = 0x02,
	.SerialNumStrIndex      = NO_DESCRIPTOR,
		
	.NumberOfConfigurations = FIXED_NUM_CONFIGURATIONS
};

/** Configuration descriptor structure. This descriptor, located in FLASH memory, describes the usage
 *  of the device in one of its supported configurations, including information about any device interfaces
 *  and endpoints. The descriptor is read out by the USB host during the enumeration process when selecting
 *  a configuration so that the host may correctly communicate with the USB device.
 */
USB_Descriptor_Configuration_t_Mouse PROGMEM ConfigurationDescriptor_Mouse =
{
	.Config = 
		{
			.Header                 = {.Size = sizeof(USB_Descriptor_Configuration_Header_t), .Type = DTYPE_Configuration},

			.TotalConfigurationSize = sizeof(USB_Descriptor_Configuration_t_Mouse),
			.TotalInterfaces        = 1,
				
			.ConfigurationNumber    = 1,
			.ConfigurationStrIndex  = NO_DESCRIPTOR,
				
			.ConfigAttributes       = (USB_CONFIG_ATTR_BUSPOWERED | USB_CONFIG_ATTR_SELFPOWERED),
			
			.MaxPowerConsumption    = USB_CONFIG_POWER_MA(100)
		},
		
	.Interface = 
		{
			.Header                 = {.Size = sizeof(USB_Descriptor_Interface_t), .Type = DTYPE_Interface},

			.InterfaceNumber        = 0x00,
			.AlternateSetting       = 0x00,
			
			.TotalEndpoints         = 1,
				
			.Class                  = 0x03,
			.SubClass               = 0x01,
			.Protocol               = 0x02,
				
			.InterfaceStrIndex      = NO_DESCRIPTOR
		},

	.MouseHID = 
		{
			.Header                 = {.Size = sizeof(USB_Descriptor_HID_t_Mouse), .Type = DTYPE_HID},
									 
			.HIDSpec                = VERSION_BCD(01.11),
			.CountryCode            = 0x00,
			.TotalReportDescriptors = 1,
			.HIDReportType          = DTYPE_Report,
			.HIDReportLength        = sizeof(MouseReport)
		},

	.MouseEndpoint = 
		{
			.Header                 = {.Size = sizeof(USB_Descriptor_Endpoint_t), .Type = DTYPE_Endpoint},
										 
			.EndpointAddress        = (ENDPOINT_DESCRIPTOR_DIR_IN | MOUSE_EPNUM),
			.Attributes             = (EP_TYPE_INTERRUPT | ENDPOINT_ATTR_NO_SYNC | ENDPOINT_USAGE_DATA),
			.EndpointSize           = MOUSE_EPSIZE,
			.PollingIntervalMS      = 0x0A
		}	
};

/** Language descriptor structure. This descriptor, located in FLASH memory, is returned when the host requests
 *  the string descriptor with index 0 (the first index). It is actually an array of 16-bit integers, which indicate
 *  via the language ID table available at USB.org what languages the device supports for its string descriptors.
 */
USB_Descriptor_String_t PROGMEM LanguageString_Mouse =
{
	.Header                 = {.Size = USB_STRING_LEN(1), .Type = DTYPE_String},
		
	.UnicodeString          = {LANGUAGE_ID_ENG}
};

/** Manufacturer descriptor string. This is a Unicode string containing the manufacturer's details in human readable
 *  form, and is read out upon request by the host when the appropriate string ID is requested, listed in the Device
 *  Descriptor.
 */
USB_Descriptor_String_t PROGMEM ManufacturerString_Mouse =
{
	.Header                 = {.Size = USB_STRING_LEN(10), .Type = DTYPE_String},
		
	.UnicodeString          = L"Deak Peter"
};

/** Product descriptor string. This is a Unicode string containing the product's details in human readable form,
 *  and is read out upon request by the host when the appropriate string ID is requested, listed in the Device
 *  Descriptor.
 */
USB_Descriptor_String_t PROGMEM ProductString_Mouse =
{
	.Header                 = {.Size = USB_STRING_LEN(18), .Type = DTYPE_String},
		
	.UnicodeString          = L"LUFA 3 gombos eger"
};

/** This function is called by the library when in device mode, and must be overridden (see library "USB Descriptors"
 *  documentation) by the application code so that the address and size of a requested descriptor can be given
 *  to the USB library. When the device receives a Get Descriptor request on the control endpoint, this function
 *  is called so that the descriptor details can be passed back and the appropriate descriptor sent back to the
 *  USB host.
 */
uint16_t CALLBACK_USB_GetDescriptor_Mouse(const uint16_t wValue, const uint8_t wIndex, void** const DescriptorAddress)
{
	const uint8_t  DescriptorType   = (wValue >> 8);
	const uint8_t  DescriptorNumber = (wValue & 0xFF);

	void*    Address = NULL;
	uint16_t Size    = NO_DESCRIPTOR;

	switch (DescriptorType)
	{
		case DTYPE_Device:
			Address = (void*)&DeviceDescriptor_Mouse;
			Size    = sizeof(USB_Descriptor_Device_t);
			break;
		case DTYPE_Configuration:
			Address = (void*)&ConfigurationDescriptor_Mouse;
			Size    = sizeof(USB_Descriptor_Configuration_t_Mouse);
			break;
		case DTYPE_String:
			switch (DescriptorNumber)
			{
				case 0x00:
					Address = (void*)&LanguageString_Mouse;
					Size    = pgm_read_byte(&LanguageString_Mouse.Header.Size);
					break;
				case 0x01:
					Address = (void*)&ManufacturerString_Mouse;
					Size    = pgm_read_byte(&ManufacturerString_Mouse.Header.Size);
					break;
				case 0x02:
					Address = (void*)&ProductString_Mouse;
					Size    = pgm_read_byte(&ProductString_Mouse.Header.Size);
					break;
			}
			
			break;
		case DTYPE_HID: 
			Address = (void*)&ConfigurationDescriptor_Mouse.MouseHID;
			Size    = sizeof(USB_Descriptor_HID_t_Mouse);
			break;
		case DTYPE_Report: 
			Address = (void*)&MouseReport;
			Size    = sizeof(MouseReport);
			break;
	}
	
	*DescriptorAddress = Address;		
	return Size;
}

