/* PICula project
 * Copyright (c) 2009 Istvan Cserny (cserny@atomki.hu)
 *
 * Az interruptos USART kezelshez felhasznltuk a
 * http://forum.sparkfun.com/viewtopic.php?t=7542
 * frumbl tvett forrskdot (ksznet rte Pantoine
 * frumtrsunknak!), amelyet mdostottunk s kiegsztettnk. 
 */

#include "picula_config.h"
#include  <delays.h>
#include  <ctype.h>	
#include  <stdio.h>

/** \file
 *  Ez az llomny tartalmazza azokat a fggvnyeket, amelyekkel az
 *  alkalmazi programbl az USB-USART kapcsolat kezelhetjk. 
 *  \see picula_usart.h a tovbbi rszletekrt.
 */
/* GLOBLIS VLTOZK az USB-USART kommunikcihoz **************************/
#define RX_BUF_SIZE 16	//USART vevoldali buffer mrete (2 hatvnya legyen!)
#define TX_BUF_SIZE 16	//USART  adoldali buffer mrete (2 hatvnya legyen!)

 
#pragma udata access fast_vars	// Ezek a globlis vltozk az ACCESS blokkban legyenek!
near unsigned char RX_inp;		// USART vevoldali buffer bemeneti mutat
near unsigned char RX_outp;		// USART vevoldali buffer kimeneti mutat
near unsigned char TX_inp;		// USART  adoldali buffer bemeneti mutat
near unsigned char TX_outp;		// USART  adoldali buffer kimeneti mutat
near unsigned char usartchar; 	// Temporary storage



#pragma udata
FILE *stdin;							// Standard input stream
unsigned char RX_buffer[RX_BUF_SIZE];	//USART vevoldali buffer (ebbl olvasunk)
unsigned char TX_buffer[TX_BUF_SIZE];	//USART  adoldali buffer (ebbe runk)


/* FGGVNY PROTOTPUSOK *********************************************/

void InitializeSystem(unsigned int usart_baud);  /**< Rendszer inicializlsa (USART csatlakoztats) **/
void USARTDeviceTasks(void);    /**< USART kezelse (interrupt vagy polling) */
int _usart_putc(char c);       /**< Egy karakter kirsa soros porton keresztl */
void outString(const rom char* psz_s); /**< Karakterfzr kiratsa */
char _usart_getc(void);      /**< Egy karakter beolvassa soros portrl */
void outdec(long data, unsigned int ndigits); /**< Decimlis kirats adott szm tizedesjegyre */
void out4hex(unsigned int t); /**< Egy 16 bites szm kirsa hexadecimlisan */
unsigned int get4hex(void);   /**< Ngyjegy hexadecimlis szm beolvassa */ 
unsigned char get2hex(void);  /**< Ktjegy hexadecimlis szm beolvassa */
void delay_ms(unsigned int d); /**< Ksleltet eljrs, ami USARTDeviceTasks()-ot is rendszeresen meghvja */
int _user_putc (char c);		/**< External dummy fggvny a C18 programknyvtrban */
char getc(FILE *f);				/**< Egy karakter beolvassa az f stream-rl */
#define _user_getc() EOF		/**< Egyelre dummy fggvny */


#pragma code

/** Egy karakter kirsa a kimeneti bufferbe.
 *  Blokkol tpus fggvny, addig vrakozik, amg 
 *  szabad helyet nem tall a kimenti bufferben.
 *  \param c a kirand karakter
 *  \return a kirt karakter kdja int tpuss konvertlva
 */
int _usart_putc(char c)
{
  unsigned char x;
  // Start of critical region - disable Tx interrupts
  PIE1bits.TXIE = 0;
  // Copy the character into the output buffer
  TX_buffer[TX_inp] = c;

  // Check for buffer wrap around
  x = (TX_inp + 1) & (TX_BUF_SIZE - 1);
  // End of critical region - re-enable Tx interrupts 
  PIE1bits.TXIE = 1;
  while(x == TX_outp) {
     #if !defined(USE_INTERRUPT)
        USARTDeviceTasks();    //Process character I/O (polling mode)
     #endif	
  }
  TX_inp = x;
  PIE1bits.TXIE = 1;		
  return (int)c;
} /* _usart_putc */


/** Beolvas egy karaktert az input bufferbl, vagy vrakozik, ha az res.
 *  Blokkol tpus fggvny, teht polling md esetn hvogatja a 
 *  USARTDeviceTasks(); fggvnyt, amg vrakozik.
 * \return a beolvasott karakter kdja (char tpus)
 */
char _usart_getc(void)
{
char c;
  // Wait for character input
  while(RX_inp == RX_outp) {
     #if !defined(USE_INTERRUPT)
        USARTDeviceTasks();    //Process character I/O (polling mode)
     #endif	 
  }	
  //=== Start of critical region - disable Rx interrupts
  PIE1bits.RCIE = 0;
  // Copy the character into the output buffer
  c =RX_buffer[RX_outp];
  // Increment pointer and check for buffer wrap around
  RX_outp = (RX_outp+1) & (RX_BUF_SIZE - 1);
  //=== End of critical region - re-enable Tx interrupts 
  PIE1bits.RCIE = 1;
  return (c);
} /* _usart_getc */

/** Beolvas egy karaktert a megadott stream-bl, s a kapott karakter 
 *  lesz a fggvny visszatrsi rtke
 * Az f paramter vlaszt, hogy az UART stream vagy az USER stream legyen a bemenet
 */
char getc(FILE *f) {
  if ((signed char) f == (signed char) _H_USART) {
      return _usart_getc();
    }
  else
    {
      return _user_getc();
    }
} 

/** Kir egy nullval lezrt szveget a ROM memribl a kimeneti bufferbe.
* Ez a fggvny a blokkol tpus _usart_putc() fggvnyt hvja!  
* \param const rom char* psz_s mutat, a ROM-ban trolt szveghez 
*/
void outString(const rom char* psz_s)
{
  char c;
  while ((c=*psz_s))
    {
      if (c == '\n')
        {
          _usart_putc(0x0D);
        }
      _usart_putc(c);
      psz_s++;
    }
}

/** Decimlis kirats adott szm tizedesjegyre. 
 * \param data a kirand szm (eljelesen)
 * \param ndigits a kirand tizedesek szma
 */
void outdec(long data, unsigned int ndigits) {
static char sign, s[12];
unsigned int i;
	i=0; sign='+';
	if(data<0) { sign='-'; data = -data;}
	do {
		s[i]=data%10 + '0';
		data=data/10;
		i++;
		if(i==ndigits) {s[i]='.'; i++;}
	} while(data>0);
	_usart_putc(sign);
	do{
		_usart_putc(s[--i]);
	} while(i);
}

/** Ngy hexadecimlis szmjegy beolvassa s talaktsa unsigned int tpusra.
 * Ez a fggvny blokkol tpus, addig vr, amg be nem rkezik ngy kararakter,
 * amelyeket a _usart_putc() eljrssal vissza is tkrznk.
 */
unsigned int get4hex(void)
{
  char c,i;
  unsigned int t;
  t=0;
  for (i=0; i<4; i++)
    {
      c=_usart_getc();
      _usart_putc(c);
      if (c>0x40)
        {
          c -=7;
        }
      t= (t<<4) + (c & 0x0F);
    }
  return t;
}

/** Egy eljel nlkli egsz szmot hexadecimlis formban kir
 * a kimeneti bufferbe. Ez a fggvny meghvja a blokkol tpus
 *  _usart_putc() fggvnyt! 
 */
void out4hex(unsigned int t)
{
  char c;
  c=(char)((t>>12) & 0x0F);
  if (c>9) c+=7;
  _usart_putc(c+'0');
  c=(char)((t>>8) & 0x0F);
  if (c>9) c+=7;
  _usart_putc(c+'0');
  c=(char)((t>>4) & 0x0F);
  if (c>9) c+=7;
  _usart_putc(c+'0');
  c=(char)(t & 0x0F);
  if (c>9) c+=7;
  _usart_putc(c+'0');
}

/** Kt hexadecimlis szmjegy beolvassa s talaktsa unsigned char tpusra
 *  Ez a fggvny blokkol tpus, addig vr, amg be nem rkezik kt kararakter,
 *  amelyeket a _usart_putc() eljrssal vissza is tkrznk. 
 */
unsigned char get2hex(void)
{
  char c,i;
  unsigned char t;
  t=0;
  for (i=0; i<2; i++)
    {
      c=_usart_getc();
      _usart_putc(c);
      if (c>0x40)
        {
          c -=7;
        }
      t= (t<<4) + (c & 0x0F);
    }
  return t;
}

/** Ksleltet eljrs, amely polling zemmd esetn az USARt adatforgalom 
 *  fenntartsrl is gondoskodik USARTDeviceTasks() rendszeres hvogatsa tjn. 
 *  A bels ciklus kb. 1 ms ksleltetst vgez.
 * \param d az elrt vrakozsi id, 1 ms egysgekben
 */
void delay_ms(unsigned int d) {
unsigned int i,j;
	for(i=0; i<d; i++) {
	#if defined(USE_INTERRUPT)
		Delay1KTCYx(5);
	#else
		for(j=0; j<6; j++) { 
			Delay100TCYx(8);
			USARTDeviceTasks();
		}
	#endif
	}
}

/*********************************************************************
* Function:        void init_usart(unsigned char usart_conf
*                                  unsigned char usart_baud)
* PreCondition:    I/O pins configured
* Input:           None
* Output:          None
* Side Effects:    Configures Rx & Tx IRQs, zero's buffer pointers
* Overview:        initialize the PIC's onboard hw USART
*********************************************************************/
void init_usart(unsigned int usart_baud)
{
  // configure USART - USART will be stdio/stderr!
  TXSTA = 0;           // Reset USART registers to POR state
  RCSTA = 0;
  BAUDCON = 0;
  BAUDCONbits.BRG16 = 1;
  RCSTAbits.CREN = 1;
  TXSTAbits.BRGH = 1;
  SPBRG = usart_baud;       // Write baudrate to SPBRG1
  SPBRGH = usart_baud >> 8; // For 16-bit baud rate generation
  TXSTAbits.TXEN = 1;  // Enable transmitter
  RCSTAbits.SPEN = 1;  // Enable receiver

  // And the USART TX and RX buffer management
  RX_inp = 0;
  RX_outp = 0;
  TX_inp = 0;
  TX_outp = 0;

  // USART interrupts
  IPR1bits.RCIP = 0;   // USART Rx on low priority interrupt
  PIE1bits.RCIE = 1;   // Enable Rx interrupts
  IPR1bits.TXIP = 0;   // USART Tx on low priority interrupt
  PIE1bits.TXIE = 0;   // Disable Tx interrupts until we need to send
} /* init_usart */ 

/*********************************************************************
* Function:        USARTDeviceTasks(void)
* PreCondition:    I/O pins configured, USART hw initialized
* Input:           None
* Output:          None
* Side Effects:    Receive to or transmit from the RX/TX buffer
* Overview:        Manage USART input/output
*********************************************************************/
void USARTDeviceTasks(void) {
 // PIC hardware USART Rx interrupt?
  if (PIR1bits.RCIF)
  {
    // Reading Rx register clears interrupt
    usartchar = RCREG;
    // TODO: Do we need to do more for framing errors?
    if (RCSTAbits.FERR);
      // printf((rom char *)"USART:Rx Framing Error, ");
    else if (RCSTAbits.OERR)
    {
      // printf((rom char *)"USART:Rx Overrun error");
      RCSTAbits.CREN = 0;  // Clearing CREN clears any Overrun (OERR) errors
      RCSTAbits.CREN = 1;  // Re-enable continuous USART receive
    }
    else // We have a valid byte :-)
    {
      RX_buffer[RX_inp++] = usartchar;
      // Check for wrap around
      RX_inp &= (RX_BUF_SIZE - 1);
      //if (RX_inp == RX_BUF_SIZE)
      //   RX_inp = 0;
      // Check for buffer overrun
      if (RX_inp == RX_outp)
         // printf((rom char *)"USART:RX buffer overrun!\n")
	  ;
    }
  }

  // PIC hardware USART Tx interrupt?
  // NB: TXIF is almost always true... so need to look at TXIE
  //       to see if the Tx interrupt should be taken and then
  //       only take it if the Tx buffer is empty
  if ((PIE1bits.TXIE) && (TXSTAbits.TRMT))
  {
    // Do we have something to transmit?
    if (TX_inp != TX_outp)
    {
      TXREG = TX_buffer[TX_outp++];
      // Do we need to wrap around to the start of the buffer?
      TX_outp &= (TX_BUF_SIZE - 1);
      //if (TX_outp == TX_BUF_SIZE)
      //   TX_outp = 0;
    }
    else
    {
      // Disable Tx interrupts until we have something to send
      // NB: re-enable interrupts in buffer-fill routine to
      // restart Tx!
      PIE1bits.TXIE = 0;
    }
  }
} //USARTDeviceTasks()


/******************************************************************************
 * Function:        Unsigned int  ReadPOT(void)
 * Output:          unsigned int - the 10-bit right justified POT value
 *
 * Side Effects:    ADC buffer value updated
 *
 * Overview:        This function reads the POT and leaves the value in the
 *                  ADC buffer register
 *****************************************************************************/
unsigned int ReadPOT(void)
{
  unsigned int w;
  mInitPOT();
  ADCON0bits.GO = 1;              // Start AD conversion
  while (ADCON0bits.NOT_DONE);    // Wait for conversion
  w = ADRESH;
  w = (w<<8) + ADRESL;
  return w;
}//end ReadPOT



/** A rendszer inicializlsa 
 * \param usart_baud az adatsebessghez tartoz SPBRGH:SPBRG rtk (unsigned int)
 */
void InitializeSystem(unsigned int usart_baud)
{
	ADCON1 |= 0x0F;					// Default all pins to digital
	mInitAllLEDs();					// Initialize all of the LED pins
	init_usart(usart_baud);			// Initilize USART
	stdout=_H_USART;				//standard output tirnytsa _usart_putc()-re
	stderr=_H_USART;				//standard error tirnytsa _usart_putc()-re
	stdin =_H_USART;				//standard input tirnytsa _usart_getc()-re
	#if defined(USE_INTERRUPT)
		RCONbits.IPEN=1;			//Ktszint megszaktsi md belltsa
		INTCONbits.GIEH=1;			//A magas priorits interrupt engedlyezse
		INTCONbits.GIEL=1;			//Az alacsony priorits interrupt engedlyezse
	#endif
}//end InitializeSystem


