/* PICula project
 * Copyright (c) 2010 Istvan Cserny (cserny@atomki.hu)
 *
 */

/** \file
 *  Ez a header llomny gondoskodik a RESET s az interupt vektorok
 *  thelyezsrl. A \ref picula_config.h tartalmtl, vagy
 *  a fordtnak kzvetlenl tadott makrodefinciktl fggen tovbbi llomnyok
 *  becsatolsra is sor kerlhet, illetve feltteles fordts segtsgvel 
 *  alternatv kdrszletek kerlnek bele a lefordtott projektbe.
 *
 *  Figyelem! Ezt a header llomnyt csak a fprogramba szabad becsatolni!
 *  Tbb forrsllomny C nyelv projektnl a tbbi llomny elejn a
 *  *** picula_config.h *** llomnyt kell becsatolni. 
 *
 *  A legfontosabb konfigurcis paramterek:
 *
 *  - <b>__18F4520:</b>  Azt jelzi, hogy a Configure/Select Device menben PIC18F4520 mikrovezrlt 
 *    vlasztottuk. Ezt a szimblumot a fordt automatikusan definilja. 
 *
 *  - <b>USE_USART:</b> Akkor definiljuk, ha hasznlni kvnjuk az USB/UART kapcsolatot (\ref picula_config.h 
 *    llomnyban vagy az MPLAB IDE Project/Build Options/Project menjben definilhatjuk)
 *
 *  - <b>USE_INTERRUPT:</b> Akkor definiljuk, ha hasznlni kvnjuk a programmegszaktst 
 *    (vagy a \ref picula_config.h llomnyban, vagy az MPLAB IDE Project/Build Options/Project 
 *    menben definilhatjuk). <b>Figyelem! Programmegszakts esetn gondoskodnunk kell a magas s az 
 *    alacsony priorits interruptok kiszolglst ellt hi_isr() s lo_isr() eljrsokrl!</b>  
 *    Ha az USE_USB s az USE_INTERRUPT szimblumok definilva vannak, akkor automatikusan az
 *    USB_INTERRUPT szimblum is definilsra kerl, ellenkez esetben pedig az USB_POLLING 
 *    kerl definilsra. A \ref hello-int.c mintaprogramban lthat, hogy az USB interruptos 
 *    kiszolglsa hogyan vgezhet.
 *
 *  - <b>Blinking_at_start:</b> azt engedlyezhetjk vele, hogy LED4 jelezze a program futst. 
 *    A lehetsges rtkek: 0 = tilts, 1 = engedlyezs. 
 * 
 *  - <b>USE_BOOTLOADER:</b> Akkor definiljuk, ha bootloadert hasznlunk (\ref picula_config.h  
 *    llomnyban vagy az MPLAB IDE Project/Build Options/Project menpontjban definilhatjuk). 
 *    Ezzel szinkronban a linker programnak is t kell adni az USE_BOOTLOADER szimbolumot 
 *    (parancssori /uUSE_BOOTLOADER opci megadsa a Project/Build Options/Project menben, 
 *    az MPLINK belltsainak lapjn).
 *    
 * 
 *    \see picula_config.h projekt szint konfigurcis belltsok
 *    \see picula-4520.h a PIC18F4520 mintaramkr hardverspecifikus makrdefincii
 */

#ifndef PICULA_ALL
#define PICULA_ALL 

/*  INCLUDE llomnyok becsatolsa ***********************************/
#include "picula_config.h"
#if defined(USE_USART)
	#include "picula_usart.h"
#endif


/** Inline fggvny, amely egy regiszter valamelyik bitcsoportjt
 * atomi mvelettel mdostja. 
 * \param reg a mdostand regiszter neve
 * \param val a berand rtk (helyirtken helyesen!)
 * \param mask a mdostand bitcsoportot kijell maszk
 */
#define ChangeBits(reg,val,mask) reg ^= ((reg^val) & mask)	///<Bitcsoport trsa atomi mvelettel


/*  Vektorok thelyezse  ********************************************/
    void hi_isr(void);     ///< Magas priorits programmegszakts kiszolglsa
    void lo_isr(void);     ///< Alacsony priorits programmegszakts kiszolglsa

#if defined(USE_BOOTLOADER)	
	#define RESET_VECTOR		0x800
	#define HIGH_INTERRUPT_VECTOR	0x808
	#define LOW_INTERRUPT_VECTOR	0x818
#else	
	#define RESET_VECTOR		0x000
	#define HIGH_INTERRUPT_VECTOR	0x008
	#define LOW_INTERRUPT_VECTOR	0x018
#endif

/*  RESET VEKTOR ****************************************************/
// Ha nem hasznlunk bootloadert, akkor a C18 startup kdja automatikuan ltrehozza!
// Ezrt az albbi sorok csak bootloader hasznlata esetn kellenek.
#if defined(USE_BOOTLOADER)		
	extern void _startup (void);
/** Az thelyzett RESET vektor.  
 *  A bootloader hasznlattl fgg abszolt cmre kerl.
 */
	#pragma code REMAPPED_RESET_VECTOR = RESET_VECTOR
	void _reset (void) 
	{
	    _asm goto _startup _endasm
	}
#endif
	
/** Az thelyzett magas priorits interrupt vektor.  
 *  A bootloader hasznlattl fgg abszolt cmre kerl.
 */
#pragma code REMAPPED_HIGH_INTERRUPT_VECTOR = HIGH_INTERRUPT_VECTOR
void Remapped_High_ISR (void)
{
	 _asm goto hi_isr _endasm
}

/** Az thelyzett alacsony priorits interrupt vektor.  
 *  A bootloader hasznlattl fgg abszolt cmre kerl.
 */
#pragma code REMAPPED_LOW_INTERRUPT_VECTOR = LOW_INTERRUPT_VECTOR
void Remapped_Low_ISR (void)
{
	 _asm goto lo_isr _endasm 
}	

#pragma code
//res eljrsokat gyrtunk az elszabadult interruptok megfkezsre,
//ha a felhasznl nem akar programmegszaktst hasznlni!
#ifndef USE_INTERRUPT
	void hi_isr()
	{
	}	
	
	void lo_isr()
	{	
	}
#endif //USE_INTERRUPT

#endif //PICULA_ALL