/************************************************************************
*  LPC932 I2C demoprogram for MICROCORE-28 demo-kit
*
*  1. Send out hello message using the on-chip UART.
*  2. Use Timer 0 to generate a 1 second timer tick (interrupt driven).
*  3. Flash main Led every second
*  4. Run leds connected to PCF8574 every second using Timer 0 tick and
*     LPC932 interrupt driven I2C driver.
************************************************************************/

#include <reg932.h>

#include "i2cexprt.h"

#define PCF8574_WR  0x40           // i2c address I/O poort write
#define PCF8574A_WR 0x70           // i2c address I/O poort write

sbit LED1 = P0^1;                  // mcore28 board led

code char hello[] = "LPC2 - I2C driver test March 2002\n";

static bit second = 0;             // flag to report one second passed
static WORD count;
static BYTE        iopBuf[1];      // I2C data buffer for I/O port
static I2C_MESSAGE iopMsg;         // I2C messg structure for I/O port


static void UART_Init(void)
{
    P1M1 &= 0xFC;                  // TXD (P1.0) -> quasi bidir

// BRG -> BRGR1,BRGR0 = (OSC / baudrate) - 16
// BRG -> BRGR1,BRGR0 = (7.373Mhz / 19200) - 16 = 0x170

    BRGR1  = 0x01;
    BRGR0  = 0x70;
    BRGCON = 3;
    AUXR1 |= 0x40;             // enable break detect reset
    SCON = 0x52;               // mode 1, receiver enable
}

static void ua_outchar(char c)
{
    while (!TI)  ;
    SBUF = c;
    TI = 0;
}

void PrintString(char code *s)
{
    while (*s)
    {
        if (*s == '\n')
            ua_outchar('\r');         // output a '\r' first
        ua_outchar(*s);
	s++;
    }
}

//   LPC2 - OSC = 7,373 MHz, PRE = 2
//   reload_value = -OSC/(PRE*500Hz) = -7373 = every 2 msec

void T0_Interrupt(void) interrupt 1
{
    TR0 = 0;                   // Stop timer 0
    TL0 = 0x33;                // set timer 0 reload value
    TH0 = 0xe3;
    TR0 = 1;                   // Restart timer 0
    if (++count > 500)         // 500 * 2 msec = 1 sec
    {
        count  = 0;
        second = 1;            // one second passed
    }
}

static void T0_Init(void)
{
    count = 0;
    TMOD = 1;                  // Timer 0 = Mode 1, 16 Bit
    TL0 = 0x33;                // set timer 0 reload value
    TH0 = 0xe3;
    ET0 = 1;                   // enable timer 0 interrupt
    TR0 = 1;                   // start timer 0
}

static void RunLeds(void)
{
  static BYTE ioport;

    switch (ioport)            // run the I2C IOport leds
    {
      case 0x7f: ioport = 0xfe;   break;
      case 0xbf: ioport = 0x7f;   break;
      case 0xdf: ioport = 0xbf;   break;
      case 0xef: ioport = 0xdf;   break;
      case 0xf7: ioport = 0xef;   break;
      case 0xfb: ioport = 0xf7;   break;
      case 0xfd: ioport = 0xfb;   break;
      case 0xfe: ioport = 0xfd;   break;
      default:   ioport = 0xfe;   break;
    }
    iopBuf[0] = ioport;
    I2C_Write(&iopMsg);
}

void main(void)
{
    TRIM = 0x33;        // no clock out at P3.0

    P0M1 &= 0xE1;       // LEDS (P0.1-4) -> push pull output
    P0M2 |= 0x1E;

    T0_Init();          // initialize Timer 0
    UART_Init();        // initialize UART
    I2C_Init();         // initialize I2C bus
    EA = 1;             // General interrupt enable

    PrintString(hello);

    iopMsg.address = PCF8574_WR;
    iopMsg.buf     = iopBuf;
    iopMsg.nrBytes = 1;
    iopBuf[0] = 0xff;
    I2C_Write(&iopMsg);

    while (1)
    {
        if (second)
        {
            second = 0;           // reset one second passed flag */
            LED1 = !LED1;         // toggle the LED
            RunLeds();
        }
    }
}
