/***************************************************************************/
/*  Name of module    : I2CSLAVE.C                                         */
/*  Creation date     : 12 March 2003                                      */
/*  Program language  : C                                                  */
/*  Name              : P.H. Seerden                                       */
/*                                                                         */
/*        (C) Copyright 2003 Philips Semiconductors B.V.                   */
/*                                                                         */
/***************************************************************************/
#include <REG932.H>
#include "ua_exprt.h"
typedef unsigned char    BYTE;
typedef unsigned short   WORD;

#define SLAVE_IDLE           0
#define SLAVE_BUSY           1
#define SLAVE_READY          2
#define SLAVE_TRX_ERROR      3
#define SLAVE_RCV_ERROR      4

#define GENERATE_STOP        0x54      // set STO, clear STA and SI
#define RELEASE_BUS_ACK      0x44      // clear STO,STA,SI and set AA (ack)
#define RELEASE_BUS_NOACK    0x40      // clear STO, STA, SI and AA (noack)
#define RELEASE_BUS_STA      0x64      // generate (rep)START, set STA

static BYTE count;                     // bytes send/received of current message
char slaveBuf[3];
static BYTE size;                      // size of slave mode buffer
static BYTE slaveStatus;               // status of the slave

void I2C_Interrupt(void) interrupt 6 using 1
{
    switch(I2STAT)
    {
      case 0x60:                         // own SLA+W received, Ack returned 
      case 0x68:                         // Addressed as slave
      case 0x70:                         // General call received, Ack returned
      case 0x78:
        slaveStatus = SLAVE_BUSY;        // Data will be received
        count = 0;
        if (size > 1)
            I2CON = RELEASE_BUS_ACK;     // return ACK on first byte
        else
            I2CON = RELEASE_BUS_NOACK;   // return NACK on first byte
        break;
      case 0x80:
      case 0x90:                         // Data received, ACK returned
        slaveBuf[count++] = I2DAT;       // read data
        if (count == size)
            I2CON = RELEASE_BUS_NOACK;   // return NACK on next byte
        else
            I2CON = RELEASE_BUS_ACK;     //* return ACK on next byte
        break;
      case 0x88:
      case 0x98:                         // data received, NACK returned
        slaveStatus = SLAVE_RCV_ERROR;
        I2CON = RELEASE_BUS_ACK;         // clr SI, set AA
        break;
      case 0xA0:                         // STOP or REP.START received, addressed as slave
        slaveStatus = SLAVE_READY;
        I2CON = RELEASE_BUS_ACK;         // clr SI, set AA
        break;
      case 0xB0:                         // Arb. lost as MST, addressed as slave transmitter
        slaveStatus = SLAVE_BUSY;
        count = 0; 
        I2DAT = slaveBuf[count++];       // Transmit next data, restart
        I2CON = RELEASE_BUS_STA;         // MST mode if bus is free again
        break;
      case 0xA8:                         // Addressed as slave transmitter
        slaveStatus = SLAVE_BUSY;
        count = 0; 
      case 0xB8:                         // Data transmitted, ACK received
        I2DAT = slaveBuf[count++];       // Transmit next data
        I2CON = RELEASE_BUS_ACK;         // clr SI, set AA
        break;
      case 0xC0:                         // Data transmitted, NOT ACK received
        slaveStatus = SLAVE_TRX_ERROR;
      case 0xC8:
        I2CON = RELEASE_BUS_ACK;         // clr SI, set AA
        break;
      default:
        break;
    }
}

//void I2C_Init(BYTE *buf, BYTE size)
void I2C_Init(void)
{
	unsigned char temp;
    slaveStatus = SLAVE_IDLE;
    slaveBuf[0] = 0xA5;
    slaveBuf[1] = 0x55;
    slaveBuf[2] = 0x99;
    size = 3;

    P1M1  |= 0x0C;             // Configure P1.2 and P1.3 to open drain
    P1M2  |= 0x0C;

//*****************************************************************************
//*  Modified from Paul's code to select an address depending on the jumper
//*  settings of A2, A1 and A0
//*****************************************************************************	
    temp = P0;
    temp &= 0x07;							// mask out non address bits 
    temp <<= 1;							// shift left one
    I2ADR = (0xE0 | temp); 				// generate I2C address depending on P0   
//*****************************************************************************
//* End modification
//*****************************************************************************	 
   
    I2SCLH = 0x05;
    I2SCLL = 0x04;
    I2CON = RELEASE_BUS_ACK;   // enable I2C hardware
    EI2C  = 1;                 // enable I2C interrupt
}

void I2C_ProcessSlave(void)
/**************************
 * Input(s)   : None.
 * Output(s)  : None.
 * Returns    : None.
 * Description: Process the slave.
 *              This function must be called by the application to check
 *              the slave status. The USER should adapt this function to
 *              his personal needs (take the right action at a certain
 *              status).
 ***************************************************************************/
{
    switch(slaveStatus)
    {
      case SLAVE_IDLE :
        /* do nothing or fill transmit buffer for transfer                 */
        break;
      case SLAVE_BUSY :
        /* do nothing if interrupt driven         */
        break;
      case SLAVE_READY :
        /* read or fill buffer for next transfer, signal application       */
        slaveStatus = SLAVE_IDLE;
        break;
      case SLAVE_TRX_ERROR :
        /* generate error message                                          */
        slaveStatus = SLAVE_IDLE;
        break;
      case SLAVE_RCV_ERROR :
        /* generate error message                                          */
        slaveStatus = SLAVE_IDLE;
        break;
    }
}
