Program I2C_PAR;
{
  This PASCAL program demonstrates the parlallel I2C adapter
  on a standard parallel port. It's basically the
  same as I2C_SER.PAS, the serial I2C demo program.
  A 8 bit multi-IO PCF8574 is used.
  There are only a few explaining comments,
  but if you know the I2C protocol, this program
  should explain itself.
  If you don't know the I2C protocol yet, find
  it on the www.philips.com internet site.
  (Philips semiconductors)

  How to use:
  Connect the parallel I2C adapter with the
  first parallel port of your PC. Connect a
  PCF8574 (set at address 20hex) to the adapter and
  of course connect a 5V power source.
  Compile this program and run it.

  version 1.0 - Arjan Strijker, the Netherlands, Oct. 1999
}

CONST
  Sda_low  =  $80; {1000 0000 : status bit 7  +busy         }
  Sda_high =  $00;
  Scl_low  =  $00;
  Scl_high =  $08; {0000 1000 : control bit 3 -select input }
  Sda_mask =  $80; {1000 0000 : data bit 7                  }
  Scl_mask =  $08; {0000 1000 : status bit 3  -error        }

  { other pins parallel port:
    Status bit 5 = pin 12 : Paper end : with 10k to Vcc
    Status bit 6 = pin 10 : Ackn      : to ground
    Status bit 4 = pin 13 : Select    : to ground

    All status bits go somewhere.
    Databits 0..6    : NC
    Controlbits 0..2 : NC
  }

  IicDelayCount : word = 100;  {delay called after each port access }

VAR
  parport                 : word;    {parallel port to use}
  IicAck                  : BOOLEAN; {acknowledge of I2C protocol }
  IcAddr                  : byte;    {addres of I2C device }
  Sda_out, Scl_out, Sda_in: Word;    {bit banging bits }
  bt                      : byte;



PROCEDURE IicOut(prt:word;data:byte);
Var t:word;
Begin
  port[prt]:=data;
  For t:=0 to IicDelayCount do ;
End;

FUNCTION IicIn (prt:word):byte;
Var t:word;
Begin
  For t:=0 to IicDelayCount do ;
  IicIn:=port[prt];
End;


PROCEDURE IicPortSet(base_addr:word);
{ base_addr is a parallel port}
BEGIN
 Sda_out := base_addr;
 Sda_in  := base_addr+1;
 Scl_out := base_addr+2;
END;

PROCEDURE Iic_prepare;
BEGIN
 IicOut(Sda_out,Sda_high);
 IicOut(Scl_out,Scl_high);
END;

PROCEDURE Iic_start;
BEGIN
 IicOut(Sda_out,Sda_low);
 IicOut(Scl_out,Scl_low);
END;

PROCEDURE Iic_restart;
BEGIN
 IicOut(Sda_out,Sda_high);
 IicOut(Scl_out,Scl_high);
 IicOut(Sda_out,Sda_low);
 IicOut(Scl_out,Scl_low);
END;


PROCEDURE Iic_stop;
BEGIN
 IicOut(Sda_out,Sda_low);
 IicOut(Scl_out,Scl_high);
 IicOut(Sda_out,Sda_high);
END;

PROCEDURE Iic_Write(Data: BYTE; VAR IicAck: BOOLEAN);
VAR i : INTEGER;
BEGIN
 FOR i:=1 TO 8 DO
 BEGIN
  {set data bit (7..0)}
  IF (Data AND $80) = 0 THEN
   IicOut(Sda_out,Sda_low)
  ELSE
   IicOut(Sda_out,Sda_high);

  {pulse clock}
  IicOut(Scl_out,Scl_high);
  IicOut(Scl_out,Scl_low);
  DATA:=DATA shl 1;
 END;
 IicOut(Sda_out,Sda_high); {set data high to enable ackn. read}
 IicOut(Scl_out,Scl_high); {puls clk high}
 IicAck := ( IicIn(Sda_in) AND Sda_mask ) <> Sda_mask; {read ackn.}
 IicOut(Scl_out,Scl_low); {clk low again}
END;

FUNCTION Iic_Read(SendAck: BOOLEAN): BYTE;
VAR i  : INTEGER;
    Rb : BYTE;
BEGIN
 Rb := 0;
 FOR i:=1 TO 8 DO
 BEGIN
  Rb:=Rb shl 1;
  IicOut(Scl_out,Scl_high);
  IF (IicIn(Sda_in) AND Sda_mask) = Sda_mask
   THEN inc(Rb);
  IicOut(Scl_out,Scl_low);
 END;
 Iic_Read := Rb;
 IF SendAck THEN IicOut(Sda_out,Sda_low);
 IicOut(Scl_out,Scl_high);
 IicOut(Scl_out,Scl_low);
 IicOut(Sda_out,Sda_high);
END;


FUNCTION Read8574(IcAddr:byte;VAR IicAck: BOOLEAN): BYTE;
VAR IicAckn: BOOLEAN;
    AddrMode : byte;
BEGIN
 IicAck   := TRUE;
 AddrMode := IcAddr shl 1; {write mode}
 {First set write mode and reset latch (make high)}
 Iic_start;
 Iic_Write(AddrMode, IicAckn); IicAck := IicAckn AND IicAck;
 Iic_Write($FF   , IicAckn); IicAck := IicAckn AND IicAck;
 Iic_stop;

 {Now set readmode and read data}
 Iic_start;
 Inc(AddrMode); {read mode}
 Iic_Write(AddrMode, IicAckn); IicAck := IicAckn AND IicAck;
 Read8574 := Iic_Read(FALSE);
 Iic_stop;
END;

PROCEDURE Write8574(IcAddr:byte;DATA:byte;VAR IicAck: BOOLEAN);
VAR IicAckn: BOOLEAN;
BEGIN
  Iic_Start;
  Iic_Write(IcAddr shl 1,IicAck);  {set write mode}
  Iic_Write(DATA,IicAck);
  Iic_Stop;
  IicAck:=IicAck and IicAckn;
END;

PROCEDURE IicAutoSpeedSet(IcAddr:byte);
{ Sets speed 100% slower than maximum speed for acknowledge. }
{ exponential rise to 1/2*Vcc : 1+ln(0.50)=0.31              }
{ exponential rise to 3/4*Vcc : 1+ln(0.75)=0.71              }
{ 0.71/0.31 = 2.3 (about 2)                                  }
Begin
  Iic_Prepare;
  IicDelayCount:=1;
  Repeat
    Iic_Start;
    Iic_Write(IcAddr,IicAck); {try getting an ackowledge}
    Iic_Stop;
    inc(IicDelayCount);
  Until (IicAck or (IicDelayCount=500));
  inc(IicDelayCount,IicDelayCount); {delay 100% more}
End;

Begin
  writeln('--- Parallel I2C DEMO -----------------');

  {Use first parallel port}
  parport:=memw[$40:$08];

  {Prepare I2C port for first use}
  Iicportset(parport);
  Iic_prepare;

  {Autodetect I2C speed}
  IcAddr:=$20;
  IicAutoSpeedSet(IcAddr shl 1); {shl 1 for write mode}
  writeln('I2C bus speed set to ',IicDelayCount,'  (1=fast.. 1000=slow)');

  {Write some data}
  bt:=$5A;
  writeln('Writing to PCF8574: ',bt);
  write8574(IcAddr,bt,IicAck);
  writeln('Acknowledge returned: ',IicAck);

  {Read some data}
  bt:=read8574(IcAddr,IicAck);
  writeln('Reading from PCF8574: ',bt);
  writeln('Acknowledge returned: ',IicAck);
  writeln('---- Arjan Strijker - Oct. 1999 ----');
End.

