/*
 * ds18x20.c
 */

#include <delays.h>

#include "types.h"
#include "1wire.h"
#include "ds18x20.h"
#include "crc.h"

/*
 * DS18x20 ROM COMMANDS
 */
#define SEARCH_ROM	(0xF0)
#define READ_ROM		(0x33)
#define MATCH_ROM		(0x55)
#define SKIP_ROM		(0xCC)
#define ALARM_SEARCH	(0xEC)

/*
 * DS18x20 FUNCTION COMMANDS
 */
#define START_CONVERSION	(0x44)
#define WRITE_SCRATCHPAD	(0x4E)
#define READ_SCRATCHPAD		(0xBE)
#define COPY_SCRATCHPAD		(0x48)
#define RECALL_E2				(0xB8)
#define READ_POWER_SUPPLY	(0xB4)

#define MAX_DEVICES	30

static unsigned char First(void);
static unsigned char Next(void);
static bool Send_MatchRom(unsigned char device);

static unsigned char ROM[8];	/* ROM Bit */
static BYTE FoundROM[MAX_DEVICES][8];
static unsigned char lastDiscrep = 0;
static bool doneFlag = false;
static unsigned char dowcrc;
static unsigned char numROMs;

/*
 * read_18B20_sensor
 */
bool read_18B20_sensor(unsigned char device, int* temp, unsigned char* rom_code)
{
	bool readDone = false;
	int tries = 3;
	int i;
	BYTE reg[10];
	S16_VAL t;

	if (device > numROMs)
		return false;

	while (readDone == false && tries-- > 0) {

		/*
		 * ROM COMMANDS
		 */
		if (Send_MatchRom(device) != true)
			continue;

		OW_write_byte(START_CONVERSION);
		// a START_CONVERSION utn a 'strong pullup'-nak 10 usec-en bell kell jnnie !
		// erre mg nem tudok gyes megoldst !!!
		OW_HI();						/* most indul a 'strong pullup' */

		/*
		 * varakozas 2 sec
		 */
		for (i = 0; i < 20; i++)	/* 20 * 100 msec (32MHz) */
			Delay10KTCYx(80);
		OW_HIZ();					/* 'strong pullup' vege */

		/*
		 * FUNCTION COMMANDS
		 */
		if (Send_MatchRom(device) != true)
			continue;
		OW_write_byte(READ_SCRATCHPAD);
		for (i = 0; i < 9; i++)
			reg[i] = OW_read_byte();

		/*
		 * ellenorzes, stb..
		 */
		if (crc_calc(&reg[0], 8) == reg[8]) {
			t.v[0] = reg[0];
			t.v[1] = reg[1];
			if (t.Val != 0x0550) {
				*temp = t.Val;
				for (i = 0; i < 8; i++)
					rom_code[i] = FoundROM[device][i];
				readDone = true;
			}
		}
	}

	return readDone;
}

unsigned char FindDevices(void)
{
	unsigned char m;

	numROMs = 0;
	if (OW_reset() != 0)
		return numROMs;

	if (First()) {	//Begins when at least one part is found
		do {
			for (m = 0; m < 8; m++) {
				FoundROM[numROMs][m] = ROM[m]; //Identifies ROM number on found device
			}
			numROMs++;
		} while (Next() && (numROMs < MAX_DEVICES)); //Continues until no additional devices are found
	}

	return numROMs;
}

static unsigned char First(void)
{
	lastDiscrep = 0;
	doneFlag = false;

	return Next();
}

static unsigned char Next(void)
{
	unsigned char m = 1; // ROM Bit index
	unsigned char n = 0; // ROM Byte index
	unsigned char k = 1; // bit mask
	unsigned char x = 0;
	unsigned char discrepMarker = 0; // discrepancy marker
	unsigned char g; // Output bit
	unsigned char nxt; // return value
	int flag;

	nxt = 0;						// set the next flag to false
	dowcrc = 0;					// reset the dowcrc
	flag = OW_reset();		// reset the 1-wire
	if(flag || doneFlag) {	// no parts -> return false
		lastDiscrep = 0;		// reset the search
		return false;
	}
	OW_write_byte(SEARCH_ROM);	// send SearchROM command

	do {		// for all eight bytes
		x = 0;
		if (OW_read_bit() == 1)
			x = 2;
//		delay(6);
		if (OW_read_bit() == 1)
			x |= 1; // and its complement
		if (x == 3) // there are no devices on the 1-wire
			break;
		else {
			if (x > 0) // all devices coupled have 0 or 1
				g = x >> 1; // bit write value for search
			else {
				// if this discrepancy is before the last
				// discrepancy on a previous Next then pick
				// the same as last time
				if (m < lastDiscrep)
					g = ((ROM[n]&k)>0);
				else // if equal to last pick 1
					g = (m == lastDiscrep); // if not then pick 0
					// if 0 was picked then record
					// position with mask k
				if (g == 0)
					discrepMarker = m;
			}
			if (g == 1) // isolate bit in ROM[n] with mask k
				ROM[n] |= k;
			else
				ROM[n] &= ~k;
			OW_write_bit(g); // ROM search write
			m++; // increment bit counter m
			k = k<<1; // and shift the bit mask k
			if(k == 0) { // if the mask is 0 then go to new ROM
			// byte n and reset mask
				ow_crc(ROM[n], &dowcrc); // accumulate the CRC
				n++; k++;
			}
		}
	} while (n < 8); //loop until through all ROM bytes 0-7

	if (m < 65 || dowcrc) // if search was unsuccessful then
		lastDiscrep=0; // reset the last discrepancy to 0
	else {
		// search was successful, so set lastDiscrep,
		// lastOne, nxt
		lastDiscrep = discrepMarker;
		doneFlag = (lastDiscrep == 0);
		nxt = true; // indicates search is not complete yet, more
		// parts remain
	}

	return nxt;
}

static bool Send_MatchRom(unsigned char device)
{
	unsigned char i;

	if (OW_reset() != 0)
		return false;

	OW_write_byte(MATCH_ROM); // match ROM
	for (i = 0; i < 8; i++)
		OW_write_byte(FoundROM[device][i]); //send ROM code

	return true;
}
