
/* NBS inductance formulas
 *
 *  Program by Steve Moshier
 *  moshier@na-net.ornl.gov
 *
 * Last rev: January, 2002
 */


#define DEBUG 0

#define SQRT2 1.41421356237309504880
#define PI 3.14159265358979323846
#define fabs(x) ( (x) < 0 ? -(x) : (x) )

double MACHEP = 2.2e-16;
double MAXNUM = 1.7e38;
double Kn, Kp, P, kl;
double sqrt(), log(), pow(), sin(), cos(), atan();
double ellpe(), ellpk(), chbevl();
double rsol(), lyle(), dwighta(), dwightb(), butterw();
double lyle2(), spielrein(), nagaoka();


/* Dispatch program for circular coil
 * of arbitrary rectangular winding cross section
 * decides which formula to use.
 */
double rsol( a, b, c, N )
double a, b, c, N;
{
double r, s, t, gam, c2a, bc, b2a, L;

if( c == 0.0 )
	{
	L = nagaoka( a, b, N );
#if DEBUG
	printf( "N" );
#endif
	goto soldone;
	}
bc = b/c;
c2a = 0.5*c/a;
b2a = 0.5*b/a;
r = a + 0.5*c;
s = a - 0.5*c;
gam = s/r;

if( b == 0.0 )
	{
	if( gam < 0.5 )
		{
#if DEBUG
		printf( "SP" );
#endif
		L = spielrein( a, b, c, N );
		goto soldone;
		}
	else
		{
#if DEBUG
		printf( "L2P" );
#endif
		L = lyle2( a, b, c, N );
		goto soldone;
		}
	}

t = c2a*c2a * (1.0 + bc*bc);
/* For flat disk coils with small but nonzero thickness, use Lyle
   instead of Butterworth.
   Thanks to Robert Strand <rstrand@ihug.com.au> for reporting this.  */
if((t <= 1.0) || ((b/r) <= 0.2))
	{
#if DEBUG
	printf( "LP" );
#endif
	L = lyle( a, b, c, N );
	goto soldone;
	}
if( (gam <= 0.4) && (r/b <= 0.4) )
	{
#if DEBUG
	printf( "DKp" );
#endif
	L = dwightb( a, b, c, N );
	goto soldone;
	}
if( (c2a <= 0.3) && (b2a > 0.7) )
	{
#if DEBUG
	printf( "Dk" );
#endif
	L = dwighta( a, b, c, N );
	goto soldone;
	}

#if DEBUG
printf( "BKp" );
#endif
L = butterw( a, b, c, N );
soldone:
return(L);
}

double log(), sqrt(), asin(), atan();
extern double fac[]; /* factorial table */


/* Circular solenoidal current sheet (infinitely thin tape winding)
 *
 * Nagaoka (1909): J. College of Science, Tokyo, 27, Art. 6, p. 18
 *
 * See also E. Jahnke and F. Emde, _Tables of Functions_,
 * 4th edition, Dover, 1945, pp 86-89.
 */
double nagaoka( a, b, N )
double a, b, N;
{
double t, f, k1e, ke, Ke, Ee, tana, Ls;


if( a <= 0.0 )
	{
	goto tiny;
	}
if( b <= 0.0 )
	{
	if( a <= 0.0 )
		{
		Kn = 0.0;
		Kp = 0.0;
		return( 0.0 );
		}
	else
		{
tiny:
		Kn = 1.0;
		Kp = 1.0;
		return( MAXNUM );
		}
	}

tana = 2.0*a/b;        /* form factor */
if( tana < 1.0e-7 )
	{
	Kn = 1.0;
	Kp = 1.0;
	P = 4.0*PI*PI*a*Kp/b;
	t = 1.0e-9 * N * N * a * P;
	return(t);
	}
t = atan( tana );
ke = sin(t);     /* modulus, k, of the elliptic integrals */
k1e = cos(t);        /* subroutine argument = 1 - k^2 */
t = k1e * k1e;
Ke = ellpk(t);      /* See elliptic integrals below */
Ee = ellpe(t);
t = tana * tana;  /* square of tan(a) */
/* Note that the constant of proportionality is exact.
 * The "permeability of the vacuum" is 4 pi 10^-9 henry/cm.
 */
f = 4.0*PI*((Ke + (t - 1.0) * Ee)/ke  -  t)/3.0;
t = 2.0e-9 * N * N * a * f; /* the inductance */
Kn = f * b / (2.0*PI*PI*a); /* Nagaoka's constant */
Kp = Kn;
P = 4.0*PI*PI*a*Kp/b;
return(t);
}



/* Single-layer circular helical solenoid of round wire
 *
 * c is the diameter of the wire.
 * There are N complete turns evenly spaced over length b.
 * b is measured from wire center at beginning of first
 *   turn to wire center at end of last turn.
 * a is the coil radius measured to the center of the wire.
 *
 * Snow, 1952
 */
double helsol( a, b, c, N )
double a, b, c, N;
{
double t, f, Ke, Ee, ke, k1e, tana, Ls, z;

/* first calculate the thin solenoid inductance */
tana = 2.0*a/b;        /* form factor */
t = atan( tana );
ke = sin(t);     /* modulus, k, of the elliptic integrals */
k1e = cos(t);        /* subroutine argument = 1 - k^2 */
t = k1e * k1e;
Ke = ellpk(t);      /* See elliptic integrals below */
Ee = ellpe(t);
t = tana * tana;  /* square of tan(a) */
f = 4.0*PI*((Ke + (t - 1.0) * Ee)/ke  -  t)/3.0;
Ls = 2.0 * N * N * a * f; /* the inductance of the solenoidal sheet */

z = PI*N*c/b;
f = 2.0*N*(0.25-log(z))
  + log(2.0*PI*N*a/b)/3.0
  - (4.0/(PI*PI)) * (Ee/ke - 1.0) * (1.0 + z*z/8.0)
  - (2.0/3.0) * ( (Ke-Ee)/ke - 0.5*ke*Ke )
  - (0.5*k1e/ke) * ( 1.0 - (k1e/ke)*asin(ke) );
f *= 2.0*PI*a;
f +=  b * (log((1.0+k1e)/(1.0-k1e)) + k1e*log(4.0));
f += Ls;
return( 1.0e-9*f );
}


/* Single-layer square solenoid
 * a = length of the side of the square coil form
 * b = length of the winding
 * N = number of turns
 *
 * Grover 1922b
 */
double squaresol( a, b, N )
double a, b, N;
{
double sum, c1, c2, c3, c4, x1, x2;

/* short coil */
if( b <= 0.25*a )
{
x1 = b/a;
c1 = (3.0-2.0*SQRT2)/24.0;
c2 = (6.0*SQRT2-5.0)/480.0;
c3 = (23.0*SQRT2-14.0)/5376.0;
c4 = -(37.0*SQRT2-28.0)/9216.0;
sum = -log(x1) + 0.72599 + x1/3.0;
x2 = x1*x1;
sum += (((c4*x2 + c3)*x2 + c2)*x2 + c1)*x2;
goto done;
}

/* long coil */
if( a < 0.25*b )
{
x1 = a/b;
x2 = x1*x1;
c1 = 0.473201004409338551642; /* (2/pi)*(log(1+sqrt(2)) - (sqrt(2)-1)/3) */
c2 = 1.0/(2.0*PI);
c3 = -1.0/(12.0*PI);
c4 = 1.0/(28.0*PI);
sum = 1.0 - c1*x1 + ((c4*x2 + c3)*x2 + c2)*x2;
sum *= 0.5*PI*x1;
goto done;
}

/* general closed-form solution */
x1 = a/b;
c1 = a*a + b*b;
c2 = sqrt(c1);
c3 = a*a + c1;
c4 = sqrt(c3);
x2 = x1*x1;
sum = log((a+c2)/(a+c4));
sum += x2 * log((a+c4)/((1.0+SQRT2)*a));
sum += 0.5*(1.0-x2) * log(c1);
sum += x2 * log(a);
sum -= log(b);
sum -= PI*x1;
x2 = (2.0*a + c4)/(SQRT2*(a + c4));
sum += 2.0*x1 * asin(x2);
if( x1 == 0.0 )
	sum += PI * x1;
else
	sum += 2.0 * x1 * atan(1/x1);
sum += (c2-c4)*x1/b;
sum += (SQRT2-1.0)*x1*x1/3.0;
sum += (c3*c4 - 2.0*c1*c2)/(3.0*a*b*b);
sum += b/(3.0*a);

done:
return( 8.0e-9*a*N*N*sum );
}


/* Butterworth's expansions for thick circular coils.
   Butterworth 1915.

   "In applying the formulae and tables, their range of application
    should be borne in mind.  They are intended to be used only when
    the inner and outer diameters of the coils differ appreciably
    (b/a < 0.8) and when the coil-lengths are not too small (c/a > 0.2)."
   In our notation,
    "b/a" means  (a - c/2) / (a + c/2)       inner radius / outer radius
    "c/a" means  b/(a + c/2),                length / outer radius

   The program was written from the description in Grover 1922a.  */

double butterw( a, b, c, N )
double a, b, c, N;
{
double r, s, x, y, z, gam, g3, zz, c2a;
double butchi(); /* see below */

r = a + 0.5*c;  /* outer radius */
s = a - 0.5*c;  /* inner radius */
gam = s/r;      /* inner radius / outer radius */
z = b/r;        /* length / outer radius  */
zz = gam * gam;
g3 = gam * zz;
if( (z > 0.0) && (gam < 1.0e-12) )
	x = 0.0;
else
	x =  zz * butchi( z/gam, 1.0 );
x -= 2.0 * butchi( z, gam );
x *= g3;
x += butchi( z, 1.0 );      /* M(c)/2 pi^2 n^2 a^5  */
/* y part */
x -=  (1.0 + zz*g3) * butchi( 0.0, 1.0 );
if( gam <= 0.0 )
	x += g3/3.0;
else
	x += 2.0 * g3 * butchi( 0.0, gam );
c2a = 0.5*c/a;
zz = 1.0 + c2a;
z = 2.0*a/b;
y = (3.*gam - 4.)*g3 + 1.0 + 3.0*z*zz*x;
zz = zz * zz;
Kp = (y*zz*zz)/(24.0*c2a*c2a);
P = 2.0*PI*PI*z*Kp;
/*y = (2.0e-9*PI*PI)*z*N*N*a*Kp;*/
y = 1.0e-9*a*N*N*P;
return(y);
}


/* Auxiliary function of Butterworth formula
   Compute chi = N / (2 pi^2 gam^3)
   where N is the mutual induction between two solid coils.

   gam = inner radius / outer radius
   z = length / outer radius

   Program checked against Butterworth's paper as well as Grover.  */

double butchi(z, gam)
double z, gam;
{
double zz, u, v, sum, c1, c2, c3, c4, alpha, beta, m;
double zet, zet2, zetp1, chi;
int n;


if( z > 4.0 )
{
zz = gam*gam;
c1 = ((((14./33.)*zz + (28./9.))*zz + (40./7.))*zz + (28./9.))*zz + (14./33.);
c2 = (((5./27.)*zz + (6./7.))*zz + (6./7.))*zz + (5./27.);
c3 = ((2./21.)*zz + (6./25.))*zz + (2./21.);
c4 = (1./15.)*zz + (1./15.);
zz = 0.25/(z*z);
chi = (((c1*zz - c2)*zz + c3)*zz - c4)*zz + (1./9.);
chi /= 2.0 * z;
/* Butterworth's formula (A)
   Said accurate to 1 in 100,000 if z > 3. */
return(chi);
}

if( z > 0.0 )
{
zet2 = 1.0 + z * z;
zet = sqrt( zet2 );
zetp1 = 1.0 + zet;
zz = 1.0/zet2;
c1 = ((( -35.*zz + 35.)*zz - 3.)*zz - 1.)*zz + 1.0/(zetp1*zetp1);
c1 *= 3./(82944.*zet);               /* D */
c2 = ( -3.*zz + 1.)*zz + (1./zetp1);
c2 *=  -1.0/(1344.*zet);             /* C */
}

if( z > gam )
{

zz = log(zetp1/z);
c3 = (zz  - 1.0/zet)/40.0;            /* B */
c4 = (0.5*z*z*zz + 0.5*zet - z)/3.0;  /* A */

if( gam <= 0.0 )
	{
	return( c4 );
	}

/*   inf      n
 *    -   (-1)  (2n-3)!            2n
 *    >   ---------------- (gam/2z)
 *    -   (2n+3) n! (n+1)!
 *   n=2
 */
zz = 0.5*gam/z;
zz = zz * zz;
v = zz * zz;
sum = 0.0;
n = 2;
do
	{
	u = v*fac[2*n-3]/((2.*n+3.)*fac[n]*fac[n+1]);
	sum += u;
	v *= -zz;
	n += 1;
	}
while( fabs(u/sum) > 1.0e-10 );

alpha = (sum*z*z)/(gam*gam);
zz = gam*gam;
chi = (( -c1*zz + c2)*zz - (c3-alpha))*zz + c4;
/* Butterworth's formula (B)
   Use in the range 4 > z > gam.  */
return(chi);
}

if( z > 0.0 )
{
/*    8         n
 *    -   4 (-1)  (2n-3)!                       2n
 *    >   ------------------------   (z / 2 gam)
 *    -   (n-1)!(n+1)!(2n+1)(2n+3)
 *   n=2
 */
zz = 0.5*z/gam;
zz = zz * zz;
v = zz*zz;
sum = 0.0;
for( n=2; n<=8; n++ )
	{
	sum += v*fac[2*n-3]/(fac[n-1]*fac[n+1]*(2.*n+1.)*(2.*n+3.));
	v *= -zz;
	}
zz = z/gam;
sum = 4.0 * zz * zz * sum;
beta = ((( -(log(2.0/zz) + (77./60.))/30.)*zz + (1./6.))*zz - (1./3.))*zz*zz;
beta = 0.5*zz*( (1./3.) + beta - sum );

zz = log( 2.0*zetp1 );
c3 = (zz - (1.0/zet) + (29./20.))/40.;            /* B' */
c4 = ( z*z*(0.5*zz - (1./3.)) - z + 0.5*zet)/3.;  /* A' */
zz = gam*gam;
chi = (( -c1*zz + c2)*zz - (c3-beta))*zz + c4;
chi += (z*z/6. - zz/40.) * log(1./gam);
/* Butterworth's formula (C)
   Use for gam > z > 0.  */
return(chi);
}

/* else z == 0 */

if( gam == 1.0 )
	return( 0.12206342136 );
if( gam <= 0.0 )
	return( 1.0/6.0 );

/*   inf                  2            2n
 *    -   (1*3*5...(2n+3))          gam
 *    >   (--------------)  ----------------------
 *    -   (2*4*6...(2n+4))  (2n+7)(2n+3)(n+3)(n+1)
 *   n=0
 */
c1 = 0.5;
v = 1.0;
zz = gam*gam;
m = 0.0;
sum = 0.0;
do
	{
	c1 *= (2.*m+3.)/(2.*m+4.);
	u = (c1*c1*v)/((2.*m+7.)*(2.*m+3.)*(m+3.)*(m+1.));
	sum += u;
	v *= zz;
	m += 1.0;
	}
while( fabs(u/sum) > 1.0e-8 );
chi = (1./6.) - zz*(log(4./gam) + (9./20.))/40. + 0.5*zz*zz*sum;
/* Butterworth's formula (D) */
return(chi);
}


/* Lyle's formumla for narrow disk coils, b=0
   Lyle 1914.  */

double lyle2( a, b, c, N )
double a, b, c, N;
{
double c1, c2, c3, c4, c2a, zz, x;

c2a = 0.5*c/a;
x = log( 4.0/c2a );
zz = c2a * c2a;

c1 = (103./(105.*1024.)) * (x + 1.1394);
c2 = (11./2880.)*(x + (96./55.));
c3 = (x + (43./12.))/24.0;
c4 = x - 0.5;

P = (((c1*zz + c2)*zz + c3)*zz + c4) * (4.0*PI);
Kp = 0.0;
x = 1.0e-9*N*N*a*P;
return(x);
}


/* Spielrein's formula for wide disk coils, b=0
   Spielrein (1915): Archiv fur Elektrotechnik 3, p. 182  */

double spielrein( a, b, c, N )
double a, b, c, N;
{
double gam, zz, S, c2a;

gam = (a - 0.5*c)/(a + 0.5*c);
S = 6.96957;
if( gam > 1.0e-12 )
	{
	zz = gam*gam;
	S += ((((0.0337*zz
	 + 0.06038)*zz
	 + 0.12494)*zz
	 + 0.33045)*zz
	 + 1.48044)*zz*zz*gam;

	S +=  -zz*gam*( 13.1595*log(1./gam) + 9.08008 );
	}
c2a = 0.5*c/a;
zz = 1.0 + c2a;
P = (zz*zz*zz*S)/(4.0*c2a*c2a);
Kp = 0.0;
zz = 1.0e-9*N*N*a*P;
return(zz);
}

/* Dwight's formula for long, thin coils
 */
double dwighta( a, b, c, N )
double a, b, c, N;
{
double ab, p, pp, c1, c2, c3, c4, c5;

ab = 2.0*a/b;
p = 1.0/sqrt( 1.0 + 4.0/(ab*ab) );
pp = p * p;
c1 = ((( (1183./96.)*pp
 - (1117./672.))*pp
 + (15./112.))*pp
 - (1./120.))*pp*p;

c2 = ((((( -(3913./128.)*pp
 + (38857./4608.))*pp
 - (1265/576.))*pp
 + (53./96.))*pp
 - (17./180.))*pp
 + (1./36.))*p;

c3 = ((((((( -(13.*68915./32768.)*pp
 + (11.*1961./2048.))*pp
 - (2135./512.))*pp
 + (217./128.))*pp
 - (95./128.))*pp
 + (1./3.))*pp
 - (5./24.))*pp
 + (1./6.))*p;

p = 0.5*c/a;
pp = p * p;

c4 = (( (4547./(350.*3584.))*pp
 + (1./400.))*pp
 - (23./12.))*pp;

c5 = (( -(23./4480.)*pp
 - (1./20.))*pp
 + 1.0)*pp;

kl = ((1./(3.*PI))*(c5*log(4./p) + c4)
 + ((c1*pp + c2)*pp + c3)*pp)*ab
 + (1./3.)*pp
 - (2./3.)*p;
kl = -kl;
p = nagaoka( a, b, N );
Kp = Kn - kl;
P = 4.0*PI*PI*Kp*a/b;
p = 1.0e-9*N*N*a*P;
return(p);
}


/* Dwight's formula for long, thick coils
   Dwight (1918): Electrical World 71, p. 300  */

double dwightb( a, b, c, N )
double a, b, c, N;
{
double r, s, gam, u, v, sum, zz;
double g3, g4, g5, g7, g9, Q;
double c1, c2, c3, c4, c5, rb;
int n;

r = a + 0.5*c;
s = a - 0.5*c;

/* Check for solid coil, inner radius zero */
if( (s < 1.0e-12) && (b > 0.0) )
{
u = 2.0*a/b;
if( u > 0.3 )
	return( butterw( a, b, c, N ) );

c1 = 197./2016.;
c2 = 113./1400.;
c3 = 1./10.;
c4 = 1./3.;
c5 = 0.7323816;
v = u * u;
sum = ((( -c1*v + c2)*v - c3)*v + c4)*v - c5 * u + 1.0;
P = 8.0*PI*PI*a*sum/(3.0*b);
Kp = P*b/(4.0*PI*PI*a);
u = 1.0e-9*a*N*N*P;
return(u);
}

gam = s/r;
/*   inf
 *    -   (2n-1)! (2n+1)!                  2n
 *    >   ---------------------- (gam / 4 )
 *    -   n!n!(n+1)!(n+2)!(2n+5)
 *   n=2
 */
sum = 0.0;
zz = 0.25*gam;
zz = zz*zz;
v = 1.0;
n = 1;
do
	{
	v *= zz;
u = (v*fac[2*n-1]*fac[2*n+1])/(fac[n]*fac[n]*fac[n+1]*fac[n+2]*(2.*n+5.));
	sum += u;
	n += 1;
	}
while( fabs(u/sum) > 1.0e-7 );

zz = gam*gam;
g3 = gam*zz;
g4 = zz*zz;
g5 = g4*gam;
g7 = g4*g3;
g9 = g4*g4*gam;
c1 = (9./112.)*(1.0-g5)*(1.0-g7) + (5./288.)*(1.0-g3)*(1.0-g9);
c2 = (9./200.)*(1.0-g5)*(1.0-g5) + (1./28.)*(1.0-g3)*(1.0-g7);
c3 = (1./10.)*(1.0-g3)*(1.0-g5);
c4 = (1./3.)*(1.0-g3)*(1.0-g3);
rb = r/b;
zz = rb*rb;
Q = ((( -c1*zz + c2)*zz - c3)*zz + c4)*zz + 3.0*g5*sum*rb;
Q = Q - 3.*rb*( 0.2441272 - (2./3.)*g3 + (0.4277559 - 0.1*log(gam))*g5);
Q = Q + 1.0 - 4.0*g3 + 3.0*g3*gam;
u = 0.5*c/a;
v = 1.0 + u;
v = v * v;
Kp = (v*v*Q)/(24.0*u*u);
P = 4.0*PI*PI*Kp*a/b;
u = 1.0e-9*N*N*a*P;
return(u);
}



/* Lyle's formula for short, thin coils
   Lyle (1914), Phil. Trans. 213A, pp. 421-435  */
#include "lyle.h"

double lyle( a, b, c, N )
double a, b, c, N;
{
double x, d, p, q, m1, m2, m3, l0, l1, l2, l3;
double b2, c2;

if( c == 0.0 )
	goto smc;
x = b/c;
if( x <= 1.0 )
	{
	x = 2.0*x - 1.0;
/*
 *	m1 = 1.0e-2*chbevl( x, lm1, 10 );
 *	m2 = 1.0e-4*chbevl( x, lm2, 10 );
 */
	m3 = 1.0e-6*chbevl( x, lm3, 10 );
	l0 = chbevl( x, ll0, 14 );
	l1 = 1.0e-2*chbevl( x, ll1, 12 );
	l2 = 1.0e-4*chbevl( x, ll2, 10 );
	l3 = 1.0e-6*chbevl( x, ll3, 10 );
	}
else
	{
smc:
	x = c/b;
	x = 2.0*x - 1.0;
/*
 *	m1 = 1.0e-2*chbevl( x, lm1a, 10 );
 *	m2 = 1.0e-4*chbevl( x, lm2a, 10 );
 */
	m3 = 1.0e-6*chbevl( x, lm3a, 8 );
	l0 = chbevl( x, ll0a, 14 );
	l1 = 1.0e-2*chbevl( x, ll1a, 10 );
	l2 = 1.0e-4*chbevl( x, ll2a, 10 );
	l3 = 1.0e-6*chbevl( x, ll3a, 10 );
	}
b2 = b * b;
c2 = c * c;
m1 = (1./96.)*(3.0*b2 + c2)/(b2 + c2);
d = b2 + c2;
m2 = (1./92160.)*(-90.*b2*b2 + 105.*b2*c2 + 22.*c2*c2)/(d*d);
d = sqrt(d);
x = d/a;
x = x * x;
p = ((m3*x + m2)*x + m1)*x + 1.0;
q = ((l3*x + l2)*x + l1)*x - l0;
P = 4.0*PI*(p * log( 8.0*a/d )  +  q);
Kp = P*b/(4.0*PI*PI*a);
x = 1.0e-9*a*N*N*P;
return(x);
}


/* compute Chebyshev polynomials up to degree n-1
 * at arg x
 * and evaluate Chebyshev expansion.
 * Note, the zero degree term is not multiplied by 1/2.
 */
double chbevl( x, ch, n )
double x;
double ch[];
int n;
{
double t[15];
double s;

int i;

t[0] = 1.0;
t[1] = x;
s = x + x;
t[2] = s*x - 1.0;
for( i=3; i<n; i++ )
	t[i] = s * t[i-1] - t[i-2];

s = 0.0;
for( i=0; i<n; i++ )
	s += t[i] * ch[n-i-1];
return(s);
}


/* Niwa's formula for a single-layer rectangular winding.
   Each turn of the coil is a rectangle of sides a, a1.
   The longer side is a.
   The length of the winding is b.
   N is the number of turns.

   Grover 1946, Chapter 10.
  */

#ifndef PI
#define PI 3.14159265358979323846
#endif
extern double asinh (double); /* Inverse hyperbolic sine.  */
extern double atan (double);  /* Inverse circular tangent. */
extern double sqrt (double);

double
rectangle (a, a1, b, N)
double a, a1, b, N;
{
  double p, q, r, s, t, L;
  double aob, a1ob, g2, b2, a2ob2, aa1, b2oaa1;

  a1ob = a1 / b;
  aob = a / b;

  t = (0.5 / a1ob) * asinh (aob);
  /* printf ("%8.5f\n", t); */
  L = t;

  t = (0.5 / aob) * asinh (a1ob);
  /* printf ("%8.5f\n", t); */
  L = L + t;

  q = a1ob * a1ob;
  r = 0.5 * (1.0 - q) / a1ob;
  s = a / (b * sqrt (1.0 + q));
  t = -r * asinh (s);
  /* printf ("%8.5f\n", t); */
  L = L + t;

  a2ob2 = aob * aob;
  r = 0.5 * (1.0 - a2ob2) / aob;
  s = a1 / (b * sqrt (1.0 + a2ob2));
  t = -r * asinh (s);
  /* printf ("%8.5f\n", t); */
  L = L + t; 

  t = -0.5 * a1ob * asinh (a / a1);
  /* printf ("%8.5f\n", t); */
  L = L + t; 

  t = -0.5 * aob * asinh (a1 / a);
  /* printf ("%8.5f\n", t); */
  L = L + t;

  b2 = b * b;
  g2 = a * a + a1 * a1;
  q = sqrt (1.0 + g2 / b2);

  aa1 = a * a1;
  b2oaa1 = b2 / aa1;
  s = 1.0 / (b2oaa1 * q);
  t = (0.5 * PI - atan (s));
  /* printf ("%8.5f\n", t); */
  L = L + t;

  r = b2oaa1 / 3.0;
  t = r * q * (1.0 - (0.5 * g2) / b2);
  /* printf ("%8.5f\n", t); */
  L = L + t;

  t = r;
  /* printf ("%8.5f\n", t); */
  L = L + t;

  t = -r * sqrt (1.0 + a2ob2) * (1.0 - 0.5 * a2ob2);
  /* printf ("%8.5f\n", t); */
  L = L + t;

  q = (a1 * a1) / b2;
  p = sqrt (1.0 + q);
  t = -r * p * (1.0 - 0.5 * q);
  /* printf ("%8.5f\n", t); */
  L = L + t;

  p = b / (6.0 * aa1);
  q = g2 * sqrt(g2);  /* g^2 = a^2 + a1^2 */
  q = (q - (a * a * a) - (a1 * a1 * a1)) / b2;
  t = p * q;
  /* printf ("%8.5f\n", t); */
  L = L + t;
  L = 8.0e-9 * N * N * L * aa1 / b;
  return L;
}

