/*   PP - KWP1281
     Palubni pocitac s komunikaci KWP1281 (pro BMM)
     autor: Z. Svoboda
     
     predloha pro KWP1281:
     Florian Schffer, http://www.blafusel.de (V 1.2)
     
Verze 2.03
17.08.09 - prvni uvolnena
18.09.09 - doplneny dalsi komentare

Fuse bits: BODLEVEL...: 00111101
           RSTDISBL...: 11001001

*/

#define PRG_NAME "  PP-KWP1281 2.03"
#define MIN_P_ODO 32         // min. meritelna perioda impulzu tachometru v us
#define MIN_P_INJ 8          // min. meritelna perioda impulzu vstriku v us
#define MIN_STISK 2          // min. pocet taktu citace 2 pro zaregistrovani stisku tlacitka
#define DL_STISK  250        // min. pocet taktu citace 2 pro dlouhy stisk tlacitka
#define NOKW_MOD_ZOBR  2     // pocet modu zobrazeni je-li komunikace KW1281 neaktivni
#define NOERR_MOD_ZOBR 5     // pocet "nonerrorovych" modu zobrazeni
#define POC_UDAJU      4     // max. pocet udaju na displeji
#define MEZ_ODO        6     // pocet pulzu tacha za 1 s, od kterych prejde displej
                             //   z udaje l/h na l/100km
#define KR_PIP         5     // doba kratkeho pipnuti a 8 ms
#define DL_PIP         20    // doba dlouheho pipnuti a 8 ms
#define ERR_PIP        40    // doba pipnuti pri chybe a 8 ms

// konstanta pro vypocet okamzite rychlosti v km/h
#define K_V    3.6e9 / MIN_P_ODO
// konstanta pro vypocet okamzite spotreby v l/h
#define K_S    (3.6 * MIN_P_INJ) / 1e9 
// konstanta pro vypocet okamzite spotreby v l/100km
#define K_S100 (MIN_P_ODO * MIN_P_INJ) / 1e16 
// konstanta pro vypocet prumerne a celkove spotreby v l
#define K_SL   8 * MIN_P_INJ / 1e12  // 8 kompenzuje shift ds_inj pri sumaci
// konstanty pro vypocet teploty
#define K_UT   4000                         // referencni napeti v mV
#define K_QT   (K_UT * 0.001) * 10/(8192 * 0.01) // smernice
#define K_KTI  2732                         // posun - napeti na cidle vnitrni teploty v mV pri 0C
#define K_KTO  2732                         // posun - napeti na cidle venkovni teploty v mV pri 0C
#define MEZ_T_OUT 4                         // mez ve C pro varovani pred namrazou 

#include <avr/io.h>
#include <stdlib.h>
#include <util/delay.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>
#include "lcd.h"
#include "uart.h"
#include "kw1281.h"
#include "xitoa.h"

/*********************************************************/
/*                   E X T E R N                         */
/*********************************************************/
extern uint8_t block_cnt, block_length, cit_rx_uart;

/*--------------------------------------------------------------------------------------------------
                                Private function prototypes
--------------------------------------------------------------------------------------------------*/
//  Function prototypes are mandatory otherwise the compiler generates unreliable code.

int main (void);
void zprac_err_uart (void);
void clr_prub(void);

typedef union
{
	uint32_t dword;
  uint16_t word[2];
	uint8_t  byte[4];
} uint32_16_8_t;

typedef struct {
  uint8_t prub_s, prub_m, prub_h;
  uint32_16_8_t prub_draha, prub_spotr;
} DATA;

typedef struct {
  uint16_t k_inj;
  DATA data;
} DATA_K;

extern volatile uint8_t cit_getc, uart_err;

DATA_K data_b, data_l, *p_data = &data_l;

DATA_K EEMEM ee_data_b = {6400, {0, 0, 0, {0}, {0}}}, // 6400 ul paliva na 1 s vstrik. pulzu
             ee_data_l = {8040, {0, 0, 0, {0}, {0}}}; // 8040 ul paliva na 1 s vstrik. pulzu

uint16_t EEMEM ee_k_odo  = 1991;  // impulzu tachometru na 1 km

uint8_t           doba_stisku = 0, mod_zobr = 0, mod_kw1281 = 0, init_kw1281 = 0,
                  tcnt2_corr = 0, cit_t = 0;          //pocet mereni teploty
volatile uint8_t  cit_pip = KR_PIP, chyby, blink = 0,
                  dl_stisk, kr_stisk = 0, mod_nast = 0,
								  zobraz = 1, nemen = 1,
                  cit_tim2 = 0, cit_odo = 0, cit_tim = 0,
                  cit1s_odo = 0,            // pocet pulzu tacha (preruseni INT1) za 1 s
									cits_odo = 0, v_ok = 0;       
uint16_t d_inj,   d_inj_pred = 0, ti_int, to_int;
volatile uint16_t T_odo, Ts_odo, k_odo,
                  ds_inj = 0, d1s_inj,
									tis_int = 0,
							    tos_int = 0;
double   t_out; 
char     texte1[15];

// Preruseni od INT0 - odpojeni napajeni
ISR (INT0_vect)
{
  eeprom_write_block((void*)&data_b.data, (void*)&ee_data_b.data, sizeof(DATA));
  eeprom_write_block((void*)&data_l.data, (void*)&ee_data_l.data, sizeof(DATA));
  delay_ms(25);
  PORTD &= ~(1 << PD6); // odpojeni napajeni z baterie

}

// Preruseni od INT1 - pulzy z tachometru
ISR (INT1_vect)
{
  T_odo = TCNT2;
  cits_odo += cit_odo;
  cit_odo = 0;
  cit1s_odo++;
  p_data->data.prub_draha.dword++;
}

// Preruseni od input capture citace 1
ISR (TIMER1_CAPT_vect)
{
  d_inj = ICR1;
  TCCR1B ^= (1 << ICES1);          //negace hrany input capture
  TIFR &= ~(1 << ICF1);
  if (!(TCCR1B & (1 << ICES1)))    //pricteni jen zaporneho pulzu
	  ds_inj += d_inj - d_inj_pred;  // pro d_inj_pred > d_inj dojde k automatickemu oriznuti vysledku
	else d_inj_pred = d_inj;         //poznamenani zacatku zaporneho pulzu
}

// Preruseni od capture citace 2 - perioda 8ms
ISR (TIMER2_COMP_vect)
{

  p_data = (PIND & (1 << PD7)) ? &data_l : &data_b; //vyber sady promennych dle stavu B/LPG

	if(cit_odo < 255) cit_odo++;
  
  cit_tim2++;
  if (cit_tim2 == 125)                             // odmereni 1s
  {
    d1s_inj = ds_inj;
		if(ds_inj)
    {
      p_data->data.prub_spotr.dword += ds_inj >> 3;
	    if (p_data->data.prub_spotr.byte[3] == 0xFF) clr_prub();
      ds_inj = 0;
	  }
	  
    if (cit1s_odo > 1) Ts_odo = ((uint16_t)(cits_odo * 250) + T_odo - tcnt2_corr)/cit1s_odo;
    else Ts_odo = 0;                      // Ts_odo = prum. perioda signalu tacha za dobu cca 1s
		tcnt2_corr = T_odo;
    v_ok = (cit1s_odo > MEZ_ODO) ? 1 : 0;
		cits_odo = 0;
    cit1s_odo = 0;
    zobraz = 1;
	  cit_tim2 = 0;
    p_data->data.prub_s++;
	  if (p_data->data.prub_s > 59)
	  { 
	    p_data->data.prub_s = 0; 
	    p_data->data.prub_m++;
      if (p_data->data.prub_m > 59)
	    { 
	      p_data->data.prub_m = 0; 
	      p_data->data.prub_h++;
        if (p_data->data.prub_h == 255) clr_prub();
      }
    }
	  if (!chyby) blink ^= 1;
	}

  if (cit_getc) cit_getc--;
  if (cit_tim) cit_tim--;
	
		
    ADMUX ^= (1 << MUX0);               // mereni teploty
    if(cit_t & 1) to_int += ADCW;       // cit_t lichy => venkovni teplota
		else ti_int += ADCW;                // cit_t sudy => vnitrni teplota
    ADCSRA |= (1 << ADSC);      
	  cit_t++;
		if(cit_t == 128) 
		{
		  cit_t = 0;
			tis_int = ti_int;
      tos_int = to_int;
			ti_int = 0;
			to_int = 0;
		}    

  if (cit_pip)                   // test povoleni tonu
  {
    cit_pip--;
	  PORTB |= (1 << PB1);
  }
  else  PORTB &= ~(1 << PB1);

  if (PINC & (1 << PC3))        // tlac. nestisknuto?
  {
    if ((doba_stisku >= MIN_STISK) && (doba_stisku < DL_STISK))
    {
	  kr_stisk = 1;
	  }
    doba_stisku = 0;
	  if (!cit_tim && !nemen)
    {
	    cit_tim = 125;
	    switch (mod_nast)
	    {
	      case 4: k_odo +=1; break;
	      case 5: k_odo -=1; break;
  	    case 6: k_odo +=100; break;
  	    case 7: k_odo -=100; break;
  	    case 8: p_data->k_inj +=1; break;
 	      case 9: p_data->k_inj -=1; break;
	      case 10: p_data->k_inj +=100; break;
	      case 11: p_data->k_inj -=100; break;
			  default: break;
      }
	  }    
  }
  else                          // tlac. stisknuto
  { 
    if (!doba_stisku) 
	  {
	    cit_pip = KR_PIP;
    }
	  if (doba_stisku < 255) doba_stisku++;
    if (doba_stisku == DL_STISK)
    { 
      dl_stisk = 1;
	    cit_pip = DL_PIP;
    }
  }

}

void zprac_err_uart (void) // zpracovani preruseni seriov. komunikace
{
  uart_disable ();
  cit_pip = ERR_PIP;
  init_kw1281 = 0;
  mod_kw1281 = 0;
  if(mod_zobr >= NOKW_MOD_ZOBR) mod_zobr = 0;
  //mod_zobr = 1;
	LcdClear();
  LcdStr("KW ERROR!");
  delay_ms(1500);
  LcdClear();
}

void clr_prub(void)
{
  p_data->data.prub_s = 0;
  p_data->data.prub_m = 0;
  p_data->data.prub_h = 0;
  p_data->data.prub_draha.dword = 0;
  p_data->data.prub_spotr.dword = 0;
}


int main (void)
{
  uint8_t   j=0;
  uint8_t   k=0;
  int8_t    i;
  uint8_t   values[12];
	uint16_t	baud_rates[] = {9600, 4800, 10400};

uint8_t	 prub_h_c, prub_m_c, prub_s_c;

uint16_t d1s_inj_c, Ts_odo_c, tis_int_c, tos_int_c;

uint32_16_8_t prub_draha_c, prub_spotr_c;

  LcdInit();
  LcdStr(PRG_NAME);

  DDRD |= (1 << DDD1)|(1 << DDD6); // PD1 vystup (TxD -> RS232), PD6 vystup
  DDRB |= (1 << DDB1); // PB1 vystup
	PORTC |= (1 << PC3); // pull-up rezistor na PC3

//nastaveni citace 2 -  CTC mod, preddelicka 256 (1 pulz = 32 us) - draha, mereni casu (preruseni a 8 ms)
  TCCR2 |= (1 << WGM21)|(1 << CS22)|(1 << CS21); 
  OCR2 = 249;

//nastaveni citace 1 - noise canceller, preddelicka 64 (1 pulz = 8 us) - palivo
  TCCR1B |= (1 << ICNC1)|(1 << CS11)|(1 << CS10); 

//nastaveni INT1 - preruseni na nabeznou hranu
//          INT0 - preruseni na sestupnou hranu
  MCUCR |= (1 << ISC11)|(1 << ISC10)|(1 << ISC01);
  GICR |= (1 << INT1)|(1 << INT0);

//nastaveni A/D prevodniku - reference: pin AREF
//                         - povoleni a spusteni A/D prevodniku, preddelicka 128
  ADCSRA |= (1 << ADEN)|(1 << ADSC)|(1 << ADPS2)|(1 << ADPS1)|(1 << ADPS0);

//prerus od inp. capt. citace 1 a compare citace 2 
  TIMSK |= (1 << OCIE2)|(1 << TICIE1);
 
	eeprom_read_block((void*)&data_b, (void*)&ee_data_b, sizeof(DATA_K));
	eeprom_read_block((void*)&data_l, (void*)&ee_data_l, sizeof(DATA_K));
  k_odo = eeprom_read_word(&ee_k_odo);

  cit_tim = 187;                 // pauza umozni pripadnou aktivaci modu nastaveni
  sei();
  while (cit_tim && !kr_stisk);
  if (kr_stisk)
  {
    kr_stisk = 0;
	mod_nast = 1;
  }

  LcdClear();
  dl_stisk = 0;

  while (1)
  { 

    if(PIND & (1 << PD2)) PORTD |= (1 << PD6); // je-li napeti od zapalovani, pripoj napeti baterie

    if (kr_stisk)    // kratky stisk tlacitka?
	  {
      kr_stisk = 0;
	    if (!mod_nast) 
	    {
	      LcdClear();
        mod_zobr++;
	      if (mod_kw1281)
        {
          if (((chyby == 0) && (mod_zobr >= NOERR_MOD_ZOBR)) || (mod_zobr > ((chyby - 1)/POC_UDAJU + NOERR_MOD_ZOBR))) mod_zobr = 0;
        }
	      else if (mod_zobr >= NOKW_MOD_ZOBR) mod_zobr = 0;
      }
	    else
	    {
	  	  if (mod_nast == 1)
	  	  {
		      mod_nast = 0;
		      LcdClear();
	      }
	      else
		    {
          if(nemen) nemen = 0;
		      else
		      {
		        if ((mod_nast & 3) == 3) mod_nast &= 0xFC;
		        else mod_nast++;
          }
		    }
	    }
    }

    if (dl_stisk)       // dlouhy stisk tlacitka?
    {
	    dl_stisk = 0;   
      if (!mod_nast)
	    {
	      if (mod_zobr == 1)
	      {
		      LcdClear();
		      mod_kw1281 ^= 1;
          if (mod_kw1281)
		      { 
		        mod_zobr = NOKW_MOD_ZOBR;
		        init_kw1281 = 1;
          }
        }
	      if (mod_zobr == 0) clr_prub();
	      if (mod_zobr >= NOERR_MOD_ZOBR) 
	      {
		      kw1281_snd_cmd (CLR_ERR);
          while (ANS_ERR == kw1281_get_data (&block_length, values)) 
		      kw1281_snd_cmd (ACK);
		      LcdClear();
        }
      }
	    else
	    {
	  	  nemen = 1;
		    if (mod_nast == 1) mod_nast = 4;
		    else
		    {
		      if (mod_nast < 8) mod_nast = 8;
		      else 
		      {
		        mod_nast = 1;
					    eeprom_write_word(&ee_data_l.k_inj, data_l.k_inj);
              eeprom_write_word(&ee_data_b.k_inj, data_b.k_inj);
            eeprom_write_word(&ee_k_odo, k_odo);
          }
		    }
      }
    }

	  LcdGotoXY (1,19);                         // zobraz pouzite palivo
    LcdChr((PIND & (1 << PD7)) ? 'L' : ' ');

	  if (mod_nast)          //zobrazeni udaju v modu nastaveni
	  {
      LcdGotoXY (1,1);
      LcdChr ((cit_tim2 < 64 && mod_nast > 3 && mod_nast < 8) ? ' ' : 'O');
      xitoa(k_odo, 10, 6, texte1);
      LcdStr (texte1);

      xitoa(p_data->data.prub_draha.dword, 10, 10, texte1);
      LcdStr (texte1);

	    LcdGotoXY (2,1);
      LcdChr ((cit_tim2 < 64 && mod_nast > 7 && mod_nast < 12) ? ' ' : 'I');
      xitoa(p_data->k_inj, 10, 6, texte1);
      LcdStr (texte1);
 
      xitoa(p_data->data.prub_spotr.dword, 10, 10, texte1);
      LcdStr (texte1);

	  }
    else
	  {

      if (uart_err) zprac_err_uart();

      if (init_kw1281)            //inicializace KW1281
	    {
	      init_kw1281 = 0;
        i = 6;
        do
        {
          LcdClear();
          xitoa (baud_rates[i%3], 10, 5, texte1); 
          LcdStr (texte1);
          LcdStr("b/s KW init");
          delay_ms (500);           
          uart_disable();         
          kw1281_init();          // 5 Baud inicializace
          uart_ini(baud_rates[i%3]);   
          i--;
        } while (!kw1281_ecu_sync() && i); // opakuj max i-krat
        if(i != 0)
	      {
	        LcdStr(" OK!");

          LcdGotoXY(2,2);
          LcdStr("get data");
          do                      // cteni identifikacnich dat
          {
            LcdChr('.');
            j = kw1281_get_data (&block_length, values);
            if (j == ASCII)  kw1281_snd_cmd (ACK);      
          } while (j == ASCII);                         // dokud je prijman ASCII Blok
          LcdClear();
        }      
	    else zprac_err_uart();         // osetreni chybne inicializace
      dl_stisk = 0;
			}

      chyby = 0;
      if (mod_kw1281)
      {
        kw1281_snd_cmd (GET_ERR);       //dotaz na chyby
        while (ANS_ERR == kw1281_get_data (&block_length, values)) //jejich vycteni
	      {
          if (!((values[0] == 0xFF) && (values[1] == 0xFF) && (values[2] == 0x88))) 
	        { 
            for (i = 0; i < (block_length - 3); i += 3)
 	          { 
		          chyby++;		    
              if (mod_zobr == ((chyby - 1)/POC_UDAJU + NOERR_MOD_ZOBR))
	            {                                  // pripadne a zobrazeni
     	          switch (chyby % POC_UDAJU) 
		            {
				          case 0: j = 2; k = 10; break; 
				          case 1: j = 1; k = 1; break; 
				          case 2: j = 1; k = 10; break; 
				          case 3: j = 2; k = 1; break;
				          default: break;
                }
	              LcdGotoXY (j,k);                           
                xitoa (values[i] * 256 + values[i + 1], 16, 5, texte1);
                LcdStr (texte1);
                xitoa (values[i + 2], 16, 3, texte1);
                LcdStr (texte1);
              } 
            }
          } 
	        kw1281_snd_cmd (ACK);
        }
      }  

      LcdGotoXY (1,20); 
      if (chyby)                            // zobrazeni poctu chyb
      {                   
        LcdChr (blink ? chyby + 48 : '!');  
        blink ^= 1;  
      }
      else LcdChr (mod_kw1281 ? '0' : ' '); 
 
			cli();                 // kopie promennych pod zakazanym prerusenim
			d1s_inj_c = d1s_inj;  
			Ts_odo_c = Ts_odo;
			prub_draha_c.dword = p_data->data.prub_draha.dword;
			prub_spotr_c.dword = p_data->data.prub_spotr.dword;
			prub_h_c = p_data->data.prub_h;
			prub_m_c = p_data->data.prub_m;
			prub_s_c = p_data->data.prub_s;
			tis_int_c = tis_int;
			tos_int_c = tos_int;
			sei();


      t_out = (K_QT * (tos_int_c>>3) - K_KTO)/10;
	  	LcdGotoXY (2,20);                    // zobrazeni varovani pred namrazou
	  	if((int8_t)(t_out) < MEZ_T_OUT) LcdChr(blink ? '!' : '*');
		  else LcdChr(' ');

	    switch (mod_zobr)              // vyber dat, jejich vypocet a zobrazeni
	    {
	      case 0:
          if(zobraz || mod_kw1281)
          {
	          zobraz = 0;
            LcdGotoXY (1,1);
 			      // prumerna spotreba
	          dtostrf ((double) (prub_draha_c.dword > k_odo>>2 ? K_SL * k_odo * p_data->k_inj * prub_spotr_c.dword * 100 / prub_draha_c.dword : 0), 5, 2, texte1);
            LcdStr (texte1);
            LcdStr ("l/100");           
		        // celkova spotreba
	          dtostrf ((double)(K_SL * 65536 * p_data->k_inj * prub_spotr_c.word[1]), 7, 1, texte1);
            LcdStr (texte1);
            LcdStr ("l"); 
						 
            LcdGotoXY (2,1);
 		        // okamzita spotreba
			      dtostrf ((double) (v_ok ? K_S100 * k_odo * p_data->k_inj * d1s_inj_c * Ts_odo_c : K_S * p_data->k_inj * d1s_inj_c), 4, 1, texte1);
			      LcdStr (texte1);
            LcdStr (v_ok ? "l/100" : "l/h  "); 
			      // celkovy cas
						xitoa(p_data->data.prub_h, 10, 4, texte1);
            LcdStr (texte1);
            LcdChr (':');
		        xitoa(p_data->data.prub_m, 10, -2, texte1);
            LcdStr (texte1);
		        LcdChr (':');
		        xitoa(p_data->data.prub_s, 10, -2, texte1);
            LcdStr (texte1);
					}
		      break;
	      case 1:
		      if(zobraz || mod_kw1281)
          {
	          zobraz = 0;
		        LcdGotoXY (1,1);
			      // prumerna rychlost
            dtostrf (prub_draha_c.dword / (k_odo * (prub_s_c/3600.0 + prub_m_c/60.0 + prub_h_c)), 5, 1, texte1);
	          LcdStr (texte1);
            LcdStr ("km/h");
						// teplota vnitrni
            dtostrf ((double) ((K_QT * (tis_int_c>>3) - K_KTI))/10, 6, 1, texte1);
            LcdStr (texte1);
						LcdStr ("Ci"); 	
				
      		  LcdGotoXY (2,1); 
			      // celkova draha
	          dtostrf ((prub_draha_c.dword / (double)k_odo), 7, 1, texte1);
            LcdStr (texte1);
            LcdStr ("km");
						// teplota venkovni
            dtostrf (t_out, 6, 1, texte1); 
						LcdStr (texte1);
						LcdStr ("Co");
		      }     
          break;
        case 2:
          kw1281_get_grp (1);
          if (kw1281_get_data (&block_length, values) == ANS_GRP)
          { 
            LcdGotoXY (1,1);
		        // Otacky
            xitoa(25*values[2], 10, 4, texte1);  // values[0] = 1  values[1] = 125
            LcdStr (texte1);
            LcdStr ("/min");
            // Tepl chlad. kapal
            dtostrf ((double)(0.9*(int8_t)(values[5]-100)), 6, 1, texte1); // values[3] = 5  values[4] = 9
            LcdStr (texte1);
            LcdStr ("Ck");
            LcdGotoXY (2,3);
		        // Lambda faktor
            dtostrf ((double)(0.002*(int8_t)(values[8]-128)+1), 6, 3, texte1); // values[6] = 11  values[7] = 20
            LcdStr (texte1);
          }
		      kw1281_get_grp (3);
          if (kw1281_get_data (&block_length, values) == ANS_GRP)
          {
            // Zatez motoru
			      if(values[3] != 2) values[5] = 0;
			      dtostrf ((double)(0.51*values[5]), 6, 1, texte1); // values[3] = 17  values[4] = 255
            LcdStr (texte1);
			      LcdStr ("%z");
		      }
		      break;
        case 3:
          kw1281_get_grp (2);
          if (kw1281_get_data (&block_length, values) == ANS_GRP)
          { 
            LcdGotoXY (1,2);
  	        // Napeti baterie
            dtostrf ((double)(0.063*values[8]), 5, 2, texte1);  // values[6] = 6  values[7] = 63
            LcdStr (texte1);
            LcdStr ("V");
            // Tepl vzduchu
            dtostrf ((double)(0.9*(int8_t)(values[11]-100)), 7, 1, texte1); // values[9] = 5  values[10] = 9
            LcdStr (texte1);
            LcdStr ("Cv"); 
            LcdGotoXY (2,1);
            // Doba vstriku
			      dtostrf ((double)(0.03*values[5]), 6, 2, texte1); // values[3] = 15  values[4] = 3
            LcdStr (texte1);
            LcdStr ("ms ");
          }
          kw1281_get_grp (4);
          if (kw1281_get_data (&block_length, values) == ANS_GRP)
          { 
    	      // Bin. bity
            xitoa(values[11], 2, -8, texte1);                // values[9] = 16  values[10] = 255
            LcdStr (texte1);
          }  
        break;
	    case 4:
  	      kw1281_get_grp (3);
          if (kw1281_get_data (&block_length, values) == ANS_GRP)
          { 
            LcdGotoXY (1,1);
	        // Uhel klapky
            dtostrf ((double)(0.334*values[8]), 6, 1, texte1); // values[6] = 3  values[7] = 167
            LcdStr (texte1);
            LcdStr ("k");
            // Predstih
			      xitoa ((int8_t)(values[11]-127), -10, 7, texte1);  // values[9] = 4  values[10] = 100
            LcdStr (texte1);
            LcdStr ("p");
          }
          kw1281_get_grp (5);  
          if (kw1281_get_data (&block_length, values) == ANS_GRP)
          { 
            LcdGotoXY (2,1);
  	        // Strida sign. ventilu nadobky s akt. uhlim
            dtostrf ((double)(0.3906*values[5]), 6, 1, texte1);  // values[3] = 23   values[4] = 100
						LcdStr (texte1);
            LcdStr ("%s");
            // Lambda
            dtostrf ((double)(0.3906*(int8_t)(values[11]-128)), 7, 1, texte1); // values[9] = 20  values[10] = 50
            LcdStr (texte1);
            LcdStr ("%l");
          }
        break;
	     default: break;
		  }
    }
  }
}
