;---------------------------------------------------------------------------;
; USI control functions
;---------------------------------------------------------------------------;

.nolist
#include <avr/io.h>	// Include device specific definitions.
.list

.macro	ldiw	dh,dl, abs
	ldi	\dl, lo8(\abs)
	ldi	\dh, hi8(\abs)
.endm

.macro	subw	dh,dl, sh,sl
	sub	\dl, \sl
	sbc	\dh, \sh
.endm

.macro	subiw	dh,dl, abs
	subi	\dl, lo8(\abs)
	sbci	\dh, hi8(\abs)
.endm

.macro	addiw	dh,dl, abs
	subi	\dl, lo8(-(\abs))
	sbci	\dh, hi8(-(\abs))
.endm

#ifdef STEREO
#if STEREO
#define B_DO	5
#else
#define B_DO	1
#endif
#endif


;---------------------------------------------------------------------------;
; Initialize USI
;
; void init_spi (void);

.global init_spi
.func init_spi
init_spi:
	ldi	r24, 0b00001000			; Enable only SCK and DI. DO is controlled in software
	out	_SFR_IO_ADDR(USICR), r24
	sbi	_SFR_IO_ADDR(PORTB), B_DO	; DO = H
	ret
.endfunc



;---------------------------------------------------------------------------;
; Receive a byte
;
; BYTE rcv_spi (void);

.global rcv_spi
.func rcv_spi
rcv_spi:
	ldi	r24, 0b000100			;PB2(SCK)
	.rept 16				;Toggle SCK 16 times
	out	_SFR_IO_ADDR(PINB), r24		;
	.endr					;/
	nop					;Read shift register
	in	r24, _SFR_IO_ADDR(USIDR)	;/
	ret
.endfunc



;---------------------------------------------------------------------------;
; Transmit a byte
;
; void xmit_spi (BYTE);

.global xmit_spi
.func xmit_spi
xmit_spi:
	ldi	r25, 0b000100			; PB2(SCK)
	in	r22, _SFR_IO_ADDR(PORTB)

	ldi	r23, 8
1:	bst	r24, 7				;DO = data bit to be sent
	bld	r22, B_DO			;
	out	_SFR_IO_ADDR(PORTB), r22	;
	lsl	r24				;/
	out	_SFR_IO_ADDR(PINB), r25		;SCK = H
	out	_SFR_IO_ADDR(PINB), r25		;SCK = L
	dec	r23				;while(--r23)
	brne	1b				;/

	sbi	_SFR_IO_ADDR(PORTB), B_DO	;DO = H
	ret
.endfunc



;---------------------------------------------------------------------------;
; Read and forward a partial data block
;
; void fwd_blk_part (void*, WORD, WORD);

.global fwd_blk_part
.func fwd_blk_part
fwd_blk_part:
	movw	r26, r24		;X = R25:R24 (memory address)
	movw	r30, r22		;Z = R23:R22 (byte offset in the sector)

	ldiw	r19,r18, 514		;R19:R18 = 514, Number of bytes to receive
	sub	r18, r30		;R19:R18 -= Z
	sbc	r19, r31		;/
1:	; Skip leading data bytes
	sbiw	ZL, 1			;Skip leading data...
	brcs	2f			;
	rcall	rcv_spi			;
	rjmp	1b			;/
2:	sbrc	r21, 7			;Destination?
	rjmp	fb_dev

fb_mem:	; Store intermediate data bytes to the memory
	sub	R18, r20		;R19:R18 -= R21:R20
	sbc	R19, r21		;/
10:	rcall	rcv_spi			;do
	st	X+, r24			; *X++ = rcv_spi()
	subi	r20, 1			;while (--r21:r20)
	sbci	r21, 0			;
	brne	10b			;/
	rjmp	fb_ret

fb_dev:	; Forward intermediate data bytes to the outgoing stream (fixed to the wave FIFO)
	andi	r21, 3			;R21:R20 &= 0x3FF
	subw	r19,r18, r21,r20	;R19:R18 -= R21:R20
	lds	r25, FifoWi		;r25 = FIFO write index

	sbic	_SFR_IO_ADDR(GPIOR0), 4	;if (16bit) R21:R20 /= 2;
	lsr	r21			;
	sbic	_SFR_IO_ADDR(GPIOR0), 4	;
	ror	r20			;/

3:	ldiw	r27,r26, Buff		;X = Buff + R25++
	add	r26, r25		;
	adc	r27, r0			;
	inc	r25			;/
	sbic	_SFR_IO_ADDR(GPIOR0), 4	;if 16bit, skip low byte
	rcall	rcv_spi			;/
4:	lds	r24, FifoCt		;wait while FIFO full
	cpi	r24, 255		;
	brcc	4b			;/
	rcall	rcv_spi			;r24 = rcv_spi()
	sbic	_SFR_IO_ADDR(GPIOR0), 4	;if 16bit, signed -> unsigned
	subi	r24, 0x80		;/
	st	X+, r24			;store r24 into FIFO
	cli				;
	lds	r24, FifoCt		;
	inc	r24			;
	sts	FifoCt, r24		;
	sei				;/
	subiw	r21,r20, 1		;while(--R21:R20)
	brne	3b			;/

	sts	FifoWi, r25		;Save FIFO write index

fb_ret:	rcall	rcv_spi			;Discard trailing data and CRC
	subi	r18, 1			;
	sbci	r19, 0			;
	brne	fb_ret			;/

	ret
.endfunc



;---------------------------------------------------------------------------;
; Read and forward the data block
;
; ISR(TIM0_COMPA_vect);


.global TIM0_COMPA_vect
.func TIM0_COMPA_vect
TIM0_COMPA_vect:
	push	r24				;Save regs.
	in	r24, _SFR_IO_ADDR(SREG)		;
	out	_SFR_IO_ADDR(GPIOR2), r24	;
	push	r30				;
	push	r31				;/

	lds	r24, FifoCt			;Check availability of the sampling data
	in	r31, _SFR_IO_ADDR(GPIOR0)	;
	andi	r31, 3				;
	sub	r24, r31			;
	brcs	9f				;/
	sts	FifoCt, r24			;Pop-out the sampling data from FIFO
	lds	r30, FifoRi			;
	add	r31, r30	 		;
	sts	FifoRi, r31			;/
	clr	r31				;Z = pointer to the top of FIFO
	addiw	r31,r30, Buff			;/

#if STEREO	// 2-ch output
	ld	r24, Z+				;Get L-ch/Mono data
	out	_SFR_IO_ADDR(OCR1B), r24	;Send it to L-ch
	sbic	_SFR_IO_ADDR(GPIOR0), 1		;Get R-ch data if available
	ld	r24, Z				;/
	out	_SFR_IO_ADDR(OCR1A), r24	;Send it to R-ch
#else		// 1-ch output
	ld	r24, Z+				;Get L-ch/Mono data
	sbis	_SFR_IO_ADDR(GPIOR0), 1		;Mix R-ch data if available
	rjmp	2f				;
	ld	r31, Z				;
	add	r24, r31			;
	ror	r24				;/
2:	out	_SFR_IO_ADDR(OCR1B), r24	;Send it to the output
#endif

9:	pop	r31				;Restore regs.
	pop	r30				;
	in	r24, _SFR_IO_ADDR(GPIOR2)	;
	out	_SFR_IO_ADDR(SREG), r24		;
	pop	r24				;/
	reti
.endfunc

