; Filename: UART.ASM
; **********************************************
; * Software half duplex UART example          *
; * Revision: 1.0                              *
; * Date      January 24, 1996                 *
; * Part:     16C54-XT/P or /SO                *
; * Fuses:    OSC:  XT (4.0 Mhz xtal)          *
; *           WDT:  OFF                        *
; * Compiled using MPASM V1.21                 *
; **********************************************
; * Include files:                             *
; *           P16C5X.INC                       *
; **********************************************************************
; * This program shows a very simple half duplex UART code example     *
; * that can be implimented in any PIC16C5X/XX device.                 *
; * uarttx - UART Transmitter code      22 Instructions, 3 Bytes RAM   *
; * uartrx - UART Receiver code         22 Instructions, 3 Bytes RAM   *
; * uartinit - UART port initialization  5 Instructions, 0 Bytes RAM   *
; * printstring - Print string table    11 Instructions, 1 Byte  RAM   *
; * printcrlf - Print carrage return     5 Instructions, 0 Byte  RAM   *
; * testuart - Test out above routines  13 Instructions, 1 Byte  RAM   *
; * String Storage - Table text strings 46 Characters, 47 Instructions *
; ********************************************************************** 
	list p=16C54
	include "c:\pictools\mplab\p16c5x.inc"
	__FUSES _CP_OFF&_WDT_OFF&_XT_OSC

; ************************************
; * Port A (RA0-RA4) bit definitions *
; ************************************
TX              EQU     3       ; RA3 is the software UART Transmit pin
RX              EQU     2       ; RA2 is the software UART Recieve pin

; ************************************
; * Port B (RB0-RB7) bit definitions *
; ************************************
; Port B is unused in this example

; *****************************************
; * Define clock speed and baud rate here *
; *****************************************

OSCCLOCK        EQU     .4000000   ; Define external crystal frequency here
BAUDRATE        EQU     .9600      ; Define desired baud rate here

; * The following assembler calculations determine delays:
; BIT_TIME (uS) = (12 + NUM_NOPs + 3 * DELAY_VALUE) * PROC_INSTRUCT_CYCLE
;
; BIT_TIME is the time for each bit or half bit = 1/(baud rate)
; PROC_INSTRUCT_CYCLE is the processor instruction cyle = (crystal freq)/4
; DELAY_VALUE is the integer delay value that is loaded into delay counter
; NUM_NOPs are the number of NOPs that have to be added to equal bit time

OVERHEAD        EQU     .11             ; Number of instructions for routine
INSTPERBIT      EQU     ((2*OSCCLOCK/(4*BAUDRATE))+1)/2 ;Instruction/bit time
INSTPERHALFBIT  EQU     ((2*OSCCLOCK/(8*BAUDRATE))+1)/2 ;Instruction/bit time

DELAYBAUD       EQU  (INSTPERBIT-OVERHEAD)/3    ;
NUMBAUDNOPS     EQU  INSTPERBIT-(3*DELAYBAUD)-OVERHEAD
DELAYHALFBAUD   EQU  (INSTPERHALFBIT-OVERHEAD)/3        ;
NUMHALFBAUDNOPS EQU  INSTPERHALFBIT-(3*DELAYHALFBAUD)-OVERHEAD

CHARBUF         EQU     10h     ; Receive and transmit shift register
BITCOUNT        EQU     11h     ; Bit counter
BAUDCOUNT       EQU     12h     ; Bit delay loop counter
TEMP            EQU     13h     ; Temp register for string pointer counting
RECCHAR         EQU     14h     ; Reieved character register

; *******************************************************************
; * string storage look-up table - place strings here               *
; *******************************************************************
stringtable             
	addwf   PCL,F           ; Add W value to PC for look-up table
stringbase
histring
	DT      "Hello World!\0"
retstring
	DT      "You pressed the following key: \0"     

; *******************************************************************
; * macro for delaying one bit time interval.                       *
; * This is used by both the uartsend and uartreceive routines.     *
; *******************************************************************
delaybit MACRO
	local dlylabels
	movlw   DELAYBAUD       ; Place baud delay value into W             
	movwf   BAUDCOUNT       ; Move baud delay value into BAUDCOUNT register
	variable nopcount
nopcount = NUMBAUDNOPS
	WHILE nopcount > 0      ; Add correct number of NOPs
	NOP                     ; Delay one additional cycle
nopcount--      
	ENDW            
dlylabels
	decfsz  BAUDCOUNT,F     ; Decrement baud delay counter, skip when zero
	goto    dlylabels       ; Jump back and delay for another count cycle
	ENDM                    ; Done with delay, so return

; *******************************************************************
; * Macro for delaying one half bit time interval.                  *
; * This is used by both the uartsend and uartreceive routines.     *
; *******************************************************************
delayhalfbit MACRO
	movlw   DELAYHALFBAUD   ; Place half baud delay value into W             
	movwf   BAUDCOUNT       ; Move baud delay value into BAUDCOUNT register
	variable nopcount
nopcount = NUMHALFBAUDNOPS
	WHILE nopcount > 0
	NOP                     ; Delay one additional cycle
nopcount--              
	ENDW            
dlyhalflabels
	decfsz  BAUDCOUNT,F     ; Decrement baud delay counter, skip when zero
	goto    dlyhalflabels   ; Jump back and delay for another count cycle
	ENDM                    ; Done with delay, so return

; *******************************************************************
; * uarttx RS-232 character output routine.                         *
; * USAGE:                                                          *
; *       Place ASCII character value into W and call uarttx        *
; *       Transmits 8 bits no parity and 1 stop bit                 *
; *******************************************************************
uarttx
	movwf   CHARBUF         ; Place output character into CHARBUF reg
	movlw   .10             ; total number of bits to send 
	movwf   BITCOUNT        ; move this to BITCOUNT reg
	bsf     STATUS,C        ; init carry with stop bit
	bcf     PORTA,TX        ; Send a start bit
	goto    loopdelay       ; Jump to send the start bit
sendbit            
	rrf     CHARBUF,F       ; place the next bit to be transmitted into carry bit
	btfsc   STATUS,C        ; Skip if next bit is zero
	goto    settxout        ; Sending a one - jump to setoutput code
	nop                     ; Add one cycle so set and clear of output synch up
	bcf     PORTA,TX        ; Transmit a 1
	goto    loopdelay       ; Done updating tx output, goto next bit
settxout
	bsf     PORTA,TX        ; Transmit a 0
	goto    $+1             ; Add 2 cycles so timing synchs up..
loopdelay
	delaybit                ; delay for one bit time
	decfsz  BITCOUNT,F      ; Decrement bit counter and skip when done
	goto    sendbit         ; Jump back to put_bit to tranmit next bit
	retlw   0               ; Done - Return back to the main program

; *******************************************************************
; * uartrx RS-232 character input routine.                          *
; * USAGE:                                                          *
; *         CALL  uartrx                                            *
; *         receive byte is placed into TEMP                        *
; *         Receives 8 bits, no parity, 1 stop bit                  *
; *******************************************************************
uartrx
	movlw   .09             ; set input bit counter
	movwf   BITCOUNT        ; place bit counter into BITCOUNT
getwait 
	btfsc   PORTA,RX        ; Skip when we recieve a start bit
	goto    getwait         ; go back and wait for a start bit
	delayhalfbit            ; delay for one half bit time
	btfsc   PORTA,RX        ; Skip if we still have the start bit 
	goto    getwait         ; Must have been noise - go back and wait for start
getloop                
	delaybit                ; delay for one bit time
	bcf     STATUS,C        ; Clear carry bit to shift in zero
	btfsc   PORTA,RX        ; Skip if the next bit is a zero
	bsf     STATUS,C        ; Set the carry bit
	goto    $+1             ; Delay 2 cycles
	nop                     ; Delay 1 cycle
	rrf     CHARBUF,F       ; Shift the next received bit into temp 
	decfsz  BITCOUNT,F      ; decrement the bit count and skip when finished
	goto    getloop         ; Go back if we still have more bits
	rlf     CHARBUF,F       ; Restore buffer after shifting in stop bit
	retlw   0

; *******************************************************************
; * printstring - print out a string of chars                       *
; *******************************************************************
printstring
	movwf   TEMP            ; Place string offset into temp
loopprint               
	movf    TEMP,W          ; Place next char to be sent into W
	call    stringtable     ; Look up the next char to send
	movwf   CHARBUF         ; Place char into CHARBUF for temp storage
	xorlw   '\0'            ; Place end of string char into W
	btfsc   STATUS,Z        ; Skip if not at end of string
	retlw   0               ; At end of string - done so go back!
	incf    TEMP,F          ; Point to next character
	movf    CHARBUF,W       ; Place print char into W
	call    uarttx          ; Send char to the screen
	goto    loopprint       ; Loop back for the next char

; *******************************************************************
; * printcrlf routine - send carrage return and line feed           *
; *******************************************************************
printcrlf
	movlw   .13             ; Place value for carrage return into W
	call    uarttx          ; Send it to the RS-232 port
	movlw   .10             ; Place value for the Line Feed into W
	call    uarttx          ; Send it to the RS-232 port
	retlw   0               ; Done, so return!

; *******************************************************************
; * inituart RS-232 port initialization routine                     *
; * USAGE:                                                          *
; *         CALL  inituart                                          *
; *******************************************************************
inituart
	movlw   b'00001100'     ; Place 11 into RA3,2
	movwf   PORTA           ; Init PORTA output latches
	movlw   b'11110111'     ; Set RA3 as an output
	tris    PORTA           ; Init PORTA tris register
	retlw   0               ; Done, so return!

; *******************************************************************
; * testuart routine - send string of chars using uarttx            *
; *******************************************************************
testuart
	call    inituart        ; Set up UART ports and TRIS bits
	movlw   histring-stringbase ; Place offset address of string into W
	call    printstring     ; Print this string to the UART
	call    printcrlf       ; Send carrage return and line feed
getnextchar
	call    uartrx          ; Get a char from the terminal
	movf    CHARBUF,W       ; Place received char into W
	movwf   RECCHAR         ; Move it to RECCHAR to display later
	movlw   retstring-stringbase ; Place offset address of string into W
	call    printstring     ; Print this string to the UART
	movf    RECCHAR,W       ; Place recieved char into W
	call    uarttx          ; Send it to the terminal
	call    printcrlf       ; Send carrage return and line feed
	goto    getnextchar     ; Jump back and do it again!

; *******************************************************************
; * reset vector - jump to testuart code from here                  *
; *******************************************************************
	ORG     1FFh
	GOTO    testuart
    END
