/*********************************************
Last Revision: 2006-10-31
P. vd Hoeven.
Target: AVR-GCC
Functions for HD44780 Lcd 1 regel * 16 charakters (2*8)
8 bits data aan PortB
Reading from the lcd is disabled (R/W connected to gnd) so we must wait long
enough for each write operation to complete.

Copyright (C) 2009 Paul van der Hoeven.
This is free software, licensed under the terms of the GNU General
Public License as published by the Free Software Foundation.
***********************************************/
#include "main.h"
#include "lcd.h"
#ifdef USE_LIB_DELAY
  #include <util/delay.h>
  #define DelayMs _delay_ms
  #define DelayUs _delay_us
#else
  #include<util/delay.h>
#endif

#if USE_LCD_WRITE_CG_RAM
#include <avr/pgmspace.h>
#endif


// This is for later, it's for LcdPutSP
// #include <avr/pgmspace.h>
// prog_uchar PacketSize[]= "ProgMem String";  


uint8_t Cursor;

#if ! LCD_BYTEWIDE
/*static inline uint8_t Swap(uint8_t byte)
{
  asm volatile(
        "swap   %0"             "\n\t"
        : "=r" (byte)
        : "0" (byte)
	: "0"
  );
    return byte;
}*/

//---------------------------------------------------------------------------
void LcdWriteHighNibble( uint8_t Data)
{
//  DelayUs(100);//2006-10-29 Delay Moved to first line of LcdWriteByte()	// Datasheet: Writing data = <37+4us(Fosc=270Khz)
  if(Data&0x10) LCD_DATA_PORT_BIT4|=_BV(LCD_DATA_BIT4); else LCD_DATA_PORT_BIT4&=~_BV(LCD_DATA_BIT4);
  if(Data&0x20) LCD_DATA_PORT_BIT5|=_BV(LCD_DATA_BIT5); else LCD_DATA_PORT_BIT5&=~_BV(LCD_DATA_BIT5);
  if(Data&0x40) LCD_DATA_PORT_BIT6|=_BV(LCD_DATA_BIT6); else LCD_DATA_PORT_BIT6&=~_BV(LCD_DATA_BIT6);
  if(Data&0x80) LCD_DATA_PORT_BIT7|=_BV(LCD_DATA_BIT7); else LCD_DATA_PORT_BIT7&=~_BV(LCD_DATA_BIT7);
  LCD_CS_PORT|=_BV(LCD_CS_BIT);		// 3 Generate Write Pulse.
  				// 2006-10-29 10us -> 2us.
   _delay_us(3);			// Datasheet: CS High for >1us.
  LCD_CS_PORT&=~_BV(LCD_CS_BIT);
}
#endif

//---------------------------------------------------------------------------
void LcdWriteByte( uint8_t Data)	// Schrijf 1 byte data naar de lcd 
{
				// 2006-10-29 100us -> 50us
  _delay_us(50);		// Datasheet: Writing data = <37+4us(Fosc=270Khz)
#if LCD_BYTEWIDE
  LCD_DATA_PORT=Data;
  LCD_CS_PORT|=_BV(LCD_CS_BIT);
				// 2009-11-14 3us is minimum lib capability.
  DelayUs(3);			// Datasheet: CS High for >1us.
  LCD_CS_PORT&=~_BV(LCD_CS_BIT);
#else
  LcdWriteHighNibble(Data);
  asm volatile("swap %0" : "=r" (Data) : "0" (Data));
  LcdWriteHighNibble(Data);	// Write the other half of the byte.
#endif
}

//---------------------------------------------------------------------------
void LcdInit(void)
{
  uint8_t Cnt;

  Cursor=0;
  #if LCD_BYTEWIDE
  LCD_DATA_DDR=0xff;		// Databits voor LCD.
  #else
  LCD_DATA_DDR_BIT4|=_BV(LCD_DATA_BIT4);// Make the 4 data bits all output.
  LCD_DATA_DDR_BIT5|=_BV(LCD_DATA_BIT5);
  LCD_DATA_DDR_BIT6|=_BV(LCD_DATA_BIT6);
  LCD_DATA_DDR_BIT7|=_BV(LCD_DATA_BIT7);
  #endif
  LCD_RS_DDR|=_BV(LCD_RS_BIT);		// RS voor LCD.
  LCD_CS_DDR|=_BV(LCD_CS_BIT);		// CS voor LCD.
  LCD_RS_PORT&=~_BV(LCD_RS_BIT);		// Writing commands now.
  _delay_ms(50);			// Datasheet: Delay > 15ms after VCC=5V (40ms @ VCC=2.7V
  for(Cnt=0; Cnt<4; Cnt++)
  {
     _delay_ms(40);		// Datasheet: >4.1ms.
#if LCD_BYTEWIDE
    LcdWriteByte(0x38);		// 30=Reset 38=Reset for 1Line * 16Char's.
  }
#else
    if(Cnt==3)
      LcdWriteHighNibble(0x20);	// Last of for loop: Set to 4bit Interface.
    else
      LcdWriteHighNibble(0x30);
  }
  LcdWriteByte(0x28);		// 4bits, 1line.
#endif
  LcdWriteByte(0x01);		// (????) Display clear.
  _delay_ms(50);
  LcdWriteByte(0x02);		// (1.52ms) Cursor home.
   _delay_ms(10);			
  LcdWriteByte(0x0c);		// (37us) Display on Cursor off blink off.
  LcdWriteByte(0x06);  		// (37us) Entry mode inc. curs. position.
  LCD_RS_PORT|=_BV(LCD_RS_BIT);	// Stop writing commands.
}

//---------------------------------------------------------------------------
void LcdCommand(uint8_t Data)
{
  LCD_RS_PORT&=~_BV(LCD_RS_BIT);	// RS low is writing commands.
  LcdWriteByte(Data);
  LCD_RS_PORT|=_BV(LCD_RS_BIT);		// Restore 'data' mode.
}

#if USE_LCD_CLEAR
void LcdClear(void)
{
  LcdCommand(1);
  Cursor=0;
  _delay_ms(3);		// 2ms works for some displays.(The smaller ones?).
}
#endif

//---------------------------------------------------------------------------
void LcdSetCursor(uint8_t Pos)
{
  Cursor=Pos;
#if   LCD_SIZE == LCD_SIZE_1x16
  LcdCommand((Pos< 8? Pos: Pos+0x38)|0x80);	// 8th char is on Pos 40.
#elif LCD_SIZE == LCD_SIZE_2x16
  LcdCommand((Pos<16? Pos: Pos+0x30)|0x80);
#elif LCD_SIZE == LCD_SIZE_2x20
  LcdCommand((Pos<20? Pos: Pos+0x2C)|0x80);
#elif LCD_SIZE == LCD_SIZE_2x40
  LcdCommand((Pos<40? Pos: Pos+0x18)|0x80);
#else
#error LCD_SIZE invalid in LcdSetCursor()
#endif
}

//---------------------------------------------------------------------------
void LcdPutC(uint8_t Data)
{
  LCD_RS_PORT|=_BV(LCD_RS_BIT);
  LcdWriteByte(Data);
  Cursor++;
#if   LCD_SIZE == LCD_SIZE_1x16
  if(Cursor== 8)
#elif LCD_SIZE == LCD_SIZE_2x16
  if(Cursor==16)
#elif LCD_SIZE == LCD_SIZE_2x20
  if(Cursor==20)
#elif LCD_SIZE == LCD_SIZE_2x40
  if(Cursor==40)
#else
#error LCD_SIZE invalid in LcdPutC()
#endif
    LcdCommand(0x80|0x40);	// Move cursor to the 2nd line.
}

//---------------------------------------------------------------------------
void LcdPutS(char *pS)
{
  while(*pS)
  {
    LcdPutC(*pS);
    pS++;
  }
}

//---------------------------------------------------------------------------
#if USE_LCD_PUT_X_HEX
//---------------------------------------------------------------------------
void LcdPut1Hex(uint8_t C)
{
  C&=0x0f;
  if(C<10)
    C+='0';
  else 
    C+='A'-10;
  LcdPutC(C);
}

//---------------------------------------------------------------------------
void LcdPut2Hex(uint8_t C)
{
  LcdPut1Hex(C>>4);
  LcdPut1Hex(C&0x0F);
}

//---------------------------------------------------------------------------
void LcdPut4Hex(uint16_t C)
{
  LcdPut2Hex(C>>8);
  LcdPut2Hex(C&0xFF);
}
#endif

//---------------------------------------------------------------------------
#if USE_LCD_WRITE_CG_RAM_P
//---------------------------------------------------------------------------
void LcdWriteCgRam_P(uint8_t const *CgRam,uint8_t Bytes)
{
  LcdCommand(0x40);			// CGram adress 0.
//  LCD_RS_PORT|=_BV(LCD_RS_BIT);		// Restore 'data' mode.
  for(;Bytes;Bytes--)
    LcdWriteByte(pgm_read_byte_near(CgRam++));
}
#endif

#if 0
/*
** macros for automatically storing string constant in program memory.
** from Peter Fleury  <pfleury@gmx.ch>  http://jump.to/fleury.
*/
#ifndef P
#define P(s) ({static const char c[] __attribute__ ((progmem)) = s;c;})
#endif
#define LcdPutSP(__s)         lcd_puts_p(P(__s))

// Writes a string from program memory to lcd.

//---------------------------------------------------------------------------
void lcd_puts_p(const char *progmem_s)
{
  register char C;
  while((C = PRG_RDB(progmem_s++)))
    LcdPutC(C);
}
#endif
/*---------------------------------------------------------------------------
              End of all code.      Below this line is only comment.
//---------------------------------------------------------------------------
Revision List:
2009-11-14 Changed a delay from 2 to 3us. (Is minimum delay from delay lib).
2007-09-11 LcdPutS( uint8_t*) changed to int8_t.
2006-10-31 LcdWriteCgRam_P() New function. for defining custom characters.
2006-10-29 Changed timing of LcdWriteHighNibble() and LcdWriteByte().
	moved DelayUs() from LcdWriteHighNibble() to LcdWriteByte().
2006-10-28 Added support for different Lcd sizes:
	LCD_SIZE_1x16	LCD_SIZE_2x16	LCD_SIZE_2X40.
2006_10-06 Added LCD_ before all #define thingies for the lcd.
2006-03-07 Removed all sbi() and cbi() macro's.
2005-10-13 Added support for 4bit databus for LCD.
	Small problem in line 54 Inline Swap() does not function properly.
2005-02-22 Removed SETRS(), CLEARRS() etc. and put all hardware dependant.
	statements together in lcd.h.
---------------------------------------------------------------------------*/


