/******************************************************************************
Display free space on a MMC/SD/SD HC card
example for the CodeVisionAVR V2.04.4+ compiler
(C) 2009 HP InfoTech s.r.l., Pavel Haiduc

The example is intended to be run on the Mega128-Dev development board
from www.priio.com.
The development board uses an ATmega128 chip running at 14.745600 MHz

The SD card socket on the development board is connected to the
following I/O ports:

SD socket    CD4050    ATmega128
pin1  /CS -----<|----- PORTD bit6
pin2  SI  -----<|----- MOSI
pin3  GND
pin4  +3.3V
pin5  SCK -----<|----- SCK
pin6  GND
pin7  SO  ------------ MISO
pin9  GND
pin10 CD  ------------ PORTG bit4 with 10k pull-up resistor to +5V
pin11 WP  ------------ PORTG bit3 with 10k pull-up resistor to +5V
pin12 GND

The CD4050 CMOS buffer is powered from a +3.3V supply and
performs logic level translation from 5V (AVR) to 3.3V (SD card).

For other I/O port connections, you need to make appropriate changes in the
"Project|Configure|C Compiler|Libraries|MMC/SD/SD HC Card and FAT Support"
menu.

The development board's serial socket P9 must be connected to the PC COM port.
The CodeVisionAVR Terminal is used for displaying data received from the
development board.
The Terminal must be configured for:
Baud Rate: 19200
Data Bits: 8
Parity: None
Stop Bits: 1
Emulation: TTY
Handshaking: None
Append LF on Reception: OFF

If you are using the CodeVisionAVR Evaluation version, you may program into
the FLASH the precompiled SDCARD3.HEX file found in the
\Examples\SDCARD3\EXE directory
******************************************************************************/

/* ATmega128 I/O register definitions */
#include <mega128.h>
/* FAT on MMC/SD/SD HC card support */
#include <ff.h>
/* printf */
#include <stdio.h>

/* Timer1 overflow interrupt frequency [Hz] */
#define T1_OVF_FREQ 100
/* Timer1 clock prescaler value */
#define T1_PRESC 1024L
/* Timer1 initialization value after overflow */
#define T1_INIT (0x10000L-(_MCU_CLOCK_FREQUENCY_/(T1_PRESC*T1_OVF_FREQ)))

/* USART Baud rate */
#define BAUD_RATE 19200
#define BAUD_INIT (_MCU_CLOCK_FREQUENCY_/(BAUD_RATE*16L)-1)

/* 100Hz timer interrupt generated by ATmega128 Timer1 overflow */
interrupt [TIM1_OVF] void timer_comp_isr(void)
{
/* re-initialize Timer1 */
TCNT1H=T1_INIT>>8;
TCNT1L=T1_INIT&0xFF;
/* card access low level timing function */
disk_timerproc();
}

/* error message list */
flash char * flash error_msg[]=
{
"", /* not used */
"FR_DISK_ERR",
"FR_INT_ERR",
"FR_INT_ERR",
"FR_NOT_READY",
"FR_NO_FILE",
"FR_NO_PATH",
"FR_INVALID_NAME",
"FR_DENIED",
"FR_EXIST",
"FR_INVALID_OBJECT",
"FR_WRITE_PROTECTED",
"FR_INVALID_DRIVE",
"FR_NOT_ENABLED",
"FR_NO_FILESYSTEM",
"FR_MKFS_ABORTED",
"FR_TIMEOUT"
};

/* display error message and stop */
void error(FRESULT res)
{
if ((res>=FR_DISK_ERR) && (res<=FR_TIMEOUT))
   printf("ERROR: %p\r\n",error_msg[res]);
/* stop here */
while(1);
}

void main(void)
{
/* FAT function result */
FRESULT res;
/* will hold the information for logical drive 0: */
FATFS fat;
/* pointer to the FATFS type structure */
FATFS *pfat;
/* number of free clusters on logical drive 0:*/
unsigned long free_clusters;
/* number of free kbytes on logical drive 0: */
unsigned long free_kbytes;
/* root directory path for logical drive 0: */
char root_path[]="0:/";

/* initialize Timer1 overflow interrupts in Mode 0 (Normal) */
TCCR1A=0x00;
/* clkio/1024 */
TCCR1B=(1<<CS12)|(1<<CS10);
/* timer overflow interrupts will occur with 100Hz frequency */
TCNT1H=T1_INIT>>8;
TCNT1L=T1_INIT&0xFF;
/* enable Timer1 overflow interrupt */
TIMSK=1<<TOIE1;

/* initialize the USART0 TX, 8N1, Baud rate: 19200 */
UCSR0A=0;
UCSR0B=1<<TXEN0;
UCSR0C=(1<<UCSZ01)|(1<<UCSZ00);
UBRR0H=BAUD_INIT>>8;
UBRR0L=BAUD_INIT&0xFF;

/* globally enable interrupts */
#asm("sei")

/* point to the FATFS structure that holds
information for the logical drive 0: */
pfat=&fat;

/* mount logical drive 0: */
if ((res=f_mount(0,pfat))==FR_OK)
   printf("Logical drive 0: mounted OK\r\n");
else
   /* an error occured, display it and stop */
   error(res);

/* get the number of free clusters */
if ((res=f_getfree(root_path,&free_clusters,&pfat))==FR_OK)
   {
   /* calculate the number of free bytes */
   free_kbytes=free_clusters *
               /* cluster size in sectors */
               pfat->csize
               /* divide by 2 to obtain the sector size in kbytes
               512 (sector size in bytes)/1024 = 1/2 kbytes
               we need to do the division by 2 directly,
               in order to prevent unsigned long multiplication
               overflow for 8GB+ SD HC cards */
               /2;
   /* display the number of free kbytes */
   printf("Free space on logical drive 0: %lu kbytes\r\n",free_kbytes);
   }
else
   /* an error occured, display it and stop */
   error(res);

/* stop here */
while(1);
}
