
#include <inttypes.h>
#include <avr/io.h>
#include <avr/pgmspace.h>
#include <util/delay.h>
#include "alcd.h"

#define CONCAT(a, b)    a ## b
#define PORT(name)      CONCAT(PORT, name)
#define PIN(name)       CONCAT(PIN, name)
#define DDR(name)       CONCAT(DDR, name)


static unsigned char lcd_pos_x;
static unsigned char lcd_pos_y;
static unsigned char lcd_pos_rows;
static unsigned char lcd_pos_cols;

static unsigned char _displayfunction = 0;
static unsigned char _displaycontrol = 0;

static void lcd_write4(unsigned char data) {
    if(data & 0x01)
        PORT(LCD_DATA0_PORT) |= (1<<LCD_DATA0_PIN);
    else
        PORT(LCD_DATA0_PORT) &= ~(1<<LCD_DATA0_PIN);
    if(data & 0x02)
        PORT(LCD_DATA1_PORT) |= (1<<LCD_DATA1_PIN);
    else
        PORT(LCD_DATA1_PORT) &= ~(1<<LCD_DATA1_PIN);
    if(data & 0x04)
        PORT(LCD_DATA3_PORT) |= (1<<LCD_DATA2_PIN);
    else
        PORT(LCD_DATA3_PORT) &= ~(1<<LCD_DATA2_PIN);
    if(data & 0x08)
        PORT(LCD_DATA3_PORT) |= (1<<LCD_DATA3_PIN);
    else
        PORT(LCD_DATA3_PORT) &= ~(1<<LCD_DATA3_PIN);
    PORT(LCD_EN_PORT) &= ~(1<<LCD_EN_PIN);
    _delay_us(1);
    PORT(LCD_EN_PORT) |= (1<<LCD_EN_PIN);
    _delay_us(1);
    PORT(LCD_EN_PORT) &= ~(1<<LCD_EN_PIN);
    _delay_us(200);
}

static void lcd_write(unsigned char data, unsigned char rs) {
    if(rs)
        PORT(LCD_RS_PORT) |= (1<<LCD_RS_PIN);
    else
        PORT(LCD_RS_PORT) &= ~(1<<LCD_RS_PIN);
    lcd_write4(data>>4);
    lcd_write4(data);
}

void lcd_init(unsigned char mode) {

    lcd_pos_x=0;
    lcd_pos_y=0;
    lcd_pos_rows = 16;
    lcd_pos_cols = 2;

#ifdef LCD_RW_PORT
    DDR(LCD_RW_PORT) |= PIN(1<<LCD_RW_PIN);
#endif
    DDR(LCD_EN_PORT) |= PIN(1<<LCD_EN_PIN);
    DDR(LCD_RS_PORT) |= PIN(1<<LCD_RS_PIN);

    DDR(LCD_DATA0_PORT) |= PIN(1<<LCD_DATA0_PIN);
    DDR(LCD_DATA1_PORT) |= PIN(1<<LCD_DATA1_PIN);
    DDR(LCD_DATA2_PORT) |= PIN(1<<LCD_DATA2_PIN);
    DDR(LCD_DATA3_PORT) |= PIN(1<<LCD_DATA3_PIN);

    _delay_ms(50);
#ifdef LCD_RW_PORT
    PORT(LCD_RW_PORT) &= ~PIN(1<<LCD_RW_PIN);
#endif

    PORT(LCD_RS_PORT) &= ~PIN(1<<LCD_RS_PIN);
    PORT(LCD_EN_PORT) &= ~PIN(1<<LCD_EN_PIN);

    _displayfunction = LCD_4BITMODE | LCD_1LINE | LCD_5x8DOTS;
    _displayfunction |= LCD_2LINE;

    lcd_write4(0x03);
    _delay_ms(5);
    lcd_write4(0x03);
    _delay_ms(5);
    lcd_write4(0x03);
    _delay_us(150);
    lcd_write4(0x02);

    lcd_write(LCD_FUNCTIONSET | _displayfunction, 0);

    _displaycontrol = LCD_DISPLAYON | LCD_CURSOROFF | LCD_BLINKOFF;

    _displaycontrol |= LCD_DISPLAYON;
    lcd_write(LCD_DISPLAYCONTROL | _displaycontrol, 0);
    lcd_clrscr();

}

void lcd_clrscr(void) {

    lcd_write(1<<LCD_CLR, 0);
    lcd_pos_x=0;
    lcd_pos_y=0;
    _delay_ms(2);
}

void lcd_home(void) {

    lcd_write(1<<LCD_HOME, 0);
    lcd_pos_x=0;
    lcd_pos_y=0;
    _delay_ms(2);
}

void lcd_gotoxy(unsigned char x, unsigned char y) {

    switch(y) {
    case 1:
        lcd_write((1<<LCD_DDRAM)+LCD_START_LINE2+x,0);
        break;
    case 2:
        lcd_write((1<<LCD_DDRAM)+LCD_START_LINE3+x,0);
        break;
    case 3:
        lcd_write((1<<LCD_DDRAM)+LCD_START_LINE4+x,0);
        break;
    case 0:
    default:
        lcd_write((1<<LCD_DDRAM)+LCD_START_LINE1+x,0);
        break;
    }
    lcd_pos_x=x;
    lcd_pos_y=y;
}

void lcd_putc(char c) {

    if(c=='\n') {
        lcd_pos_x=0;
        lcd_pos_y++;
        if(lcd_pos_y>lcd_pos_cols)
            lcd_pos_y=lcd_pos_cols;
        lcd_gotoxy(lcd_pos_x, lcd_pos_y);
        return;
    }
    lcd_write(c, 1);
    lcd_pos_x++;
    if(lcd_pos_x>lcd_pos_rows) {
        lcd_pos_x=0;
        if(lcd_pos_y>lcd_pos_cols)
            lcd_pos_y=lcd_pos_cols;
    }
    lcd_gotoxy(lcd_pos_x, lcd_pos_y);
}

void lcd_puts_p(const char *progmem_s) {
    register char c;

    while ( (c = pgm_read_byte(progmem_s++)) ) {
        lcd_putc(c);
    }
}

void lcd_puts(const char *s) {
    register char c;

    while ( (c = *s) ) {
        lcd_putc(c);
        s++;
    }
}

void lcd_setchar(unsigned char pos, const char *progmem_s) {
    unsigned char cmd = LCD_SETCGRAMADDR | (pos << 3);
    unsigned char c;
    int i;

    pos &= 0x7;
    for (i=0; i<8; i++){
        c=pgm_read_byte(progmem_s++);
        lcd_write(cmd++, 0);
        lcd_write(c, 1);
    }
}
