/****************************************************************************/
/* File        : tasks.c                                                    */
/* Description : system kernel                                              */
/* Author      : ...                                                        */
/* Data        :                                                            */
/* Versione    : 0.1                                                        */
/****************************************************************************/





#include <avr/sleep.h>
#include <inttypes.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include "Com.h"
#include "tasks.h"

#define F_CPU                       4000000UL
#define TIMER_TICK_FREQUENCY        1000L // 1000kHz = 1ms tick size
#define TIMER_TICK_PERIOD_MS        (1000/TIMER_TICK_FREQUENCY) // Tick Period in ms

#define TIMER_PRESCALER_COUNT 256L    
#define TIMER_PRESCALER_CODE  4  

#define TICK_RELOAD           (256L - (F_CPU/(TIMER_TICK_FREQUENCY * TIMER_PRESCALER_COUNT)))

#define PUSH(r) asm volatile("push r"#r);
#define POP(r)  asm volatile("pop  r"#r);

//#define PROVA1
#define MAX_TASKS           4
#define START_SHOT        0xFF

enum TASK_STATE
{
   
    READY = 0,        /* it's ready to run                  */
    RUN,              /* running task                       */
    WAITEVNT,         /* wait an event                      */
    WAITMSG,          /* wait a message                     */
    WAITPOST,         /* attesa di consegnare un messaggio  */
	STOP              /* stopped                            */
};

/*-- local variables  --*/

static  TAG_EVNT  event[10];
static  BYTE      evnt_cnt[10];    /* events counter      */
static  BYTE      event_flag;



static volatile BYTE currentTask = START_SHOT;

static volatile Task *tasks[MAX_TASKS];
static   WORD tick_cnt;
volatile WORD DelayTimerTicks = 0 ;
static   Task *current = 0;
static   BYTE Total_Task;

	

/*-- local functions --*/

static void scheduler(void);


/***********************************************
* Funzione   : taskInit()
* Purpose    : kernel init
* Argomenti  :
* Ritorno    :
* Note       : 
************************************************/
void taskInit(void)
{
  volatile BYTE tmp;
  
  tick_cnt = 0;
  DelayTimerTicks = 0;
  event_flag= 0;
  Total_Task = 0;

  
  /*-- Get the Current Timer1 Count and Add the offset for the Compare target --*/
  TCNT0 = (BYTE) TICK_RELOAD ;

  /*-- Clear the Overflow interrupt status bit and enable the overflow interrupt --*/
  sbi(TIFR, TOV0) ;
  sbi(TIMSK, TOIE0) ; 

  /*-- Set Timer Clock Source Fclk/256 --*/
  TCCR0 = (TCCR0 & 0xF8) | TIMER_PRESCALER_CODE ;
  
  /*-- events init--*/
  for(tmp = 0; tmp < 10; tmp++)
  {
    event[tmp].evnt =  tmp +1;
    event[tmp].bFree = TRUE;
	evnt_cnt[tmp] = 0;
	
  }
 
}



/***************************************************************************
* Funzione    : taskCreate                                               
* Purpose     : task init                                                 
* Arguments   : task function pointer, stackarea pointer, stackarea size  
*               and mailbox size                                          
* Ritorno     :                                                           
* Note        :                                                           
***************************************************************************/
BYTE taskCreate(Task *t, PFUNC f, BYTE* stackArea, BYTE stackSize, BYTE nMsg)
{
	
	signed char a;
	
	cli();
	BYTE *s = (BYTE *)(stackArea + stackSize - 3);
	for (a=31; a>=0; a--) *s-- = 0;
	t->sreg = 0;
	
	t->sp = (BYTE *)(stackArea + stackSize - 3);		
	stackArea[stackSize -2] = ((WORD)f)>>8;
	stackArea[stackSize -1] = ((WORD)f)&0xff;
	
	t->state   = READY;
	t->event   = GOS_NULLEVNT;
	t->timeOut = GOS_NULLEVNT;
	
	  /*----- message queue init -----*/
    t->pQueue = (GOS_MSG*)stackArea;
    t->pIn    = t->pQueue;
    t->pOut   = t->pQueue;
    t->nMaxMsg= nMsg;
    t->nCntMsg= 0;

	
	for (a=0; a< MAX_TASKS; a++)
	{
		if (!tasks[a])
		{
			tasks[a] = t;
			break;
		}
	}
	Total_Task++;
	sei();
	return (BYTE)a;
}
/*****************************************************
* Function   : task_gethandle
* Purpose    : get the running task handle
* Arguments  :
* Ritorno    :
* Note       :
******************************************************/
HANDLE task_gethandle(void)
{
  return (BYTE)currentTask;
}

/*****************************************************
* Function   : task_signal
* Purpose    : signal the event occurence
* Arguments  :
* Ritorno    :
* Note       :
******************************************************/
void task_signal(BYTE evnt)
{

   ComPuts(" signal\n");
   if(evnt != GOS_NULLEVNT)
     evnt_cnt[evnt]++;
   event_flag = evnt;
}
/*****************************************************
* Function   : task_poll
* Descrizione: release the controll to other tasks
* Arguments  :
* Ritorno    :
* Note       :
******************************************************/
void task_poll(void)
{
    current->state = READY;
	/*--switch the next task --*/
	scheduler();
}
/************************************************************
* Function   : task_post
* Purpose    : put a message in the task message queue
* Arguments  : task Handle, mesage pointer
* Ritorno    : Esito dell'operazione 
*              (-> se c' posto nella coda dei messaggi)
* Note       :
*************************************************************/
BYTE task_post(HANDLE hTask, GOS_MSG* pMsg)
{
  Task* prc;

  /*----- determino il processo destinatario -----*/
  prc = tasks[hTask];
	
  /*----- controllo se c'e' posto  nella coda -----*/
  if(prc->nCntMsg == prc->nMaxMsg)
    return GOS_QUEUEFULL;
  else
    (prc->nCntMsg)++;
  /*----- accodo il messaggio -----*/
  prc->pIn->nCode  = pMsg->nCode; 
  prc->pIn->bParam = pMsg->bParam; 
  prc->pIn->wParam = pMsg->wParam; 
  prc->pIn->xParam = pMsg->xParam;
  /*----- incremento il puntatore al messaggio in ingresso -----*/
  if(++prc->pIn == prc->pQueue + prc->nMaxMsg)
    prc->pIn = prc->pQueue;
  /*----- -----*/
  return GOS_SUCCESS;
}
/***********************************************************
* Function   : task_pick
* Purpose    : pick a message from message queue
* Arguments  : message to get
* Ritorno    : Esito -> se ci sono messaggi
* Note       :
************************************************************/
BYTE task_pick(GOS_MSG* pMsg)
{
  /*----- check if there are messages in the queue -----*/
  if(!current->nCntMsg)
    return GOS_QUEUEEMPTY;
  else
    current->nCntMsg--;
  /*----- get a message from queue -----*/
  pMsg->nCode  = current->pOut->nCode;
  pMsg->bParam = current->pOut->bParam;
  pMsg->wParam = current->pOut->wParam;
  pMsg->xParam = current->pOut->xParam;
  /*----- incremento il puntatore al messaggio in uscita -----*/
  if(++current->pOut == current->pQueue + current->nMaxMsg)
    current->pOut = current->pQueue;
  /*----- -----*/
  return GOS_SUCCESS;
}
/*******************************************************************
* Function   : task_getevent()
* Arguments  : get an event to code still unitilzed
* Argomenti  : 
* Ritorno    : event code 
* Note       :
********************************************************************/
BYTE task_getevent(void)
{
  BYTE i;

  for(i = 0; i < 10; i++)
  {
    if(event[i].bFree)
    {
      event[i].bFree = 0;
      task_resetevent(event[i].evnt);
      return event[i].evnt;
    }
  }
  return GOS_NULLEVNT;
}
/*******************************************************************
* Function   : ios_releaseevent
* Descrizione: Rilascia un evento
* Arguments  : L'evento da rilasciare
* Ritorno    : 
* Note       :
********************************************************************/
void task_releaseevent(BYTE evnt)
{
  BYTE i;

  for(i = 0; i < 10; i++)
  {
    if(event[i].evnt == evnt)
    {
      event[i].bFree = 1;
      task_resetevent(event[i].evnt);
      break;
    }
  }
}
/*******************************************************************
* Function   : task_resetevent
* Descrizione: event reset
* Arguments  : event code to reset
* Ritorno    :
* Note       :
********************************************************************/
void task_resetevent(BYTE evnt)
{
  evnt_cnt[evnt] = 0;
}
/*******************************************************************
* Function   : task_wait
* Purpose    : pause the actual task untill the next event or timeout
* Arguments  : event code, timeout (ms)
* Ritorno    :
* Note       :
********************************************************************/
BYTE task_wait(BYTE evnt, WORD time_out)
{
   
    current->state = WAITEVNT;	
	current->event = evnt;
	if(time_out != GOS_NULLTIME) 
	{
      current->timeOut = tick_cnt + time_out;
	}
    else
      current->timeOut = GOS_NULLTIME;
	  
    /*-- start  scheduler --*/
    scheduler();
	current->timeOut = 0;
	if(evnt_cnt[evnt])
    {
	   return evnt;
    }
	else
	   return GOS_TIMEOUTEVNT;
  
}
/***********************************************
* Function   : task_switch()
* Purpose    : start sheduler
* Arguments  :
* Ritorno    :
* Note       : 
************************************************/
void task_switch()
{
  scheduler();
}
/***********************************************
* Function   : task_delay()
* Purpose    : delay function
* Arguments  : delay amount in ms
* Ritorno    :
* Note       : 
************************************************/
void task_delay(WORD ms)
{
	DelayTimerTicks = ms ;
	
	while( DelayTimerTicks )
		; // Sit and wait for the interrupt handler to decrement the ticks to 0
}




/**************************************************************************/
/*                                                                        */
/*                          Inner Functions                               */
/*                                                                        */
/**************************************************************************/
/*------------------------------------------------------------------------*/
/* Function    : scheduler                                                */
/* Purpose     : start another task                                       */
/* Arguments   :                                                          */
/* Ritorno     :                                                          */
/* Note        : viene chiamata implicitamente pushandone l'indirizzo     */
/*               come ultima istruzione                                   */
/*------------------------------------------------------------------------*/
static void scheduler(void)
{

    BOOL go = TRUE;
	
	
    cli();
	
  
  	PUSH(31);PUSH(30);PUSH(29);PUSH(28);PUSH(27);PUSH(26);PUSH(25);PUSH(24);
	PUSH(23);PUSH(22);PUSH(21);PUSH(20);PUSH(19);PUSH(18);PUSH(17);PUSH(16);
	PUSH(15);PUSH(14);PUSH(13);PUSH(12);PUSH(11);PUSH(10);PUSH( 9);PUSH( 8);
	PUSH( 7);PUSH( 6);PUSH( 5);PUSH( 4);PUSH( 3);PUSH( 2);PUSH( 1);PUSH( 0);
	
	
	
	
	if (currentTask != START_SHOT)
	{
		current = tasks[currentTask];
		current->sreg = SREG;
		current->sp = (BYTE *)(SP + 32);
	}
	/*-- start here task scheduling --*/
    while(go)
    {
	    currentTask++;
		if (currentTask > (Total_Task-1))
			currentTask = 0;		
		current = tasks[currentTask];		
		switch(current->state)
		{		  
		  case WAITEVNT:
		    if((evnt_cnt[event_flag]>0 && current->event == event_flag))
			{			  
			  /*-- event occurrence --*/
			  current->event = 0;
			  go = FALSE;
			}
			if((current->timeOut && current->timeOut < tick_cnt))
			  go = FALSE; // timeout occorrence
		  break;	
          case WAITMSG:
		    if(current->nCntMsg)
			  go = FALSE;
          break;	
		  case READY:		    
			  go = FALSE;
		  break;
		}
	
	}	
	current->state = RUN;
	SREG = current->sreg;
	SP = (WORD)(current->sp - 32);
	
	POP( 0);POP( 1);POP( 2);POP( 3);POP( 4);POP( 5);POP( 6);POP( 7);
	POP( 8);POP( 9);POP(10);POP(11);POP(12);POP(13);POP(14);POP(15);
	POP(16);POP(17);POP(18);POP(19);POP(20);POP(21);POP(22);POP(23);
	POP(24);POP(25);POP(26);POP(27);POP(28);POP(29);POP(30);POP(31);
	asm volatile("reti");

  sei();
  
}

/*---------------------------------------------------------------------------
Function: interrupt SIG_OVERFLOW0 
Purpose:  called when TCNT0 counter is in overflow
----------------------------------------------------------------------------*/
SIGNAL(SIG_OVERFLOW0)
{
  TCNT0 = (BYTE) TICK_RELOAD ;

  tick_cnt++;
  
  /*-- decrementa la variabile, delay only --*/
  if( DelayTimerTicks )
	DelayTimerTicks--;
}
<?show();P("\r\n\r\n// generated by AvrWiz on "+System.DateTime.Now);?>