	list	b=4
;************************************************************************
;																   		*
;	Filename:		UART_receiver.asm										   	*
;																	 	*
;************************************************************************
;																		*
;	Files required: none												*
;																	 	*
;************************************************************************
;																		*
;	Can be used on:														*
;					16F1826											 	*
;																		*
;************************************************************************
;																	 	*
;	Notes:															   	*
;		Pin assignment													*
;	  		Port A											 			*
;				0..3 = Data output 0..3						output		*
;				4    = Data output 8						output		*
;				5    = Not used	/ MCLR						input only	*
;				6..7 = Address input 10..11					output		*
;																	 	*
;	  		Port B 														*
;				0 = Data received 							output		*
;				1 = UART Rx									input		*
;				2 = UART Tx									input		*
;				3 = Data output	10							output		*
;				4..7 = Data output 4..7						output		*
;																	 	*
;************************************************************************

	errorlevel	-302

	list	p=16f1826		; 16F1826 can be used
	#include <p16f1826.inc>	; processor specific variable definitions


		__CONFIG _CONFIG1, _CP_OFF & _WDTE_OFF & _BOREN_ON & _PWRTE_ON & _FOSC_INTOSC & _MCLRE_OFF & _CLKOUTEN_OFF
		__CONFIG _CONFIG2, _WRT_OFF & _STVREN_OFF & _BORV_LO & _LVP_OFF

	__idlocs		0x0000


F_OSC		EQU	.16000000
OSC			EQU	F_OSC/4

;BaudRate	EQU	.4800
BaudRate	EQU	.9600
;BaudRate	EQU	.19200

#define	BufferLength	.16			; Must be a power of 2 and BufferLength <= .32

	cblock	0x0020
		UartRxChar:	.16				; Uart Receiver fifo
	endc

	cblock	0x0070
		Temp						; Scratch register

		Rs232Flags					; Veriable for receive state machine
		Rs232Temp					; Temporary save for WREG 
		Rs232Status					; Temporary save for STATUS

		UartRxChWp					; UART receive buffer write pointer 
		UartRxChRp					; UART receive buffer read  pointer 
		UartRxChNum					; Number of characters in buffer
		Command:2					; Command code from telegram
	endc

;PORTB
bDataReady			EQU	0
;Telegram format; c<command>
; where
;  command: a 12 bit command sent as 3 hexadecimal character.

;Rs232Flags
bWaitCommandPrefix	EQU	7			; Waiting for a commad prefix (c)
bWaitCommand		EQU	6			; Waiting for command characters
;					EQU	5
;					EQU	4
;					EQU	3
;					EQU	2
bNibbleCount1		EQU	1			; Number of command charactes to receive
bNibbleCount0		EQU	0


	org		0x0000					; Reset vector
;***********************************
; Initialization

		goto	Init


	org		0x0004					; Interrupt vector
;***********************************
; Interrupt service routine

Int
;		No register save			; WREG, STATUS, BSR saved by the hardware
									; FSR1 used only in ISR

		banksel	PIE1
		btfss	PIE1,RCIE			; Receive interrupt enabled
		goto	RcIntEnd			; No, leave transmit ISR routine
		banksel	PIR1
		btfss	PIR1,RCIF			; Receive interrupt requested
		goto	RcIntEnd			; No, leave transmit ISR routine

		banksel	UartRxChWp
		movf	UartRxChWp,w		; Put receive char to buffer if no error
		movwf	FSR1L
		movlw	high(UartRxChar)
		movwf	FSR1H				; Get write pointer
		banksel	RCSTA
		movf	RCSTA,w				; Check for error
		andlw	(1 << FERR) | (1 << OERR)
		btfss	STATUS,Z
		goto	UART_ERR			; If error not to store char
		movf	RCREG,w				; Read character
		movwf	INDF1				; Write it to buffer
		incf	FSR1L,w				; Move pointer in circular way
		andlw	BufferLength-1
		iorlw	low(UartRxChar)
		banksel	UartRxChWp
		movwf	UartRxChWp			; Store modified write pointer
		incf	UartRxChNum,f		; Sign one more char in buffer
		goto	RcIntEnd

UART_ERR							; Receive error, reenable receive
		bcf		RCSTA,CREN			; Turn off receiver
		movf	RCREG,w				; Clear receive interrupt flag, drop character
		bsf		RCSTA,CREN			; Turn on  receiver

RcIntEnd
;		No register restore			; WREG, STATUS, BSR restored by the hardware
		retfie						; Exit from ISR

;***********************************
; Initialization

Init
		banksel	OSCCON				; Set frequency of internal oscillator
		movlw	(1<<IRCF3)|(1<<IRCF2)|(1<<IRCF1)|(1<<IRCF0)		; 16MHz
		movwf	OSCCON

		banksel	ANSELA
		clrf	ANSELA				; Set ports to digital mode
		clrf	ANSELB

		banksel	TRISA				; Set port directions
		movlw	0x20
		movwf	TRISA				; RA5 input only pin
		movlw	0x06
		movwf	TRISB				; UART uses RB1 and RB2 

		banksel	LATB				; Clear data ready signal
		clrf	LATA
		clrf	LATB

		banksel	RCSTA				; Set up UART
	if	BaudRate<.9600				; Set up Baud rate
		bcf		TXSTA,BRGH
		movlw	(F_OSC/.64/BaudRate)-1
	else
		bsf		TXSTA,BRGH
		movlw	(F_OSC/.16/BaudRate)-1
	endif

		movwf	SPBRG				; Store Baud divide ratio
		bcf		TXSTA,TXEN			; Disable transmitter
		bsf		RCSTA,CREN			; Enable receiver
		bsf		RCSTA,SPEN			; Enable UART

		banksel	UartRxChWp			; Init buffer pointers
		movlw	UartRxChar			; Get address of buffer
		movwf	UartRxChWp			; Store it in write pointer
		movwf	UartRxChRp			; Store it in read  pointer
		clrf	UartRxChNum			; No character in buffer

		clrf	Rs232Flags			; Initialize receive state machine
		bsf		Rs232Flags,bWaitCommandPrefix

		banksel	PIE1
		bsf		PIE1,RCIE			; Enable receiver interrupt

		banksel	INTCON
		bsf		INTCON,PEIE			; Enable peripheral interrupts
		bsf		INTCON,GIE			; Enable global interrupts

MainLoop:
		call	UART_Receive		; Look for character received
		btfsc	STATUS,C
		goto	MainLoop			; No character
		
		btfsc	Rs232Flags,bWaitCommandPrefix
		goto	CommandPrefix		; Process command prefix

		call	ConvToNibble		; Convert to hex nibble
		btfsc	STATUS,C
		goto	WaitForStart		; Wait for a new telegram if no hex digit character found 


SetCommand							; Process nibble as command
		btfss	Rs232Flags,bWaitCommand
		goto	WaitForStart
		movwf	Temp
		swapf	Command+1,f
		swapf	Command,w			; Shift left 4 bits
		xorwf	Command+1,w
		andlw	0x0F
		xorwf	Command+1,f
		swapf	Command,f
		movlw	0xF0
		andwf	Command,f
		movf	Temp,w
		iorwf	Command,f			; Combine with new nibble

		decf	Rs232Flags,w		; Decrease count of nibbles to received
		xorwf	Rs232Flags,w		; Modify only the bit 1..0
		andlw	(1 << bNibbleCount1)|(1 << bNibbleCount0)
		xorwf	Rs232Flags,f

		movf	Rs232Flags,w		; Get count of niobbles to receive
		andlw	(1 << bNibbleCount1)|(1 << bNibbleCount0)
		btfss	STATUS,Z			; Process telegram if no more nibbles to receive
		goto	MainLoop

		banksel	Command				; Dispatch data to ports
		swapf	Command+1,w
		movwf	Temp
		banksel	LATA				; High nibble to PORTA
		xorwf	LATA,w
		andlw	0xF0
		xorwf	LATA,f
		banksel	Command
		lsrf	Temp,f				; Bit 10 to PORTB3 since PORTA.5 is an input only pin
		lsrf	Temp,w
		banksel	LATA
		xorwf	LATB,w
		andlw	0x08
		xorwf	LATB,f
		movf	Command,w
		banksel	LATA				; Low nibble to PORTA
		xorwf	LATA,w
		andlw	0x0F
		xorwf	LATA,f
		banksel	Command
		movf	Command,w
		banksel	LATB				; Middle nibble to PORTB
		xorwf	LATB,w
		andlw	0xF0
		xorwf	LATB,f
		bsf		LATB,bDataReady		; Sign data ready
		nop
		bcf		LATB,bDataReady
		banksel	Command

WaitForStart						; Wait for a new telegram
		clrf	Rs232Flags
		bsf		Rs232Flags,bWaitCommandPrefix; Command prefix to receive
		goto	MainLoop

CommandPrefix
		xorlw	'c'					; Command prefix received
		btfss	STATUS,Z
		goto	WaitForStart
		clrf	Rs232Flags			; No, wait for a new telegram
		bsf		Rs232Flags,bWaitCommand
		clrf	Command				; Initialize command value
		clrf	Command+1
		movlw	3
		xorwf	Rs232Flags,w		; A command needs 3 nibbles to receive
		andlw	3
		xorwf	Rs232Flags,f
		goto	MainLoop

;*************************************************************
; Convert a hexadecimal character to nibble in WREG
; Input:	character in Rs232Temp
; output:	nibble in W
;			C = 1 if no hexadecimal digit character processed

ConvToNibble						; Convert to nibble
		movf	Rs232Temp,w
		addlw	-'0'
		movwf	Temp
		btfsc	Temp,7
		goto	NoHexDigit			; Character < '0'
		sublw	.9
		btfsc	STATUS,C
		goto	Number				; Character >= '0' and <='9'
		movf	Rs232Temp,w
		addlw	-'A'
		movwf	Temp
		btfsc	Temp,7
		goto	NoHexDigit			; Character < 'A'
		sublw	.5
		btfss	STATUS,C
		goto	NoHexDigit			; Character > 'F'

Number								; A hexadecimal character found
		movf	Rs232Temp,w
		btfsc	Rs232Temp,6			; If character A..F
		addlw	-.7					;  substract 7 form code
		addlw	-'0'				; Substract '0' form code
		andlw	0x0F				; Keep bits 3..0 only
		bcf		STATUS,C			; No error
		return

NoHexDigit
		bsf		STATUS,C			; Sign error
		retlw	0

;**********************************************************************
; Get a character from receive buffer

UART_Receive						; Gets char into w, C == 1 if no character in fifo
		banksel	Rs232Status
		bsf		STATUS,C			; Set no character received
		swapf	STATUS,w			; Save STATUS
		movwf	Rs232Status

		banksel	PIE1
		bcf		PIE1,RCIE			; Disable RC interrupt

		banksel	Rs232Temp
		clrf	Rs232Temp
		banksel	UartRxChNum
		movf	UartRxChNum,w		; Check number of available chars
		btfsc	STATUS,Z
		goto	NO_REC_CHAR			; Jump if no character available

		movf	UartRxChRp,w		; Get read pointer
		movwf	FSR0
		movlw	high(UartRxChar)
		movwf	FSR0H
		movf	INDF0,w				; Read it from buffer
		movwf	Rs232Temp
		incf	FSR0,w				; Move the pointer
		andlw	BufferLength-1		; in circular way
		iorlw	low(UartRxChar)
		movwf	UartRxChRp
		decf	UartRxChNum,f		; Decrement number of available characters
		bcf		Rs232Status,C+4		; Valid character received - !Status was swapped!

NO_REC_CHAR
		banksel	PIE1
		bsf		PIE1,RCIE			; Enable receive interrupt

		banksel	Rs232Status
		swapf	Rs232Status,w		; Restore STATUS
		movwf	STATUS
		movf	Rs232Temp,w			; Get char just received
		return

	END
