/*********************************************************************
 *
 *                 1-Wire Communication Protocol
 *
 *********************************************************************
 * FileName:        1wire.c
 * Dependencies:
 * Processor:       PIC18
 * Complier:        MCC18 v3.13
 * Company:         Microchip Technology, Inc.
 *
 * Software License Agreement
 *
 * Copyright  2004-2007 Microchip Technology Inc.  All rights reserved.
 *
 * Microchip licenses to you the right to use, copy and distribute Software 
 * only when embedded on a Microchip microcontroller or digital signal 
 * controller and used with a Microchip radio frequency transceiver, which 
 * are integrated into your product or third party product (pursuant to the 
 * sublicense terms in the accompanying license agreement).  You may NOT 
 * modify or create derivative works of the Software.  
 *
 *
 * You should refer to the license agreement accompanying this Software for 
 * additional information regarding your rights and obligations.
 *
 * SOFTWARE AND DOCUMENTATION ARE PROVIDED AS IS WITHOUT WARRANTY OF ANY 
 * KIND, EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION, ANY WARRANTY 
 * OF MERCHANTABILITY, TITLE, NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR 
 * PURPOSE. IN NO EVENT SHALL MICROCHIP OR ITS LICENSORS BE LIABLE OR OBLIGATED 
 * UNDER CONTRACT, NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF 
 * WARRANTY, OR OTHER LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR 
 * EXPENSES INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, 
 * PUNITIVE OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF 
 * PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY 
 * THIRD PARTIES (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER 
 * SIMILAR COSTS.
 *
 * Author               Date        Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Sasha     			12/20/07    Original
 ********************************************************************/

 /****** I N C L U D E S **********************************************************/
#include <stdio.h>
#include "onewire.h"
#include "c_onewire.h"
#include "i_onewire.h"
#include "w_onewire.h"

#include "oneWireCRC.h"
#include "mal.h"

//****** V A R I A B L E S ********************************************************/
unsigned char macro_delay = 0;
unsigned char timerDown = 0;
OW_ADDRESSING_MODE OW_addressing_mode = OW_SKIP_ROM_ADDRESS;

void setCustomTimer(unsigned long timeuS);
void setCustomTimerWaitFor(unsigned long timeuS);
void waitForTimer(void);

void init_oneWire(void) {
	#ifdef __dsPIC30F4013__
		//TMR3 Custom
		T3CON = 0;
		PR3 = 3200 - 1;
	 	IFS0bits.T3IF = 0;
		IPC1bits.T3IP = 4;
	 	IEC0bits.T3IE = 1;
		T3CONbits.TCS = 0;
		T3CONbits.TGATE = 0;
		T3CONbits.TCKPS = 0b00;
		T3CONbits.TON = 0;
	#else
	#endif
}

void set_OW_addressing_mode(OW_ADDRESSING_MODE new_OW_addressing_mode) {
	OW_addressing_mode = new_OW_addressing_mode;
}

/**********************************************************************
* Function:        void drive_OW_low (void)
* PreCondition:    None
* Input:		   None	
* Output:		   None	
* Overview:		   Configure the OW_PIN as Output and drive the OW_PIN LOW.	
***********************************************************************/
void drive_OW_low (void) {
	OW_TRIS_PIN = OUTPUT;
	Nop();
	OW_LAT_PIN = LOW;
}

/**********************************************************************
* Function:        void drive_OW_high (void)
* PreCondition:    None
* Input:		   None	
* Output:		   None	
* Overview:		   Configure the OW_PIN as Output and drive the OW_PIN HIGH.	
***********************************************************************/
void drive_OW_high (void) {
	OW_TRIS_PIN = INPUT;
}

/**********************************************************************
* Function:        unsigned char read_OW (void)
* PreCondition:    None
* Input:		   None	
* Output:		   Return the status of OW pin.	
* Overview:		   Configure as Input pin and Read the status of OW_PIN 	
***********************************************************************/
unsigned char read_OW (void) {
	unsigned char read_data = 0;
	OW_TRIS_PIN = INPUT;
	if (HIGH == OW_PORT_PIN) {
		read_data = SET;
	} else {
		read_data = CLEAR;
	}
	return read_data;
}

/**********************************************************************
* Function:        unsigned char OW_reset_pulse(void)
* PreCondition:    None
* Input:		   None	
* Output:		   Return the Presense Pulse from the slave.	
* Overview:		   Initialization sequence start with reset pulse.
*				   This code generates reset sequence as per the protocol
***********************************************************************/
unsigned char OW_reset_pulse(void) {
	unsigned char presence_detect = 2;
  	drive_OW_low(); 				// Drive the bus low
	setCustomTimerWaitFor(463);	  			// delay 480 microsecond (us) need to substract every time 2*12 to be correct with timing
	//not that setCustomTimerWaitFor will release bus so it goes to high, second call is only timing, side effect is that it also release bus
	//but since it is already released it is no problem
	setCustomTimerWaitFor(64);				// delay 70 microsecond (us) 

	if (!read_OW()) {
		presence_detect--;			//Sample for presence pulse from slave
	}
 	setCustomTimerWaitFor(386);	  			// delay 410 microsecond (us)
	
	//drive_OW_high ();		    	// Release the bus this is done by setCustomTimerWaitFor

	if (read_OW()) {
		presence_detect--;			//Sample for presence pulse from slave
	}
	
	return presence_detect;
}

/**********************************************************************
* Function:        void OW_write_bit (unsigned char write_data)
* PreCondition:    None
* Input:		   Write a bit to 1-wire slave device.
* Output:		   None
* Overview:		   This function used to transmit a single bit to slave device.
*				   
***********************************************************************/
void OW_write_bit (unsigned char write_bit) {
	if (write_bit) {
		//writing a bit '1'
		lock_isr();
		drive_OW_low(); 				// Drive the bus low
		wait(DELAY_6Us);				// delay 6 microsecond (us)
		drive_OW_high ();  				// Release the bus
		unlock_isr();
		setCustomTimerWaitFor(58);				// delay 64 microsecond (us) 
	} else {
		//writing a bit '0'
		drive_OW_low(); 				// Drive the bus low
		setCustomTimerWaitFor(54);				// delay 60 microsecond (us) 
		setCustomTimerWaitFor(4);				// delay 10 microsecond (us) 
	}
}	

/**********************************************************************
* Function:        unsigned char OW_read_bit (void)
* PreCondition:    None
* Input:		   None
* Output:		   Return the status of the OW PIN
* Overview:		   This function used to read a single bit from the slave device.
*				   
***********************************************************************/
unsigned char OW_read_bit (void) {
	unsigned char read_data; 
	//reading a bit 

	lock_isr();

	drive_OW_low(); 						// Drive the bus low
	wait(DELAY_6Us);						// delay 6 microsecond (us)
	drive_OW_high();  						// Release the bus
	wait(DELAY_9Us);						// delay 9 microsecond (us)

	read_data = read_OW();					//Read the status of OW_PIN

	unlock_isr();
	setCustomTimerWaitFor(49);				// delay 55 microsecond (us) 

	return read_data;
}

/**********************************************************************
* Function:        void OW_write_byte (unsigned char write_data)
* PreCondition:    None
* Input:		   Send byte to 1-wire slave device
* Output:		   None
* Overview:		   This function used to transmit a complete byte to slave device.
*				   
***********************************************************************/
void OW_write_byte (unsigned char write_data) {
	unsigned char loop = 0;
	for (loop = 0; loop < 8; loop++) {
		OW_write_bit(write_data & 0x01); 	//Sending LS-bit first
		write_data >>= 1;					// shift the data byte for the next bit to send
	}	
}	

/**********************************************************************
* Function:        unsigned char OW_read_byte (void)
* PreCondition:    None
* Input:		   None
* Output:		   Return the read byte from slave device
* Overview:		   This function used to read a complete byte from the slave device.
*				   
***********************************************************************/
unsigned char OW_read_byte (void) {
	unsigned char loop = 0;
	unsigned char result = 0;
	
	for (loop = 0; loop < 8; loop++) {
		result >>= 1; 				// shift the result to get it ready for the next bit to receive
		if (OW_read_bit()) {
			result |= 0x80;				// if result is one, then set MS-bit
		}
	}
	return result;					
}	

void OW_address_device(OW_device ow_device) {
	switch (OW_addressing_mode) {
		case OW_SKIP_ROM_ADDRESS : {
			OW_write_byte(OW_SKIP_ROM);
			break;
		}
		case OW_MATCH_ROM_ADDRESS : {
			unsigned char i = 0;
			OW_write_byte(OW_MATCH_ROM);
			for (i = 0; i < 8; i++) {
				OW_write_byte(ow_device.byte[i]);
			}
			break;
		}
	}
}

 unsigned char OW_read_rom(OW_device *ow_device) {
	unsigned char result = 0;
	if (ow_device != NULL) {
		if (!OW_reset_pulse()) {
			OW_write_byte(OW_READ_ROM);
			ow_device->value.family = OW_read_byte();
			ow_device->value.serial_number[0] = OW_read_byte();
			ow_device->value.serial_number[1] = OW_read_byte();
			ow_device->value.serial_number[2] = OW_read_byte();
			ow_device->value.serial_number[3] = OW_read_byte();
			ow_device->value.serial_number[4] = OW_read_byte();
			ow_device->value.serial_number[5] = OW_read_byte();
			ow_device->value.crc = OW_read_byte();
			{
				unsigned char myCRC = 0;
				unsigned char receivedCRC = ow_device->value.crc;
				myCRC =  calculateOneWireCRC(ow_device->byte, 7);
				if (myCRC == receivedCRC) {
					result = 1;
				} else {
				}
			}
		} else {
		}
	} else {
	}
	return result;
}

void setCustomTimer(unsigned long timeuS) {
	#ifdef __dsPIC30F4013__
		unsigned long pr3Value = timeuS;
		timerDown = 0;
		pr3Value *= PR3CONVERTERVALUE;
		pr3Value -= 1;
		PR3 = pr3Value;
		TMR3 = 0;
		T3CONbits.TON = 1;
	#else
		wait(timeuS);
	#endif
}

void setCustomTimerWaitFor(unsigned long timeuS) {
	#ifdef __dsPIC30F4013__
		setCustomTimer(timeuS);
		waitForTimer();
	#else
		wait(timeuS);
	#endif
}

void waitForTimer(void) {
	#ifdef __dsPIC30F4013__
		while (1) {
			unsigned char temp = 0;
			//lock_isr();
			temp = timerDown;
			//unlock_isr();
			if (temp) {
				break;
			}
		}
	#else
	#endif
}

void isr_appCustom(void) {
	#ifdef __dsPIC30F4013__
		T3CONbits.TON = 0;
		timerDown = 1;
		drive_OW_high();  					// Release the bus
	#else
	#endif
}

/********************************************************************************************
                  E N D     O F     1 W I R E . C  
*********************************************************************************************/
