/************************************************************************************
 *  Launchpad demprogram: ADC_simple
 *
 * Egyszer mrs ADC-vel, ahol a referencia a kb.3,5 V-os tpfeszltsg.
 * Az A5 (P1.5 lb) analg bemenetre 0 - 3,5 V kztti jelet vigynk.
 * Emellett megmrjk a fl tpfeszltsget (Chan 11) s a bels
 * hmr jelt (Chan 10).
 *
 * A mrs eredmnyeit egyirny szoftveres UART kezelssel (csak adatklds)
 * kiratjuk. Paramterek: 9600 baud, 8 adatbit, nincs paritsbit, 2 stopbit.
 * A program elejn az ADC-re vonatkoz gyri kalibrcis adatokat
 * is kiratjuk, amelyeket a tovbbi programokban felhasznlhatunk.
 *
 * A CPU gyrilag kalibrlt 1 MHz-es DCO rajelet hasznlja.
 * ACLK = n/a, MCLK = SMCLK = DCO 1 MHz
 *
 * Hardver kvetelmnyek:
 *  - Launchpad MSP430G2xxx mikrovezrlvel
 *  - Az jabb kiads (v1.5) krtyn az RXD,TXD tktseket
 *    SW llsba kell helyezni (a tbbi tktssel prhuzamosan)
 *
 *
 *                MSP430G2xxx
 *             -----------------
 *         /|\|              XIN|-
 *          | |                 |
 *          --|RST          XOUT|-
 *            |                 |
 *     TxD <--|P1.1         P1.5|<-- 0-3,5 V
 *
 *
 * Megjegyzsek:
 *  - A programban hivatkozott gyri kalibrcis adatok
 *    az MSP430G2231 mikrovezrlbe nincsenek megadva, ennl
 *    a mikrovezrlnl hagyjuk el a kalibrcis adatokat
 *    lilistz programsorokat!
 *
 *  - A hmr kalibrcis adataihoz a fejlc llomnyokban nincs
 *    definilva szimbolikus nv. Az albbi sorokat adjuk hozz az
 *    io430g2452.h s io430g2553.h llomnyokhoz:
 *  __no_init unsigned __READ int CAL_ADC_25T85 @ 0x10EA;
 *  __no_init unsigned __READ int CAL_ADC_25T30 @ 0x10E8;
 *  __no_init unsigned __READ int CAL_ADC_25VREF_FACTOR @ 0x10E6;
 *  __no_init unsigned __READ int CAL_ADC_15T85 @ 0x10E4;
 *  __no_init unsigned __READ int CAL_ADC_15T30 @ 0x10E2;
 *  __no_init unsigned __READ int CAL_ADC_15VREF_FACTOR @ 0x10E0;
 *  __no_init unsigned __READ int CAL_ADC_OFFSET @ 0x10DE;
 *  __no_init unsigned __READ int CAL_ADC_GAIN_FACTOR @ 0x10DC;
 *
 *  I. Cserny
 *  MTA ATOMKI, Debrecen
 *  Sept 2012
 *  Fejleszti krnyezet: Texas Instruments Code Composer Studio 5.20.1
 ************************************************************************************/

#include "msp430.h"
#include "stdint.h"

#define TXD       BIT1                 // TXD a P1.1 lbon

/*-------------------------------------------------------------
 * Egyszeri mrs egy ADC csatornban, AVCC a referencia
 *-------------------------------------------------------------
 * chan - csatornavlaszt bitek (a csatorna sorszma << 12)
 */
uint16_t ADC_single_meas(uint16_t chan) {
  ADC10CTL0 &= ~ENC;                   //Az ADC letiltsa jrakonfigurlshoz
  ADC10CTL0 = ADC10SHT_3               //mintavtel: 64 rats
             | ADC10ON                 //Az ADC bekapcsolsa
             | SREF_0;                 //VR+ = AVCC s VR- = AVSS
  ADC10CTL1 = ADC10SSEL_0 + chan;      //csatorna = 'chan', ADC10OSC az rajel
  ADC10CTL0 |= ENC + ADC10SC;          //Konverzi engedlyezse s indtsa
  while (ADC10CTL1 & BUSY);            //Vrakozs a konverzi vgre
  return ADC10MEM;                     //Visszatrsi rtk a konverzi eredmnye
}

/**------------------------------------------------------------
 *   Ksleltet eljrs (1 - 65535 ms)
 *-------------------------------------------------------------
 * delay - a ksleltets ms egysgben megadva
 */
void delay_ms(uint16_t delay) {
  uint16_t i;
  for(i=0; i<delay; i++) {             //"delay"-szer ismteljk
    __delay_cycles(1000);              //1 ms ksleltets
  }
}

/**------------------------------------------------------------
 *   Egy karakter kikldse a soros portra
 *   SW_UART: 9600 bit/s, 8, N, 2 formtum
 *   DCO = 1 MHz (Bitid = 104.167 usec)
 *-------------------------------------------------------------
 * c - a kikldeni kvnt karakter kdja
 */
void sw_uart_putc(char c) {
  uint8_t i;
  uint16_t TXData;
  TXData = (uint16_t)c | 0x300;        //Add 2 stop bits (mark)
  TXData = TXData << 1;                //Add 1 start bits (space)
  for(i=0; i<11; i++) {
    if(TXData & 0x0001) {              //Soron kvetkez bit vizsglata
      P1OUT |= TXD;                    //Ha '1'
    } else {
      P1OUT &= ~TXD;                   //Ha '0'
    }
    TXData = TXData >> 1;              //Adatregiszter lptets jobbra
    __delay_cycles(89);                //<== Itt kell hangolni!
  }
  P1OUT |= TXD;                        //Az alaphelyzet: mark
}

/**------------------------------------------------------------
 *  Karakterfzr kirsa a soros portra
 *-------------------------------------------------------------
 * p_str - karakterfzr mutat (nullval lezrt stringre mutat)
 */
void sw_uart_puts(char* p_str) {
  char c;
  while ((c=*p_str)) {
      sw_uart_putc(c);
      p_str++;
  }
}

/**------------------------------------------------------------
 * Eljel nlkli egsz szm kirsa hexadecimlis formban
 *-------------------------------------------------------------
 * t - a kirand szm (uint16_t tpus)
 */
void sw_uart_out4hex(uint16_t t)
{
  char c;
  c=(char)((t>>12) & 0x0F);
  if (c>9) c+=7;
  sw_uart_putc(c+'0');
  c=(char)((t>>8) & 0x0F);
  if (c>9) c+=7;
  sw_uart_putc(c+'0');
  c=(char)((t>>4) & 0x0F);
  if (c>9) c+=7;
  sw_uart_putc(c+'0');
  c=(char)(t & 0x0F);
  if (c>9) c+=7;
  sw_uart_putc(c+'0');
}

/**------------------------------------------------------------
 * Decimlis kirats adott szm tizedesjegyre.
 *-------------------------------------------------------------
 * data - a kirand szm (eljelesen)
 * ndigits - a kirand tizedesek szma
 */
void sw_uart_outdec(int32_t data, uint8_t ndigits) {
  static char sign, s[12];
  uint8_t i;
  i=0; sign='+';
  if(data<0) { sign='-'; data = -data;}
    do {
      s[i]=data%10 + '0';
      data=data/10;
      i++;
      if(i==ndigits) {s[i]='.'; i++;}
    } while(data>0);
    sw_uart_putc(sign);
    do {
      sw_uart_putc(s[--i]);
    } while(i);
}

void main(void) {
uint16_t data;
int32_t temp;
  WDTCTL = WDTPW + WDTHOLD;            //Letiltjuk a watchdog idztt
  DCOCTL = CALDCO_1MHZ;                // DCO belltsa a gyrilag kalibrlt
  BCSCTL1 = CALBC1_1MHZ;               // 1 MHz-es frekvencira
  P1DIR |= TXD;                        //TXD legyen digitlis kimenet
  P1OUT |= TXD;                        //TXD alaphelyzete: mark
//--- P1.3 bels felhzs engedlyezse -----------------------
  P1DIR &= ~BIT3;                      //P1.3 legyen digitlis bemenet
  P1OUT |= BIT3;                       //Felfel hzzuk, nem lefel
  P1REN |= BIT3;                       //Bels felhzs engedlyezse
//--- Analg csatornk engedlyezse --------------------------
  ADC10AE0 |= BIT5;                    //P1.5 legyen analg bemenet
//--- dvzl sor kiratsa (rjuk t az MCU nevt!) ----------
  sw_uart_puts("\r\nADC_simple program on MSP430G2553");
//--- Az ADC gyri kalibrcis adatainak kilistzsa ----------
  sw_uart_puts("\r\nCAL_ADC_25T85 = ");
  sw_uart_out4hex(CAL_ADC_25T85);
  sw_uart_puts("\r\nCAL_ADC_25T30 = ");
  sw_uart_out4hex(CAL_ADC_25T30);
  sw_uart_puts("\r\nCAL_ADC_25VREF_FACTOR = ");
  sw_uart_out4hex(CAL_ADC_25VREF_FACTOR);
  sw_uart_puts("\r\nCAL_ADC_15T85 = ");
  sw_uart_out4hex(CAL_ADC_15T85);
  sw_uart_puts("\r\nCAL_ADC_15T30 = ");
  sw_uart_out4hex(CAL_ADC_15T30);
  sw_uart_puts("\r\nCAL_ADC_15VREF_FACTOR = ");
  sw_uart_out4hex(CAL_ADC_15VREF_FACTOR);
  sw_uart_puts("\r\nCAL_ADC_OFFSET = ");
  sw_uart_out4hex(CAL_ADC_OFFSET);
  sw_uart_puts("\r\nCAL_ADC_GAIN_FACTOR = ");
  sw_uart_out4hex(CAL_ADC_GAIN_FACTOR);
  sw_uart_puts("\r\n");

  while(1) {
    delay_ms(2000);                    //2 s vrakozs
//--- Az A5 bemenjelnek mrse s kiratsa ------------------
    data = ADC_single_meas(INCH_5);
    sw_uart_puts("chan 5 = ");
    sw_uart_out4hex(data);
    sw_uart_puts(" chan 11 = ");
//--- Az A11 csatorna (VDD/2) mrse s kiratsa --------------
    data = ADC_single_meas(INCH_11);
    sw_uart_out4hex(data);
    sw_uart_puts(" chan 10 = ");
//--- Az A10 csatorna (hmr) mrse s kiratsa -------------
    data = ADC_single_meas(INCH_10);
    sw_uart_out4hex(data);
    temp = ((uint32_t)data*2467L - 711040L) >> 8; //tizedfok egysgben
    sw_uart_puts(" temp = ");
    sw_uart_outdec(temp,1);            //kirats egy tizedesjegyre
    sw_uart_puts(" C\r\n");
  }
}
