/***********************************************************

	LCD-Routinen
	fr Samsung 2138a mit M50530-Controller
	24 x 8 Zeichen

	Copyright 2003-04 by Daniel Jelkmann


	Zur Ansteuerung wird der 4Bit-Mode verwendet.
	Neben den 4 Datenleitungen (Data0-3) werden 4 weitere
	Steuerleitungen (Read/Write, Execute, OC1, OC2)
	bentigt. Damit werden am AVR 8 Pins bentigt.

	
	Anschluss:
	
	Alle 8 LCD-Signale mssen an einem Port des AVR
	angeschlossen werden. Gem folgender Tabelle mssen
	die 4 Datenleitungen an den niederwertigsten Bits
	anliegen. Die Reihenfolge der restlichen 4 Steuerleitungen
	und der verwendete Port ist ber die Konstanten im 
	Quellcode festgelegt und muss ggf. angepasst werden.
	Die restlichen LCD-Signale sind in folgender Tabelle
	zu finden.

		LCD	Beschreibung		Anschluss
		-------------------------------------------------------------------------
		Pin 01	Masse			Masse
		Pin 02	Data0			nicht angeschlossen
		Pin 03	Data1			nicht angeschlossen
		Pin 04	Data2			nicht angeschlossen
		Pin 05	Data3			nicht angeschlossen
		Pin 06	Data4			AVR Port Bit 0
		Pin 07	Data5			AVR Port Bit 1
		Pin 08	Data6			AVR Port Bit 2
		Pin 09	Data7			AVR Port Bit 3
		Pin 10	EX (Execute)		AVR Port Bit 4*
		Pin 11	R/W (Read/Write)	AVR Port Bit 5*
		Pin 12	OC2				AVR Port Bit 7*
		Pin 13	OC1				AVR Port Bit 6*
		Pin 14	Kontrast-Regelung, ungefhr 8,2 Volt, am besten mittels Poti anschlieen
		Pin 15	Versorgungsspannung, +5 Volt
		Pin 16	Masse			Masse


	
	* Die Reihenfolge dieser Leitungen kann ber die
	Konstanten im Quellcode angepasst werden.
	
	
	Es wird keine Gewhr fr die Vollstndigkeit, Korrektheit,
	Funktionsfhigkeit oder fr sonstige Eigenschaften des
	Codes bernommen. Haftung ausgeschlossen.
	
*************************************************************/


#include <avr/io.h>
//#include <avr/delay.h>
#include <avr/pgmspace.h>

#include "lcd-m50530.h"

#include <stdlib.h>


/*
// wartet die angegebene Zeit in Millisekunden
// F_MCU muss entsprechend gesetzt sein
// (bentigt delay.h)
void delay(const unsigned int ms)
{
	for (unsigned char i = ms; i > 0; --i)
		// 4 cycles per loop * 250 = 1.000 cycles; 1.000 x F_MCU = 1 ms
		_delay_loop_2(250*F_MCU);
}*/


// sendet den angelegten Befehl zum LCD
// setzt kurzzeitig das EX-Signal und nimmt es anschlieend wieder zurck
void LCD_execute(void)
{
	delay_short();
	// EX-Signal setzen
	LCD_PORT |= (1 << LCD_EX_PIN);
	delay_short();
	// EX-Signal lschen
	LCD_PORT &= ~(1 << LCD_EX_PIN);
	delay_short();
}


// wartet solange bis das Busy-Flag nicht mehr gesetzt ist
// und das LCD weitere Befehle entgegennimmt
void LCD_waitReady(void)
{
	/*
		READ BUSY FLAG & FUNCTION FLAGS (RB)
		input:
			OC1		0
			OC2		0
			RW		1
		output:
			DB7		busy-flag (1 = busy)
	*/

	// Datenleitungen auf Eingang schalten
	// dazu die Bits der Datenleitungen auf 0 setzen
	LCD_DDR &= 240;

	// RB-Befehl senden
	//LCD_PORT = (1 << LCD_OC1_PIN) | (1 << LCD_OC2_PIN) | (1 << LCD_RW_PIN);
	LCD_PORT = 1 << LCD_RW_PIN;
	
	unsigned char flags = 0;
	while (1)
	{
		do
		{
			delay_short();
			// EX-Signal setzen
			LCD_PORT |= (1 << LCD_EX_PIN);
			delay_short();
			// flags einlesen
			flags = LCD_PIN;
			// EX-Signal lschen
			LCD_PORT &= ~(1 << LCD_EX_PIN);
		} while (bit_is_set(flags, 3));
		
		if (flags & 6)
			// bits 1 oder 2 gesetzt, also entweder sind wie im 8bit-mode oder wir haben gerade den zweiten Teil im 4bit-mode gelesen
			break;
	}
		
	
/*	unsigned char busy;
	do
	{
		delay_short();
		LCD_execute();
		// busy-Signal einlesen
		busy = bit_is_set(LCD_PIN, 3);
		LCD_execute();
	} while (busy);
*/
	
	// alle Leitungen als Ausgnge schalten
	LCD_DDR = 0xFF;
}


// bertrgt das bergebene Byte an das LCD
void LCD_sendByte(const unsigned char byte, unsigned char control)
{
	LCD_waitReady();
	// die unteren 5 Bits im Control-Byte ausblenden
	control &= 224;
	// oberes Nibble bertragen
	LCD_PORT = (byte >> 4) | control;
	LCD_execute();
	// unteres Nibble bertragen
	LCD_PORT = (byte & 15) | control;
	LCD_execute();
}



// lscht die Anzeige und setzt die Display + Cursor-Adresse auf 0,0
void LCD_clear(void)
{
	/*
		CLEAR DISPLAY, MOVE DISPLAY/CURSOR ADDRESS HOME
		input:
			OC1		0
			OC2		0
			RW		0
			DB7-DB1	0
			DB0		1
	*/
	LCD_sendByte(1, 0);	// DB0
	delay(2);
}


// setzt 4-Bit-Mode und initialisiert das LCD
// muss als erstes aufgerufen werden, bevor das LCD angesteuert werden kann
void LCD_init(void)
{
	// alle Leitungen auf Ausgang schalten
	LCD_DDR = 0xFF;

	// auf 4-Bit-Mode umschalten
	/*
		SET FUNCTION MODE
		input:
			OC1		0
			OC2		0
			RW		0
			DB7,DB6	1
			DB5		I/O 8/4	(1 = 8 bit, 0 = 4 bit)
			DB4		FONT 8/12 (1 = 5x8 fonts, 0 = 5x12 fonts)
			DB3,DB2	DUTY
			DB1,DB0	RAM
	*/
	LCD_sendByte(216, 0);	// DB7, DB6, DB4, DB3
	LCD_clear();
}


// schreibt die bergebene Zeichenkette an die aktuelle Cursor-Position
void LCD_write(const unsigned char * c)
{
    while (*c) 
      LCD_writeChar(*c++);
}


// schreibt die bergebene Zeichenkette an die aktuelle Cursor-Position (fr String im ROM)
void LCD_write_P(const unsigned char * progmem_string)
{
	register unsigned char c;
	while ((c = pgm_read_byte(progmem_string++)))
		LCD_writeChar(c);
}


// setzt den Cursor an die angegebene Position (y = Zeile, x = Spalte)
void LCD_setCursorPos(const unsigned char y, const unsigned char x)
{
	// Adresse aus x und y berechnen
	// Adressen sind wie folgt: Zeile 0 = 0; Zeile 1 = 64; Zeile 2 = 128; Zeile 3 = 192; Zeile 4 = 0+24; Zeile 5 = 64+24; Zeile 6 = 128+24; Zeile 7 = 192+24
	unsigned char address = x + ((y % 4) << 6);
	if (y > 3)
		address += 24;
	LCD_sendByte(address, (1<<LCD_OC1_PIN) | (1<<LCD_OC2_PIN));
}
