/***********************************************************

	LCD-Routinen
	fr Samsung 2138a mit M50530-Controller
	24 x 8 Zeichen

	Copyright 2003 by Daniel Jelkmann


	Zur Ansteuerung wird der 4Bit-Mode verwendet.
	Neben den 4 Datenleitungen (Data0-3) werden 4 weitere
	Steuerleitungen (Read/Write, Execute, OC1, OC2)
	bentigt. Damit werden am AVR 8 Pins bentigt.

	
	Anschluss:
	
	Alle 8 LCD-Signale mssen an einem Port des AVR
	angeschlossen werden. Gem folgender Tabelle mssen
	die 4 Datenleitungen an den niederwertigsten Bits
	anliegen. Die Reihenfolge der restlichen 4 Steuerleitungen
	und der verwendete Port ist ber die Konstanten im 
	Quellcode festgelegt und muss ggf. angepasst werden.
	Die restlichen LCD-Signale sind in folgender Tabelle
	zu finden.

		LCD	Beschreibung		Anschluss
		-------------------------------------------------------------------------
		Pin 01	Masse			Masse
		Pin 02	Data0			nicht angeschlossen
		Pin 03	Data1			nicht angeschlossen
		Pin 04	Data2			nicht angeschlossen
		Pin 05	Data3			nicht angeschlossen
		Pin 06	Data4			AVR Port Bit 0
		Pin 07	Data5			AVR Port Bit 1
		Pin 08	Data6			AVR Port Bit 2
		Pin 09	Data7			AVR Port Bit 3
		Pin 10	EX (Execute)		AVR Port Bit 5*
		Pin 11	R/W (Read/Write)	AVR Port Bit 4*
		Pin 12	OC2				AVR Port Bit 7*
		Pin 13	OC1				AVR Port Bit 6*
		Pin 14	Kontrast-Regelung, ungefhr 8,2 Volt, am besten mittels Poti anschlieen
		Pin 15	Versorgungsspannung, +5 Volt
		Pin 16	Masse			Masse


	
	* Die Reihenfolge dieser Leitungen kann ber die
	Konstanten im Quellcode angepasst werden.
	
	
	Es wird keine Gewhr fr die Vollstndigkeit, Korrektheit,
	Funktionsfhigkeit oder fr sonstige Eigenschaften des
	Codes bernommen. Haftung ausgeschlossen.
	
*************************************************************/

#ifndef LCD_M50530_H
#define LCD_M50530_H

#include <avr/delay.h>


//#define F_MCU 				5		/* Takt-Frequenz in MHz */


// wartet die angegebene Zeit in Millisekunden
// XTAL_CPU muss entsprechend gesetzt sein
// (bentigt delay.h)
//extern void delay(unsigned int ms);
#define delay(ms)		for (unsigned int i = ms; i > 0; --i)	\
							_delay_loop_2(XTAL_CPU / 4000);
//							_delay_loop_2(250*F_MCU);


/****************************************************************
Ggf. hier den Port anpassen, an dem das LCD angeschlossen ist! 
****************************************************************/
// Data-Direction-Register, am dem das LCD angeschlossen ist
#define LCD_DDR DDRA
// Port-Register, am dem das LCD angeschlossen ist
#define LCD_PORT PORTA
// Pin-Register, am dem das LCD angeschlossen ist
#define LCD_PIN PINA


/*************************************************
Ggf. hier die Bits der 4 Steuerleitungen anpassen!
*************************************************/
// Die Datenleitungen des LCDs mssen an den Pins 0-3 anliegen (Data0 auf Pin0, Data1 auf Pin1 usw.)
// Die anderen 4 Steuerleitungen sind hier einzustellen!
// Port-Pin, an dem der RW-Pin des LCDs liegt
#define LCD_RW_PIN 4
// Port-Pin, an dem der EX-Pin des LCDs liegt
#define LCD_EX_PIN 5
// Port-Pin, an dem der OC1-Pin des LCDs liegt
#define LCD_OC1_PIN 6
// Port-Pin, an dem der OC2-Pin des LCDs liegt
#define LCD_OC2_PIN 7


// ein paar Konstanten zur besseren Lesbarkeit des Quelltextes :-)
#define LCD_DISPLAY_ON		16
#define LCD_CURSOR_ON		8
#define LCD_UNDERLINE		4
#define LCD_CURSOR_BLINK	2
#define LCD_CHARACTER_BLINK	1


#define LCD_CURSOR_ADDRESS_NO_CHANGE	64
#define LCD_CURSOR_ADDRESS_INC_RD		72
#define LCD_CURSOR_ADDRESS_INC_WD		80
#define LCD_CURSOR_ADDRESS_INC_RDWD		88
#define LCD_CURSOR_ADDRESS_DEC_RD		104
#define LCD_CURSOR_ADDRESS_DEC_WD		112
#define LCD_CURSOR_ADDRESS_DEC_RDWD		120

#define LCD_DISPLAY_START_ADDRESS_NO_CHANGE	64
#define LCD_DISPLAY_START_ADDRESS_INC_RD	65
#define LCD_DISPLAY_START_ADDRESS_INC_WD	66
#define LCD_DISPLAY_START_ADDRESS_INC_RDWD	67
#define LCD_DISPLAY_START_ADDRESS_DEC_RD	69
#define LCD_DISPLAY_START_ADDRESS_DEC_WD	70
#define LCD_DISPLAY_START_ADDRESS_DEC_RDWD	71


#define LCD_SPECIAL_AE						(char)144	// , dezimal 144, oktal 220
#define LCD_SPECIAL_OE						(char)145	// , dezimal 145, oktal 221
#define LCD_SPECIAL_UE						(char)197	// , dezimal 197, oktal 305
#define LCD_SPECIAL_ae						(char)160	// , dezimal 160, oktal 240
#define LCD_SPECIAL_oe						(char)161	// , dezimal 161, oktal 241
#define LCD_SPECIAL_ue						(char)17	// , dezimal 17, oktal 21
#define LCD_SPECIAL_ss						(char)163	// ,  dezimal 163, oktal 243


// wartet einen kurzen Moment, 4 cycles per loop * 8 = 32 cycles, macht bei 16 MHz 2 s
#define delay_short()	_delay_loop_2(8)


// sendet den angelegten Befehl zum LCD
// setzt kurzzeitig das EX-Signal und nimmt es anschlieend wieder zurck
extern void LCD_execute(void);


// wartet solange bis das Busy-Flag nicht mehr gesetzt ist
// und das LCD weitere Befehle entgegennimmt
extern void LCD_waitReady(void);


// bertrgt das bergebene Byte an das LCD
extern void LCD_sendByte(const unsigned char byte, unsigned char control);


// setzt 4-Bit-Mode und initialisiert das LCD
// muss als erstes aufgerufen werden, bevor das LCD angesteuert werden kann
extern void LCD_init(void);

// lscht die Anzeige und setzt die Display + Cursor-Adresse auf 0,0
extern void LCD_clear(void);


// setzt den Entry-Mode des LCDs, also die automatische Erhhung der Adresse nach einem Lese- oder Schreibbefehl
// man verwende die oben deklarierten Konstanten ;)
#define LCD_setEntryMode(mode)			LCD_sendByte(mode, 0)


// schreibt das bergebene Zeichen an die aktuelle Cursor-Position
#define LCD_writeChar(c)				LCD_sendByte(c, 1 << LCD_OC2_PIN)


// schreibt die bergebene Zeichenkette an die aktuelle Cursor-Position
extern void LCD_write(const unsigned char * c);

// schreibt die bergebene Zeichenkette an die aktuelle Cursor-Position (fr String im ROM)
extern void LCD_write_P(const unsigned char * progmem_string);


// setzt den Display-Modus
// man verwende die oben deklarierten Konstanten ;)
#define LCD_setDisplay(mode)			LCD_sendByte(mode | 32, 0)


// setzt den Cursor an die angegebene Adresse
#define LCD_setCursorAddress(address)	LCD_sendByte(address, (1<<LCD_OC1_PIN) | (1<<LCD_OC2_PIN))


// setzt den Cursor an die angegebene Position (y = Zeile, x = Spalte)
extern void LCD_setCursorPos(const unsigned char y, const unsigned char x);

#endif
