/*
 * ATtiny2313_Thermostat.c
 *
 * Created: 10/28/2011 7:01:10 PM
 *  Author: Phil Levchenko
 *	 Email: phil at jumperone dot com
 *    Site: www.JumperOne.com
 * License: Licensed under Creative Commons Attribution-ShareAlike License 3.0
 *          http://creativecommons.org/licenses/by-sa/3.0/
 *
 *    This firmware comes WITHOUT ANY WARRANTIES and may contain MISTAKES,
 *    so use it AT YOUR OWN RISK!!!
 *
 *    Don't forget to change fuses on your MCU to work from
 *    internal 8MHz RC oscillator and eliminate frequency
 *    division by 8 (which is set by default).
 */ 


#define F_CPU 8000000UL // MCU Frequency (mostly used by delay.h)

#include <avr/io.h>
#include <util/delay.h>
#include <inttypes.h>
#include "types.h"


#ifndef _BV
#define _BV(x) (1<<x)
#endif

#define BUTTON_1_PIN	PIND
#define BUTTON_1_PN		PD4
#define BUTTON_2_PIN	PIND
#define BUTTON_2_PN		PD5
#define BUTTON_3_PIN	PIND
#define BUTTON_3_PN		PD6

#define RELAY_PORT		PORTD
#define RELAY_PN		PD0

#define LED_PORT		PORTB
#define LED_PN			PB2

#define RELAY_ON() {RELAY_PORT |= _BV(RELAY_PN);}
#define RELAY_OFF() {RELAY_PORT &= ~_BV(RELAY_PN);}
#define RELAY_TOGGLE() { RELAY_PORT ^= _BV(RELAY_PN);}

#define LED_ON() {LED_PORT |= _BV(LED_PN);}
#define LED_OFF() {LED_PORT &= ~_BV(LED_PN);}
#define LED_TOGGLE() {LED_PORT ^= _BV(LED_PN);}

#define TSENS_PORT			PORTD
#define TSENS_PIN			PIND
#define TSENS_DDR			DDRD
#define TSENS_PN			PD1
#define TSENS_BIT			(TSENS_PIN & _BV(TSENS_PN))
#define TSENS_PULL_LOW()	{ TSENS_DDR |= _BV(TSENS_PN);  }
#define TSENS_LISTEN()		{ TSENS_DDR &= ~_BV(TSENS_PN); }

#define DISP_DELAY 2

#define DISP_OFF() {PORTB &= 0x1F;}
#define DISP_DIGIT1() {PORTB &= 0x1F; PORTB |= 0x20;}
#define DISP_DIGIT2() {PORTB &= 0x1F; PORTB |= 0x40;}
#define DISP_DIGIT3() {PORTB &= 0x1F; PORTB |= 0x80;}
#define DISP_DIGIT_ALL() {PORTB |= 0xE0;}

#define DISP_A_PORT		PORTB
#define DISP_A_PN		_BV(PB4)
#define DISP_B_PORT		PORTA
#define DISP_B_PN		_BV(PA1)
#define DISP_C_PORT		PORTB
#define DISP_C_PN		_BV(PB0)
#define DISP_D_PORT		PORTB
#define DISP_D_PN		_BV(PB1)
#define DISP_E_PORT		PORTA
#define DISP_E_PN		_BV(PA0)
#define DISP_F_PORT		PORTB
#define DISP_F_PN		_BV(PB3)
#define DISP_G_PORT		PORTD
#define DISP_G_PN		_BV(PD3)
#define DISP_DP_PORT	PORTD
#define DISP_DP_PN		_BV(PD2)


// All those display port definitions take less space than if to use hex values to set multiple pins at a time.
// Stupid compiler! )) This firmware needs assembly lang. face lift..

//#define DISP_SET_ZERO() { PORTA &= 0xFC; PORTB &= 0xE4; PORTD &= 0xF3; } // This one takes more space
#define DISP_SET_ZERO() { PORTA &= ~_BV(PA1); PORTA &= ~_BV(PA0); PORTB &= ~_BV(PB4); PORTB &= ~_BV(PB3); PORTB &= ~_BV(PB1); PORTB &= ~_BV(PB0); PORTD &= ~_BV(PD3); PORTD &= ~_BV(PD2); }
#define DISP_ALL_ON() { PORTA |= _BV(PA1); PORTA |= _BV(PA0); PORTB |= _BV(PB4); PORTB |= _BV(PB3); PORTB |= _BV(PB1); PORTB |= _BV(PB0); PORTD |= _BV(PD3); PORTD |= _BV(PD2); }
#define DISP_DP_ON() { DISP_DP_PORT |= DISP_DP_PN; }
#define DISP_DP_OFF() { DISP_DP_PORT &= ~DISP_DP_PN; }
	
	
void DispS1(void){ DISP_SET_ZERO(); DISP_B_PORT |= DISP_B_PN; DISP_C_PORT |= DISP_C_PN; }
void DispS2(void){ DISP_SET_ZERO(); DISP_A_PORT |= DISP_A_PN; DISP_B_PORT |= DISP_B_PN; DISP_G_PORT |= DISP_G_PN; DISP_D_PORT |= DISP_D_PN; DISP_E_PORT |= DISP_E_PN; }
void DispS3(void){ DISP_SET_ZERO(); DISP_A_PORT |= DISP_A_PN; DISP_B_PORT |= DISP_B_PN; DISP_C_PORT |= DISP_C_PN; DISP_D_PORT |= DISP_D_PN; DISP_G_PORT |= DISP_G_PN;}
void DispS4(void){ DISP_SET_ZERO(); DISP_B_PORT |= DISP_B_PN; DISP_C_PORT |= DISP_C_PN; DISP_F_PORT |= DISP_F_PN; DISP_G_PORT |= DISP_G_PN;} 
void DispS5(void){ DISP_SET_ZERO(); DISP_A_PORT |= DISP_A_PN; DISP_C_PORT |= DISP_C_PN; DISP_D_PORT |= DISP_D_PN; DISP_F_PORT |= DISP_F_PN; DISP_G_PORT |= DISP_G_PN;}
void DispS6(void){ DISP_SET_ZERO(); DISP_A_PORT |= DISP_A_PN; DISP_C_PORT |= DISP_C_PN; DISP_D_PORT |= DISP_D_PN; DISP_E_PORT |= DISP_E_PN; DISP_F_PORT |= DISP_F_PN; DISP_G_PORT |= DISP_G_PN;}
void DispS7(void){ DISP_SET_ZERO(); DISP_A_PORT |= DISP_A_PN; DISP_B_PORT |= DISP_B_PN; DISP_C_PORT |= DISP_C_PN;}
void DispS8(void){ DISP_SET_ZERO(); DISP_A_PORT |= DISP_A_PN; DISP_B_PORT |= DISP_B_PN; DISP_C_PORT |= DISP_C_PN; DISP_D_PORT |= DISP_D_PN; DISP_E_PORT |= DISP_E_PN; DISP_F_PORT |= DISP_F_PN; DISP_G_PORT |= DISP_G_PN;}
void DispS9(void){ DISP_SET_ZERO(); DISP_A_PORT |= DISP_A_PN; DISP_B_PORT |= DISP_B_PN; DISP_C_PORT |= DISP_C_PN; DISP_D_PORT |= DISP_D_PN; DISP_F_PORT |= DISP_F_PN; DISP_G_PORT |= DISP_G_PN;}
void DispS0(void){ DISP_SET_ZERO(); DISP_A_PORT |= DISP_A_PN; DISP_B_PORT |= DISP_B_PN; DISP_C_PORT |= DISP_C_PN; DISP_D_PORT |= DISP_D_PN; DISP_E_PORT |= DISP_E_PN; DISP_F_PORT |= DISP_F_PN;}
void DispSE(void){ DISP_SET_ZERO(); DISP_A_PORT |= DISP_A_PN; DISP_D_PORT |= DISP_D_PN; DISP_E_PORT |= DISP_E_PN; DISP_F_PORT |= DISP_F_PN; DISP_G_PORT |= DISP_G_PN;}
void DispSr(void){ DISP_SET_ZERO(); DISP_E_PORT |= DISP_E_PN; DISP_G_PORT |= DISP_G_PN;}
void DispSH(void){ DISP_SET_ZERO(); DISP_B_PORT |= DISP_B_PN; DISP_C_PORT |= DISP_C_PN; DISP_E_PORT |= DISP_E_PN; DISP_F_PORT |= DISP_F_PN; DISP_G_PORT |= DISP_G_PN;}
void DispSL(void){ DISP_SET_ZERO(); DISP_D_PORT |= DISP_D_PN; DISP_E_PORT |= DISP_E_PN; DISP_F_PORT |= DISP_F_PN;}
void DispSF(void){ DISP_SET_ZERO(); DISP_A_PORT |= DISP_A_PN; DISP_E_PORT |= DISP_E_PN; DISP_F_PORT |= DISP_F_PN; DISP_G_PORT |= DISP_G_PN;}

void (*g_disp_number_function_arr[])(void) = {	DispS0,	DispS1,
												DispS2,	DispS3,
												DispS4,	DispS5,
												DispS6,	DispS7,
												DispS8,	DispS9, 
												DispSE, DispSr, 
												DispSH, DispSL, 
												DispSF
											 };


// Info to display
// 3 digits + 3 decimal points (0 - no point, >0 - point)
u08 g_display_buff[6] = {0, 0, 0, 0, 0, 0};

#define DISP_MODE_TEMPERATURE	0
#define DISP_MODE_SETPOINT		1
#define DISP_MODE_HYST			2
u08 g_disp_mode = DISP_MODE_TEMPERATURE;
u16 g_disp_mode_timer = 0;

// Mode state variable for setting temperature in 1s steps instead of 0.1s
//u08 bQuickStepMode = FALSE;

#define VAL_TO_DISPLAY_THERM		0
#define VAL_TO_DISPLAY_SETPOINT		1
#define VAL_TO_DISPLAY_HYST			2


typedef struct{
	u16  current_temp;
	u08  current_temp_sign;
	u16  set_point_temp;
	u08  set_point_temp_sign;
	u16  hysteresis;
} ThermStruct;

ThermStruct g_therm_struct;

// Temperature readings from DS18B20
u16convert g_temperature;

// Initialize all that crap :)
void Init(void) {
	// Initialize all ports
	DDRA = _BV(PA1)|_BV(PA0);
	DDRB = 0xFF; // all pins to output
	DDRD = _BV(PD3)|_BV(PD2)|_BV(PD0);
	
	PORTA = 0x00;
	PORTB = 0x00;
	PORTD = 0x00;
}

// Write to EEPROM
void EEPROMWrite(u08 address, u08 data) {
	// Wait for completion of previous write 
	while(EECR & (1<<EEPE)) {}
	// Set up address and data registers 
	EEAR = address;
	EEDR = data;
	// Write logical one to EEMPE 
	EECR |= (1<<EEMPE);
	// Start eeprom write by setting EEPE 
	EECR |= (1<<EEPE);
}

// Read from EEPROM
u08 EEPROMRead(u08 address) {
	// Wait for completion of previous write 
	while(EECR & (1<<EEPE)) {}
	// Set up address register 
	EEAR = address;
	// Start eeprom read by writing EERE 
	EECR |= (1<<EERE);
	// Return data from data register 
	return EEDR;
}

// Moves temperature readings to display buffer.
void ThermToDisplay(ThermStruct *therm, u08 val_to_display) {
	u16 buff = 0;
	buff = therm->current_temp;
	
	switch (val_to_display){
	case VAL_TO_DISPLAY_THERM:
		buff = therm->current_temp;
		break;
	case VAL_TO_DISPLAY_SETPOINT:
		buff = therm->set_point_temp;
		break;
	case VAL_TO_DISPLAY_HYST:
		buff = therm->hysteresis;
		break;
	default:
		break;
	}	
	
	// Clean Display Buffer
	g_display_buff[0] = g_display_buff[1] = g_display_buff[2] = g_display_buff[3] = g_display_buff[4] = g_display_buff[5] = 0;
	
	if (buff > 999) {
		// Show an overload OFL (Over F Load)
		
		g_display_buff[1] = 14;
		g_display_buff[2] = 13;
	} else {
		// Single digit
		if (buff < 10) {
			g_display_buff[2] = buff;
			g_display_buff[4] = 1; // Decimal point
		}
		// Two digits
		else if (buff >= 10 && buff <= 99) {
			g_display_buff[1] = buff / 10;
			g_display_buff[2] = buff % 10;
			g_display_buff[4] = 1; // Decimal point
		}
		// Three digits
		else {
			g_display_buff[0] = buff / 100;
			buff %= 100;
			g_display_buff[1] = buff / 10;
			g_display_buff[2] = buff % 10;
			g_display_buff[4] = 1; // Decimal point
		}
		
		if (val_to_display == VAL_TO_DISPLAY_THERM && therm->current_temp_sign)	{
			// If temperature is negative
			g_display_buff[5] = 1;
		}
	}
}


// Display digits stored in g_display_buff
void DisplayUpdate(void) {
	DISP_OFF();
	g_disp_number_function_arr[ g_display_buff[0] ]();
	if (g_display_buff[3]) DISP_DP_ON();
	DISP_DIGIT1();
	_delay_ms(DISP_DELAY);
	
	DISP_OFF();
	g_disp_number_function_arr[ g_display_buff[1] ]();
	if (g_display_buff[4]) DISP_DP_ON();
	DISP_DIGIT2();
	_delay_ms(DISP_DELAY);
	
	DISP_OFF();
	g_disp_number_function_arr[ g_display_buff[2] ]();
	if (g_display_buff[5]) DISP_DP_ON();
	DISP_DIGIT3();
	_delay_ms(DISP_DELAY);
}


// Check buttons state (if any were pressed)
void ButtonsChk(void) {
	u08 buttons_value;
	static u08 is_button_pressed = FALSE;
	static u08 buttons_old_state = 0x70;
	// Read all three buttons state at once
	// If none is pressed the value will be 0x70
	buttons_value = PIND & 0x70; // Mask out all other pins
	
	// ON BUTTON PRESS
	if (buttons_value != 0x70 && !is_button_pressed) {
		buttons_old_state = buttons_value;
		is_button_pressed = TRUE;
	}
	// ON BUTTON RELEASE
	else if (buttons_value == 0x70 && is_button_pressed) {
		g_disp_mode_timer = 0;
		is_button_pressed = FALSE;
		switch (buttons_old_state){
		case 0x60: // Button one
			if (g_disp_mode != DISP_MODE_HYST) {
				if ( g_therm_struct.set_point_temp > 1 && g_therm_struct.set_point_temp > (g_therm_struct.hysteresis/2) )
							g_therm_struct.set_point_temp--;
				g_disp_mode = DISP_MODE_SETPOINT;
			} else {
				if ( g_therm_struct.hysteresis > 2 )
							g_therm_struct.hysteresis--;
			}
			break;
		case 0x50: // Button two
			if (g_disp_mode != DISP_MODE_HYST) {
				// 97.5 degrees, because of the max hysteresis/2 == 2.5 degrees
				if (g_therm_struct.set_point_temp < 975)
								g_therm_struct.set_point_temp++;
				g_disp_mode = DISP_MODE_SETPOINT;
			} else {
				if ( g_therm_struct.hysteresis < 50 && g_therm_struct.set_point_temp > ((g_therm_struct.hysteresis/2)+2) )
								g_therm_struct.hysteresis++;
			}
			break;
		case 0x30: // Button three
			g_disp_mode = DISP_MODE_HYST;
			break;
		case 0x40: // Buttons 1 & 2
			break;
		case 0x10: // Buttons 2 & 3
			break;
		case 0x20: // Buttons 1 & 3
			break;
		default:
			break;
		}
	}
}

// Returns 0 if the sensor is present, 1 otherwise
u08 OneWireReset(void) {
	u08 status;
	TSENS_PULL_LOW();
	_delay_us(485);
	TSENS_LISTEN();
	_delay_us(72);
	status = TSENS_BIT;
	_delay_us(425);
	return status;
}

// Read Bit Function
u08 OneWireReadBit(void) {
	u08 bit;
	TSENS_PULL_LOW();
	_delay_us(3);
	TSENS_LISTEN();
	_delay_us(15);
	bit = TSENS_BIT;
	return bit;
}

// Write bit Function
void OneWireWriteBit(u08 bit) {
	TSENS_PULL_LOW();
	_delay_us(2);
	if (bit == 1) TSENS_LISTEN();
	_delay_us(105);
	TSENS_LISTEN();
}

// Read BYTE Function
u08 OneWireReadByte(void) {
	u08 byte = 0;
	u08 i;
	for (i=0;i<8;i++) {
		if (OneWireReadBit()) byte|=0x01<<i;
		_delay_us(120);
	}
	return byte;
}

// Write BYTE Function
void OneWireWriteByte(u08 byte) {
	u08 i;
	u08 temp;
	
	for (i=0;i<8;i++) {
		temp = byte>>i;
		temp &= 0x01;
		OneWireWriteBit(temp);
	}
	_delay_us(105);
}

// Process temperature readings from DS18B20
// which should be stored in g_temperature, prior
// to calling this function
void ExtractTherm(u16 sensor_data, ThermStruct *therm) {
	// This function is designed for working with 12bit
	// temperature reading from DS18B20 sensor.	
	
	u16 decimal_part;

	// If temperature is negative
	if (sensor_data & 0x8000) {
		// 1 if negative temperature
		therm->current_temp_sign = 1;
		// Remove sign bits
		sensor_data &= 0x0FFF;
		// Two's complement
		sensor_data = (~sensor_data) + 1;
	} else {
		// 0 if positive temperature
		therm->current_temp_sign = 0;
	}
	
	// Extract number after decimal point
	// and multiply it by 0.0625 (625)
	decimal_part = sensor_data & 0x000F;
	decimal_part *= 625;
	decimal_part /= 1000;
	// Shift to leave whole degree
	sensor_data &= 0x0FFF;
	sensor_data >>= 4;
	
	// Shift decimal point to the right.
	// Example: 15.2 degrees will be converted to 152
	therm->current_temp = ((u08)sensor_data * 10) + (u08)decimal_part;
}

// Check if temperature is reached the set point
// and control the relay accordingly
void SetPointTemperatureCheck(ThermStruct *therm) {
	static u08 is_relay_on = FALSE;
	
	// If current temperature is positive
	if (!therm->current_temp_sign) {
		// If relay is off
		if (is_relay_on == FALSE) {
			if (therm->current_temp <= therm->set_point_temp - (therm->hysteresis/2)) {
				is_relay_on = TRUE;
			}
		}
		// If relay is on
		else {
			if (therm->current_temp >= therm->set_point_temp + (therm->hysteresis/2)) {
				is_relay_on = FALSE;
			}
		}
	}
	// If the temperature is negative
	else {
		is_relay_on = TRUE;
	}


	if (is_relay_on == TRUE) {
		RELAY_ON();
		LED_ON();
	}
	else {
		RELAY_OFF();
		LED_OFF();
	}
}

void DisplayTherm(void) {
	static u08 conversion_time_counter = 0;
	
	if (conversion_time_counter == 0) {
		// Start Temperature Conversion
		if (!OneWireReset()) {		
			// Skip ROM command
			OneWireWriteByte(0xCC);
			// Start Conversion command
			OneWireWriteByte(0x44);
			conversion_time_counter++;
		}
	}
	else if (conversion_time_counter > 0 && conversion_time_counter <= 200)	{
		conversion_time_counter++;
	}
	else if (conversion_time_counter > 200) {
		// Read Temperature After Conversion is Complete
		conversion_time_counter = 0;
		if (!OneWireReset()) {
			// Skip ROM command
			OneWireWriteByte(0xCC);
			// Read Scratchpad command
			OneWireWriteByte(0xBE);
			g_temperature.bytes.low  = OneWireReadByte();
			g_temperature.bytes.high = OneWireReadByte();
			
			// Convert DS18B20 data into whole temperature,
			// decimal part and sign and put it to baTherm
			ExtractTherm(g_temperature.value, &g_therm_struct);
			// Move baTherm to display buffer
			ThermToDisplay(&g_therm_struct, VAL_TO_DISPLAY_THERM);
			// Main thermostat function
			SetPointTemperatureCheck(&g_therm_struct);
		}
	}
}

/*****************************************************************************/
int main(void)
{
	g_disp_mode_timer = 0;
	
	_delay_ms(100);
	Init();
	
	// Check if MCU EEPROM has any previous settings stored or not
	// 0x88 if settings is there
	u08 first_time_flag = 0;
	first_time_flag = EEPROMRead(5);
	u16convert temperature_buf;
	
	// If EEPROM is empty
	if (first_time_flag != 0x88) {
		// Setpoint Temperature to 10.0
		EEPROMWrite(0,0x00);
		EEPROMWrite(1,0x64);
		// Hysteresis to 1.0
		EEPROMWrite(2,0x00);
		EEPROMWrite(3,0x0A);
		// Sign to positive
		EEPROMWrite(4,0x00);
		// And first_time_flag
		EEPROMWrite(5,0x88); 
		// Set the initial setpoint params.
		g_therm_struct.hysteresis = 4; //(0x000A) 1.0 degrees C
		g_therm_struct.set_point_temp = 10; //(0x0064) 25.0 degrees C
		g_therm_struct.set_point_temp_sign = 0;
	}
	// If some data in EEPROM is present (6th byte == 0x88)
	else {
		// Read Set Point Temperature
		temperature_buf.bytes.high = EEPROMRead(0);
		temperature_buf.bytes.low  = EEPROMRead(1);
		g_therm_struct.set_point_temp = temperature_buf.value;
		// Read Hysteresis value
		temperature_buf.bytes.high = EEPROMRead(2);
		temperature_buf.bytes.low  = EEPROMRead(3);
		g_therm_struct.hysteresis = temperature_buf.value;
		// Read sign
		g_therm_struct.set_point_temp_sign = EEPROMRead(4);
	}
	
	//
	// Main program cycle
	//	
    while(1) {
		DisplayUpdate();
		ButtonsChk();
		
		switch (g_disp_mode){
		case DISP_MODE_TEMPERATURE:
			DisplayTherm();
			break;
		case DISP_MODE_SETPOINT:
			ThermToDisplay(&g_therm_struct, VAL_TO_DISPLAY_SETPOINT);
			break;
		case DISP_MODE_HYST:
			ThermToDisplay(&g_therm_struct, VAL_TO_DISPLAY_HYST);
			break;
		default:
			break;
		}
		
		// If no buttons is pressed, wait for some time and switch to Temperature display
		if (g_disp_mode != DISP_MODE_TEMPERATURE && g_disp_mode_timer < 350) {
			g_disp_mode_timer++;
		}
		else if (g_disp_mode != DISP_MODE_TEMPERATURE && g_disp_mode_timer >= 350) {
			g_disp_mode = DISP_MODE_TEMPERATURE;
			g_disp_mode_timer = 0;
			
			// Write settings to EEPROM
			temperature_buf.value = g_therm_struct.set_point_temp;
			EEPROMWrite(0, temperature_buf.bytes.high);
			EEPROMWrite(1, temperature_buf.bytes.low);
			temperature_buf.value = g_therm_struct.hysteresis;
			EEPROMWrite(2, temperature_buf.bytes.high);
			EEPROMWrite(3, temperature_buf.bytes.low);
			EEPROMWrite(4, g_therm_struct.set_point_temp_sign);
		}

	}
	
	return 0;	
}

