; Disabling error messages
	errorlevel -302					; Disable "Register in operand not in bank 0"
	errorlevel -306					; Disable "Crossing page boundary"


	list	p=16f887				; 16F887 can be used
	#include <p16f887.inc>			; processor specific variable definitions

	__CONFIG _CONFIG1, _CP_OFF &_CPD_OFF & _WDT_OFF & _BOR_ON & _PWRTE_ON & _HS_OSC & _LVP_OFF & _IESO_OFF & _FCMEN_OFF & _MCLRE_OFF
	__CONFIG _CONFIG2, _WRT_OFF & _BOR40V

; Serial port Baud rate selection
;BaudRate	EQU	.2400				; Set the BaudRate of RS232 communication
;BaudRate	EQU	.4800				; Set the BaudRate of RS232 communication
BaudRate	EQU	.9600				; Set the BaudRate of RS232 communication
;BaudRate	EQU	.19200				; Set the BaudRate of RS232 communication
;BaudRate	EQU	.38400				; Set the BaudRate of RS232 communication

TXBUFFERSIZE	EQU	.32				; Must be a power of 2
RXBUFFERSIZE	EQU	.32				; Must be a power of 2

F_OSC	EQU	.20000000				; Oscillator frequency in Hz
OSC		EQU	F_OSC/4

	ifndef BaudRate
		error "No BaudRate defined"
	endif
	if ( (BaudRate!=.2400) && (BaudRate!=.4800) && (BaudRate!=.9600) && (BaudRate!=.19200) && (BaudRate!=.38400) )
		error "Nonstandard BaudRate defined"
	endif

;***** VARIABLE DEFINITIONS

;Vars in shared memory. This memory is available from any bank

  cblock	0x070
	pclath_temp						; variable used for context saving
	w_temp							; variable used for context saving
	status_temp						; variable used for context saving
	fsr_temp						; variable used for context saving
LastCommon:
  endc

	IF LastCommon > 0x7F
		ERROR "To many variables used in Common RAM"
	ENDIF

  cblock	0x20
	UartRxChNum						; Uart Rx fifo number of available chars
	UartRxChWp						; Uart Rx fifo write pointer
	UartRxChRp						; Uart Rx fifo read  pointer

	UartTxChNum						; Uart Tx fifo number of available chars
	UartTxChWp						; Uart Tx fifo write pointer
	UartTxChRp						; Uart Tx fifo read  pointer

	Rs232StrPtr						; 16 bit pointer to string to be sent
	Rs232StrPtrH
	Rs232Temp						; Temporary storage in UART routines
	Rs232Temp2						; Temporary storage in UART routines
	Rs232Fsr						; Save for FSR in UART routines
	Rs232Status						; Save for STATUS in UART routines

	temp
	counterl
	counterh
LastBank0:
  endc

	IF LastBank0 > 0x70
		ERROR "To many variables used in Bank 0"
	ENDIF

  cblock	0x120					; Uart buffers

	UartTxChar:	TXBUFFERSIZE		; Uart Tx fifo

	UartRxChar:	RXBUFFERSIZE		; Uart Rx fifo

 	; Not to use memory 0x170..0x17F  -  it will overwrite 0x070..0x07F
LastBank2:
  endc

	IF LastBank2 > 0x17F
		ERROR "To many variables used in Bank 2"
	ENDIF

;**********************************************************************
; PortC
bTxD			EQU	6				; UART's TxD line
bRxD			EQU	7				; UART's RxD line

;**********************************************************************
		org     0x0000           	; processor reset vector
		goto    main            	; go to beginning of program

;**********************************************************************
		org     0x0004           	; interrupt vector location

		movwf	w_temp				; context saveing
		swapf	w_temp, f			; Save W to common RAM
		swapf	STATUS, w			; Get STATUS
		clrf 	STATUS				; select Bank0, IRP=0 for interrupt stuff
		movwf	status_temp			; Save STATUS
		movf	FSR,w				; Save FSR
		movwf	fsr_temp
		movf	PCLATH,w			; Needed only if more Pages used not only Page0
		movwf	pclath_temp
		clrf	PCLATH

;--------
INT_UART

		bsf		STATUS,IRP			; Buffers are on Bank2

		btfss	PIR1,RCIF			; Is it a receive interrupt
		goto	INT_UART_TX

		movf	UartRxChWp,w		; Put receive char to buffer if no error
		movwf	FSR
		movf	RCSTA,w				; Check for error
		andlw	(1<<FERR) | (1<<OERR)
		btfss	STATUS,Z
		goto	UART_ERR			; If error not to read char
		movf	RCREG,w				; Read character
		movwf	INDF				; Write it to buffer
		incf	FSR,w				; Move pointer in circular way
		andlw	RXBUFFERSIZE - 1
		addlw	low(UartRxChar)
		movwf	UartRxChWp
		incf	UartRxChNum,f		; Sign one more char in buffer
		goto	INT_UART_TX

UART_ERR							; Receive error, reenable receive
		movf	RCREG,w				; Clear receive interrupt flag, drop character
		bcf		RCSTA,CREN			; Turn off receiver
		bsf		RCSTA,CREN			; Turn on  receiver

INT_UART_TX							; Is it a transmit interrupt
		btfss	PIR1,TXIF
		goto	INT_UART_EX

		movf	UartTxChNum,f		; Get next char from buffer if there is one
		btfsc	STATUS,Z
		goto	INT_TX_NOCHAR		; No more chars in buffer

		movf	UartTxChRp,w		; Get pointer to buffer
		movwf	FSR
		movf	INDF,w				; Get character from buffer
		movwf	TXREG				; Send it
		incf	FSR,w				; Move pointer
		andlw	TXBUFFERSIZE - 1	; in circular way
		addlw	low(UartTxChar)
		movwf	UartTxChRp
		decfsz	UartTxChNum,f		; Sign one char read from buffer
		goto	INT_UART_EX			; Skip disabling transmit interrupt if more chars in buffer

INT_TX_NOCHAR						; No more chars to transmit
		bsf 	STATUS,RP0
		bcf		PIE1,TXIE			; Disable transmit int

INT_UART_EX
		clrf	STATUS				; Bank0, IRP=0

;--------
INT_EXIT
		movf	pclath_temp,w
		movwf	PCLATH				; Needed only in more Pages used not only Page0
		movf	fsr_temp,w			; Restore FSR
		movwf	FSR
		swapf	status_temp,w		; Restore context for main program
		movwf	STATUS				; This will also restore bank selection and IRP
		swapf	w_temp,w			; Restore W from common RAM
		retfie                  	; Return from interrupt

;**********************************************************************
; Initialization

main
		bsf		STATUS,RP0			; Bank1

		movlw	0xFF				; Set PORTA 0..7 as input
		movwf	TRISA
		movlw	0xFF
 		movwf	TRISB				; Set PORTA 0..7 as input
		movlw	0xFF
		movwf	TRISC				; Set PORTA 0..7 as input

		clrf	PIE1				; dissable all possible interrupt flag

		bsf		STATUS,RP1			; Bank3
		clrf	ANSELH
		clrf	ANSEL				; Use all pins as digital i/o

		clrf	STATUS				; Bank0, IRP=0

		call	UART_INIT			; Init uart and it's buffers

; Low power

SendMessage
		movlw	(1<<GIE) | (1<<PEIE)
		movwf	INTCON 				; Enable global, extended peripheral and TMR0 interrupt

		call	SEND_MESSAGE		; Send message

;================================================================================
;	Main loop

MainLoop
		call	UART_RECEIVE
		btfsc	STATUS,C
		goto	MainLoop

		movwf	temp
		andlw	0x60
		xorlw	0x60
		btfss	STATUS,Z
		goto	MainSendBack

		movlw	0x20
		xorwf	temp,f

MainSendBack
		movf	temp,w
		call	UART_SEND

		movlw	0x10
		movwf	counterh
counth
		clrf	counterl
countl
		decfsz	counterl,f
		goto	countl
		decfsz	counterh,f
		goto	counth

		goto	MainLoop

;******************************************************************************
;	UART functions
;******************************************************************************

;**********************************************************************
; Init uart

UART_INIT:
		clrf	UartRxChNum			; Init receive fifo
		movlw	low(UartRxChar)
		movwf	UartRxChWp
		movwf	UartRxChRp

		clrf	UartTxChNum			; Init transmit fifo
		movlw	low(UartTxChar)
		movwf	UartTxChWp
		movwf	UartTxChRp

		bsf 	STATUS,RP0			; Bank1

	if	BaudRate<.9600				; Set up Baud rate
		bcf		TXSTA,BRGH
		movlw	(F_OSC/.64/BaudRate)-1
	else
		bsf		TXSTA,BRGH
		movlw	(F_OSC/.16/BaudRate)-1
	endif

		movwf	SPBRG
		bsf		TXSTA,TXEN			; Enable transmission

		bsf		PIE1,RCIE			; Enable  receive interrupt
		bcf		PIE1,TXIE			; Disable transmit interrupt

		bcf 	STATUS,RP0			; Bank0

		bsf		RCSTA,SPEN			; Enable receive
		bsf		RCSTA,CREN

		return

;**********************************************************************
; Send a character (w), write it to uart send buffer
; Keeps IRP, RP1, RP0, DC, C bits of STATUS
; Keeps FSR

UART_SEND							; Sends char in w
		movwf	Rs232Temp			; Save character to send
UART_Send1
		movf	UartTxChNum,w		; Wait for at least 1 free location
		xorlw	TXBUFFERSIZE
		btfsc	STATUS,Z			; Test number of chars in buff
		goto	UART_Send1

		swapf	STATUS,w
		movwf	Rs232Status			; Save Status
		movf	FSR,w				; Save FSR
		movwf	Rs232Fsr

		bsf 	STATUS,RP0			; Bank1
		bcf		PIE1,TXIE			; Disable TX interrupt
		bcf 	STATUS,RP0			; Bank0

		bsf		STATUS,IRP			; Buffer in on Bank2

		movf	UartTxChWp,w		; Get write pointer
		movwf	FSR

		movf	Rs232Temp,w			; Get char to send
		movwf	INDF				; Store it in out buffer
		incf	FSR,w				; Move the pointer
		andlw	TXBUFFERSIZE - 1	; in circular way
		addlw	low(UartTxChar)
		movwf	UartTxChWp
		incf	UartTxChNum,f		; Increment number of available characters

		bsf 	STATUS,RP0			; Bank1
		bsf		PIE1,TXIE			; Enable TX interrupt

UART_Exit
		bcf 	STATUS,RP0			; Bank0
		movf	Rs232Fsr,w			; Restore FSR
		movwf	FSR
		swapf	Rs232Status,w		; Restore STATUS
		movwf	STATUS				; Restore IRP, Bank selection
		movf	Rs232Temp,w			; Get char sent / just received
		return

;**********************************************************************
; Get a character from receive buffer
; Keeps IRP, RP1, RP0, DC, C bits of STATUS
; Keeps FSR

UART_RECEIVE						; Gets char into w, C == 1 if no char was in fifo

		bsf		STATUS,C			; Set no character received
		swapf	STATUS,w			; Save STATUS
		movwf	Rs232Status
		movf	FSR,w				; Save FSR
		movwf	Rs232Fsr

		bsf		STATUS,IRP			; Buffer in on Bank2

		bsf 	STATUS,RP0			; Bank1
		bcf		PIE1,RCIE			; Disable RC interrupt
		bcf 	STATUS,RP0			; Bank0

		clrf	Rs232Temp
		movf	UartRxChNum,w		; Check number of available chars
		btfsc	STATUS,Z
		goto	NO_REC_CHAR			; Jump if no character available

		movf	UartRxChRp,w		; Get read pointer
		movwf	FSR

		movf	INDF,w				; Read it from buffer
		movwf	Rs232Temp
		incf	FSR,w				; Move the pointer
		andlw	TXBUFFERSIZE - 1	; in circular way
		addlw	low(UartRxChar)
		movwf	UartRxChRp
		decf	UartRxChNum,f		; Decrement number of available characters
		bcf		Rs232Status,C+4		; Valid character received - !Status was swapped!

NO_REC_CHAR
		bsf 	STATUS,RP0			; Bank1
		bsf		PIE1,RCIE			; Enable RC interrupt

		goto	UART_Exit			; Restore STATUS and FSR

;**********************************************************************
; Send init message

SEND_MESSAGE
		movlw	high(InitMsg)		; Init string pointer high part
		movwf	Rs232StrPtrH
		movlw	low(InitMsg)		; Init string pointer low  part

;**********************************************************************
; Send a string (high part of address must be in Rs232StrPtrH)

SEND_STRING
		movwf	Rs232StrPtr			; Save string pointer low  part
lString0
		pagesel	MsgTabl
		call	MsgTabl				; Get next character
		clrf	PCLATH
		movwf	Rs232Temp2			; Save it
		andlw	0x7F				; Mask bit7 off
		call	UART_SEND			; Send character
		btfss	Rs232Temp2,7		; Check for end of string mark
		goto	lString0			; Loop for all characters
		return

;******************************************************************************
; String tables

InitMsg								; String tables (last character of string has bit7 set)
	dt	"Hello Word"
CrLf
	retlw	0x0a
	retlw	0x8d					; Sign end of string - bit7 set

	org		0x7F8

;******************************************************************************
; Get next character of a string

MsgTabl
		movf	Rs232StrPtrH,w		; Get string pointer high byte
		movwf	PCLATH				; Load it to PCLATH
		movf	Rs232StrPtr,w		; Get string pointer low  byte
		incf	Rs232StrPtr,f		; Increment pointer as
		btfsc	STATUS,Z
		incf	Rs232StrPtrH,f		; 16 bit number
lString1
		movwf	PCL					; Computed goto to string table
		return						; If movwf	PCL cannot be the last instruction

; Why this message comes if last instruction is a PC modifying instruction ??? 

;"CORE-W0014: Halted due to PC incrementing over the Maximum PC address and wrapping back to Zero"

;====== E N D   O F   2 K  P A G E ============================================

	END                     		; directive 'end of program'
