using System;
using System.Threading;
using System.IO.Ports;
using Microsoft.SPOT;
using Microsoft.SPOT.Hardware;
using System.Text;

namespace test
{
    public class MBSerialSlave
    {
        #region Constant
        private const byte SLAVE_ADDRESS = 1;

        // size in bytes
        private const UInt16 INPUT_IMAGE_SIZE = 2;
        private const UInt16 OUTPUT_IMAGE_SIZE = 2;
        private const UInt16 FLAG_IMAGE_SIZE = 512;
        private const UInt16 NORMAL_IMAGE_SIZE = INPUT_IMAGE_SIZE + OUTPUT_IMAGE_SIZE + FLAG_IMAGE_SIZE;
        private const UInt16 RETAIN_IMAGE_SIZE = 512;

        private const UInt16 PROCESS_IMAGE_SIZE = NORMAL_IMAGE_SIZE + RETAIN_IMAGE_SIZE;
        private const UInt32 PROCESS_IMAGE_NR_BITS = PROCESS_IMAGE_SIZE * 8;

        private const UInt16 IN_BEGIN_ADDR = 0;
        private const UInt16 OUT_BEGIN_ADDR = INPUT_IMAGE_SIZE;
        private const UInt16 FLAG_BEGIN_ADDR = INPUT_IMAGE_SIZE + OUTPUT_IMAGE_SIZE;
        private const UInt16 RETAIN_BEGIN_ADDR = NORMAL_IMAGE_SIZE;

        // RS232-RS485 ADU buffer size: 256 bytes
        private const int ADU_BUFFER_SIZE = 256;
        #endregion

        private Thread m_workerThread;
        private SerialPort m_serialPort;
        private static OutputPort m_led1 = null;
        private Timer m_ledTimer = null;
        private Timer m_t35Timer = null;

        private enum State
        {
            m1_wait_for_t35,
            m1_wait_for_data
        };
        private State m1_state = State.m1_wait_for_t35;

        // Command buffer
        private UInt16 cmd_ptr = 0;
        private byte[] cmd_buf = new byte[ADU_BUFFER_SIZE+1];

        private UInt16 mb_startingadr;  // Starting(or Output or Register) Address
        private UInt16 mb_data;         // Quantity(or Output or Register Value)

        private byte[] PROCESS_IMAGE = new byte[PROCESS_IMAGE_SIZE];

        private enum Exception_code : byte
        {
            MB_EX_NONE = 0,
            MB_EX_ILLEGAL_FUNCTION = 1,
            MB_EX_ILLEGAL_DATA_ADDRESS = 2,
            MB_EX_ILLEGAL_DATA_VALUE = 3
        };
        private Exception_code exception_code = Exception_code.MB_EX_ILLEGAL_FUNCTION;

        #region IS_PROP
        private bool is_valid_slave_address { get { return cmd_buf[0] == SLAVE_ADDRESS; } }
        private bool is_broadcast_address { get { return cmd_buf[0] == 0; } }
        private bool is_mb_fc_1 { get { return cmd_buf[1] == 1; } }
        private bool is_mb_fc_2 { get { return cmd_buf[1] == 2; } }
        private bool is_mb_fc_3 { get { return cmd_buf[1] == 3; } }
        private bool is_mb_fc_4 { get { return cmd_buf[1] == 4; } }
        private bool is_mb_fc_5 { get { return cmd_buf[1] == 5; } }
        private bool is_mb_fc_6 { get { return cmd_buf[1] == 6; } }
        private bool is_mb_fc_15 { get { return cmd_buf[1] == 15; } }
        private bool is_mb_fc_16 { get { return cmd_buf[1] == 16; } }

        // FC1: Read Coils
        private bool is_fc1_startingadr_ok
        {
            get
            {
                return ((mb_startingadr >= (INPUT_IMAGE_SIZE * 8))
                    && (mb_startingadr + mb_data <= (PROCESS_IMAGE_SIZE * 8)));
            }
        }
        
        // FC1: Read Coils
        private bool is_fc1_quantity_ok
        {
            get
            {
                return ((mb_data >= 0x0001) && (mb_data <= 0x07D0));
            }
        }

        // FC2: Read Discrete Inputs
        private bool is_fc2_startingadr_ok
        {
            get
            {
                return ((mb_startingadr >= 0) 
                    && (mb_startingadr + mb_data <= (INPUT_IMAGE_SIZE * 8)));
            }
        }

        // FC2: Read Discrete Inputs
        private bool is_fc2_quantity_ok
        {
            get
            {
                return ((mb_data >= 0x0001) && (mb_data <= 0x07D0));
            }
        }

        // FC3: Read Holding Registers
        private bool is_fc3_startingadr_ok
        {
            get
            {
                return ((mb_startingadr >= OUT_BEGIN_ADDR) 
                    && (mb_startingadr + mb_data <= PROCESS_IMAGE_SIZE));
            }
        }

        // FC3: Read Holding Registers
        private bool is_fc3_quantity_ok
        {
            get
            {
                return ((mb_data >= 0x0001) && (mb_data <= 0x007D));
            }
        }

        // FC4: Read Input Registers
        private bool is_fc4_startingadr_ok
        {
            get
            {
                return ((mb_startingadr >= 0) 
                    && (mb_startingadr + mb_data <= INPUT_IMAGE_SIZE));
            }
        }

        // FC4: Read Input Registers
        private bool is_fc4_quantity_ok
        {
            get
            {
                return ((mb_data >= 0x0001) && (mb_data <= 0x007D));
            }
        }

        // FC5: Write Single Coil
        private bool is_fc5_output_adr_ok
        {
            get
            {
                return true; // javitani !
            }
        }

        // FC5: Write Single Coil
        private bool is_fc5_output_value_ok
        {
            get
            {
                return ((mb_data == 0x0000) || (mb_data == 0xFF00));
            }
        }

        // FC6: Write Single Register
        private bool is_fc6_register_adr_ok
        {
            get
            {
                return true; // javitani !
            }
        }
        #endregion

        //
        // Constructor
        //
        public MBSerialSlave()
        {
            // thread
            m_workerThread = new Thread(this.DoWork);
            
            // serial port
            m_serialPort = new SerialPort(Microsoft.SPOT.Hardware.STM32F4.SerialPorts.COM3);
            m_serialPort.BaudRate = 9600;
            m_serialPort.Parity = Parity.None;
            m_serialPort.DataBits = 8;
            m_serialPort.StopBits = StopBits.None;
            m_serialPort.Handshake = Handshake.None;
            m_serialPort.ReadTimeout = Timeout.Infinite;
            m_serialPort.Open();

            m_led1 = new OutputPort(Microsoft.SPOT.Hardware.STM32F4.Pins.GPIO_PIN_D_12, false);
            m_ledTimer = new Timer
            (
                new TimerCallback(LedTimeout),
                m_led1,
                Timeout.Infinite,
                Timeout.Infinite
            );

            m_t35Timer = new Timer
            (
                new TimerCallback(T35Timeout),
                this,
                Timeout.Infinite,
                Timeout.Infinite
            );

            SetBit(260);
            SetBit(261);
            SetBit(262);
        }

        //
        // Start
        //
        public void Start()
        {
            m_workerThread.Start();

            // Loop until worker thread activates.
            while (!m_workerThread.IsAlive);
        }

        //
        // LedTimeout
        //
        private static void LedTimeout(object obj)
        {
            (obj as OutputPort).Write(false);
        }

        //
        // T35Timeout
        //
        private static void T35Timeout(object obj)
        {
            (obj as MBSerialSlave).RtuFsm(0, true);
        }

        //
        // SetBit
        //
        private void SetBit(int bit)
        {
            int byteNr = bit / 8;
            int bitIndex = bit % 8;
            PROCESS_IMAGE[byteNr] |= (byte)(1 << bitIndex);
        }

        //
        // ClearBit
        //
        private void ClearBit(int bit)
        {
            int byteNr = bit / 8;
            int bitIndex = bit % 8;
            PROCESS_IMAGE[byteNr] &= (byte)~(byte)(1 << bitIndex);
        }

        //
        // GetBit
        //
        private bool GetBit(int bit)
        {
            int byteNr = bit / 8;
            int bitIndex = bit % 8;
            return ((PROCESS_IMAGE[byteNr] & (byte)(1 << bitIndex)) != 0);
        }

        //
        // Read Coils
        //
        private Exception_code exec_fc1(ref byte pdu_len)
        {
            if (!is_fc1_quantity_ok) return Exception_code.MB_EX_ILLEGAL_DATA_VALUE;
            else if (!is_fc1_startingadr_ok) return Exception_code.MB_EX_ILLEGAL_DATA_ADDRESS;

            // read byte count
            cmd_buf[2] = ((mb_data % 8) != 0) ? (byte)(mb_data / 8 + 1) : (byte)(mb_data / 8);

            // mb_startingadr -> bit kezdocim
            // mb_data        -> bit darabszam
            for (UInt16 i = 0; i < mb_data; ++i)
            {
                // bitCopy
                bool bit = GetBit(mb_startingadr);
                if (bit)
                {
                    cmd_buf[3 + i / 8] |= (byte)(1 << (i % 8));
                }
                else
                {
                    cmd_buf[3 + i / 8] &= (byte)~(byte)(1 << (i % 8));
                }
                ++mb_startingadr;
            }
            pdu_len = (byte)(cmd_buf[2] + 2);
            return Exception_code.MB_EX_NONE;
        }

        //
        // Read Discrete Inputs
        //
        private Exception_code exec_fc2(ref byte pdu_len)
        {
            if (!is_fc2_quantity_ok) return Exception_code.MB_EX_ILLEGAL_DATA_VALUE;
            else if (!is_fc2_startingadr_ok) return Exception_code.MB_EX_ILLEGAL_DATA_ADDRESS;

            // read byte count
            cmd_buf[2] = ((mb_data % 8) != 0) ? (byte)(mb_data / 8 + 1) : (byte)(mb_data / 8);

            // mb_startingadr -> bit kezdocim
            // mb_data        -> bit darabszam
            for (UInt16 i = 0; i < mb_data; ++i)
            {
                // bitCopy
                bool bit = GetBit(mb_startingadr);
                if (bit)
                {
                    cmd_buf[3 + i / 8] |= (byte)(1 << (i % 8));
                }
                else
                {
                    cmd_buf[3 + i / 8] &= (byte)~(byte)(1 << (i % 8));
                }
                ++mb_startingadr;
            }
            pdu_len = (byte)(cmd_buf[2] + 2);
            return Exception_code.MB_EX_NONE;
        }

        //
        // Read Holding Registers
        //
        private Exception_code exec_fc3(ref byte pdu_len)
        {
            if (!is_fc3_quantity_ok) return Exception_code.MB_EX_ILLEGAL_DATA_VALUE;
            else if (!is_fc3_startingadr_ok) return Exception_code.MB_EX_ILLEGAL_DATA_ADDRESS;

            mb_startingadr *= 2; // word kezdocim atalakitasa byte kezdocimme
            mb_data *= 2;        // word darabszam atalakitasa byte darabszamma

            // read_byte_count
            cmd_buf[2] = (byte)mb_data;

            for (UInt16 i = 0; i < mb_data; ++i)
            {
                cmd_buf[3 + i] = PROCESS_IMAGE[mb_startingadr + i];
            }

            pdu_len = (byte)(cmd_buf[2] + 2);
            return Exception_code.MB_EX_NONE;
        }

        //
        // Read Input Registers
        //
        private Exception_code exec_fc4(ref byte pdu_len)
        {
            if (!is_fc4_quantity_ok) return Exception_code.MB_EX_ILLEGAL_DATA_VALUE;
            else if (!is_fc4_startingadr_ok) return Exception_code.MB_EX_ILLEGAL_DATA_ADDRESS;

            mb_startingadr *= 2; // word kezdocim atalakitasa byte kezdocimme
            mb_data *= 2;        // word darabszam atalakitasa byte darabszamma

            // read_byte_count
            cmd_buf[2] = (byte)mb_data;

            for (UInt16 i = 0; i < mb_data; ++i)
            {
                cmd_buf[3 + i] = PROCESS_IMAGE[mb_startingadr + i];
            }

            pdu_len = (byte)(cmd_buf[2] + 2);
            return Exception_code.MB_EX_NONE;
        }

        //
        // Write Single Coil
        //
        private Exception_code exec_fc5(ref byte pdu_len)
        {
            if (!is_fc5_output_value_ok) return Exception_code.MB_EX_ILLEGAL_DATA_VALUE;
            else if (!is_fc5_output_adr_ok) return Exception_code.MB_EX_ILLEGAL_DATA_ADDRESS;

            if (mb_data == 0)
                ClearBit(mb_startingadr);
            else
                SetBit(mb_startingadr);

            //
            // elso 5 byte visszakuldese
            //
            pdu_len = 5;
            return Exception_code.MB_EX_NONE;
        }

        //
        // Write Single Register
        //
        private Exception_code exec_fc6(ref byte pdu_len)
        {
            if (!is_fc6_register_adr_ok) return Exception_code.MB_EX_ILLEGAL_DATA_ADDRESS;

            mb_startingadr *= 2; // convert to byte address
            PROCESS_IMAGE[mb_startingadr] = (byte)(mb_data >> 8);
            PROCESS_IMAGE[mb_startingadr + 1] = (byte)(mb_data & 0xff);

            //
            // elso 5 byte visszakuldese
            //
            pdu_len = 5;
            return Exception_code.MB_EX_NONE;
        }

        //
        // restart_t35_timer
        //
        private void restart_t35_timer()
        {
            m_t35Timer.Change(50, Timeout.Infinite);
        }

        //
        // RtuFsm
        //
        private void RtuFsm(byte ch, bool timeout = false)
        {
            switch (m1_state)
            {
                case State.m1_wait_for_t35:
                    if (timeout)
                    {
                        cmd_ptr = 0;
                        m1_state = State.m1_wait_for_data;
                    }
                    else
                    {
                        restart_t35_timer(); // data received, restart t35 timer
                    }
                    break;
                case State.m1_wait_for_data:
                    if (timeout == false)
                    {
                        if (cmd_ptr < ADU_BUFFER_SIZE)
                        {
                            cmd_buf[cmd_ptr] = ch;
                            ++cmd_ptr;
                            restart_t35_timer();
                        }
                        else
                        {
                            m1_state = State.m1_wait_for_t35;
                            restart_t35_timer();
                        }
                    }
                    else
                    {
                        if (cmd_ptr >= 8 && (MBCrc.Calc(cmd_buf, cmd_ptr) == 0)
                            && (is_valid_slave_address || is_broadcast_address))
                        {
                            //
                            //  reset exc. code for default ILLEGAL_FUNCTION
                            // 
                            exception_code = Exception_code.MB_EX_ILLEGAL_FUNCTION;

                            mb_startingadr = (UInt16)((cmd_buf[2] << 8) | cmd_buf[3]);
                            mb_data = (UInt16)((cmd_buf[4] << 8) | cmd_buf[5]);

                            byte pdu_len = 0;
                            if (is_mb_fc_1) exception_code = exec_fc1(ref pdu_len);
                            else if (is_mb_fc_2) exception_code = exec_fc2(ref pdu_len);
                            else if (is_mb_fc_3) exception_code = exec_fc3(ref pdu_len);
                            else if (is_mb_fc_4) exception_code = exec_fc4(ref pdu_len);
                            else if (is_mb_fc_5) exception_code = exec_fc5(ref pdu_len);
                            else if (is_mb_fc_6) exception_code = exec_fc6(ref pdu_len);

                            if (is_valid_slave_address)
                            {
                                //
                                // ebben az esetben valaszolni is kell
                                //

                                if (exception_code != Exception_code.MB_EX_NONE)
                                {
                                    //
                                    // cmd_buf[1] tartalmazza az fc-t, csak hiba eseten modositom,
                                    // egyebkent nem valtoztatom meg, marad az, amit kaptunk
                                    //
                                    cmd_buf[1] += 0x80;
                                    cmd_buf[2] = (byte)exception_code;
                                    pdu_len = 2;
                                }

                                byte adu_len = (byte)(1 + pdu_len);
                                UInt16 calc_crc = MBCrc.Calc(cmd_buf, adu_len);
                                cmd_buf[adu_len++] = (byte)(calc_crc & 0xff);
                                cmd_buf[adu_len++] = (byte)(calc_crc >> 8);
                                m_serialPort.Write(cmd_buf, 0, adu_len);
                                m_serialPort.Flush();

                                // diag
                                m_led1.Write(true);
                                m_ledTimer.Change(50, Timeout.Infinite);
                            }
                        }
                        m1_state = State.m1_wait_for_t35;
                        restart_t35_timer();
                    }
                    break;
                default:
                    m1_state = State.m1_wait_for_t35;
                    restart_t35_timer();
                    break;
            }
        }

        //
        // Working Thread
        //
        private void DoWork()
        {
            byte[] rxBytes = new byte[1];
            int readCount = 0;

            restart_t35_timer();
            while (true)
            {
                readCount = m_serialPort.Read(rxBytes, 0, 1);
                if (readCount > 0)
                {
                    RtuFsm(rxBytes[0]);
                }
            }
        }
    }
}
