using System;
using Microsoft.SPOT;
using System.Threading;
using System.IO.Ports;
using System.Text;
using Microsoft.SPOT.Hardware;

namespace test
{
    public class MBSerialMaster
    {
        #region Constant
        // RS232-RS485 ADU buffer size: 256 bytes
        private const int ADU_BUFFER_SIZE = 256;
        #endregion

        byte[] m_txBuffer = new byte[ADU_BUFFER_SIZE + 1];
        byte[] m_rxBuffer = new byte[ADU_BUFFER_SIZE + 1];

        private Thread m_workerThread;
        private SerialPort m_serialPort;
        private static InterruptPort m_userButton = null;
        private bool m_running = false;

        //
        // Constructor
        //
        public MBSerialMaster()
        {
            // thread
            m_workerThread = new Thread(this.DoWork);

            // serial port
            m_serialPort = new SerialPort(Microsoft.SPOT.Hardware.STM32F4.SerialPorts.COM2);
            m_serialPort.BaudRate = 9600;
            m_serialPort.Parity = Parity.None;
            m_serialPort.DataBits = 8;
            m_serialPort.StopBits = StopBits.None;
            m_serialPort.Handshake = Handshake.None;
            m_serialPort.ReadTimeout = 1000; // msec
            m_serialPort.Open();

            m_userButton = new InterruptPort
            (
                Microsoft.SPOT.Hardware.STM32F4.Pins.GPIO_PIN_A_0,
                true,
                Port.ResistorMode.Disabled,
                Port.InterruptMode.InterruptEdgeHigh
            );
            m_userButton.EnableInterrupt();

            m_userButton.OnInterrupt += (uint port, uint state, DateTime time) =>
            {
                m_running = !m_running;
            };
        }

        //
        // Start
        //
        public void Start()
        {
            m_workerThread.Start();

            // Loop until worker thread activates.
            while (!m_workerThread.IsAlive) ;
        }

        //
        // BuildQuery
        //
        private int BuildQuery(byte[] buf, byte slave, byte fc, UInt16 addr, UInt16 quantity)
        {
            if (fc >= 1 && fc <= 4)
            {
                buf[0] = slave;
                buf[1] = fc;
                buf[2] = (byte)(addr >> 8);
                buf[3] = (byte)(addr & 0xff);
                buf[4] = (byte)(quantity >> 8);
                buf[5] = (byte)(quantity & 0xff);
                UInt16 calc_crc = MBCrc.Calc(buf, 6);
                buf[6] = (byte)(calc_crc & 0xff);
                buf[7] = (byte)(calc_crc >> 8);

                return 8;
            }

            return 0;
        }

        //
        // CalcRespLen
        //
        private int CalcRespLen(byte fc, UInt16 quantity)
        {
            if (fc == 1 || fc == 2)
            {
                int byte_count = ((quantity % 8) != 0) ? (byte)(quantity / 8 + 1) : (byte)(quantity / 8);
                return 5 + byte_count;
            }
            else if (fc == 3 || fc == 4)
            {
                return 5 + 2 * quantity;
            }

            return 0;
        }

        //
        // Working Thread
        //
        private void DoWork()
        {
            int state = 1;

            int good_response = 0;
            int bad_response = 0;
            int no_response = 0;

            byte slave = 1;
            byte fc = 1;
            UInt16 addr = 250;
            UInt16 quantity = 258;

            while (true)
            {
                if (m_running)
                {
                    switch (state)
                    {
                        case 1:
                            slave = 1;
                            fc = 1;
                            addr = 258;
                            quantity = 258;
                            state = 2;
                            break;
                        case 2: // bad response test
                            slave = 1;
                            fc = 2;
                            addr = 25;
                            quantity = 5;
                            state = 3;
                            break;
                        case 3:
                            slave = 1;
                            fc = 3;
                            addr = 30;
                            quantity = 10;
                            state = 4;
                            break;
                        case 4:
                            slave = 1;
                            fc = 4;
                            addr = 0;
                            quantity = 1;
                            state = 5;
                            break;
                        case 5: // no response test
                            slave = 2;
                            fc = 4;
                            addr = 0;
                            quantity = 1;
                            state = 1;
                            break;
                        default:
                            break;
                    }

                    // prepare query
                    int txLen = BuildQuery(m_txBuffer, slave, fc, addr, quantity);

                    // send query and wait to flush
                    m_serialPort.Write(m_txBuffer, 0, txLen);
                    m_serialPort.Flush();

                    // calc expected Bytes
                    int expectedBytes = CalcRespLen(fc, quantity);

                    // wait for response
                    int readCount = m_serialPort.Read(m_rxBuffer, 0, expectedBytes);

                    // check response
                    if ((readCount > 0) && (MBCrc.Calc(m_rxBuffer, readCount) == 0) 
                        && (readCount == expectedBytes || readCount == 5))
                    {
                        if (readCount == expectedBytes)
	                    {
                            ++good_response;
                            Debug.Print(
                                good_response.ToString()
                                + "/"
                                + bad_response.ToString()
                                + "/"
                                + no_response.ToString()
                                + ": "
                                + BitConverter.ToString(m_rxBuffer, 0, readCount));
                  	    }
                        else
                    	{
                            // readCount == 5
                            ++bad_response;
                            Debug.Print(
                                good_response.ToString()
                                + "/"
                                + bad_response.ToString()
                                + "/"
                                + no_response.ToString()
                                + ": "
                                + BitConverter.ToString(m_rxBuffer, 0, readCount));
                        }
                    }
                    else
                    {
                        ++no_response;
                        Debug.Print(
                            good_response.ToString()
                            + "/"
                            + bad_response.ToString()
                            + "/"
                            + no_response.ToString());
                    }
                }

                Thread.Sleep(150);
            }
        }
    }
}
