/*
 * Forrs: AN997
 */

/** I N C L U D E S ************************************************/
#include <p18cxxx.h>
#include "i2c.h"

/** P R O T O T Y P E S ********************************************/
void bit_in(unsigned char *data);   // Bit input function
void bit_out(unsigned char data);   // Bit output function

/********************************************************************
 * Function:        void bstart(void)
 *
 * Description:     This function generates an I2C Start condition.
 *******************************************************************/
void bstart(void)
{
    SDA_TRIS = 1;                   // Ensure SDA is high
    SCL = 1;                        // Ensure SCL is high
    SDA_TRIS = 0;                   // Configure SDA as an output
    SDA = 0;                        // Pull SDA low
    SCL = 0;                        // Pull SCL low
    
} // end bstart(void)

/********************************************************************
 * Function:        void bstop(void)
 *
 * Description:     This function generates an I2C Stop condition.
 *******************************************************************/
void bstop(void)
{
    SCL = 0;                        // Ensure SCL is low
    SDA_TRIS = 0;                   // Configure SDA as an output
    SDA = 0;                        // Ensure SDA low
    SCL = 1;                        // Pull SCL high
    SDA_TRIS = 1;                   // Allow SDA to be pulled high
} // end bstop(void)

/********************************************************************
 * Function:        void bit_out(unsigned char data)
 *
 * Description:     This function outputs a bit to the I2C bus.
 *******************************************************************/
void bit_out(unsigned char data)
{
    SCL = 0;                        // Ensure SCL is low
    if (data & 0x80)                // Check if next bit is high
    {
        SDA_TRIS = 1;               // Release SDA to be pulled high
    }
    else
    {
        SDA_TRIS = 0;               // Configure SDA as an output
        SDA = 0;                    // Pull SDA low
    }
    SCL = 1;                        // Pull SCL high to clock bit
    SCL = 0;                        // Pull SCL low for next bit
} // end bit_out(unsigned char data)

/********************************************************************
 * Function:        void bit_in(unsigned char *data)
 *
 * Description:     This function inputs a bit from the I2C bus.
 *******************************************************************/
void bit_in(unsigned char *data)
{
    SCL = 0;                        // Ensure SCL is low
    SDA_TRIS = 1;                   // Configure SDA as an input
    SCL = 1;                        // Bring SCL high to begin transfer
    *data &= 0xFE;                  // Assume next bit is low
    if (SDA)                        // Check if SDA is high
    {
        *data |= 0x01;              // If high, set next bit
    }
    SCL = 0;                        // Bring SCL low again
} // end bit_in(unsigned char *data)

/********************************************************************
 * Function:        unsigned char byte_out(unsigned char data)
 *
 * Description:     This function outputs a byte to the I2C bus.
 *                  It also receives the ACK bit and returns 0 if
 *                  successfully received, or 1 if not.
 *******************************************************************/
unsigned char byte_out(unsigned char data)
{
    unsigned char i;                // Loop counter
    unsigned char ack;              // ACK bit

    ack = 0;
    for (i = 0; i < 8; i++)         // Loop through each bit
    {
        bit_out(data);              // Output bit
        data = data << 1;           // Shift left for next bit
    }
    bit_in(&ack);                   // Input ACK bit

    return ack;
} // end byte_out(unsigned char data)

/********************************************************************
 * Function:        unsigned char byte_in(unsigned char ack)
 *
 * Description:     This function inputs a byte from the I2C bus.
 *                  Depending on the value of ack, it will also
 *                  transmit either an ACK or a NAK bit.
 *******************************************************************/
unsigned char byte_in(unsigned char ack)
{
    unsigned char i;                // Loop counter
    unsigned char retval;           // Return value

    retval = 0;
    for (i = 0; i < 8; i++)         // Loop through each bit
    {
        retval = retval << 1;       // Shift left for next bit
        bit_in(&retval);            // Input bit
    }
    bit_out(ack);                   // Output ACK/NAK bit

    return retval;
} // end byte_in(void)

/********************************************************************
 * Function:        void ACK_Poll(void)
 *
 * Description:     This function implements Acknowledge polling.
 *
 * Dependencies:    'control' contains the control byte
 *******************************************************************/
void ACK_Poll(unsigned char control)
{
    unsigned char result;           // Polling result

    result = 1;                     // Initialize result
    do
    {
        bstart();                   // Generate Start condition
        result = byte_out(control); // Output control byte
    } while (result == 1);
    bstop();                        // Generate Stop condition
} // end ACK_Poll(void)
