#include <stdlib.h>
#include <stdint.h>
#include <avr/io.h>
#include <avr/pgmspace.h>
#include "crc8.h"
#include "ds1820.h"

inline __attribute__((gnu_inline)) void therm_delay(uint16_t delay){
while(delay--) asm volatile("nop"); }

uint8_t therm_reset() {

uint8_t i;

THERM_LOW();
THERM_OUTPUT_MODE();
therm_delay(us(480));

THERM_INPUT_MODE();
therm_delay(us(60));

i=(THERM_PIN & (1<<THERM_DQ));
therm_delay(us(420));

//Return the value read from the presence pulse (0=OK, 1=WRONG)
return i;
}

void therm_write_bit(uint8_t bit){

THERM_LOW();
THERM_OUTPUT_MODE();
therm_delay(us(1));

if(bit) THERM_INPUT_MODE();

therm_delay(us(60));
THERM_INPUT_MODE();
}

uint8_t therm_read_bit(void){

uint8_t bit = 0;

THERM_LOW();
THERM_OUTPUT_MODE();
therm_delay(us(1));

THERM_INPUT_MODE();
therm_delay(us(14));

if(THERM_PIN&(1<<THERM_DQ)) { bit=1; }

therm_delay(us(45));

return bit;
}

uint8_t therm_read_byte(void){

uint8_t i = 8, n = 0;

while(i--) {

	n>>=1;
	n|=(therm_read_bit()<<7);
}

return n;
}


void therm_write_byte(uint8_t byte){

uint8_t i = 8;

while(i--){

	therm_write_bit(byte&1);
	byte>>=1;
}


}


uint16_t DS1820_read(uint8_t spad[]){

uint8_t i;
uint16_t measure;

therm_reset();
therm_write_byte(THERM_CMD_SKIPROM);
therm_write_byte(THERM_CMD_CONVERTTEMP);

//Wait until conversion is complete
while(!therm_read_bit());

//Reset, skip ROM and send command to read Scratchpad
therm_reset();
therm_write_byte(THERM_CMD_SKIPROM);
therm_write_byte(THERM_CMD_RSCRATCHPAD);

//Read Scratchpad
for ( i = 0;i < 9;i++) {
	spad[i]=therm_read_byte();
}
therm_reset();

if ( crc8( &spad[0], 9 ) ) {
// CRC FAIL	
} else {
// CRC Ok.
}

measure = spad[0] | (spad[1] << 8);
measure &= (uint16_t)0xfffe;
measure <<= 3;
measure += ( 16 - spad[6] ) - 4;

return measure;
}


uint8_t DS1820_format_temp( uint16_t measure, char str[], uint8_t format) {

uint8_t  negative;
int32_t  tempfull;
int16_t  tempdec;
uint16_t fract;

int32_t maxrange = 0;
int32_t  minrange = 0;
int32_t temperaturevalue = 0;

char temp[10];
int8_t temp_loc = 0;
uint8_t str_loc = 0;
ldiv_t ldt;
uint8_t ret;

if ( measure & 0x8000 )  {
	negative = 1;       // mark negative
	measure ^= 0xffff;  // convert to positive => (twos complement)++
	measure++;	

} else { negative = 0;};

tempfull  = (measure >> 4);
tempdec = tempfull;
tempfull *= 10000;
tempfull +=( measure & 0x000F ) * 625;

if ( negative ) {
	tempfull = -tempfull;
}

tempdec *= 10;

// tempdec += ((measure & 0x000F) * 640 + 512) / 1024;
// 625/1000 = 640/1024
fract = ( measure & 0x000F ) * 640;
if ( !negative ) {
	fract += 512;
}
fract /= 1024;
	tempdec += fract;

if ( negative ) {
	tempdec = -tempdec;
}

if ( format == 0 ) {
	maxrange = 1250000;
	minrange = -550000;
	temperaturevalue = tempfull;
	
} else if ( format == 1 ) {
	maxrange = 1250;
	minrange = -550;
	temperaturevalue = tempdec;
	
}

	if ( temperaturevalue > minrange && temperaturevalue < maxrange ) {

		if ( temperaturevalue < 0) {
			temperaturevalue = -temperaturevalue;
		}

		do {
			ldt = ldiv( temperaturevalue, 10 );
			temp[temp_loc++] = ldt.rem + '0';
			temperaturevalue = ldt.quot;
		} while ( temperaturevalue > 0 );

		if ((temp_loc <= 4) && (temp_loc >= 0) && (format == 0)) {
			temp[temp_loc++] = '0';
		}
		else if ((temp_loc == 1) && (format == 1)) {
			temp[temp_loc++] = '0';
		}


		if ( negative ) {
			temp[temp_loc] = '-';
		} else {
			temp[temp_loc] = ' ';
		}

		while ( temp_loc >= 0 ) {
			str[str_loc++] = temp[(uint8_t)temp_loc--];
			
			if (( temp_loc == 3 ) && (format == 0)) {
				str[str_loc++] = '.';
			}
			else if (( temp_loc == 0 ) && (format == 1)) {
				str[str_loc++] = '.';
			}
		}
		str[str_loc] = '\0';

		ret = 0;
	} else {
		ret = 1;
	}
	
	return ret;
}
