#include "microthreads.h"

#include <setjmp.h>

typedef struct {
  union {
    void (*startingPoint)(void);
    jmp_buf savedState;
  } address;
  ThreadState state;
} ThreadDescriptor;

jmp_buf          microthreads_threadResetHandle;
ThreadDescriptor microthreads_threadDescriptors[ MAX_THREADS ];
int8_t           microthreads_inited = 0;
int8_t           microthreads_currentThread = 0;

void microthreads_beginThread(int space)
{
  int8_t buffer[space];
  // do not optimize the space out
  buffer[space-1] = 1;
  buffer[STACK_SIZE-1]--;
  void (*startingPoint)(void) = microthreads_threadDescriptors[microthreads_currentThread].address.startingPoint;
  microthreads_enableInterrupts();
  (*startingPoint)();
  microthreads_threadDescriptors[microthreads_currentThread].state = MT_Unallocated;
  microthreads_scheduleNext();
}

void microthreads_threadStarter(void)
{
  // this will be the main thread
  int8_t buffer[STACK_SIZE];
  // do not optimize the space out
  buffer[STACK_SIZE-1] = 1;
  buffer[STACK_SIZE-1]--;
  if( ! setjmp(microthreads_threadResetHandle) )
    return;
  
  microthreads_beginThread( microthreads_currentThread * STACK_SIZE + 1 );
}

void microthreads_init(void)
{
  if( !microthreads_inited )
  {
    microthreads_disableInterrupts();
    int8_t i;
    
    microthreads_threadDescriptors[0].state = MT_Main;
    
    for( i=1; i < MAX_THREADS; i++ )
    {
      microthreads_threadDescriptors[i].state = MT_Unallocated;
    }
    microthreads_currentThread = 0;
    microthreads_inited = 1;
    microthreads_enableInterrupts();
    microthreads_threadStarter();
  }
}

int8_t microthreads_getCurrentThread(void)
{
  return microthreads_currentThread;
}

void microthreads_scheduleNext(void)
{
  if( !microthreads_inited )
    return;
  
  microthreads_disableInterrupts();
  if( !setjmp(microthreads_threadDescriptors[microthreads_currentThread].address.savedState) )
  {
    do {
      if((++microthreads_currentThread) >= MAX_THREADS)
	microthreads_currentThread = 0;
    }while(microthreads_threadDescriptors[microthreads_currentThread].state == MT_Unallocated);
    
    longjmp(microthreads_threadDescriptors[microthreads_currentThread].address.savedState, 1);
  }
  microthreads_enableInterrupts();
}

int8_t microthreads_startThread( void (*pointer)(void) )
{
  int8_t i;
  
  microthreads_disableInterrupts();
  
  for(i=1; i < MAX_THREADS; i++)
  {
    if( microthreads_threadDescriptors[i].state == MT_Unallocated )
    {
      if( !setjmp(microthreads_threadDescriptors[microthreads_currentThread].address.savedState))
      {
	microthreads_currentThread = i;
	microthreads_threadDescriptors[microthreads_currentThread].address.startingPoint = pointer;
	microthreads_threadDescriptors[i].state = MT_Running;
	longjmp(microthreads_threadResetHandle, 1);
      }
      microthreads_enableInterrupts();
      return i;
    }
  }
  return -1;
}

void microthreads_killThread(int8_t number)
{
  if((number > 0) && (number < MAX_THREADS))
  {
    microthreads_disableInterrupts();
    if(microthreads_threadDescriptors[number].state == MT_Running)
      microthreads_threadDescriptors[number].state = MT_Unallocated;
    microthreads_enableInterrupts();
    if( number == microthreads_currentThread )
      microthreads_scheduleNext();
  }
}

