/*
 * main.c
 *
 *  Created on: 14.10.2010
 *  (C) 2010 D.Goersch
 */


#include <avr/io.h>
#include <avr/pgmspace.h>
#include <avr/interrupt.h>
#include <util/delay.h>
#include <stdint.h>
#include "soft-pwm.h"

#define DELAY_FADE 11 			// wait XX ms between each fade step
#define DELAY_LED 20			// wait XX ms between each LED
#define DURATION_SHORT 15		// autofade off after XX sec if jumper is open (high)
#define DURATION_LONG 25		// autofade off after XX sec if jumper is closed (low)
#define LTR 0					// Fade fom PortA0 forwards
#define RTL 1					// Fade from the last (amount) to PortA0 backwards
#define OFF 0					// State of
#define ON 1					// the LEDs

uint8_t amount=0;					// Amount of LEDs
uint8_t time;					// Fade-Off time
uint8_t direction;				// 0 - LTR 1 - RTL
uint8_t state=OFF;				// 0 - OFF 1 - ON
volatile uint8_t int0=0, int1=0;// flags for interrupts
volatile uint8_t tenms=0;   	// +1 each 10ms
volatile uint8_t running=0;		// +1 each second

// dimming steps
uint8_t pwmtable[32] PROGMEM = {0, 1, 2, 2, 2, 3, 3, 4, 5, 6, 7, 8, 10, 11,
                                13, 16, 19, 23, 27, 32, 38, 45, 54, 64, 76,
                                91, 108, 128, 152, 181, 215, 255};


// patchtable for outputs
// uint8_t patchtable[24] PROGMEM = {0, 1, 2, 3, 4, 5, 6, 7, 8, 9,
//                                  10, 11, 12, 13, 14, 15, 16,
//                                  17, 18, 19, 20, 21, 22, 23};

// patchtable for outputs (u.wehner)
uint8_t patchtable[24] PROGMEM = {0, 1, 2, 3, 4, 5, 6, 7, 23, 22,
				  21, 20, 19, 18, 17, 16, 8, 9,
                                  10, 11, 12, 13, 14, 15};


void init(void);
void fade_on(uint8_t direction);
void fade_off(uint8_t direction);

int main(void)
{
	init();

	for(;;)
	{

		if(state==OFF)
		{
			if(int0 || int1) amount = ~((PIND & 0x03) | ((PIND >>2) & 0x1C)) & 0x1f;		// read PD0,1,4,5,6 as a dec. number
			if(int0)									// if LEDs off, turn on LTR
			{
				state = ON;
				direction = LTR;
				fade_on(direction);
				int0 = 0;

				TCNT0 = 100;							// Preload
				TCCR0 |= (1<<CS00) | (1<<CS02);			// Timer0 prescaler 1024
			}
			else if(int1)								// LEDs off, turn on RTL
			{
				state = ON;
				direction = RTL;
				fade_on(direction);
				int1 = 0;

				TCNT0 = 100;							// Preload
				TCCR0 |= (1<<CS00) | (1<<CS02);			// Timer0 prescaler 1024
			}
		}
		else if(state==ON)
		{
			if((!time & (running >= DURATION_SHORT)) | (time & (running >= DURATION_LONG))) {	// turn of automaticly
				TCCR0 = 0;								// disable timer0
				tenms = 0; running = 0;					// reset turnoff-timer
				fade_off(direction);
				state = OFF;

			}
			if(int0 & (direction==LTR)) int0=0;			// LEDs LTR on, left switch ignored
			else if(int1 & (direction==RTL)) int1=0;	// LEDS RTL on, right switch ignored
			else if(int0 & (direction==RTL))			// LEDs RTL on, fade off on left switch
			{
				TCCR0 = 0;								// disable timer0
				tenms = 0; running = 0;					// reset turnoff-timer
				fade_off(direction);
				state = OFF;
				int0 = 0;
			}
			else if(int1 & (direction==LTR))			// LEDs LTR on, fade off on right switch
			{
				TCCR0 = 0;								// disable timer0
				tenms = 0; running = 0;					// reset turnoff-timer
				fade_off(direction);
				state = OFF;
				int1 = 0;
			}
		}
	}
}

// do basic initialisation
void init(void)
{
	DDRA = 0xff;	// PortA
	DDRB = 0xff;	// PortB
	DDRC = 0xff;	// PortC as Output
	DDRD = 0x00;	// PortD as Input
	PORTD = 0xff;	// PullUps on PortD

//	moved to mainloop
//	amount = ~((PIND & 0x03) | ((PIND >>2) & 0x1C)) & 0x1f;		// read PD0,1,4,5,6 as a dec. number
	time   = PIND & 0x20;										// read PinD5 as fadeout mode

//	activated in mainloop after LEDs on
//	TCCR0 |= (1<<CS00) | (1<<CS02);	// Timer0 prescaler 1024
	TCNT0 = 100;					// Preload
	TIMSK |= (1<<OCIE0);			// enable Timer0

	TCCR1B = 1;             	// Timer1 runs full system clock
    TIMSK |= (1<<OCIE1A);   	// enable Timer1

    MCUCR |= (1<<ISC01);		// INT0 on falling edge
    MCUCR |= (1<<ISC11);		// INT1 on falling edge
    GICR |= (1<<INT0);			// enable INT0
    GICR |= (1<<INT1);			// enable INT1

    sei();
}

// fade the LEDs on
void fade_on(uint8_t direction)
{
	if (direction == LTR)
	{
		for(uint8_t led=0;led<amount;led++)
		{
			for(uint8_t bri=0;bri<32;bri++)
			{
				pwm_setting[pgm_read_byte(patchtable+led)] = pgm_read_byte(pwmtable+bri);
//				pwm_setting[patchtable[led]] = pwmtable[bri];
				_delay_ms(DELAY_FADE);
			}
			_delay_ms(DELAY_LED);
		}
	}
	else
	{
		for(uint8_t led=amount;led>0;led--)
		{
			for(uint8_t bri=0;bri<32;bri++)
			{
				pwm_setting[pgm_read_byte(patchtable+(led-1))] = pgm_read_byte(pwmtable+bri);
				_delay_ms(DELAY_FADE);
			}
			_delay_ms(DELAY_LED);
		}
	}
}

// fade the LEDs off
void fade_off(uint8_t direction)
{
	if (direction == LTR)
	{
		for(uint8_t led=0;led<amount;led++)
		{
			for(uint8_t bri=32;bri>0;bri--)
			{
				pwm_setting[pgm_read_byte(patchtable+led)] = pgm_read_byte(pwmtable+bri-1);
				_delay_ms(DELAY_FADE);
			}
			_delay_ms(DELAY_LED);
		}
	}
	else
	{
		for(uint8_t led=amount;led>0;led--)
		{
			for(uint8_t bri=32;bri>0;bri--)
			{
				pwm_setting[pgm_read_byte(patchtable+(led-1))] = pgm_read_byte(pwmtable+bri-1);
				_delay_ms(DELAY_FADE);
			}
			_delay_ms(DELAY_LED);
		}
	}
}

// set flags on interrupts
ISR(INT0_vect) { int0 = 1; }
ISR(INT1_vect) { int1 = 1; }

// timer for auto-off
ISR(TIMER0_COMP_vect)
{
	tenms++;
	if (tenms >= 100)
	{
		running++;
		tenms = 0;
	}
}
