/*
 * simple nixie clock - control program for pic 16f84a
 * (c) 2003 ch. klippel  -  ck@mamalala.de
 *
 * this implements a clock using nixie tubes. the clock can be free
 * running or synchronized to a dcf time telegram.
 * additionally, it implements an alarm function and a counter mode
 * to count signals on the io pin.
 *
 * version 1.4
 *
 * changelog:
 *
 * v1.5: - modified the resync timing
 *         it now has two stages: if we had no valid time telegram so far,
 *         the resync will be tried every 4 seconds (if there is a signal
 *         detected at the dcf input). once a valid time was received, the resync
 *         will only happen every 4 minutes for the remaining of that hour.
 *         the reason for that is, with very noisy signals and thus permanent
 *         sync attempts, the clock's free running time gets skewed because
 *         of the internal timer-resync. usually that is only a fraction of a second,
 *         but done repeatedly causes a big shift.
 *	   it didnt affect the clock when no signal is at the dcf input, but it was
 *         annoying in situations with really noisy signal reception, like at my place.
 *
 * v1.4: - slight change to the timer correction.
 *         now, when the correction values are set to 50 in the clock setup,
 *         after the timer is ready to count again its value is where it would
 *	   be when no timer was written. this is because we need certain instructions
 *	   to reach the point where the timer is adjusted. writing to the timer,
 *	   when the :2 prescaler is used, lets stay the counter at the written
 *	   value for 6 instructions, including the writing instruction.
 *	   to be in sync with the internal timer increment, the write must fall
 *	   into the beginning of the timer increment (see the microchip docs)
 *	   this makes the whole adjustment stuff better calculatable.	
 *
 * v1.3: - not really bugfixes. instead saved some codewords, before it
 *         used 1020, now only 1008. this is because some code movement
 *	   to avoid redundant checks, and then because we "clear" the used
 *         variables with the pre-programmed ee-prom contens. since we read
 *         it anyways, this saves to implement the clearRAM() stuff...
 *         its good to have some codewords left free in case some bug appears
 *         and needs some to fix it ....
 *         also, the clock parameters are now prefilled with something 
 *         meaningfull.
 *         the only real change in functionality is that the free-running time 
 *         update is now completely moved into the interrupt routine.
 *
 * v1.2: - time skew correction of loop a and b can now adjusted in the
 *	   basic setup
 *       - added second page to basic setup
 *	 - values in basic setup can now be cleared with Key_4
 *       - Key_6 in basic setup moves to second page, from there Key_8 exits
 *	 - improved eeprom load/store mechanism
 *
 * v1.1: - fixed a bug in key handling that prevented the counter alarm
 *         from beeing displayed correctly
 *       - modifyed key scanning. now uses more ram but less codewords.
 *       - improved dcf reception stability, 3 levels of checking selectable
 *       - length of auto-date display can now be set in 5 second steps 
 *       - the new parameters are saved to the internal eeprom also
 *	 - fixed some typos in the source code comentary
 *
 * v1.0: - initial release
 *
 * this code is released under the terms of the gpl, version 2 or newer.
 * see the file COPYING for more information about the license.
 *
 * compiler used is the cc5x from b. knudsen data.
 * if you dont have that compiler, dont worry. it has generated an
 * assembly file also, which should be included in the package you have
 * downloaded.
 *
 * the code is written for use with a 10.240 mhz oscillator.
 * this gives 2.560 mhz prescaler :2 input, 1.280 mhz tmr0 rate
 * tmr0 irq each 256 tmr0 steps = 5.000 khz irq-rate
 * each time the irq is called "a" is increased. if it reaches 125, "b" is
 * increased and "a" reset to zero. if b reaches 40, it is set to zero and 
 * one second passed. one increment of a equals to 0.200 ms, thus one 
 * increment of b equals to 25 ms.
 * after the start of a second, if b = 6 we can check for the dcf bit. 
 * if dcf-in is 1, the dcf-bit is 0, otherwise it is 1
 *
 * (in other words we could say that the internal secon-clock is a 40th of
 *  a 125th of a 256th of the half of the quarter crystal frequency ;-)
 *
 * after startup, we wait 4 seconds, after which we enable the irq for
 * pin rb0. if that pin changes from high to low, there is probably the start
 * of a dcf pulsed second and we sync the timer to it.
 * after that we wait for the missing 59th pulse to sync with the start of
 * a dcf time telegram. if this mark is found, we process each incomming
 * bit and set the current second to 59. from now on each dcf bit is
 * processed, and with the start of the 20th bit we start to decode the
 * time telegram.
 *
 * if for any reason the input rb0 is high after b=2, something went wrong
 * with the signal reception, and all sync bits are cleared.
 * each 4 seconds the sync procedure repeats until a sync can be established.
 * the internal clock the runs with oscillator accuracy.
 * the received time telegram is checked with the contained parity bits,
 * and if all is ok, the time and date is updated at the following second 0.
 */

/* define chiptype */

#pragma chip PIC16F84A

/* turn on compiler optimizations */

#pragma optimize=1

/* set chip configuration : xt oscillator, power up timer on, watchdog off */

#pragma config |= 0x3FF1

/* preset the data eeprom for a usable clock setup, also inits all variables ;-) */

#pragma cdata[0x2100] = 60, 15,  0,  0,  6, 45,  0,  0,  0, 50, 50,  3, 11,  3,  0,  0
#pragma cdata[0x2110] =  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3, 11,  3,  0
#pragma cdata[0x2120] =  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0
#pragma cdata[0x2130] =  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0


/* include stuff for interrupt handling */

#include "int16CXX.h"

/* variables that are read/written from/to the eeprom */

static char EE_Array[11]   	@ 0x0C;
static char Blend_Time	   	@ EE_Array[0];
static char Snooze_Time    	@ EE_Array[1];				/* configures blend- and snooze-time */
static char DCF_Mode	   	@ EE_Array[2];				/* level of dcf signal checking. 0=minimal, 1=extended, 2=paranoid */
static char Auto_Date	   	@ EE_Array[3];				/* seconds to display the date before a new minute */
static char AL_Hour	   	@ EE_Array[4];				/* alarm time */
static char AL_Minute	  	@ EE_Array[5];				
static char Counter_Set_a  	@ EE_Array[6];				/* trigger counter Alarm_On at these values */
static char Counter_Set_b  	@ EE_Array[7];
static char Counter_Set_c  	@ EE_Array[8];
static char Loop_a_Corr	   	@ EE_Array[9];
static char Loop_b_Corr	   	@ EE_Array[10];				

/* global variables */

static char Year		@ 0x17;
static char Month		@ 0x18;
static char Day			@ 0x19;			/* actual time and date */
static char Key_Bits[6]    	@ 0x1A;			/* this wastes 3 bytes, but we have enough... */
static char Key_Bits1      	@ Key_Bits[2];					/* each key row now has it own byte to simplify handling */
static char Key_Bits2     	@ Key_Bits[4];
static char Key_Bits3     	@ Key_Bits[1];
static char Hour		@ 0x20;
static char Minute		@ 0x21;
static char Second		@ 0x22;
static char Old_Hour		@ 0x23;
static char Old_Minute		@ 0x24;
static char Old_Second		@ 0x25;			/* old date _or_ values to display when not showing the time */
static char DCF_Hour		@ 0x26;
static char DCF_Minute		@ 0x27;
static char DCF_Second		@ 0x28;
static char DCF_Year		@ 0x29;
static char DCF_Month		@ 0x2A;
static char DCF_Day 		@ 0x2B;			/* received dcf time, all in bcd except the second */
static char Snooze_Remain	@ 0x2C;			/* how many snooze time left ? */
static char DCF_BITS		@ 0x2D;			/* number of even bits of current dcf part */
static char Unsynced		@ 0x2E;			/* seconds without sync */
static char Day_Test		@ 0x2F;					/* for calendaring */
static char Status_Bits		@ 0x30;
static char More_Bits		@ 0x31;		/* more status bits */
static char More_Bits2		@ 0x32;
static char Counter_a		@ 0x33;
static char Counter_b		@ 0x34;
static char Counter_c		@ 0x35;		/* actual counter values */
static char Display_Num		@ 0x36;
static char cc			@ 0x37; 	/* temporary variables */
static char ca			@ 0x38;
static char cb			@ 0x39;
static char cd			@ 0x3A;
static char ce			@ 0x3B;
static char cf			@ 0x3C;
static char cg			@ 0x3D;			/* temporary variables */

/* specific bits from the variables above */

static bit Key_1           	@ Key_Bits1.1;
static bit Key_2           	@ Key_Bits1.2;
static bit Key_3           	@ Key_Bits1.3;
static bit Key_4          	@ Key_Bits2.1;
static bit Key_5          	@ Key_Bits2.2;
static bit Key_6          	@ Key_Bits2.3;
static bit Key_7         	@ Key_Bits3.1;
static bit Key_8         	@ Key_Bits3.2;
static bit Key_9        	@ Key_Bits3.3;		
static bit DCF_PBIT 		@ DCF_BITS.0;		/* to check the parity of the time telegram */
static bit Second_Sync		@ Status_Bits.0;		/* 1 = sync to dcf second pulse,   0 = out of sync */
static bit DCF_Sync 		@ Status_Bits.1;		/* 1 = in sync with dcf telegram,  0 = out of sync */
static bit DCF_OK 		@ Status_Bits.2;		/* 1 = time telegram valid so far, 0 = error */
static bit Scan_Keys 		@ More_Bits.1;		/* time to check the keys for autorepeat */
static bit Last_DCF_OK 		@ More_Bits.2;		/* indicates if the last dcf datagram was ok or not */
static bit Had_DCF		@ More_Bits.3;
static bit DCF_Bit 		@ More_Bits.6;		/* currently received dcf bit */
static bit Has_DCF_Bit 		@ More_Bits.7;		/* flags the arrival of a new dcf bit */
static bit Last_IO 	  	@ More_Bits2.0;		/* last state of the counter input pin */
static bit Alarm_On 	  	@ More_Bits2.1;		/* is the alarm time reached ? */
static bit Alarm_Active  	@ More_Bits2.2;
static bit Alarm_Changed 	@ More_Bits2.3;		/* indicates that the alarm time has changed */
static bit Test_Bit 	  	@ More_Bits2.5;		/* for general testing */
static bit Count_Reached 	@ More_Bits2.6;		/* is the programmed counter value reached ? */
static bit Enable_Blend  	@ More_Bits2.7;		/* blend between old and new numbers or not */
static bit Toggle_Bit 	  	@ cg.0;			/* to half the rate of the blend time by two */

/* assign names to port pins */

bit in  @ PORTB.0;	/* dcf input, open collector, logic 1 = 0v, logic 0 = +5v (via pullup) */
bit in1 @ PORTB.1;	/* in1 - in3 are the input pins for the keyboard */
bit in2 @ PORTB.2;
bit in3 @ PORTB.3;
bit AL  @ PORTA.0;	/* alarm output */
bit IO  @ PORTA.1;	/* io-pin, used for counter input */

void Check_Parity   ( void );	/* calculate the parity of the dcf datagram */
void Process_DCF_Bit( void );	/* process the actual dcf-bit, if any */
void Update_Display ( void );	/* update the nixie display and read in the keys */
void Check_Counter(void);
void Advance_Day(void);
char BCD_2_Dec( char BCD );

#pragma origin 4	/* start of the irq vector */
interrupt ISR(void)	/* interrupt service routine */
{
  static char a @ 0x3F, b @ 0x40;	/* internal counters, if a = 125 and b = 40 then one second passed */
  int_save_registers    /* save working registers */

  if ( T0IF )		/* timer interrupt occured */
  {
    a++;		/* increase the "a" counter */

    if(a>124)				/* 125 counts done ? (0 to 124 = 125 steps = 25 ms) */
    {					/* if so .... */
      nop();	        		/* to sync the timer adjust to its internal stepping */
      TMR0 = Loop_a_Corr + 216; 	/* when corr = 50, this means "no change" (see microchip datasheets) */

      a = 0;		/* clear counter a */
      b++;		/* and increase counter b */
      
      if (b == 2 && in && Second_Sync) 	/* shortly after a second started, there must be the dcf bit */
      {					/* if it is _not_ there, and we are in sync with the dcf second pulse */
        if ( DCF_Second != 59)		/* ... check for the missing 59th second (datagram start/end mark) */
        {				/* if it is not .... */
          if(DCF_Sync)			/* but we had synced to the dcf datagram before .... */
	  {
	    Second_Sync = 0;		/* ... then reset the sync flags, because we are out of sync */
	    DCF_Sync = 0;
	  }
	  else				/* but if had no dcf datagram sync before .... */
	    DCF_Second = 59;		/* ... then we just found it ;-) */
          DCF_OK = 0;			/* reset the dcf datagram status, because if we are here, we need to */
        }

        if ( DCF_Second == 59)		/* and when we found the 59th second */
           DCF_Sync = 1;		/* .... we set the status to in-sync */
      }
     
      if (b == 6)			/* a bit more later, the state of the input is the actual dcf-bit */
        DCF_Bit = !in;			/* but inverted, so we invert it here */


      if(!(b & 0b.0000.0111))		/* after 200 ms .... */
        Scan_Keys = 1;			/* flag for scanning the keys */

      if (b == 40)			/* has the b counter reached 40, i.e. one second is over ? */
      {
        TMR0 = Loop_b_Corr + 228;       /* again, this means no adjust when corr = 50 */
				
        b = 0;				/* reset the b counter */

        if(Second_Sync)			/* are we in sync with the dcf second pulse ? */
          Has_DCF_Bit = 1;		/* if so, flag it */
        else
          if (!Unsynced.2 && !Had_DCF)		/* if we are not in sync with the dcf second pulse */
            Unsynced++;			/* increse the unsynced counter */

        if(Enable_Blend)
          Old_Second = Second;		/* save the old second */
        cb = 0;				/* restart the blend-counter */

        Second++;			/* and increase the actual second */
        if (Second > 59)				/* after 60 seconds .... */
        {
          Second = 0;				/* ... set to second 0 .... */
          Minute++;					/* ... and add a minute */

          if(Snooze_Remain != 0)			/* is snooze active ? (that is, alarm-pause */
            Snooze_Remain--;			/* if we snooze, one minute less is left to snooze */

          if (!Unsynced.2 && Had_DCF)		/* if we are not in sync with the dcf second pulse */
            Unsynced++;			/* increse the unsynced counter */
        }

        if (Minute > 59)				/* and after 60 minutes .... */
        {
          Minute  = 0;				/* ... the minute will be 0 .... */
          Had_DCF = 0;
          Hour++;					/* ... and a new hour started .... */
        }
      
        if (Hour > 23)				/* the 25'th hour ....*/
        {
          Hour = 0;				/* ... doesnt exist, so it is 0 ... */
          Advance_Day();			/* but means we have a new day ... */
        }

        if (!Status_Bits)	/* do we miss any dcf sync bits ? */
        {
          if (Unsynced.2)				/* if so, and at least 4 seconds passed without any sync ...*/
          {
            Unsynced = 0;				/* reset unsync-counter */
            INTF = 0;				/* reset dcf-io irq flag */
            INTE = 1;				/* and enable the dcf-io irq. when that happens, it must */
   						/* be the start of a second */
          }
        }
      }
    }

    if(Last_IO && !IO)			/* check if we had a high/low transition on the counter input */
    {
      Counter_a++;			/* if so, increase the 10's and ones of the counter */
      Check_Counter();                  /* check the counter for overflows */
      Last_IO = 0;			/* this pulse is done, so clear its flag */

      					/* now lets check if the counter reached the programmed */
					/* value */
      if(Counter_a == Counter_Set_a && Counter_b == Counter_Set_b && Counter_c == Counter_Set_c )
        Count_Reached = 1;		/* ....and if we reched the count, flag it */
    }

    Last_IO = IO;			/* remember the actual state of the io pin */ 

    if (b > 8 && !in && DCF_Second != 59)	/* if the dcf input not as it should be at this time .... */
    {
      if (DCF_Mode.1)				/* ... and we do paranoid cheking every 0.2 ms ... */
        goto resetdcf;				/* reset the dcf state */
      
      if(!a && DCF_Mode.0)			/* ... or extended checking every 25 ms ... */
        goto resetdcf;				/* reset the dcf state */
      
      goto donecheck;				/* but if the signal seems ok, continue */

      resetdcf:					/* reset the dcf flags */
      Status_Bits = 0;
   
      donecheck:
    }   

    if(Alarm_Active || Count_Reached)
      AL = 1;
    else
      AL = 0;

    while(TMR0 > 216) 			/* sanity check if the timer adjust was to heavy */
      {}				/* this is to avoid a unwanted timer irq */

    RTIF = 0;  				/* reset the timer interrupt flag */
  }

  if( INTF )				/* check if we had a level change on the dcf input */
  {
    if( INTE ) 				/* if so, and we expected it .... */
    {
      INTF = 0;				/* clear the interrupt flag */
      INTE=0;				/* and disable this interrupt */
      Second_Sync = 1;			/* we suspect te reason for this irq a dcf second start */
      a = 0;				/* because of that we clear counter a */
      b = 0;				/* and counter b to sync the internal clock */
    }
  }

  int_restore_registers			/* irq done, resore the working registers */
}

char BCD_2_Dec( char BCD )		/* convert a 8 bit bcd number to decimal */
{
   static char cx @ 0x41,cy @ 0x42;			/* temporary variables */
   cx = BCD & 0b.11110000;		/* get the bcd 10's of the bcd num */
   					/* the following code equals to : */
					/* make it bcd-ones and multiply by ten */
   cy = cx >> 1;			/* imaginary "one's * 8" */
   cx >>= 4;				/* 10's to ones */
   cy += cx;				/* plus ones = one's * 9 */
   cy += cx;				/* plus ones = one's * 10 */
   return((BCD & 0b.00001111) + cy); 	/* return bcd 1's plus decimal 10's */
}

void Save_Old_Time ( void )		/* remember the just passed time */
{
  Old_Hour = Hour;			/* the hour .... */
  Old_Minute = Minute;			/* .... the minute ... */
  Old_Second = Second;			/* ... and finally the second */
}

void Advance_Day( void )		/* advance the date by one day, to keep it current when */
					/* there is no dcf reception */
{
  Day++;				/* advance the current day */
  Day_Test = 32;			/* check for the 32st day, which means a new month is reached */

  if(Month == 4 || Month == 6 || Month == 9 || Month == 11) /* but first check if the actual month ... */
    Day_Test = 31;			/* only has 30 days, i.e. we have the 31th day */

  if(Month == 2)			/* but in februrary .... */
  {
    Day_Test = 30;			/* ... we have only 29 days, so if this is the 30th day .... */
      if(Year.0 || Year.1)		/* ... except for the special 4th year ... */
    Day_Test--;			/* ... which only has 28 days, so check for the 29th day reached */
  }
 
  if(Day == Day_Test)		/* so, if we have reached the next month .... */
  {
    Day = 1;			/* start the month at day 1 */
    Month++;			/* and advance the month */
  }

  if(Month > 12)			/* but if we passed the december */
  {
    Month = 1;			/* ... it must be now the first month */
    Year++;			/* HAPPY NEW YEAR ! ;-) */
  }

  if(Year > 99)
    Year = 0;
}

void Write_EE(void)
{
  for(EEADR = 0; EEADR < 11; EEADR++)
  {
    EEDATA = EE_Array[EEADR];
    GIE = 0;				/* first, disable irq's during the write */
    WREN = 1;				/* enable the eeprom-write */
    EECON2 = 0x55;			/* this is the sequence needed to write the eeprom */
    EECON2 = 0xaa;			/* see the microchip datasheet */
    WR = 1;				/* do the write */
    GIE = 1;				/* re-enable the irq's */
    EEIF = 0;				/* clear the ee-write done flag */
    while(!EEIF)			/* until the flag is set, the write is in progress, so wait */
      Update_Display();
  }
}

void Read_EE(void)
{
  for(EEADR = 0; !EEADR.6; EEADR++)
  {
    RD = 1;
    EE_Array[EEADR] = EEDATA;
  }

}

void Check_Counter(void)		/* check the 3 counter variables for overflow and adjust them */
{
  if(Counter_a > 99)			/* check if the counter 10's and ones overflowed */
  {
    Counter_a = 0;			/* reset them if so */
    Counter_b++;				/* and advance the 100's and 1000's */
  }

  if(Counter_b > 99)			/* check if the 100's and 1000's overflowed */
  {
    Counter_b = 0;			/* reset them if so */
    Counter_c++;				/* advance the 10000's and 100000's */
  }
  
  if(Counter_c > 99)			/* check if the 10000's and 100000's overflowed */
    Counter_c = 0;			/* reset them, if so */
}
void main( void)			/* the big blob of code that does nothing magical .... */
{
restart:				/* location to start all that stuff */

  TRISA = 0b.0000.0010;  		/* setup portb, 1 = input, 0 = output */
  TRISB = 0b.0000.1111;  		/* setup portb, 1 = input, 0 = output */
  OPTION = 128; 		 	/* prescaler divide by 2  */
  INTEDG = 0; 		 		/* rb0 interrupt on falling edge */
  INTCON = 160;				/* enable the irq's we want */
  AL = 0;				/* we dont want the alarm on at startup */

  Read_EE();

Update_Display();
  if(!Key_8)				/* if no setup-key (Key_1) is pressed at startup... */
    goto start;				/* go to the main stuff */

basicsetup:

  Enable_Blend = 0;					/* otherwise do the setup-loop */
  Old_Hour = Auto_Date;
  Old_Minute = Blend_Time;
  Old_Second = Snooze_Time;

  Update_Display();			/* and show the values */

  if(Scan_Keys)				/* time to scan for autorepeat ? */
  {
    if(Key_4)
    {
      Auto_Date = 0;
      Blend_Time = 1;
      Snooze_Time = 1;
    }

    Auto_Date += Key_1;			/* Key_1 adjusts automatic date display time in 5 sec steps */
    Blend_Time += Key_2;		/* Key_2 adjust the blend time */
    Snooze_Time += Key_3;		/* Key_3 sets the snooze time */

    if(Auto_Date > 60)			/* auto-date time too high ? */
      Auto_Date = 0;			/* then reset it */

    if(Blend_Time > 99)			/* blend time too high ? */
      Blend_Time = 1;			/* then restart blend time from 1 */

    if(Snooze_Time > 99)		/* more than 99 minutes snooze time ? */
      Snooze_Time = 1;			/* snooze only one minute */

    Scan_Keys = 0;			/* we are done with checking the keys */
  }
 
if(!Key_6)
  goto basicsetup;				/* but if dont exit the setup, redo it until its done */

basicsetup2:					/* otherwise do the setup-loop */
  Old_Hour = DCF_Mode;
  Old_Minute = Loop_a_Corr;
  Old_Second = Loop_b_Corr;

  Update_Display();			/* and show the values */

  
  if(Key_4)
  {
    Loop_a_Corr = 0;
    Loop_b_Corr = 0;
  }

  if(Scan_Keys)				/* time to scan for autorepeat ? */
  {
    DCF_Mode += Key_1;			/* Key_1 adjusts automatic date display time in 5 sec steps */
    Loop_a_Corr += Key_2;		/* Key_2 adjust the blend time */
    Loop_b_Corr += Key_3;		/* Key_3 sets the snooze time */

    if(DCF_Mode > 2)			/* auto-date time too high ? */
      DCF_Mode = 0;			/* then reset it */

    if(Loop_a_Corr > 99)			/* blend time too high ? */
      Loop_a_Corr = 0;			/* then restart blend time from 1 */

    if(Loop_b_Corr > 99)		/* more than 99 minutes snooze time ? */
      Loop_b_Corr = 0;			/* snooze only one minute */

    Scan_Keys = 0;			/* we are done with checking the keys */
  }

if(!Key_8)
  goto basicsetup2;				/* but if dont exit the setup, redo it until its done */

  Write_EE();					/* save the setup to the eeprom */
  Read_EE();

start:					/* here the real code starts */

  while (1)				/* we never finish that big loop, for sure */
  {

    if(Has_DCF_Bit)				/* do we have a fresh dcf bit ? */
      Process_DCF_Bit();			/* if we have, process it of course ... */

    if (!DCF_Second)				/* if we have 60 seconds received by the dcf */
    {
      Last_DCF_OK = DCF_OK;			/* remember the current status of the dcf datagram */

      if(Status_Bits == 7) 	/* and all is in sync ... */
      {
         Hour = BCD_2_Dec(DCF_Hour);		/* ... update the hour to the dcf hour */
         Minute = BCD_2_Dec(DCF_Minute);	/* and the mite */
         Second = DCF_Second;			/* not to forget the second */
         Year = BCD_2_Dec(DCF_Year);		/* but also the year */
         Month = BCD_2_Dec(DCF_Month);	/* the month */
         Day = BCD_2_Dec(DCF_Day);		/* day too */
         Last_DCF_OK = 1;			/* and flag that the last datagram was ok */
         Had_DCF = 1;
      }
    }

    if (cb == Blend_Time)			/* are we finished with fading away the old numbers ? */
    {
      Save_Old_Time();				/* then make the old time the current one, which means */
      cb = 0;					/* to blend between the same numbers, and that is no blending at all ;-) */
    }

    if(Key_7)					/* is the alarm-switch turned on ? */
    {
      if(Key_Bits1 != 0)				/* and is the snooze-button pressed ? */
        Snooze_Remain = Snooze_Time;		/* then set to the configured snooze time */
      
    if(Hour == AL_Hour && Minute == AL_Minute) /* if we have reached the alarm time ... */
      Alarm_On = 1;

    }

    if(!Key_7)
      Alarm_On = 0;

    if(!Snooze_Remain && Alarm_On)		/* if we have no snooze, but a flagged alarm */
      Alarm_Active = 1;
    else
      Alarm_Active = 0;

    Enable_Blend = 1;				/* eneble blending of the numbers */

    if(Scan_Keys)				/* check if a autorepeat intervall has occured */
    {
      Scan_Keys = 0;				/* ... reset the autorepeat flag then ... */

      if(Key_8)					/* when it was Key_8, we can adjust the time */
      {
        if(Key_4) Hour++;				/* advance the hour if Key_4 is pressed */
        if(Key_5) Minute++;			/* advance the minute if Key_5 is pressed */

	if(Key_6)				/* reset time with Key_6 */
	{
	  Hour = 0;
	  Minute = 0;
	  Second = 0;
	}

	if(Minute > 59)				/* check if the minute overflowed */
	  Minute = 0;				/* clear minute, if so .... */

	if(Hour > 23)				/* check if the hour overflowed */
	  Hour = 0;				/* ... and reset, if .... */
        goto nextkey8;
      }

      if(Key_9)					/* process keys when Key_9 is active (counter mode) */
      {
        if(Key_4) Counter_c ++;			/* advance the 10000's and 100000's if Key_6 is pressed */
        if(Key_5) Counter_b ++;			/* advance 100's and 1000's if Key_5 is pressed */
	if(Key_6) Counter_a ++;			/* advance 10's and ones if Key_4 is pressed */
	Check_Counter();			/* check for counter overflows and adjust if needed */
        
        if(Key_3)
        {
          Counter_Set_a = Counter_a;		/* store counter content as counter-alarm point */
          Counter_Set_b = Counter_b;
          Counter_Set_c = Counter_c;
          Write_EE();
          Enable_Blend = 0;
          while(Key_3)
          {
            Old_Second = Counter_a;
            Update_Display();
          }            
        }
        goto nextkey9;
      }

      if(Key_2)					/* check for Key_2, adjust alarm when pressed */
      {
        if(Key_4) AL_Hour++;			/* advance alarm hour if Key_4 is pressed */
        if(Key_5) AL_Minute++;			/* advance alarm minute if Key_5 is pressed */

	if(Key_6)				/* clear alarm time with Key_6 */
	{
	  AL_Hour = 0;
	  AL_Minute = 0;
	}

	if(AL_Minute > 59)			/* check if alarm minute overflowed */
	  AL_Minute = 0;			/* and clear it, if so */

	if(AL_Hour > 23)			/* check if the alarm hour overflowed */
	  AL_Hour = 0;				/* and clear it, if so */

        if(Key_Bits2) 				/* if one of the set keys was pressed .... */
          Alarm_Changed = 1;			/* flag the change of the alarm time */
        goto nextkey2;
      }

      if(Key_1)					/* check if Key_1 is pressed, then we can adjust the date */
      {
   	if(Key_3)			/* if Key_4 and Key_5 are pressed at the same time ... */
 	{
	  Day = 1;				/* reset the date */
	  Month = 1;
          Year = 0;
          goto nextkey1;          
        }

        if(Key_4)				/* ... with Key_4 we set the day .... */
        {
	  Advance_Day();			/* ... by advancing it and checking for month-length */
        }

	if(Key_5)				/* check if Key_5 is pressed ... */
	{
	  Month++;				/* ... advance the month, if so */
	  Day = 1;				/* and reset the day */
	}

	if(Key_6)				/* check if Key_6 is pressed ... */
	{
	  Year++;				/* ... advance the year, if so ... */
	  Day = 1;				/* and reset the month */
	}

        if(Month > 12)				/* check if the month overflowed */
	  Month = 1;				/* and reset it, if so */

        if(Year > 99)				/* now check if the year overflowed */
	  Year = 0;				/* and reset it, if so */
      }
    goto nextkey1;
    }

    if(Key_9)
    {
nextkey9:
      Old_Hour = Counter_c;			/* show the counter value by default */
      Old_Minute = Counter_b;
      Old_Second = Counter_a;

      if(Key_1)					/* when Key_1 is pressed in counter-mode */
      {
        Old_Hour = Counter_Set_c;			/* show the counter-alarm instead */
        Old_Minute = Counter_Set_b;
        Old_Second = Counter_Set_a;
      }

      if(Key_2)				/* if Key_2 is pressed in counter mode ... */
      {
         Counter_a = 0;			/* clear the counters */
         Counter_b = 0;
         Counter_c = 0;
         Count_Reached = 0;			/* and clear the counter alarm also */
      }
 
      Enable_Blend = 0;				/* turn of number blending */
      goto keysdone;
    }

nextkey1:
    cf = 60 - Auto_Date;			/* at what second should the date display start */
    if(Second >= cf && Key_Bits1 == 0)		/* if that time is reached, and no accountable keys pressed ... */
      Key_1 = 1;				/* emulate the pressing of Key_1 to show the date */

    if (Key_1)					/* if Key_1 is pressed */
    { 
      Old_Second = Year;			/* show the date */
      Old_Minute = Month;
      Old_Hour = Day;
      Enable_Blend = 0;				/* and turn of number blending */
goto keysdone;
    }
 
    if(Key_2)					/* if Key_2 is pressed */
    {
nextkey2:
      Old_Hour = AL_Hour;			/* show the alarm time */
      Old_Minute = AL_Minute;
      Old_Second = 0;
      Enable_Blend = 0;				/* disable number blending */
goto keysdone;
    }

    if (Key_3)					/* if Key_3 is pressed */
    {
      Old_Hour = Second_Sync;
      Old_Hour += DCF_Sync;
      Old_Hour += DCF_OK;

      if(Last_DCF_OK)				/* if the last dcf datagram was ok */
        Old_Hour += 10;				/* show it as 1 one the hour 10's */

      Old_Minute = 0;				/* set minute display to 00 */

      if(DCF_Bit)				/* if the received dcf bit is 1 */
        Old_Minute = 10;			/* show it in the minute 10's */

      if(!in)					/* if the dcf input low (logic 1 for that signal) */
        Old_Minute ++;				/* show it in minutes 1 */

      Old_Second = DCF_Second;			/* show the actual/estimated dcf second in the seconds display */
      Enable_Blend = 0;				/* disable number blending */
    }

nextkey8:
    if(Key_8 && Key_1)				/* is Key_8 and Key_1 pressed at the same time ? */
      goto basicsetup;				/* if so, jump to the initial setup */

keysdone:

    if(Alarm_Changed && Key_Bits1 == 0)		/* if the alarm was changed, and the keys released ... */
    {
      Alarm_Changed = 0;
      Write_EE();
    }

    Update_Display();				/* finally, show the stuff */
  }
}

char Div_by_10(char what)			/* fixed division by 10, saves codewords */
{
static char cx @ 0x43;				/* working variable */

  cx = 0;					/* init variable to 0 */
  while(what >= 10)				/* as long as the number is bigger than 10 */
  {
    what -= 10;					/* substract 10 */
    cx++;					/* and increase the working variable */
  }
  return cx;					/* return the working variable to the caller */
}

char Modulo_10(char what)			/* fixed modulo 10, saves codewords */
{
  while(what >= 10)				/* as long as the number is bigger than 10 */
  {
    what -= 10;					/* substract 10 */
  }
  return what;					/* return the remaining number to the caller */
}

void Update_Display( void )			/* display output and key scanning routine */
{
  static   char col @ 0x44, oldport @ 0x45, real_col @ 0x46, s @ 0x47;	/* local variables */
  col = 5;
  while(col != 255)
  {
    #pragma computedGoto 1			/* computed goto's are like a switch() in c, but less codewords */
    s = col << 3;
    s -= col;
    skip(s);					/* select what we have to display */
    case0:  Display_Num = Modulo_10(Second); 	/* one's of the actual second */
            cc = Modulo_10(Old_Second); 		/* one's of the old second */
            goto done;				/* continue with display */
    case1:  Display_Num = Div_by_10(Second); 	/* 10's of the actual second */
            cc = Div_by_10(Old_Second); 		/* 10's of the old second */
            goto done;				/* continue with display */
    case2:  Display_Num = Modulo_10(Minute); 	/* now the one's of the minute */
            cc = Modulo_10(Old_Minute); 		/* and the old minute one's */
            goto done;				/* continue with display */
    case3:  Display_Num = Div_by_10(Minute); 	/* 10's of the minute */
            cc = Div_by_10(Old_Minute); 		/* 10's of the old minute */
            goto done;				/* continue with display */
    case4:  Display_Num = Modulo_10(Hour); 	/* the hour one's */
            cc = Modulo_10(Old_Hour); 		/* and the old hour one's */
            goto done;				/* continue with display */
    case5:  Display_Num = Div_by_10(Hour); 	/* the hour 10's */
            cc = Div_by_10(Old_Hour); 		/* and finally the old hour 10's */
    done:
    #pragma computedGoto 0

    Display_Num <<=4;		/* combine the old and new number to display, actual value in the higher 4 bit */
    Display_Num += cc;		/* the old stuff goes to the lower 4 bits */

    oldport = PORTA & 227;	/* remember actual porta bits, but mask out the column bits */

    col <<= 2;			/* move the colum-counter bits to the right position */
    real_col = oldport + col;	/* add the column to the old, masked porta bits, store in a new variable */
    col >>=2;			/* shift back the column-counter bits */
    oldport += 28;		/* set all column bits on the old, stored porta value */
    ca = 0;			/* initialize the blend-time counter */
    PORTA = oldport;		/* blank the display */

    if(Enable_Blend)		/* if blending is enabled */
    {
      PORTB.4 = Display_Num.4;	/* set the number-bits of the output to the actual number bits */
      PORTB.5 = Display_Num.5;
      PORTB.6 = Display_Num.6;
      PORTB.7 = Display_Num.7;
      PORTA = real_col;		/* turn on the display at the right column */
      for (;ca < cb; ca++) ;	/* show it until the blend-time has reached the blend counter */
    }
    else				/* otherwise */
      cb = 0;			/* reset the blend counter */

    PORTA = oldport;		/* blank the display */
    PORTB.4 = Display_Num.0;	/* set the number bit of the output to the old number bits */
    PORTB.5 = Display_Num.1;
    PORTB.6 = Display_Num.2;
    PORTB.7 = Display_Num.3;
    PORTA = real_col;		/* turn on again at the same column */
    for(; ca < Blend_Time; ca++); /* display it for the remaining blend time */

    Key_Bits[col] = PORTB & 14;

    col--;
  }

  cg++;						/* toggle the toggle bit, its bit0 @ cg */
  cb+= Toggle_Bit;				/* and decrease the blend timer, if the toggle is 1 */
}

void First_Check_Parity( void )	/* check parity bit for dcf hour and minute */
{
  if (!DCF_PBIT)		/* if no parity bit is set */
      DCF_OK = 1;		/* then clear the dcf status bit */
}

void Check_Parity( void )	/* check parity bit for dcf hour and minute */
{
  if (DCF_PBIT)		/* if no parity bit is set */
      DCF_OK = 0;		/* then clear the dcf status bit */
  DCF_PBIT = 0;		/* clear the parity bit */
}

void Process_DCF_Bit( void)		/* process an incomming dcf bit */
{
 static  char s @ 0x48;			/* internal jump variable */

  				/* if we are in sync with the dcf signal ... */
    if(DCF_Sync && DCF_Second > 20)			/* ... and more than 20 dcf seconds elapsed */
    {					/* we can start the processing */
      if(DCF_Bit) DCF_BITS++;		/* if the actual bit is 1, increase the dcf-bits counter, toggling the parity bit */
      #pragma computedGoto 1
      s = DCF_Second - 21;		/* prepare adress to jump to */
      s <<= 2;
      skip(s);				/* select which bit to process */
      Case21 : DCF_Minute.0 = DCF_Bit; goto end;	/* minute bit0 */
      Case22 : DCF_Minute.1 = DCF_Bit; goto end;	/* minute bit1 */
      Case23 : DCF_Minute.2 = DCF_Bit; goto end;	/* ... */
      Case24 : DCF_Minute.3 = DCF_Bit; goto end;
      Case25 : DCF_Minute.4 = DCF_Bit; goto end;
      Case26 : DCF_Minute.5 = DCF_Bit; goto end;
      Case27 : DCF_Minute.6 = DCF_Bit; goto end;
      Case28 : First_Check_Parity(); nop(); nop(); goto end;	/* minute complete, check for parity */
      Case29 : DCF_Hour.0 = DCF_Bit; goto end;		/* hour */
      Case30 : DCF_Hour.1 = DCF_Bit; goto end;
      Case31 : DCF_Hour.2 = DCF_Bit; goto end;
      Case32 : DCF_Hour.3 = DCF_Bit; goto end;
      Case33 : DCF_Hour.4 = DCF_Bit; goto end;
      Case34 : DCF_Hour.5 = DCF_Bit; goto end;
      Case35 : Check_Parity(); nop(); nop(); goto end;		/* hour complete, check for parity */
      Case36 : DCF_Day.0 = DCF_Bit; goto end;		/* day */
      Case37 : DCF_Day.1 = DCF_Bit; goto end;
      Case38 : DCF_Day.2 = DCF_Bit; goto end;
      Case39 : DCF_Day.3 = DCF_Bit; goto end;
      Case40 : DCF_Day.4 = DCF_Bit; goto end;
      Case41 : DCF_Day.5 = DCF_Bit; goto end;
      Case42 : nop(); nop(); nop(); goto end;		/* unused, is: day of week */
      Case43 : nop(); nop(); nop(); goto end;
      Case44 : nop(); nop(); nop(); goto end;
      Case45 : DCF_Month.0 = DCF_Bit; goto end;		/* month */
      Case46 : DCF_Month.1 = DCF_Bit; goto end;
      Case47 : DCF_Month.2 = DCF_Bit; goto end;
      Case48 : DCF_Month.3 = DCF_Bit; goto end;
      Case49 : DCF_Month.4 = DCF_Bit; goto end;
      Case50 : DCF_Year.0 = DCF_Bit; goto end;		/* year */
      Case51 : DCF_Year.1 = DCF_Bit; goto end;
      Case52 : DCF_Year.2 = DCF_Bit; goto end;
      Case53 : DCF_Year.3 = DCF_Bit; goto end;
      Case54 : DCF_Year.4 = DCF_Bit; goto end;
      Case55 : DCF_Year.5 = DCF_Bit; goto end;
      Case56 : DCF_Year.6 = DCF_Bit; goto end;
      Case57 : DCF_Year.7 = DCF_Bit; goto end;
      Case58 : Check_Parity(); nop(); nop();		/* date complete, check parity */
      end:
      #pragma computedGoto 0
  }
  else 
    DCF_PBIT = 0;


  if(Has_DCF_Bit)	/* if we were flagged for a new bit, which means a new dcf second .... */
    DCF_Second++;	/* ... then increase the dcf second */

  if(DCF_Second > 59)	/* check if the dcf second overflowed */
  {
    DCF_Second = 0;	/* if so, clear it */
    DCF_PBIT = 0;	/* and also clear the parity bit */
  }
  Has_DCF_Bit = 0;	/* clear new-dcf-bit flag */
}

/* thats all folks, thanks for reading ;-) */
