;************************************************************************
; Following the example of Bob Blick's original propeler clock I made 	*
; my own version of this fancy clock and have re-written the code from	*
; scratch. I used a bit more powerfull version of PIC cpu and added 	*
; these features:														*
;  - the clock keeps time AND DATE and Day of week     					*
;  - can display analogue hands or digital time and mixed mode where	*
;    time is displayed as analogue hands and digital date				*
;  - setting the time/date or switch mode is done by RC5 remote control	*
;  - possibility to display a message programmed in ROM or EEPROM		*
;                                                                 		*
;************************************************************************
;                                                               		*
;    Filename:	    prop_bright.asm                                    	*
;    StartDate:     02/12/2001                                        	*
;    LastUpdate:    20/10/2009											*
#define	VersionMajor	0x03	;										*
#define	VersionMinor	0x25	;										*
;#define	DebugI2C	; comment this line to compile release version	*
;                                                                     	*
;  Based on:                                                         	*
;    Author:        Soubry Henk                                       	*
;    Company:       Soubry Software Service                           	*
;                                                                     	*
;  Ported to  16F886 and extended with:				             		*
;                                                                     	*
;	 Hardware can be used with 28 pin 18Fxxxx controllers				*
;    Modified to drive more outputs: 20 LEDs							*
;    Separate (may be different colour) led for ticks, second hand		*
;	 8 dot heigh characters												*
;	 Displays day of week (1 = Monday), century							*
;    Name of weekdays on 16 languages (see Daynames.asm)				*
;	 Selectable date formats: (CC)YY-MM-DD or DD-MM-(CC)YY				*
;    12/24 hour digital/analogue display mode							*
;	 All digital display can use Binary Coded Decimal format			*
;	 Individualy controllable analogue hands							*
;	 Dot/arc second hand, bar/dot minute and hour hand option			*
;	 Doubble/single tick at 12'o clock and 0'o clock in 24 hour format	*
;	 III, VI, IX, XII signs on analogue clock in 12 hour mode			*
;	 Temperature, humidity measurement dislaied with day of week		*
;	 Buffer voltage, rotation speed displaied with internal variables	*
;			o	- DispOffset											*
;			s	- ScrollSpeed											*
;			p	- PictNum												*
;			l	- Language												*
;			a	- RemoteAddr											*
;			w	- PixelWidth											*
;			f	- flags													*
;			g	- flags3												*
;			h	- flags4												*
;			i	- RC5_flags												*
;		Only for brightness controll version							*
;			x	- External light measurement (only if option used)		*
;			t	- Light threshold value									*
;			c	- Current Brightness value								*
;			m	- Brightness controll mode								*
;					00 : Daytime setting always							*
;					01 : 08 - 20 Daytime, 20 - 08 Night					*
;					02 : Night setting always							*
;					03 : Light measure based controll	- 16F886 only	*
;			d	- Brightness value for daytime							*
;			n	- Brightness calue for night							*
;	 Two scrolling message with adjustable speed						*
;	 Scrolling display with												*
;		date, time, name of weekday, temperature, humidity				*
;	 Static picture display - 32 picture can be stored					*
;		00 : Zodiac ring with actual symbol at top						*
;		01 : "Propeller Clock" 											*
;		02 - 0F : User pictures											*
;		10 - 1F : Slideshow	picture										*
;	 Slide show option with pictures 16..31: plays Towers of Hanoi		*
;	 Store time and settings in a PCF8583 I2C RTC						*
;  Options:																*
;	 Calibration pin can be used to select motor rotation direction		*
;		pull up/down with 10k resistor									*
;	 PWM brightness controll with night and daytime settings			*
;	 Light measurement for automatic brightness controll - 16F886 only	*
;	 Light controll for higher/lower code meas darker					*
;                                                                     	*
;************************************************************************
;                                                                    	*
;    Files required: CharGen.asm                                      	*
;                    Keys.asm  	                                      	*
;                    DayNames.asm                                      	*
;                    Zodiac.asm                                      	*
;                                                                     	*
;************************************************************************
;                                                                     	*
;    Notes:                                                           	*
;    	Pin assignment                                                	*
;      		Port A                                             			*
;				0 = Temperature measurement A/D channel	analogue		*
;       		1 = Humidity measurement 	A/D channel	analogue		*
;       		2 = Second hand led										*
;       		3 = Voltage measurement 	A/D channel	analogue		*
;       		4 = Ir receiver 					input /open drain 	*
;       		5 = Analogue inner leds        		       				*
;			  6.7 = 20MHZ Oscillator 									*
;					- RA6 can be freed up using external oscillator		*
;																		*
;      		Port B 														*
;				0 = Index								input			*
;               1..7 = Display led's (1=inner, 7=outer)					*
;                                                                     	*
;      		Port C 														*
;				0 = Inner display led (because RB0 - INT is index)		*
;               1 = Analogue ticks led									*
;               2 = Calibration output - Rot dir input at reset			*
;											H =	CounterClockWise		*
;											L = ClockWise				*
;											use 4k7-10k pull up/down 	*
;               3 = I2C SCL		use 2k7 pull up to Vdd	must be input	*
;               4 = I2C SDA 	use 2k7 pull up to Vdd	must be input	*
;               5 = Outer led	    		                        	*
;               6 = TxD 								must be input	*
;               7 = RxD 		use 10k pull up to Vdd	must be input	*
;                                                                     	*
;    	Pin assignment with brightness setting			             	*
;      		Port A                                             			*
;				0 = Temperature measurement A/D channel	analogue		*
;       		1 = Humidity measurement 	A/D channel	analogue		*
;       		2 = Light measurement		A/D channel analogue 16F886	*
;       		3 = Voltage measurement 	A/D channel	analogue		*
;       		4 = Ir receiver 					input /open drain 	*
;       		5 = Analogue inner leds        		       				*
;			  6.7 = 20MHZ Oscillator 									*
;					- RA6 can be freed up using external oscillator		*
;																		*
;      		Port B 														*
;				0 = Index								input			*
;               1..7 = Display led's (1=inner, 7=outer)					*
;                                                                     	*
;      		Port C 														*
;				0 = Inner display led (because RB0 - INT is index)		*
;               1 = Analogue ticks led									*
;               2 = CCP1 PWM brightness controll output					*
;               3 = I2C SCL		use 2k7 pull up to Vdd	must be input	*
;               4 = I2C SDA 	use 2k7 pull up to Vdd	must be input	*
;               5 = Calibration output - Rot dir input at reset			*
;											H =	CounterClockWise		*
;											L = ClockWise				*
;											use 4k7-10k pull up/down 	*
;               6 = Second hand led										*
;               7 = Outer led											*
;************************************************************************
;																		*
;  RTC Ram layout														*
;	00		01		02		03		04		05		06		07			*
;	Cmd		1/10	Sec		Minute	Hour	Day		Month	Timer		*
;			1/100							Year1.0	Wday	10Day/Day	*
;			sec												not used	*
;																		*
;	08		09		0A		0B		0C		0D		0E		0F			*
;	A	L	A	R	M		R	E	G	I	S	T	E	R	S			*
;						Not used										*
;																		*
;	10		11		12		13		14		15		16		17			*
;	Year	~Year	Century	Flags	Disp	Scroll	Flags3	Flags4		*
;									offs	speed						*
;																		*
;	18		19		1A		1B		1C		1D		1E		1F			*
;	Lang.	Flags5	Scroll	Pict	Bright	Bright	Bright	Light		*
;					flags			mode	Daytime	Night	level		*
;************************************************************************
;	Ir Remote control keys												*
;																		*
;	See keys.asm and keys.txt											*
;																		*
; Afther time setting Toggle Outer Led command stores time to RTC		*
;************************************************************************

; Upgrade hints on 16F886
;  - RC5 receiver can be moved to RE3 - MClr pin	=> One more output 
;  - RA5 <-> RC5									=> LEDs on PortC

	errorlevel	-302

;***** VARIABLE DEFINITIONS

;#define MotorDirFromCalibration	; un-comment this line if get motor direction from calibration pin
#define	MotorCounterClockWise	; un-comment this line if motor is running counter clockwise
;#define ReversTextScroll		; un-comment this line if your text must scroll from left to right (e.g. Hebrew)
;#define SlashedZero			; comment this line to use normal zero
#define RC5AddrCheck			; comment this line if RC5 Address checking not required
#define	HumidityMeasure			; comment this line if humidity measure not required
#define	LightMeasure			; comment this line if light measure not required
;#define	GreatherMeansDarker		; comment this line if least code means darker 

#define DISPLAYOFFSET	0x37	; Initial Display Offset for the hardware (display memory address) !!!

	list	p=16f886            ; list directive to define processor
	#include <p16f886.inc>      ; processor specific variable definitions

	#include "CharGen.asm"
	#include "Keys.asm"
	#include "DayNames.asm"
	#include "Zodiac.asm"

	__CONFIG _CONFIG1, _CP_OFF &_CPD_OFF & _WDT_OFF & _BOR_ON & _PWRTE_ON & _HS_OSC & _LVP_OFF & _IESO_OFF & _FCMEN_OFF & _MCLRE_ON
	__CONFIG _CONFIG2, _WRT_OFF & _BOR40V

	__idlocs		0xA000 | (VersionMajor << 8) | (VersionMinor)

;F_OSC	EQU	.20000000		; Oscillator frequency in Hz

;***** RTC VARIABLE DEFINITIONS

RTC_ADDR	EQU	0xA0		; Address of RTC

;#define	RTC_Control	0x00	; Control register
#define RTC_Tenms		0x01	; Address of 1/100 Seconds
#define	RTC_Second		0x02	; Address of Second
;#define	RTC_Minute	0x03	; Address of Minute
;#define	RTC_Hour	0x04	; Address of Hour / AM-PM flag and mode
;#define	RTC_Day		0x05	; Address of Day  / 2 lsb of Year
;#define	RTC_Month	0x06	; Address of Week day / Month
#define RTC_Year        0x10    ; Address of Year
#define RTC_YearNeg     0x11    ; Address of negated value of Year
#define RTC_century		0x12	; Address of century
#define	RTC_flags		0x13	; Address of flags
#define	RTC_disp_off	0x14	; Address of DispOffset
#define RTC_ScrollSp	0x15	; Address of Scroll speed
#define	RTC_flags3		0x16	; Address of flags3
#define	RTC_flags4		0x17	; Address of flags4
#define	RTC_Language	0x18	; Address of Language
#define	RTC_flags5		0x19	; Address of flags5
#define	RTC_Scrollflags	0x1A	; Address of Scroll flags
#define	RTC_PictNum		0x1B	; Address of Picture number
#define	RTC_BrightMode	0x1C	; Address of Brightness mode
#define	RTC_BrightDay	0x1D	; Address of Brightness Daytime
#define	RTC_BrightNight	0x1E	; Address of Brightness Night
#define	RTC_LightLevel	0x1F	; Address of Light level


;Display memory		Low byte	High byte
;DispI			    0x020		0x120	; 2*40 bytes from 0x20 -> 0x48 and 0x120 -> 0x148
;DispII			    0x0A0		0x1A0	; 2*80 bytes from 0xA0 -> 0xF0 and 0x1A0 -> 0x1F0

; Common memory 0x070-0x07F, 0x0F0-0x0FF, 0x170-0x17F, 0x1F0-0x1FF

;Vars in shared memory. This memory is available from any bank
  cblock	0x70
	Scratch						; memory locations for general use
	Scratch2					;
	Scratch3					;
	Scratch4					;

; Rotation measurement in binary format - make it easy to copy from TMR1

	rotation_L					; Rotation count
	rotation_H

;Vars for building display-content
	BCD							; Binary coded decimal
	BCD_High					;   high part
	digitindex					; index digit to display
	dotindex					; index dot to display
	tdd							; temporary display data
	tddH						;   high part

	flags						; Digital display flags
	flags2						; System  flags

	RC5_flags					; RC5 receiver state bits

; No more room
	w_temp						; variable used for context saving
  endc

	IF w_temp > 0x7F
		ERROR "To many variables used in Common area"
	ENDIF

;Vars for display timing routines and interrupts
  cblock	0x20
    DispI: .40					; Display data 0..39
	PeriodCnt_H					; 16 bit counter counts number of TMR0 interrupt
	PeriodCnt_L					; between index-pulses (one revolution)

  ;!! Keep variable order - indirect access used to them v
	StorePCnt1_H				; first  16 bit storage for PeriodeCounter
	StorePCnt1_L				;
	StorePCnt2_H				; second 16 bit storage for PeriodeCounter
	StorePCnt2_L				;
  ;!! Keep variable order - indirect access used to them ^

	SubSec_H					; 16 bit sub-second counter
	SubSec_L					; there are 2500 sub-seconds in 1/2 second

	PixelWidth					; Pixel-width = number of TMR0 interrupts in one ON pixel
	PixelOff					; 8 bit down-counter to time PixelWidth

	PixelPitch_H				; PixelPitch = 16 bit number of interrupts between two pixel
	PixelPitch_L				;
	PixelPitch_F				; Fraction of PixelPitch needed for accuracy and to avoid jitter

	NewPPitch_H					; New pixel pitch calculated by main programm, will be
	NewPPitch_L					; copied to PixelPitch by interrupt routine
	NewPPitch_F					;

	NextPixel_H					; Next pixel @ PeriodCnt = NextPixel
	NextPixel_L					;
	NextPixel_F					;
	RoundedNext_L				; RoundedNext = Round(NextPixel)
	RoundedNext_H				;

	DispOffset					; Display offset compared to indexpuls

	TmrScroll					; Count Down timer for scroll delay

;Vars for time-keeping
  ;!! Keep variable order - indirect access used to access them v
	Second2						; 0..119 counting half seconds
	Minute						; 0.. 59
	Hour						; 0.. 23
	Day							; 1.. Dmon-1
	Month						; 1.. 12
	Year						; 0.. 99
	Century						; 0.. 99
	WDay						; Day of week 0..6 - displayed as 1..7 : Monday is 1
	Tempr						; Temperature
	Humidity					; Humidity
	PictNum						; 32 pictures: Pict0 is Zodiac ring, Pict1: "Propeller Clock", 
  ;!! Keep variable order - indirect access used to access them ^

	DMon						; days in current Month + 1

	BrightMode					; Brightness controll mode
	LightLimit					; External light limit

	dpi							; 0..119 display index for main program

	flags3						; Analogue display flags

; No more room
	iFSR						; copy of FSR used by interrupt routine
  endc

	IF iFSR >= 0x70
		ERROR "To many variables used in Bank 0"
	ENDIF

; start a new block of vars in bank1
  cblock	0xA0
	DispII:	.80					; Display data 40..119
; No more room
  endc

; start a new block of vars in bank2

  cblock	0x110				; Bank 2 - 16 bytes
; 16 more room
  endc

  cblock	0x120				; Bank 2 - 16F876 80 bytes
	DispIHigh:	.40				; Display high data 0..39

;	ch_dot_point_H				; pointer the the current dot# in a character			- BCD_High
;	ch_dot_point_L				; points the the chargenerator table					- BCD
;	ch_dot_index				; index of dot# in a char								- dotindex
;	ch_blanking					; counter for blanking the display when scrolling text	- EEADRH
	ch_flags					; Scroll flags	- Used by ScrollText
	ch_flags_RC5				; Scroll flags	- Set by RC5 receiver

	ch_slidecnt					; Slidshow timing counter

	Language_sc					; A copy of language code during scrolling

	rot_bcd:5					; Rotation in BCD 1/min

	Voltage						; Buffer's voltage
	Voltage_F					;   fraction part

	Light						; External light

	Selector					; Parameter selector

	AD_state					; A/D delay state

	BrightDay					; Brightness value for daytime
	BrightNight					; Brightness value for night

; 4 more room
	pclath_temp					; variable used for context saving
	fsr_temp					; variable used for context saving
	status_temp					; variable used for context saving

  endc

	IF status_temp >= (Second2+0x100)
		ERROR "To many variables used in Bank 2"
	ENDIF

  cblock	(Second2+0x100)
  ; Second2_sc must be at address 0x100 + Second2
  ;!! Keep variable order - indirect access used to access them v
	Second2_sc					; 0..119 counting half seconds
	Minute_sc					; 0.. 59
	Hour_sc						; 0.. 23
	Day_sc						; 1.. Dmon-1
	Month_sc					; 1.. 12
	Year_sc						; 0.. 99
	Century_sc					; 0.. 99
	WDay_sc						; Day of week 0..6 - displayed as 1..7 : Monday is 1
	Tempr_sc					; Temperature
	Humidity_sc					; Humidity
	PictNum_sc
  ;!! Keep variable order - indirect access used to access them ^

;Vars for RC5 decoding
	RC5_Tmr						; Time counter
	RC5_BitCnt					; Bit  counter
								; Shift register for interrupt routine
	RC5_ShrA					;  for RC5 address
	RC5_ShrC					;  for RC5 command
	RC5_Cmd						; Received command code
	RC5_Cmd2					; storage for previous cmd

; No more room
  endc

	IF  RC5_Cmd2>= 0x170
		ERROR "To many variables used in Bank 2"
	ENDIF

; start a new block of vars in bank3
  cblock	0x190				; Bank 3 - 16 bytes
; 16 more room
  endc

  cblock	0x1A0				; Bank 3 - 80 bytes
	DispIIHigh:	.80				; Display high data 40..119
; No more room
  endc


;**********************************************************************
; give meaningfull names to the scratch locations

; for the 16bit/8bit divide routine
DIV_HI		EQU	Scratch			; 16 bit Divident
DIV_LO		EQU	Scratch2		; needed for 16 bit / 8 bit division
DIV_Q		EQU	Scratch3		; Quotient
DIVISOR		EQU	Scratch4		; Divisor

ch_dot_index	EQU	dotindex
ch_blanking		EQU	EEADRH

;**********************************************************************
; define Port bits
; Port A
bTempCh		EQU	0		; bit 0 analogue input	Temperature
bHumCh		EQU	1		; bit 1 analogue input	Humidity
bLightCh	EQU	2		; bit 2 analogue input	Light
bVoltCh		EQU	3		; bit 3 analogue input	Voltage
bRC5inp		EQU	4
bInnerLED	EQU	5
;						; bit 6..7 used for oscillator

; Port B
bIndexPuls	EQU	0
;						; bit 7..1 used for pixel data output
bMinLED		EQU	7
bHourLED	EQU	6

; Port C
bPixel0		EQU	0		; Lsb of pixel data
bTicksLED	EQU	1
;bSCL		EQU	3		; used by MSSI
;bSDA		EQU	4		; used by MSSI
bBrightPWM	EQU	2		; PWM brightness controll output
bCalibration	EQU	5	; Square wave output 2.5 kHz / Motor dir input at reset - 10k pull up/down
bSecLED		EQU	6
bOuterLED	EQU	7

; TMR1L, TMR1H	used to count rotation speed

; T1CON
bMotorDirCC	EQU	1		; Direction of motor rotation  -  Read at reset from PORTC,bCalibration
bTimeSet	EQU 2		; Time setting commands were received
bNewRot		EQU	4		; New rotation count avaible
bPStorage	EQU	5		; Rotation time measurement buffer select

; CCP1CON				; Brightness controll
;			EQU	0		; Used by 18F886 extended CCP modul
;			EQU	1
;bPWMmode0	EQU	2		; PWM mode set Mode b00xx1100
;bPWMmode1	EQU	3		;
bNight		EQU	4		; Use night setting
bDark		EQU	5		; Light measure reading was dark
;				6		; Not implemented on normal CCP module.  Used by 18F886 extended CCP modul
;				7		; Not implemented on normal CCP module.  Used by 18F886 extended CCP modul


; CCPR1H				; Used by PWM hardware
Brightness	EQU	CCPR1L	; Stores actual Brightness code for PWM

Language	EQU	CCPR2L	; Stores language code
ScrollSpeed	EQU	CCPR2H	; Stores Scrolling speed

; define flag3 bits		; Analogue Display flags
bShow36912		EQU	0	; Show  / hide drawing III, VI, IX, XII around the clock
bSecMode		EQU	1	; Dot   / Arc second  hand
bMinMode		EQU	2	; Dot   / Bar minute  hand
bHourMode		EQU	3	; Dot   / Bar hour    hand
bShowSecHand	EQU	4	; Show  / hide second hand
bShowMinHand	EQU	5	; Show  / hide minute hand
bShowHourHand	EQU	6	; Show  / hide hour   hand
bAnalMode24h	EQU	7	; 24    / 12 hour mode

flags4			EQU	RCSTA

; define flags4	bits	; More analogue display flags
bShowTicks		EQU	4	; Show   / hide ticks
bSTick12		EQU	5	; Single / double tick at 12'o clock

;;; Common

; define flag bits		; Digital Display flags
bShowOuter		EQU	0	; Show / hide outer ring
bShowDayName	EQU	1	; Show   day's name instead day of week, temperature, humidity
bDisp12Mode		EQU	2	; 12   / 24 hour mode
bDateFormat		EQU	3	; Use YYYY-MM-DD / DD-MM-YYYY or YY-MM-DD / DD-MM-YY date format
bShowWDay		EQU	4	; Show / hide day of week, temperature, humidity
bShowDTime		EQU	5	; Show / hide digital time
bShowDDate		EQU	6	; Show / hide digital date
bShowCentury	EQU	7	; Show / hide century

; define flag2 bits		; System flags
bDspEnabled		EQU	0	; Blank the display - motor slow or not stabilized
bNewPCnt		EQU	1	; New pitch calculated
bNewTime		EQU	2	; New time calculated
bNewPPitch		EQU	3	; New pitch measured
bText			EQU	4	; Text on display
bScrollOn		EQU	5	; Scrolling text on display
bDemo			EQU	6	; Demo mode active
bTimeInv		EQU	7	; Time invalid

; RC5 flags
bRC5_WaitStart	EQU	0
bRC5_DataReady	EQU	1
bRC5_HalfBit	EQU	2
bRC5_Idle		EQU	3
bRC5_prev_inp	EQU	bRC5inp	; Has to be on same bit as bRC5inp on PORT
bRC5_ReSynced	EQU	5
							; Interrupt routine keeps the values of these bits
							; Not RC5 related bits, but must be in common ram
bBinMode		EQU	6		; Binary display mode
bUseBinFont		EQU	7		; Numerical character to display in Binary mode

;;; Bank2

; define ch_flags
bScrollDateTime		EQU	0		; Date, time, day name, temperature, humidity in scrolling text
bDemo_sc			EQU	bDemo	; A copy of demo mode flag

;**********************************************************************
; Port A
;#define	TempCh		PORTA,0				; A/D input
#define 	RC5inp		PORTA,bRC5inp
#define 	InnerLED	PORTA,bInnerLED

; Port B
;#define 	IndexPuls	PORTB,bIndexPuls
#define		MinLED		PORTB,bMinLED

; Port C
;#define	Pixel0		PORTC,bPixel0
#define		TicksLED	PORTC,bTicksLED
;#define	Calibration	PORTC,bCalibration
#define 	OuterLED	PORTC,bOuterLED
#define 	SecLED		PORTC,bSecLED

; T1CON
#define		MotorDirCC	T1CON,bMotorDirCC
#define		NewRot		T1CON,bNewRot
#define		TimeSet		T1CON,bTimeSet
#define		PStorage	T1CON,bPStorage

; TMR1 		registers used for rotation measureing

; CCPR2H 	register  used to store scroll speed
; CCPR2L 	register  used to store language code

; CCP1CON	Brightness controll
#define		Night		CCP1CON,bNight
#define		Dark		CCP1CON,bDark

; Flags
#define		ShowOuter	flags,bShowOuter
#define		ShowDayName	flags,bShowDayName
#define		Digit12Mode	flags,bDisp12Mode
#define		DateFormat	flags,bDateFormat
#define		ShowWDay	flags,bShowWDay
#define		ShowDTime	flags,bShowDTime
#define		ShowDDate	flags,bShowDDate
#define		ShowCentury	flags,bShowCentury

; Flags2
#define		fDspEnabled	flags2,bDspEnabled
#define		fNewPCnt	flags2,bNewPCnt
#define		fNewTime	flags2,bNewTime
#define		fNewPPitch	flags2,bNewPPitch
#define		fText		flags2,bText
#define		fScrollOn	flags2,bScrollOn
#define		fDemo		flags2,bDemo
#define		fTimeInv	flags2,bTimeInv

; Flags3
#define		Show36912	flags3,bShow36912
#define		SecMode		flags3,bSecMode
#define		MinMode		flags3,bMinMode
#define		HourMode	flags3,bHourMode
#define		ShowSecHand	flags3,bShowSecHand
#define		ShowMinHand	flags3,bShowMinHand
#define		ShowHourHand	flags3,bShowHourHand
#define		AnalMode24h		flags3,bAnalMode24h

; RCSTA	Flags4
#define		ShowTicks	flags4,bShowTicks
#define		STick12		flags4,bSTick12

; RC5 Flags
#define		RC5_WaitStart	RC5_flags,bRC5_WaitStart
#define		RC5_DataReady	RC5_flags,bRC5_DataReady
#define		RC5_HalfBit		RC5_flags,bRC5_HalfBit
#define		RC5_Idle		RC5_flags,bRC5_Idle
#define		RC5_prev_inp	RC5_flags,bRC5_prev_inp
#define		RC5_ReSynced	RC5_flags,bRC5_ReSynced
#define		BinMode			RC5_flags,bBinMode
#define		UseBinFont		RC5_flags,bUseBinFont

; ch_flags
#define		ScrollDateTime	ch_flags,bScrollDateTime

;******************************************************************************
;	Define macro's for a Select - Case
;******************************************************************************
select_w	macro
sel_last	set 0                     ;setup variable only
         	endm

case     	macro   val
	        xorlw   val ^ sel_last
			btfsc	STATUS,Z
sel_last set val
         	endm

;
;******************************************************************************
;	Define EEPROM content for initial message
;******************************************************************************

	ORG     0x2100           ; Start of EEPROM page 0
		de	"===> This version of Bob Blick's"
		de	" Propeller Clock was built with "
		de	"16F886"
		de	", PCF8583 RTC, light and temperature sensors. ",0x83,0x84," <==="
		de	0x00

	ORG     0x2180           ; Start of EEPROM page 1
		de	"===> Uses 20 leds, 8 dot high characters,"
		de	" 16 bit wide display memory, advanced graphics,"
		de	" measures rotation speed. "
		de	"Ver."	,VersionMajor | 0x30 ,".", ((VersionMinor>>4) & 0x0F) | 0x30, ((VersionMinor) & 0x0F) | 0x30, " <==="
		de	0x00

;**********************************************************************
		ORG     0x0000          ; processor reset vector

		clrf	PORTA
		clrf	PORTB
		clrf	PORTC
		goto    main            ; go to beginning of program

;**********************************************************************
		ORG     0x0004          ; interrupt vector location

		movwf	w_temp			; context saveing
		swapf	w_temp,f
		swapf	STATUS,w		; Save STATUS
		bcf		STATUS,RP0
		bsf		STATUS,RP1		; Context saving variables are on Bank2
		movwf	status_temp
		swapf	PCLATH,w		; Save PCLATH
		movwf	pclath_temp		; Needed only if >2K program memory used
		movf	FSR,w			; context saving
		movwf	fsr_temp		;

		clrf	PCLATH			; IT routine is in 0x000..0x7FF
		clrf	STATUS			; select Bank0 & IRP=0 for interrupt stuff

;**********************************************************************

INT_RB0
		btfss	INTCON,INTF		; interrupt on RB0?
		goto	INT_TMR0		; nope, TMR0 is next

;----------	RB0 interrupt ! == Index Sensor
		movlw	0xff			;
		movwf	TMR0			; force TMR0 to 0xFF

		movlw	StorePCnt1_H	; Get the address of storage to use
		btfss	PStorage
		movlw	StorePCnt2_H
		movwf	FSR
		movf	PeriodCnt_H,w	; Save Period counter
		movwf	INDF
		incf	FSR,f
		movf	PeriodCnt_L,w	;
		movwf	INDF
		movlw	1<<bPStorage	; Switch to next storage
		xorwf	T1CON,f

;StoreDone:
		bsf		fNewPCnt		; signal new stored PeriodCount to main program
		clrf	PeriodCnt_H		; PeriodCnt = 0
		clrf	PeriodCnt_L		;
		clrf	NextPixel_H		; Next_Pixel = 0 (thus now)
		clrf	NextPixel_L		;
		clrf	NextPixel_F		;
		clrf	RoundedNext_L	; RoundedNext = 0
		clrf	RoundedNext_H	;
		clrf	PixelOff		; display next pixel no matter what.
		movf 	DispOffset,w	; iDispIndex = DispOffset
		movwf	iFSR			;
		btfss	fNewPPitch		; is there a new value calculated by main prg?
		goto	lINT_RB0		; no, continue
		movf	NewPPitch_H,w	; PixelPitch = NewPPitch
		movwf	PixelPitch_H	;
		movf	NewPPitch_L,w	;
		movwf	PixelPitch_L	;
		movf	NewPPitch_F,w	;
		movwf	PixelPitch_F	;
		bcf		fNewPPitch		;

lINT_RB0
		incfsz	TMR1L,f
		goto	INT_RB0_2
		incf	TMR1H,f

INT_RB0_2
		bcf		INTCON,INTF		; clear RB0 interrupt flag
		goto	lDisp_1			; do display routine but skip Period increment
;--------
INT_TMR0
		btfss	INTCON,T0IF		; Test if a TMR0 interrupt occured
		goto	INT_TMR2		; nope, TMR2 is next

		bsf		TMR0,7			; TMR0 = TMR0 + 128 => Test to double TMR0 speed

		incfsz	PeriodCnt_L,f	; Increment 16-bit period counter
		goto	lDisp_1			;
		incfsz	PeriodCnt_H,f	;
		goto	lDisp_1			; if overflow in MSB period counter = speed to low

		bcf		flags2,bDspEnabled
		bcf		PORTC,bOuterLED	; Other bit of PortA, PortB bits and PortC 0 will be cleared by lDisp_1

lDisp_1
		movf	RoundedNext_L,w	; PeriodCnt = RoundedNextPixel ?
		xorwf	PeriodCnt_L,w	;
		btfss	STATUS,Z		;
		goto	lPixelOn		; no, check if there is a pixel on
		movf	RoundedNext_H,w	;
		xorwf	PeriodCnt_H,w	;
		btfss	STATUS,Z		;
		goto	lPixelOn		; no,  check if there is a pixel on
								; yes, display next pixeldata

		movf	iFSR,w			; load new memory pointer in w
		movwf	FSR				; load File Select register with Display Index
		bsf		STATUS,IRP		; Point to high data

		movlw	0x00			; Blank the display if rotation too slow
		btfsc	flags2,bDspEnabled
		movf	INDF,w			; Get display high data

		xorwf	PORTA,w			; Update only output bits
		andlw	(1<<bInnerLED)
		xorwf	PORTA,f

		movlw	0x00			; Blank the display if rotation too slow
		btfsc	flags2,bDspEnabled
		movf	INDF,w			; Get display high data
		xorwf	PORTC,w			; Update only output bits
		andlw	(1<<bSecLED)|(1<<bTicksLED)
		xorwf	PORTC,f

		bcf		STATUS,IRP		; Point to low data
		movlw	0x00
		btfsc	flags2,bDspEnabled
		movf	INDF,w			; Get display data

		movwf	PORTB			; PortB = DisplayData(index)

		xorwf	PORTC,w
		andlw	(1 << bPixel0)
		xorwf	PORTC,f			; dispatch bit 0 to PORTC

	ifdef	MotorDirFromCalibration
		btfsc	MotorDirCC			; Get direction of motor rotation
		call	CheckDecrementFSR	; decrement FSR, check correct progress of display memory pointer
		btfss	MotorDirCC			; Get direction of motor rotation
		call	CheckIncrementFSR	; increment FSR, check correct progress of display memory pointer
	else

	  ifdef MotorCounterClockWise
		call	CheckDecrementFSR	; decrement FSR, check correct progress of display memory pointer
	  else
		call	CheckIncrementFSR	; increment FSR, check correct progress of display memory pointer
	  endif
	endif

		movwf	iFSR			; Store it in i(nterrupt)FSR

		movf	PixelPitch_F,w 	; NextPixel = NextPixel + PixelPitch
		addwf	NextPixel_F,f	;
		btfss	STATUS,C		;
		goto	no_overflow		;
		incf	NextPixel_L,f	; fraction overflow, add 1 to Low byte
		btfsc	STATUS,Z		;
		incf	NextPixel_H,f	; low byte roll-over, add 1 to High byte
no_overflow:
		movf	PixelPitch_L,w	;
		addwf	NextPixel_L,f	;
		btfsc	STATUS,C		;
		incf	NextPixel_H,f	; low byte overflow, add 1 to High byte

		movf	NextPixel_L,w	; RoundedNext = NextPixel
		movwf	RoundedNext_L	;
		movf	NextPixel_H,w	;
		movwf	RoundedNext_H	;
		btfss	NextPixel_F,7	; IF NextPixel_Fraction >= 128 then
		goto	NotRoundUp		;
		incf	RoundedNext_L,f	; ROUND UP
		btfsc	STATUS,Z		;
		incf	RoundedNext_H,f	;
NotRoundUp:
		movf	PixelWidth,w	;
		movwf	PixelOff		; PixelOff = PixelWidth

		goto	lDispDone		;

lPixelOn
		movf	PixelOff,w		; Is there a pixel on?
		btfsc	STATUS,Z		; PixelOff = 0 ?
		goto	lDispDone		; no, jump

		decfsz	PixelOff,f		; pixel is on, countdown
		goto	lDispDone		;

		clrf	PORTB			; turn off display LED's
		bcf		PORTC,bPixel0	; also turn off PortC LED

lDispDone
		bcf		INTCON,T0IF		; clear TMR0 interrupt flag before return

;--------
INT_TMR2
		btfss	PIR1,TMR2IF		; interrupt on TMR2?
		goto	INT_EXIT		; nope, interrupt is done!

								; do the TMR2 stuff, we get here every 200uSec

		movlw   1<<bCalibration	; toggle PORTC bit, use scope or freq counter
		xorwf	PORTC,f			; to calibrate to 2.5 kHz

								; Text-scroll timer
		movf	TmrScroll,f		; TmrScroll == 0 ?
		btfss	STATUS,Z		;
		decf	TmrScroll,f		; no, TmrScroll --

								; real time sub-second counter
		incfsz	SubSec_L,f		; Increment 16-bit Sub Second counter
		goto	lTime_1			; stil counting
		incfsz	SubSec_H,f		;
		goto	lTime_1			; stil counting

		incf	Second2,f		; 1/2 second has passed
		bsf		fNewTime		; Sign time need checking

		call	InitSubSecCnt	; reload counter SubSecond = 0x10000 - .2500 = 0xF63C
								; returns 0
		btfss	fDspEnabled		; Displaying disabled 
		btfsc	fDspEnabled		; Displaying disabled 
		movlw	1<<bOuterLED	; Toggle outerled if time invalid
		btfsc	fTimeInv
		xorwf	PORTC,f

		movf	Second2,w		; Minute passed
		xorlw	.120
		btfss	STATUS,Z
		goto	lTime_1			; No, goto RC5 task

		movf	TMR1L,w			; Read rotation from Timer1
		movwf	rotation_L		; Store it
		movf	TMR1H,w
		movwf	rotation_H		; rotation_L & rotation_H are in common area

		clrf	TMR1H			; Reset Timer 1
		clrf	TMR1L

		bsf		NewRot			; Sign new rotation data avaible

lTime_1
		bsf		STATUS,RP1		; Bank2
		movf	AD_state,f
		btfss	STATUS,Z
		decf	AD_state,f

lRC5_Stuff						; Start RC5 stuff here, Variables are on Bank2
		movlw	PORTA			; As on Bank2 use indirect access to PORTA
		movwf	FSR

		btfsc	RC5_DataReady	;
		goto	lRC5_Exit		;
		btfss	RC5_Idle		;
		goto	lRC5_Not_Idle	;
		decfsz	RC5_Tmr,f		;
		goto	lRC5_Exit		;
		btfsc	INDF,bRC5inp	; Test input
		bcf		RC5_Idle		; input = high, cancel Idle state
		incf	RC5_Tmr,f		; continue Idle state until input = high
		goto	lRC5_Exit		;

lRC5_Not_Idle
		btfss	RC5_WaitStart	;
		goto	lRC5_ReSync		;

lRC5WaitStart
		btfsc	INDF,bRC5inp	; test input from PORTA
		goto	lRC5_Exit		; no startbit

		bcf		RC5_WaitStart	; start received
		movlw	.13				; 13 bits to receive
		movwf	RC5_BitCnt		;
		clrf 	RC5_ShrA		;
		clrf	RC5_ShrC		;
		movlw	.6				;
		goto	lRC5_Reload		;

lRC5_ReSync						;
		movf	INDF,w			; Read PORTA
		xorwf	RC5_flags,w		;
		andlw	1<<bRC5inp		;
		btfsc	STATUS,Z		;
		goto	lRC5_no_sync	;
		bsf		RC5_ReSynced	;

		xorwf	RC5_flags,f		; Modify stored input bit

		movlw	.6				; re-sync the timer
		btfss	RC5_HalfBit		;
		movlw	.2				;
		movwf	RC5_Tmr			;

lRC5_no_sync
		btfsc	RC5_HalfBit		;
		goto	lRC5_2nd_Half	;

lRC5_1st_Half
		decfsz	RC5_Tmr,f		;
		goto	lRC5_Exit		;
		bcf		STATUS,C		;
		btfsc	INDF,bRC5inp	; Test PORTA,bRC5inp
		bsf		STATUS,C		; C = RC5inp
		rlf		RC5_ShrC,f		;
		rlf		RC5_ShrA,f		;
		bsf		RC5_HalfBit		; indicate that the first half bit is received
		bcf		RC5_ReSynced	;
		movlw	.4				; reload timer
		goto	lRC5_Reload

lRC5_2nd_Half
		btfsc	RC5_ReSynced	;
		goto	lReSyncOK		;
		decfsz	RC5_Tmr,f		;
		goto	lRC5_Exit		;

;lRC5_Error
		movf	RC5_flags,w		; Clear RC5 status bits but keep binary mode flags
		andlw	(1<<bUseBinFont) | (1<<bBinMode) | (1<<bRC5_DataReady)
		iorlw	(1<<bRC5_WaitStart) | (1<<bRC5_Idle)		;
		movwf	RC5_flags
		movlw	.128			;

lRC5_Reload
		movwf	RC5_Tmr			;
		goto	lRC5_Exit		;

lReSyncOK
								; test second bit half
		bcf		RC5_HalfBit		;
		decfsz	RC5_BitCnt,f	;
		goto	lRC5_Exit		;
		rlf		RC5_ShrC,f		; Shift left to complete ShrA
		rlf		RC5_ShrA,f		;
		rlf		RC5_ShrC,w		; RC5_ShrC remains unchanged
		rlf		RC5_ShrA,f		; Complete address
		bcf		RC5_ShrC,7		;
		btfss	RC5_ShrA,6		;
		bsf		RC5_ShrC,7		; Bit 7 of RC5_ShrC = Inv Bit6 in RC5_ShrA
								; RC5_ShrA and RC5_ShrC was cleared and shifted left 13+2 times, so C = 0
		btfsc	RC5_ShrA,5		;
		bsf		STATUS,C		; C = ToggleBit
		rrf		RC5_ShrC,w		; ToggleBit in bit 7 of RC5_ShrC
		movwf	RC5_Cmd			;
		movf	RC5_ShrA,w
	ifdef	RC5AddrCheck
		xorlw	RemoteAddr		; If RC5_ShrA = RemoteAddr
		andlw	0x1F			; Only 5 bits to test
		btfsc	STATUS,Z		;
	endif
		bsf		RC5_DataReady	; Sign new command received
lRC5_Exit
		clrf	STATUS			; Bank0

		bcf		PIR1,TMR2IF		; clear TMR2 interrupt flag
;--------
INT_EXIT
		bsf		STATUS,RP1		; Context saving variables are on Bank2 except w_temp
		movf	fsr_temp,w		; context recall
		movwf	FSR				;
		swapf	pclath_temp,w	; Needed only if >2K program memory used
		movwf	PCLATH
		swapf	status_temp, w	; restore context for main program
		movwf	STATUS			;  this will also restore bank selection
		swapf	w_temp, w
		retfie                  ; return from interrupt

;******************************************************************************
;	Init sub second counter	--	SubSecond = 0x10000 - 2500 = 0xF63C

InitSubSecCnt
		movlw	0x3C			; load counter SubSecond = 0x10000 - 2500 = 0xF63C
		movwf	SubSec_L		;
		movlw	0xF6			;
		movwf	SubSec_H		;
		retlw	0

;**********************************************************************

div16X8Divisor					; Store divisor
		movwf	DIVISOR

div16X8							; DIV_HI and DIV_LO / DIVISOR.  result to DIV_Q
								; remainder in DIV_LO
								; does not deal with divide by 0 case
		clrf 	DIV_Q
div_1
		movf 	DIVISOR, W
		subwf 	DIV_LO, F
		btfsc 	STATUS, C		; if negative skip
		goto 	div_2
div_borrow
		movlw 	.1
		subwf 	DIV_HI, F		; DIV_HI = DIV_HI - 1
		btfss 	STATUS, C		; if no borrow occurred
		goto	div_done
div_2
		incf 	DIV_Q, F
		goto 	div_1
div_done
		movf 	DIVISOR, W		; re-add DIVISOR to DIV_LO to get
		addwf 	DIV_LO, W		; remainder in DIV_LO
		movwf	DIV_HI			; shift remainder to DIV_HI
		movf	DIV_Q,w			; Return with quotient in w

		return

;**********************************************************************
; Calculate pixel pitch

CalcPixelPitch					; that is a hard one. We have to divide by 120 !!!
								; PixelPitch = PeriodCnt / 120

		movf 	StorePCnt1_H,w	; check if two period counters
		movwf	DIV_LO			;
		xorwf	StorePCnt2_H,w	; are the same. If not, don't
		btfss	STATUS,Z		; calculate new pixelpitch because
		goto	CalcDone		; the rotation speed was not stable
		movf	StorePCnt1_L,w	;
		xorwf	StorePCnt2_L,w	;
		btfss	STATUS,Z		;
		goto	CalcDone		;

		clrf	DIV_HI			; Divider = 00:StorePCnt1_H

		movlw	.120			; Init divisor
		call 	div16X8Divisor
		movwf	NewPPitch_H		;
		movf	StorePCnt1_L,w	;
		movwf	DIV_LO			;

		call	div16X8			; Divider = Remainder:StorePCnt1_L
		movwf	NewPPitch_L		;
		clrf	DIV_LO			;

		call	div16X8			; Divider = Remainder:00
		movwf	NewPPitch_F		;

								; start calculation for pixel width
		clrc					; Clear Carry
		rrf		NewPPitch_L,w	;
		movwf	PixelWidth		; W = PixelWidth = NewPPitch_L * 4 / 8
		clrc
		rrf		PixelWidth,f	; PixelWidth = NewPPitch_L * 2 / 8
		clrc
		rrf		PixelWidth,f	; PixelWidth = NewPPitch_L * 1 / 8
		addwf	PixelWidth,f	; PixelWidth = NewPPitch_L * 5 / 8

		bsf		flags2,bDspEnabled ; If bDspEnabled int. routine clears PORTA,bOuterLED
		call	SetOuterLed		; set Port for Outerled equal to flags OuterLED bit

NotON:
		bsf		fNewPPitch		; signal new Pitch value to interrupt routine

CalcDone:
		bcf		fNewPCnt		; clear flag

		return

;******************************************************************************
;	Check correct progress of sec, min, hour, day, day of week, month, year
;******************************************************************************

TimeCheck
		movlw	Second2
		movwf	FSR
								; FSR -> Second2
		movlw	.120			; Check for second overflow
		call	CheckAndIncNext
								; FSR -> Minute
		movlw	.60				; Check for minute overflow
		call	CheckAndIncNext
								; FSR -> Hour
		movlw	.24				; Check for hour overflow
		call	CheckAndIncNext
								; FSR -> Day
		btfsc	STATUS,C        ; if Day incremented increment WDay as well
		incf	WDay,f

		movlw	.7				; if Day of week == 7
		subwf	WDay,w
		btfsc	STATUS,C        ; if Day of week < 7 then c=0 (borrow)
		movwf	WDay			; Day of week = 0

		movlw	.32				; DMon = 32
		movwf	DMon
		movf	Month,w			;
		movwf	Scratch			; For months <8 odd  months have 31 days
		btfsc	Month,3
		incf	Scratch,f		; For months >7 even months have 31 days
		btfss	Scratch,0
		decf	DMon,f
		xorlw	.2				; Month = February ?
		btfss	STATUS,Z		;
		goto	lNotFeb			; continue
		decf	DMon,f			; February has max 29 days

		movf	Year,w
		btfsc	STATUS,Z		; Year == 0 check Century instead
		movf	Century,w
		andlw	0x03
		btfss	STATUS,Z
		decf	DMon,f			; Year is not a LeapYear, dec DMon for Feb

lNotFeb
		movf	DMon,w
		call	CheckAndIncNext	; Check for day overflow
		btfsc	STATUS,C        ; if day cleared, correct day
		incf	Day,f
								; FSR -> Month
		movlw	.13				; Check for month overflow
		call	CheckAndIncNext
		btfsc	STATUS,C        ; if month cleared, correct month
		incf	Month,f
								; FSR -> Year
		movlw	.100			;
		call	CheckAndIncNext
								; FSR -> Century
		movlw	.100			;
		subwf	Century,w		; Century < 100 ?
		btfsc	STATUS,C		;
		clrf	Century			;

		bcf		fNewTime		; Clear new time flag
		return

CheckAndIncNext
		subwf	INDF,w			; test INDF value to w
		btfsc	STATUS,C        ; if INDF < w then c=0 (borrow)
		movwf	INDF			; Reset value
		incf	FSR,f			; Move to next unit
		btfsc	STATUS,C        ; If reseted, increment next unit
		incf	INDF,f
		return					; C = 1, if carry was to next unit


;******************************************************************************
;	Routines to display Time and Date
;******************************************************************************

;-------- Update Display memory with name of day

LoadBCDDay
		movf	Language,w		; Get Language code
		call	DayAddress
		movf	digitindex,w	; Read names backward
		sublw	.9
		call	AddBCD16
		movlw	high(DayNames)	; Add address of table
		call	CharPattern		; Get next character
		clrf	PCLATH			; Use page 0

ConvToInternalNotBin
		bcf		UseBinFont		; Not numeric character

ConvToInternal					; Converts ascii character to internal address
		addlw	-' '			; Convert ascii code to inernal
ConvToAddress
		clrc
		movwf	BCD				; Save internal code
		clrf	BCD_High
		btfsc	BCD,7
		clrf	BCD				; If code was > 0x9F - use space instead
		rlf		BCD,f			; Multiply by 6
		rlf		BCD,w
		call	RlfAddBCD16
		movf	BCD,w
		return					; Store new code

DayAddress
		movwf	Scratch2		; Store language code
		call	ConvToAddress	; Multiply by 6
		addwf	Scratch2,w		; Add ones more  - Multiply by 7
		addwf	WDay,w			; Add day of week

Mult10
		movwf	BCD				; Multiply by 10
		clrf	BCD_High		; result is 16 bit wide
		addwf	BCD,f
		movf	BCD,w
		rlf		BCD,f
		rlf		BCD_High,f
		rlf		BCD,f
RlfAddBCD16
		rlf		BCD_High,f
AddBCD16						; Add w to BDC_High:BCD
		addwf	BCD,f
		btfsc	STATUS,C
		incf	BCD_High,f
		return

CharPatternTab
		btfsc	UseBinFont		; Is this char a numeric char in bin mode
		goto	CharPatternBin	; Use binary table
CharTable
		movlw	high(CharTab)
CharPattern						; If calls out from lower 2K, afther call PCLATH has to be set to 0
		addwf	BCD_High,w		; Get high byte
GetXII
		movwf	PCLATH			; Store it in PCLATH
		movf	BCD,w			; Get low byte
		movwf	PCL				; Get next character

;=======================================================
; No execution passes here


;-------- Update Display memory with Digital Time Display

DTime
		movf	dpi,f			;
		btfss	STATUS,Z		;
		goto	l_DTime_1		; display index > 0

		clrf	digitindex		; display index = 0, load start parameters for
		clrf	dotindex		; digit displaying

l_DTime_1
		movlw	.60				;
		subwf	dpi,w			;
		btfsc	STATUS,C		;
		return					; display index >= 60

		movf	dotindex,w		;
		btfss	STATUS,Z		;
		goto	l_DTime_3

		movlw	.6
		addwf	dotindex,f
		bcf		UseBinFont

		btfsc	ShowDTime
		call	LoadBCDTime		; dotindex = 0, load new digit

		btfss	ShowDTime
		call	LoadBCDTest		; dotindex = 0, load new digit

l_DTime_3
		decfsz	dotindex,f		;
		goto	l_DTime_2

		incf	digitindex,f	; select next digit
		return					; to get a gap between digits

l_DTime_2
		call	CharPatternTab	; get the dot pattern for this column
		clrf	PCLATH

IorToTdd
		iorwf	tdd,f			; Load pixed data to temp disp data

IncBCD16
		movlw	1
		goto	AddBCD16

;-------- Update Display memory with Digital Date Display

DDate
		movlw	.60
		subwf	dpi,w			;
		btfss	STATUS,C		;
		return					; display index < 60
		btfss	STATUS,Z		;
		goto	l_DDate_1		; display index > 60

		clrf	digitindex		; display index = 60, load start parameters for
		clrf	dotindex		; digit displaying

l_DDate_1
		movf	dotindex,w		;
		btfss	STATUS,Z		;
		goto	l_DDate_11

		movlw	.6
		addwf	dotindex,f
		bcf		UseBinFont

		btfss	ShowDDate
		call	LoadBCDVoltage

		btfss	ShowDDate
		goto	l_DDate_10

		btfss	ShowWDay
		goto	l_DDate_9

		btfss	Second2,3
		call	LoadBCDWDay
		btfsc	Second2,3
l_DDate_9
		call	LoadBCDDate		; dotindex = 0, load new digit

l_DDate_10
		movlw	.4				; Read dot patterns backward
		call	AddBCD16

l_DDate_11
		decfsz	dotindex,f		;
		goto	l_DDate_2		; dotindex < 6, display data

		incf	digitindex,f	; select next digit
		return					; to get a gap between digits

l_DDate_2
		call	CharPatternTab	; get the dot pattern for this column
		clrf	PCLATH

l_Date_3
		movwf	Scratch2		; Flip the dot pattern
		clrf	Scratch			; Clear bit 8
		movlw	7				; Characters are 7 bit height
l_Date_4
		rrf		Scratch2,f		; Rotate (right) next bit to   C
		rlf		Scratch,f		; Rotate (left ) next bit from C
		addlw	-1				;
		btfss	STATUS,Z		; Loop for 7 dots
		goto	l_Date_4
		movf	Scratch,w		; Get flipped dot pattern

		iorwf	tdd,f			; Load pixed data to temp disp data

DecBCD16
		decf	BCD,f			; Move table pointer backward
		incf	BCD,w
		btfsc	STATUS,Z
		decf	BCD_High,f
		return					;

;-------- Update Display memory with Hands

Hands
		movlw	.30				; difference in offset for hands and digital display
		subwf	dpi,w			; scratch4 = dpi - 30
		btfss	STATUS,C		; if scratch4 < 0 then
		addlw	.120			;    scratch4 = scratch4 + 120
		movwf	Scratch4		;

		btfss	ShowSecHand		; Second hand disabled
		goto	MinuteHand

		movf	Second2,w		;
		andlw	0xFE			; filter out 1/2 second

		subwf	Scratch4,w
ArcSecHand						; Draw arc if Scratch4<= (Second2 and 0xFE)
		btfss	SecMode			; Set C if dot second hand mode
		bsf		STATUS,C
		btfss	STATUS,C
		bsf		tddH,bSecLED	; Sec led - second hand
SecHand
		btfsc	STATUS,Z		;
		bsf		tddH,bSecLED	; Sec led - second hand

MinuteHand
		bcf		STATUS,C		;
		rlf		Minute,w		; Clears C, because Minute < .60
		movwf	Scratch			; Scratch = minute * 2

		xorwf	Scratch4,w		;
		btfsc	STATUS,Z
		btfss	ShowMinHand		; Minute hand disabled
		goto	HourHand

		btfsc	MinMode			;
		goto	DotMinHand

		movlw	B'00001111'		;
		iorwf	tdd,f			; Turn on minute hand
		bsf		tddH,bInnerLED	; Turn on analogue innner Leds
		goto	HourHand

DotMinHand
		bsf		tdd,bMinLED		; Turn on minute dot

HourHand
		btfss	ShowHourHand	; Hour hand disabled
		return

		rlf		Hour,w			; Get hour
		movwf	Scratch2		; Scratch2 = hour * 2
		rlf		Scratch2,f		; Scratch2 = hour * 4
		rlf		Scratch2,f		; Scratch2 = hour * 8
		addwf	Scratch2,f		; Scratch2 = hour * 10

		movlw	.12				;
l_hour_div12					; Divide (2 * minute) by 12
		subwf	Scratch,f		;
		btfss	STATUS,C		;
		goto	l_div12_done	;
		incf	Scratch2,f		; advance hour a bit
		goto	l_hour_div12	;

l_div12_done					; C=0
		movlw	.120			; Scratch2 = (10 * Hour) + (2 * Minute / 12)

		btfsc	AnalMode24h		; In 24 hour mode
		rrf		Scratch2,f		;  divide by 2

l_120
		subwf	Scratch2,f		;
		btfsc	STATUS,C		; result > 0 ?
		goto	l_120
		addwf	Scratch2,w		; result was negative, re-add 120 and load it in w

		xorwf	Scratch4,w		;
		btfss	STATUS,Z
		return

		btfss	HourMode
		bsf		tddH,bInnerLED	; Turn on analogue innner Leds
		btfsc	HourMode
		bsf		tdd,bHourLED
		return


;-------- Get character Digital Date Display

LoadBCDDate						; load BCD with date digits
		movlw	high(DateTable0)
		movwf	PCLATH
		movf	digitindex,w	;
		addwf	digitindex,w	;
		btfss	ShowCentury
		addlw	DF_YYMMDD-DF_YYYYMMDD
		btfsc	DateFormat
		addlw	DF_DDMMYYYY-DF_YYYYMMDD

DateTable0
		addwf	PCL,f			;

; DateFormat: YYYY-MM-DD
DF_YYYYMMDD
;0--
		movf	Day,w			; Day     1 digit
		goto	l_convert		;
;1--
		movf	Day,w			; Day    10 digit
		goto	l_convert_swap	;
;2--
		movlw	'-'				; '-'
		goto	l_char			;
;3--
		movf	Month,w			; Month   1 digit
		goto	l_convert		;
;4--
		movf	Month,w			; Month  10 digit
		goto	l_convert_swap	;
;5--
		movlw	'-'				; '-'
		goto	l_char			;
;6--
		movf	Year,w			; Year    1 digit
		goto	l_convert		;
;7--
		movf	Year,w			; Year   10 digit
		goto	l_convert_swap
;8--
		movf	Century,w		; Year  100 digit
		goto	l_convert		;
;9--
		movf	Century,w		; Year 1000 digit
		goto	l_convert_swap

;  DateFormat: DD-MM-YYYY
DF_DDMMYYYY
;10--
		movf	Year,w			; Year    1 digit
		goto	l_convert		;
;11--
		movf	Year,w			; Year   10 digit
		goto	l_convert_swap	;
;12--
		movf	Century,w		; Year  100 digit
		goto	l_convert		;
;13--
		movf	Century,w		; Year 1000 digit
		goto	l_convert_swap	;
;14--
		movlw	'-'				; '-'
		goto	l_char			;
;15--
		movf	Month,w			; Month   1 digit
		goto	l_convert		;
;16--
		movf	Month,w			; Month  10 digit
		goto	l_convert_swap	;
;17--
		movlw	'-'				; '-'
		goto	l_char			;
;18--
		movf	Day,w			; Day     1 digit
		goto	l_convert		;
;19--
		movf	Day,w			; Day    10 digit
		goto	l_convert_swap

; DateFormat: YY-MM-DD
DF_YYMMDD
;20--
		movlw	' '				; ' '
		goto	l_char			;
;21--
		movf	Day,w			; Day     1 digit
		goto	l_convert		;
;22--
		movf	Day,w			; Day    10 digit
		goto	l_convert_swap	;
;23--
		movlw	'-'				; '-'
		goto	l_char			;
;24--
		movf	Month,w			; Month   1 digit
		goto	l_convert		;
;25--
		movf	Month,w			; Month  10 digit
		goto	l_convert_swap	;
;26--
		movlw	'-'				; '-'
		goto	l_char			;
;27--
		movf	Year,w			; Year    1 digit
		goto	l_convert		;
;28--
		movf	Year,w			; Year   10 digit
		goto	l_convert_swap
;29--
		movlw	' '				; ' '
		goto	l_char			;

;  DateFormat: DD-MM-YY
DF_DDMMYY
;30--
		movlw	' '				; ' '
		goto	l_char			;
;31--
		movf	Year,w			; Year    1 digit
		goto	l_convert		;
;32--
		movf	Year,w			; Year   10 digit
		goto	l_convert_swap	;
;33--
		movlw	'-'				; '-'
		goto	l_char			;
;34--
		movf	Month,w			; Month   1 digit
		goto	l_convert		;
;35--
		movf	Month,w			; Month  10 digit
		goto	l_convert_swap	;
;36--
		movlw	'-'				; '-'
		goto	l_char			;
;37--
		movf	Day,w			; Day     1 digit
		goto	l_convert		;
;38--
		movf	Day,w			; Day    10 digit
		goto	l_convert_swap
;39--
		goto	l_Space			;

DateTable1


	IF high(DateTable0) != high(DateTable1)
		ERROR "Date jump table page error"
        ENDIF

LoadBCDTime						; load BCD with time digits
		movf	Hour,w
		btfss	Digit12Mode		; 24 hour mode
		goto	lBCDHour
		addlw	-.12			; Prepare to 12 hour mode
		btfss	STATUS,C		; If hour > .12: add -.12
		movf	Hour,w			; Reload hour if hour < 12
		btfsc	STATUS,Z
		addlw	.12				; Correct 0 to 12
lBCDHour
		movwf	BCD

		movlw	high(TimeTable0)
		movwf	PCLATH
		movf	digitindex,w	;
		addwf	digitindex,w	; Clears C

TimeTable0
		addwf	PCL,f			;
;0--
		movf	BCD,w
		goto	l_PM			;
;1--
		movf	BCD,w			; Hour   10 digit
		goto	l_leading_swap_PM	;
;2--
		movf	BCD,w			; Hour    1 digit
		goto	l_convert		;
;3--
		movlw	':'				; ':'
		goto	l_char			;
;4--
		movf	Minute,w		; Minute 10 digit
		goto	l_convert_swap	;
;5--
		movf	Minute,w		; Minute  1 digit
		goto	l_convert		;
;6--
		movlw	':'				; ':'
		goto	l_char			;
;7--
		rrf		Second2,w		; Second 10 digit
		goto	l_convert_swap	;
;8--
		rrf		Second2,w		; Second  1 digit
		goto	l_convert		;
;9--
		goto	l_Space			;


TimeTable1

	IF high(TimeTable0) != high(TimeTable1)
		ERROR "Time jump table page error"
        ENDIF

LoadBCDWDay
		btfsc	ShowDayName
		goto	LoadBCDDay

LoadBCDTemp
		comf	Tempr,w			; Take Temperature negative
		addlw	1				; Complement it
		btfss	Tempr,7
		movf	Tempr,w			; Reload if positive
		movwf	BCD
		movlw	high(TempTable0)
		movwf	PCLATH
		movf	digitindex,w	;
		addwf	digitindex,w	; Clears C

TempTable0
		addwf	PCL,f			;
	ifdef	HumidityMeasure
;0--
l_Space
		movlw	' '				; ' '
		goto	l_char_clr		;
;1--
		incf	WDay,w			; Day of week
		goto	l_convert_sc	;
;2--
		movlw	0x95			; 'C'
		goto	l_char			;
;3--
		movf	BCD,w			; Temperature  1
		goto	l_convert		;
;4--
		movf	BCD,w			; Temperature 10
		goto	l_leading_swap	;
;5--
		movlw	'-'				; if Tempr < 0 '-' else ' '
		goto	l_convert_sign	;
;6--
		movlw	'%'				; '%'
		goto	l_char			;
;7--
		movf	Humidity,w		; Humidity     1
		goto	l_convert		;
;8--
		movf	Humidity,w		; Humidity    10
		goto	l_leading_swap
;9--
		goto	l_Space			;
	else
;0--
l_Space
		movlw	' '				; ' '
		goto	l_char_clr		;
;1--
		movlw	' '				; ' '
		goto	l_char_clr		;
;2--
		incf	WDay,w			; Day of week
		goto	l_convert_sc	;
;3--
		movlw	0x95			; 'C'
		goto	l_char			;
;4--
		movf	BCD,w			; Temperature  1
		goto	l_convert		;
;5--
		movf	BCD,w			; Temperature 10
		goto	l_leading_swap	;
;6--
		movlw	' '				; ' '
		goto	l_char_clr		;
;7--
		movlw	'-'				; if Tempr < 0 '-' else ' '
		goto	l_convert_sign	;
;8--
		movlw	' '				; ' '
		goto	l_char_clr		;
;9--
		goto	l_Space			; ' '
	endif

TempTable1

	IF high(TempTable0) != high(TempTable1)
		ERROR "Temp jump table page error"
        ENDIF

;-------- Update Display memory with rotation measurement

LoadBCDTest						; load BCD with test digits
		movlw	high(TestTable0+1)
		movwf	PCLATH
		movf	digitindex,w	;
		addwf	digitindex,w	; Clears C
		bsf		STATUS,RP1		; Bank2

TestTable0
		addwf	PCL,f			;
;0--
		movlw	' '				; ' '
		goto	l_char			;
;1--
		movf	rot_bcd+4,w		; Rotation digit 10000
		goto	l_BCD			;
;2--
		movf	rot_bcd+3,w		; Rotation digit  1000
		goto	l_BCD			;
;3--
		movf	rot_bcd+2,w		; Rotation digit   100
		goto	l_BCD			;
;4--
		movf	rot_bcd+1,w		; Rotation digit    10
		goto	l_BCD			;
;5--
		movf	rot_bcd,w		; Rotation digit     1
		goto	l_BCD			;
;6--
		movlw	'R'				; 'R'
		goto	l_char			;
;7--
		movlw	'/'				; '/'
		goto	l_char			;
;8--
		movlw	'm'				; 'm'
		goto	l_char			;
;9--
		goto	l_Space			;

TestTable1

	IF high(TestTable0) != high(TestTable1)
		ERROR "Test jump table page error"
        ENDIF

LoadBCDVoltage					; load BCD with voltage digits
		call	ParameterBCD	; Select parameter to display
								; Need to clear PCLATH. but next instructions will overwrite it	
		movwf	BCD				; Copy it to BCD
		movlw	high(VoltTable0+1)
		movwf	PCLATH
		movf	digitindex,w	;
		addwf	digitindex,w	; Clears C
		bsf		STATUS,RP1		; Bank2

VoltTable0
		addwf	PCL,f			;
;0--
		movf	BCD,w			; Parameter low  nibble
		goto	l_hex			;
;1--
		swapf	BCD,w			; Parameter high nibble
		goto	l_hex			;
;2--
		call	ParameterName	; Get name code of parameter
		goto	l_char			;
;3--
		movlw	' '				; ' '
		goto	l_char			;
;4--
		movlw	'V'				; 'V'
		goto	l_char			;
;5--
		movf	Voltage_F,w		; Voltage digit  0.01
		goto	l_convert		;
;6--
		movf	Voltage_F,w		; Voltage digit  0.1
		goto	l_convert_swap	;
;7--
		movlw	'.'				; '.'
		goto	l_char			;
;8--
		movf	Voltage,w		; Voltage digit  1
		goto	l_convert		;
;9--
		movf	Voltage,w		; Voltage digit 10
		goto	l_leading_swap	;

VoltTable1

	IF high(VoltTable0) != high(VoltTable1)
		ERROR "Volt jump table page error"
        ENDIF

l_char_clr
		bcf		UseBinFont		;
l_char
		btfsc	fScrollOn
		return
		bcf		STATUS,RP1		; Bank 0 --- For entry from DrawTest
		goto	ConvToInternal

l_leading_swap_PM
		call	l_convert_swap	; Convert to BCD and swapp
		movf	Scratch2,f
		btfss	STATUS,Z		; If not 0 - display the number
		return
		goto	l_pm_ap			; Check for pm

l_leading_swap
		call	l_convert_swap	; Convert to BCD and swapp
		movf	Scratch2,f		; Remove leading 0
		btfsc	STATUS,Z
		goto	l_Space
		return

l_convert_sign					; Convert sign
		btfsc	Tempr,7
		goto	l_char			; if <0   - "-"
		goto	l_Space			; if >= 0 - space

l_PM
		call	Conv2BCD		; Display pm indicator
		andlw	0xF0
		btfsc	Digit12Mode
		btfsc	STATUS,Z		; If not 0, display number
		goto	l_Space
l_pm_ap
		movlw	.12				; Check for pm
		subwf	Hour,w
		btfss	STATUS,C		;
		goto	l_Space
		movlw	"'"				; If pm display apstroph
		goto	l_char_clr		; Not to use binary font

l_hex
		andlw	0x0F			; Use low nibble only
		movwf	BCD
		sublw	0x09
		movlw	'0'+0x07		; Convert hex codes to A..F
		btfsc	STATUS,C		; If nibble > 9 add 7, and use upper case character
l_char_add0
		movlw	'0'				; Convert to character code
		addwf	BCD,w
		goto	l_char

l_convert_swap
		call	Conv2BCD		; Convert to BCD
l_BCD_swap
		movwf	BCD				; Swap nibbles
		swapf	BCD,w			;
		goto	l_BCD

l_convert_sc					; Blanks data for scroll routines
		btfsc	fScrollOn
		goto	l_Space

l_convert
		call	Conv2BCD		; Convert to BCD
l_BCD
		andlw	0x0F			; Use low nibble
		movwf	Scratch2		; Store it in Scratch2 for leading 0 test
		movwf	BCD				;
		sublw	9
		btfss	STATUS,C		; Check for decimal character (A..F converted to space)
		goto	l_Space
		btfsc	BinMode			; Test for binary mode
		bsf		UseBinFont

		goto	l_char_add0		; Convert to character code, C=0

ParameterName					; Called with Bank2 active
		movf	Selector,w		; Get parameter selector
		addlw	low(ParChar)
		movwf	BCD
		movlw	high(ParChar)
		call	GetXII			; Get name code
		clrf	PCLATH			; Page0
		return

ParameterBCD
		bsf		STATUS,RP1		; Copy parameters on Bank2 to common ram
		movf	Light,w
		movwf	Scratch
		movf	BrightDay,w
		movwf	BCD
		movf	BrightNight,w
		movwf	Scratch2
		movlw	high(ParamTabl)	; Selet the desired one
		movwf	PCLATH
		rlf		Selector,w		; Two instructions for a parameter
		bcf		STATUS,RP1
		andlw	0x1E
		addlw	low(ParamTabl)
		movwf	PCL

;--------
Conv2BCD						; Convert w to BCD (0 <= w <= 99)
		clrf	Scratch2		; Clear high nibble
l_bcd_1
		addlw	-.10			; Sub .10
		incf	Scratch2,f		; Inc high nibble
		btfsc	STATUS,C		; Loop if no carry
		goto	l_bcd_1
		addlw	.10				; Re add .10, C=1
		decf	Scratch2,f		; Correct high nibble
		swapf	Scratch2,f		; Swap to high part of byte
		addwf	Scratch2,w		; Add to low nibble, C=0
		return

;-------- Update Display memory with Time Ticks every 5 minutes

Ticks
		movf	dpi,w
		xorwf	Scratch3,w
		btfss	STATUS,Z		; At a tick position
		return

		bsf		tddH,bTicksLED	; Turn on analogue tick Leds

		btfsc	STick12
		goto	Ticks10			; If single tick at 12 always add .10

		movf	Scratch3,w		; Dpi=30 at 12 o clock
TicksCikl
		xorlw	.20
		btfsc	STATUS,Z		; Draw " at 12, ' at other
		goto	Ticks9
		xorlw	.20^.29
		btfsc	STATUS,Z
		goto	Ticks2
		xorlw	.29^.31
		btfsc	STATUS,Z
		goto	Ticks9

		btfss	AnalMode24h		; If analogue hands operate in 24 hour mode
		goto	Ticks10

		xorlw	.31
		addlw	-.60
		btfsc	STATUS,C
		goto	TicksCikl

Ticks10
		movlw	.1
Ticks9
		addlw	.7				; W = 0 if jump to here
Ticks2
		addlw	.2				; W = 0 if jump to here
		addwf	Scratch3,f
		return

;-------- Update Display memory with XII   III  VI  IX around the clock

Draw36912
		movlw	(1<<bShowDDate)|(1<<bShowDTime)
		andwf	flags,w
		btfss	AnalMode24h		; Hide if analogue hands operate in 24 hour mode
		btfss	STATUS,Z		; Hide if digital displaing on
		return

		movf	dpi,f
		btfss	STATUS,Z
		goto	l_Draw_0

		movlw	5
		movwf	dotindex
		movlw	(IX-XII)+low(XII)+2
		movwf	BCD
l_Draw_0
		btfss	dotindex,3
		goto	l_Draw_2
		movf	dpi,w			;
		xorlw	.27				; Are we at XII
		btfsc	STATUS,Z		;
		goto	l_Start_XII		; Start draw it
		xorlw	.27^.58			; Are we at III
		btfsc	STATUS,Z
		goto	l_Start_III		; Start draw it
		xorlw	.58^.88			; Are we at VI
		btfsc	STATUS,Z		;
		goto	l_Start_VI		; Start draw it
		xorlw	.88^.118		; Are we at IX
		btfss	STATUS,Z		;
		goto	l_Draw_1		; No, continue drawing

l_Start_IX						; Calculate index of char
		movlw	IX-VI
l_Start_VI
		addlw	VI-III			; W = 0 if jump to here
l_Start_III
		addlw	III-XII			; W = 0 if jump to here
l_Start_XII
		addlw	low(XII)
		movwf	BCD				; Store table offset, W = 0 if jump to here
		clrf	dotindex		; Start counting dots
l_Draw_1
		btfsc	dotindex,3
		return					; No more dots to draw
l_Draw_2
		incf	dotindex,f		; Move to next dot
		movlw	high(XII)
		call	GetXII			; get the dot pattern for this column
		clrf	PCLATH
		goto	IorToTdd

CharPatternBin					; Binary font is simple
		movf	dotindex,w
		xorlw	0x03			; If dotidex != 3
		btfss	STATUS,Z
		retlw	0				; Return 0
		movf	BCD,w
		addlw	-(0x60)			; Else return (BCD-0x60) / 6
		clrf	Scratch
lBin1
		addlw	-6
		incf	Scratch,f
		btfsc	STATUS,C
		goto	lBin1
		decf	Scratch,w
		return

;******************************************************************************
;	Processing of RC5 command's, called by main program if a command is received
;******************************************************************************

ProcessRC5
		bsf		STATUS,RP1		; Bank2

		movf	RC5_Cmd,w		;
		xorwf	RC5_Cmd2,f		;
		btfsc	STATUS,Z		;
		goto	ProcessRC5Done	; Process command only once, at least toggle bit changes for a new command 

		andlw	0x7F			;

		select_w
								; Bank2

		case	TScrollMode		;
		  goto	ToggleScrollMode; Toggle scroll mode

		bcf		STATUS,RP1		; Bank0

		case	TOuterLine 		;
		  goto	ToggleOuterLed	; Outer Line
		case	SEC_CL			;
		  goto	ClrSecond		; Adjust time : Clear Seconds
		case	SEC_UP			;
		  goto	IncSecond		; Adjust time : Increment Seconds
		case	SEC_DN			;
		  goto	DecSecond		; Adjust time : Decrement Seconds
		case	MIN_UP			;
		  goto	IncMinute		; Adjust time : Increment Minutes
		case	MIN_DN			;
		  goto	DecMinute		; Adjust time : Decrement Minutes
		case	HOUR_UP			;
		  goto	IncHour			; Adjust time : Increment Hours
		case	HOUR_DN			;
		  goto	DecHour			; Adjust time : Decrement Hours
		case	DAY_UP			;
		  goto	IncDay			; Adjust Date : Increment Days
		case	DAY_DN			;
		  goto	DecDay			; Adjust Date : Decrement Days
		case	WDAY_UP			;
		  goto	IncWDay			; Adjust Date : Increment Day of week
		case	WDAY_DN			;
		  goto	DecWDay			; Adjust Date : Decrement Day of week
		case	MON_UP			;
		  goto	IncMonth		; Adjust Date : Increment Month
		case	MON_DN			;
		  goto	DecMonth		; Adjust Date : Decrement Month
		case	YEAR_UP			;
		  goto	IncYear			; Adjust Date : Increment Year
		case	YEAR_DN
		  goto	DecYear			; Adjust Date : Decrement Year
		case	INDEX_UP		;
		  goto	DecDispOffset	; Adjust index sensor Offset, rotate display left
		case	INDEX_DN		;
		  goto	IncDispOffset	; Adjust index sensor Offset, rotate display right
		case	DigitalMode		;
		  goto	ChgDigitMode	; Change digital display mode
		case	AnalogueMode	;
		  goto	ChgAnalogMode	; Change analogue display mode
		case	AnalogueHands	;
		  goto	ChgAnalogHands	; Change analogue hand options
		case	DemoM			;
		  goto	ToggleDemo		; Demo Mode
		case	TextMode		;
		  goto	ToggleText		; Text Mode
		case	TextMode		;
		  goto	ToggleText		; Scrolling Text Mode
		case	TStaticText		;
		  goto	ToggleStaticText; Static Text Mode
		case	TWDayName		;
		  goto	ToggleName		; Togge day name display and digit display mode
		case	SET_SP			;
		  goto	SetScrollSp		; Set Scrolling speed
		case	TBinMode		;
		  goto	ToggleBinMode	; Toggle binay mode
		case	TLANGUAGE		;
		  goto	SetLanguage		; Set Language
		case	TPICT			;
		  goto	SetPictNum		; Set Picture number
		case	TSlideShow		;
		  goto	ToggleSlideShow	; Toggle slide show mode
		case	TTicks			;
		  goto	ToggleTicks		; Toggle Ticks mode
		case	BRIGHT_UP		;
		  goto	BrightnessUp	; Brightness up
		case	BRIGHT_DN		;
		  goto	BrightnessDown	; Brightness down
		case	BRIGHT_MD		;
		  goto	BrightnessMode	; Brightness mode
		case	LIGHTLIM_UP		;
		  goto	LightLimitUp	; Light limit up
		case	LIGHTLIM_DN		;
		  goto	LightLimitDown	; Light limit down
		case	STANDBY		;
		  bcf	fDspEnabled		; Disable displaying
		goto	ProcessRC5Done

ToggleScrollMode				; Set Scroll mode - Bank2 active
		movlw	RTC_Scrollflags
		movwf	tddH
		incf	ch_flags_RC5,w
		andlw	0x01
		movwf	ch_flags_RC5
		goto	WriteToRtc		; Switch to Bank0, sets IRP 0

SetPictNum
		movlw	RTC_PictNum
		movwf	tddH
		incf	PictNum,f
		bcf		PictNum,5		; Only 32 pictures - Keeps bit 7
		movf	PictNum,w
		call	WriteToRtc		;
		goto	UpdatePict

ToggleSlideShow
		movlw	0x80			; Toggles bit 7 of PictNum 
		xorwf	PictNum,f
		goto	ProcessRC5Done

BrightnessMode
		movlw	RTC_BrightMode	; Get address in RTC
		movwf	tddH
		incf	BrightMode,w	;
		andlw	3
		movwf	BrightMode
		goto	WriteToRtc		;

BrightnessDown
		movlw	-2
BrightnessUp
		addlw	1				; If jump was here w=0
		addwf	Brightness,f
		movlw	low(BrightDay)
		movwf	FSR
		movlw	RTC_BrightDay
		btfsc	Night
		addlw	1
		btfsc	Night
		incf	FSR,f
		movwf	tddH
		movf	Brightness,w
		bsf		STATUS,IRP
		movwf	INDF
		goto	WriteToRtc		; Switcher to Bank0, sets IRP 0

LightLimitDown
		movlw	-2
LightLimitUp
		addlw	1				; If jump was here w=0
		addwf	LightLimit,f
		movlw	RTC_LightLevel
		movwf	tddH
		movf	LightLimit,w
		goto	WriteToRtc		;

SetScrollSp
		movlw	RTC_ScrollSp	; Address of scrolling speed in RTC
		movwf	tddH
		movlw	0xF0			; Set scrolling speed
		addwf	ScrollSpeed,w
		iorlw	0x0F			; Speeds are 0x0F, 0x1F, ..., 0xEF, 0xFF
		movwf	ScrollSpeed
		goto	WriteToRtc		;

SetLanguage						; Set language
		movlw	RTC_Language
		movwf	tddH
		incf	Language,w
		andlw	0x0F			; Only 16 language
		movwf	Language
		goto	WriteToRtc		;

ToggleBinMode
		movlw	RTC_flags5
		movwf	tddH
		movlw	(1 << bBinMode)
		xorwf	RC5_flags,f
		movf	RC5_flags,w
		andlw	(1 << bBinMode)
		goto	WriteToRtc		;


ToggleName
		movlw	1 << bShowDayName	;
		addwf	flags,w
		xorwf	flags,w
		andlw	(1<<bDateFormat) | (1<<bDisp12Mode) | (1<<bShowDayName)
		goto	XorToFlags

ToggleOuterLed
		btfsc	TimeSet			; If time modified
		call	StoreTime		; Write it into RTC
		bcf		TimeSet			; Clear modification flag
		movlw	1 << bShowOuter	;

XorToFlags
		xorwf	flags,f			;

FlagsToRTC						; Save flags settings to RTC ram
		movlw	RTC_flags		; Address Flags
		movwf	tddH
		movf	flags,w
WriteToRtc
		clrf	STATUS			; Bank0, IRP=0
		call	I2CByteWrite
		goto	ProcessRC5Done	;

ChgDigitMode
		movlw	1 << bShowWDay
		addwf	flags,f
		goto	FlagsToRTC		;

ToggleTicks
		movlw	RTC_flags4
		movwf	tddH
		movlw	1<<bShowTicks
		addwf	flags4,w
		andlw	(1<<bShowTicks) | (1<<bSTick12)
		movwf	flags4
		goto	WriteToRtc		;

ChgAnalogHands
		movlw	1 << bShowSecHand	;
		addwf	flags3,f
		goto	Flags3ToRTC

ChgAnalogMode
		movlw	1 << bShow36912	;
		addwf	flags3,w
		xorwf	flags3,w
		andlw	(1<<bHourMode) | (1<<bMinMode) | (1<<bSecMode) | (1 << bShow36912)

XorToFlags3
		xorwf	flags3,f

Flags3ToRTC						; Save flags3 settings to RTC ram
		movlw	RTC_flags3		; Address Flags3
		movwf	tddH
		movf	flags3,w
		goto	WriteToRtc		;

IncDispOffset
		incf	DispOffset,w	;
		call 	CheckIncrement 	;
		goto	SetDispOffset

DecDispOffset
		decf	DispOffset,w	;
		call 	CheckDecrement	;

SetDispOffset					; Save flags settings to RTC ram
		movwf	DispOffset		;
		movlw	RTC_disp_off	; Address DispOffset
		movwf	tddH
		movf	DispOffset,w	;
		goto	WriteToRtc		;

IncSecond
		incf	Second2,f		; Inc seconds
		incf	Second2,f		;
		movlw	.120			;
		subwf	Second2,w		;
		btfsc	STATUS,C		;
ClrSecond						; Clear seconds
		clrf	Second2			;
		goto	TimeSetRC5Done	;

DecSecond
		movlw	.2				; Dec seconds
		subwf	Second2,f		;
		movlw	.120			;
		btfss	STATUS,C		;
		addwf	Second2,f		;
		goto	TimeSetRC5Done	;

IncMinute
		incf	Minute,f		; Inc minute
		movlw	.60				;
		xorwf	Minute,w		;
		btfsc	STATUS,Z		;
StoreMinute
		movwf	Minute			;
		goto	TimeSetRC5Done

DecMinute
		decf	Minute,f		; Dec minute
		btfss	Minute,7
		goto	TimeSetRC5Done	;
		movlw	.59
		goto	StoreMinute

IncHour
		incf	Hour,f			; Inc hour
		movlw	.24				;
		xorwf	Hour,w			;
		btfsc	STATUS,Z		;
StoreHour
		movwf	Hour			;
		goto	TimeSetRC5Done	;

DecHour
		decf	Hour,f			; Dec hour
		btfss	Hour,7
		goto	TimeSetRC5Done	;
		movlw	.23
		goto	StoreHour

IncDay
		decf	DMon,w			; Inc day
		xorwf	Day,w			;
        btfsc	STATUS,Z		;
		clrf	Day				;
		incf	Day,f			;
		goto	TimeSetRC5Done	;

DecDay
		decfsz	Day,f			; Dec day
		goto	ProcessRC5Done	;
		decf	DMon,w			;
		movwf	Day				;
		goto	TimeSetRC5Done	;

IncWDay
		incf	WDay,f			; Inc day of week
		movlw	.7
		xorwf	WDay,w
 	    btfsc	STATUS,Z		;
StoreWDay
		movwf	WDay			;
		goto	TimeSetRC5Done	;

DecWDay
		decf	WDay,f			; Dec day of week
		btfss	WDay,7
		goto	TimeSetRC5Done	;
		movlw	.6
		goto	StoreWDay

IncMonth
		movlw	.12				; Inc month
		xorwf	Month,w			;
		btfsc	STATUS,Z		;
StoreMonth
		movwf	Month			;
		incf	Month,f			;
		goto	TimeSetRC5Done	;

DecMonth
		decfsz	Month,f			; Dec month
		goto	TimeSetRC5Done	;
		movlw	.11				;
		goto	StoreMonth		;

DecYear							; Dec year
		decf	Year,f
		movlw	.99				;
		btfss	Year,7
		goto	TimeSetRC5Done	;
		movwf	Year
		decf	Century,f
		goto	TimeSetRC5Done	;

IncYear							; Increment year -> TimeCheck will correct
		incf	Year,f

TimeSetRC5Done
		bcf		fTimeInv
		bsf		fNewTime		; force display update
		bsf		TimeSet			;

ProcessRC5Done
		bsf		STATUS,RP1		; Bank2
		movf	RC5_Cmd,w		;
		movwf	RC5_Cmd2		;
		bcf		RC5_DataReady	;
		bcf		STATUS,RP1		; Bank0

SetOuterLed
		btfss	fDspEnabled		; Displaying disabled
		goto	OuterLedOff
		btfsc	ShowOuter		;
		bsf		OuterLED		;
		btfss	ShowOuter		;
OuterLedOff
		bcf		OuterLED		;
		return					;

ToggleStaticText				; Static text
		bcf		fScrollOn		; Scrolling OFF
		movlw	1 << bText		; toggle Text flag
		xorwf	flags2,f		;
UpdatePict
		btfsc	fText
		call	PrintDisp		; Print static message
		goto	ProcessRC5Done	;

ToggleText
		call	TextON_OFF		; Scrolling Text mode
		goto	ProcessRC5Done	;

ToggleDemo
		movlw	1 << bDemo		;
		xorwf	flags2,f		;
		goto	ProcessRC5Done	;

;******************************************************************************
; Display mode routines

DemoMode
		movf	Second2,f		;
		btfss	STATUS,Z		;
		return

TextON_OFF
		bcf		fScrollOn		; Scrolling OFF
		movlw	1 << bText		; toggle Text flag
		xorwf	flags2,f		;
		btfss	fText			; test Text flag
		return					;
		bsf		fScrollOn		; Scrolling ON

ClearDisplay
		bsf		STATUS,RP1		; Bank 2
		movlw	0x20			; Set FSR to 0x20
		movwf	EEDATA			; First character is a space
		movwf	FSR				;
		movwf	ch_slidecnt		; Init slide show counter
		movlw	.120
		movwf	ch_dot_index
cldisp
		clrf	INDF			; clear display memory
		bsf		STATUS,IRP
		clrf	INDF			; clear display high memory
		bcf		STATUS,IRP
		call	CheckIncrementFSR; Increment and test pointer
		movwf	FSR
		decfsz	ch_dot_index,f	; Loop for .120 positions
		goto	cldisp			; Loop clears ch_dot_index

      	clrf	ch_blanking		; init stuff needed for Text Scroll function

		movlw	0x80
		andwf	EEADR,f			; Clear address for next read

		movf	ch_flags_RC5,w
		movwf	ch_flags		; Copy ch_flags

		btfsc	fDemo			; Copy demo mode flag
		bsf		ch_flags,bDemo_sc

RetBank0
		clrf	STATUS			; Bank 0
		return

;******************************************************************************
;	Read temperature measurement
;******************************************************************************

ReadAd
		bsf		STATUS,RP1		; Bank2
		decf	AD_state,w		; Get A/D state
		bcf		STATUS,RP1		; Bank0
		btfsc	STATUS,Z
		goto	StartAD			; If A/D state == 1 -> Start A/D
		addlw	1
		btfss	STATUS,Z		; If A/D state == 1 -> Read result
		return

		movf	ADRESH,w		; Get the 10 bit result
		movwf	DIV_HI
		bsf		STATUS,RP0		; Bank1
		movf	ADRESL,w
		movwf	DIV_LO
		bcf		STATUS,RP0		; Bank0

		btfsc	ADCON0,CHS1		; A/D channel 2 or 3 selected
		goto	AD_VBuff		; Buffer voltage measured
		btfsc	ADCON0,CHS0		; A/D channel 1 selected
		goto	AD_Humidity		; Humidity measured

AD_TEMP
								; A/D res = (T+273.15)/100/5*1024    0 C ==> 560 = 512 + 48
		call	Div2			; T C = (AD-512)/2
		addlw	-.26			; add -48/2, -2 correction because self heating
		movwf	Tempr			; Store temperature in C (-26 ... 230)

		goto 	SelNextCh		; Next channel is 1 - Humidity

AD_Humidity						; Measured value   0% - 0.55637V - 113.83
								;                100% - 2.536  V - 518.86
		call	Div4			; Divide result by 4
		addlw	-.114/.4
		movwf	Humidity		; Store humidity in %
	ifdef	LightMeasure
		goto	SelNextCh		; Next channel is 2 - Light
	else
		movlw	(1<<CHS1)		; Skipp channel 2
		goto	SelCh			; Next channel is 3 - Buffer voltage
	endif

AD_VBuff
	ifdef	LightMeasure
		btfss	ADCON0,CHS0		; A/D channel was 3 selected
		goto	AD_Light		; 
	endif
								; Measured value  0V - 0.0000V -   0
								;                15V - 2.9297V - 600
		movlw	.40				; Divide by 40
		call	div16X8Divisor
		bsf		STATUS,RP1		; Bank2
		movwf	Voltage			; Store integer part
		clrc
		rlf		DIV_HI,w		; Multiply fractional part with 2.5
;								; DIV_HI < .40, 2*DIV_HI < .80, so C=0
		rrf		DIV_HI,f
		addwf	DIV_HI,w
		movwf	Voltage_F		; Store fractional part
		bcf		STATUS,RP1		; Bank0

	ifdef	LightMeasure
		goto	SelNextCh		; Next channel is 0 - Tempareture

AD_Light						; A/D channel was 2 selected
		call	Div4			; Convert to 8 bits
		bsf		STATUS,RP1		; Bank2
		movwf	Light			; Store measured value
		bcf		STATUS,RP1		; Bank0
		bcf		Dark			; Clear lighting state
		subwf	LightLimit,w	; Compare to limit
	ifdef	GreatherMeansDarker
		btfss	STATUS,C
	else
		btfsc	STATUS,C
	endif
		bsf		Dark			; Dark
	endif

SelNextCh
		movlw	(1<<CHS0)
SelCh
		addwf	ADCON0,f		; Select next channel
		bcf		ADCON0,CHS2
		movlw	3
		goto	SetADState

StartAD
		bsf		ADCON0,GO		; Start A/D conversion
SetADState
		bsf		STATUS,RP1		; Bank2
		movwf	AD_state
		goto	RetBank0		; Bank0

Div4							; Divide result by 4
		call	Div2
		movwf	DIV_LO
Div2							; Divide result by 2
		clrc
		rrf		DIV_HI,f
		rrf		DIV_LO,w
		return

;******************************************************************************
;	Main program
;******************************************************************************

main

InitIO
;		clrf	PORTA
;		clrf	PORTB			; PortA, PortB, PortC cleared at reset vector
;		clrf	PORTC			; all LED OFF

	ifdef	MotorDirFromCalibration
		btfsc	PORTC,bCalibration	; Read Calibration input
		bsf		MotorDirCC		; Set direction of motor rotation
	endif

		bsf		STATUS,RP0		; Bank1

		movlw	0x84			; PORTA 5..4,2 bits are digital I/O, 3, 1, 0 is analog, A/D right justified
		movwf	ADCON1

		movlw	0xFF ^ ((1<<bInnerLED)); Set 5 as output, 7,6,4,3,2,1,0  input
		movwf	TRISA
		movlw	(1<<bIndexPuls)	; Set 7..6 as output, 0 as input
 		movwf	TRISB			;
		movlw	0xFF ^ ((1<<bSecLED)|(1<<bBrightPWM)|(1<<bCalibration)|(1<<bOuterLED)|(1<<bTicksLED)|(1<<bPixel0))
		movwf	TRISC			;

	    clrf    SSPSTAT			; Disable SMBus inputs
    	bsf     SSPSTAT,SMP		; Disable slew rate control
	    movlw   0x31			; Divide FOSC/4 by .50
    	movwf   SSPADD			; Setup 100 kHz I2C clock
	    clrf    SSPCON2			; Clear control bits

;		clrf	PIE1			; dissable all possible peripheral interrupt enable
		movlw	b'10011000'		; set up timer. prescaler(bit3)bypassed, Int on falling Edge RB0
		movwf	OPTION_REG		;
		movlw	.249			; TMR2 period=250 PR2=249
		movwf	PR2				;
		bsf		PIE1,TMR2IE		; enable Peripheral interrupt from TMR2

		bsf		STATUS,RP1		; Bank3
		clrf	EECON1			; Clear EECON - Data EEProm access select

		movlw	0x0F			; Only RA3, 2, 1, 0  analogue
		movwf	ANSEL
		clrf	ANSELH

		bcf		STATUS,RP0		; Bank2

		movlw	0x0F			; May be removed

		movwf	rot_bcd			; Blank rotation BCD data
		movwf	rot_bcd+1		; Wait min. 2 minutes to get first measurement
		movwf	rot_bcd+2
		movwf	rot_bcd+3
		movwf	rot_bcd+4

		movwf	AD_state

		clrf	Voltage
		clrf	Voltage_F
		clrf	Light
		clrf	EEADR

		clrf	Selector
		clrf	PictNum_sc

		clrf	RC5_flags		; clear RC5_flags
		bsf		RC5_WaitStart	; RC5 is waiting for startbit
		clrf	RC5_Tmr			;

		bcf		STATUS,RP1		; Bank0

		clrf	Tempr
		clrf	Humidity

;		clrf	Brightness
		movlw	0x0C			; PWM mode bit 1..0 are don't care: can be used for data storage
		movwf	CCP1CON

		clrf	TMR1H
		clrf	TMR1L

		movlw	b'00101000'
		movwf	SSPCON			; Enable SSP, select I2C Master mode

		movlw	0x81			; A/D on , Fosc/32, Ch 0
		movwf	ADCON0

InitRam
		clrf	PixelWidth		;
		clrf	PixelOff		;

		call	InitSubSecCnt	; reload counter SubSecond = 0x10000 - .2500 = 0xF63C, returns 0x81

		clrf	flags2			; clear flags2 - holds TimeInv set by InitTime

		call	InitTime		; Init time counter variables and Display flags

		movf	DispOffset,w
		movwf	iFSR

InitDisp
		call 	ClearDisplay	; clear display content
InitTmr
		clrf	INTCON			; clear and dissable all possible interrupt flag
		clrf	PIR1			; clear all pripheral interrupt flag
		clrf	PIR2

		movlw	b'00000101'		; TMR2=ON, Prescaler = 1/4,  Postscaler = 1
		movwf	T2CON			;

		clrf	TMR0			; start timer

		movlw	(1<<T0IE) | (1<<INTE) | (1<<PEIE) | (1<<GIE)
		movwf	INTCON

MainLoop
		btfsc	RC5_DataReady	; RC5_flags is in common area
		call	ProcessRC5		;

		movf	TmrScroll,f		;
		btfss	STATUS,Z		;
		goto	NoScrolling		;
		movf	ScrollSpeed,w	; Get Scrolling speed
		movwf	TmrScroll		; Reinit scrolling timer

		btfsc	fScrollOn		;
		call	ScrollText		;

		btfss	fText
		goto	NoScrolling

		btfsc	PictNum,7		; Slideshow
		call	SlideShow

NoScrolling
		btfsc	fNewPCnt		; test Period counter flag
		call	CalcPixelPitch  ; calculate new pixel pitch

		btfss	ADCON0,GO		; Test for A/D conversion ready
		call	ReadAd			; Read temperature

		btfss	fNewTime		; test new time flag
		goto  	BrightControl	;
								; one second past. Do timecheck and if text=off, also a display update.
		call	TimeCheck		;

		btfsc	Second2,0
		goto	TestDemo

		bsf		STATUS,RP1		; Bank2
		incf	Selector,f		; Increment parameter selector
		movf	Selector,w
		xorlw	ParEnd-ParChar
		btfsc	STATUS,Z		; Check limit
		movwf	Selector
		bcf		STATUS,RP1		; Bank0

TestDemo
		btfsc	fDemo			; Check for demo mode
		call	DemoMode		;

		btfsc	fText			; Test Image flag
		goto  	BrightControl	;

ConvRot
		btfsc	NewRot			; Convert rotation measurement
		call	Word2Bcd5

ReDrawClock						; start display memory update
		clrf	dpi				; dpi = 0
		clrf	Scratch3		; keep index for 5 min timeticks
		movlw	0x20			; start of display memory

lUpdateDisplay
		movwf	FSR

		clrf	tdd				;
		clrf	tddH

		btfsc	ShowDTime		;
		call	DTime			; Display digital Time

		btfsc	ShowDDate		;
		call	DDate			; Display digital Date

		call	Hands			; Analogue clock's hands

		btfsc	ShowTicks		;
		call	Ticks			; Analogue clock's ticks

		btfsc	Show36912		;
		call	Draw36912		; XII, III, VI, IX around the clock

		movf	flags,w			; Check for show rotation speed and buffer voltage...
		andlw	(1<<bShowDDate)|(1<<bShowDTime)
		btfss	STATUS,Z
		goto	StoreTdd
		movf	flags3,w
		andlw	(1<<bShowHourHand)|(1<<bShowMinHand)|(1<<bShowSecHand)|(1<<bShow36912)
		btfss	STATUS,Z
		goto	StoreTdd
		call	DTime			; Display digital Rotation
		call	DDate			; Display digital voltage

StoreTdd
		movf	tdd,w			; store data in
		movwf	INDF			; display memory
		movf	tddH,w			; Store high data
		bsf		STATUS,IRP		; Ind will use Bank 2 & 3
		movwf	INDF			; high display memory
		bcf		STATUS,IRP		; Ind will use Bank 0 & 1

		incf	dpi,f			;
		movlw	.120			;
		xorwf	dpi,w			;
		btfsc	STATUS,Z		;
		goto  	BrightControl	;
		call	CheckIncrementFSR ; Move pointer
		goto	lUpdateDisplay


BrightControl
		movlw	low(BrightDay)
		movwf	FSR
		btfss	BrightMode,0
		goto	SetStatic

	  ifdef	LightMeasure
		btfsc	BrightMode,1
		goto	AutoPwm
	  endif

		bsf		Night
		movlw	.8
		subwf	Hour,w
		btfsc	STATUS,C
		bcf		Night

		movlw	.20
		subwf	Hour,w
		btfsc	STATUS,C
SetNight
		bsf		Night
		btfsc	Night
		incf	FSR,f
SetBr
		bsf		STATUS,IRP
		movf	INDF,w
		bcf		STATUS,IRP
		movwf	Brightness

		goto	MainLoop

SetStatic
		btfss	BrightMode,1
		goto	SetDay
		goto	SetNight
		
	  ifdef	LightMeasure
AutoPwm
		btfsc	Dark
		goto	SetNight	
SetDay
		bcf		Night
		goto	SetBr

	  endif

;******************************************************************************
;	Text display functions
;******************************************************************************

ScrollText
		bsf		STATUS,RP1		; Bank2

      	movf	ch_dot_index,w	;
		btfss	STATUS,Z		; dot_index == 0 ?
		goto	Scroll_0

		movf	ch_blanking,w	;
		btfsc	STATUS,Z		;
		goto	Scroll_read_ee	;
		decfsz	ch_blanking,f	;
		goto	Scroll_2		; insert one more " "

		movlw	0x80			; Change message
		xorwf	EEADR,f			;
		andwf	EEADR,f			; start of message
		movlw	' '
		movwf	EEDATA

		movf	ch_flags_RC5,w
		xorwf	ch_flags,w
		andlw	0x01
		xorwf	ch_flags,f

		btfss	ch_flags,bDemo_sc; in demo mode?
		goto	Scroll_read_ee	; re-read char

		bcf		STATUS,RP1		; Bank0
		goto	TextON_OFF		; at end of line, turn text off!

Scroll_read_ee
		movf	EEDATA,w
		btfss	STATUS,Z		; if Z, end of message
		goto	Scroll_1		;

		movlw	0x0F			; insert 15 " " at end of string to clear display
		movwf	ch_blanking		;
Scroll_2
		movlw	" "				; w = " "
Scroll_1

		call	ConvToInternalNotBin
		movlw	6
		movwf	ch_dot_index

Scroll_0
		call	CharTable
		clrf	PCLATH			; Call / Goto in first 2K
		movwf	Scratch			;
		call	IncBCD16		; Move index of character table

	ifdef ReversTextScroll
		movlw	0xE4			; Start at begin of 7 o'clock character
	else
		movlw	0xBF			; Start at end   of 5 o'clock character
	endif

		movwf	FSR				;
ScrollLoop
		movf	INDF,w			; Get current display data = Previous_INDF
		xorwf	Scratch,w		; W = (Previous_INDF ^ Scratch)
		xorwf	INDF,f			; INDF = (Previous_INDF ^ Scratch) ^ Previous_INDF = Scratch
		xorwf	Scratch,f		; Scratch = (Previous_INDF ^ Scratch) ^ Scratch = Previous_INDF

	ifdef ReversTextScroll

		call	CheckIncrementFSR	; Move pointer
		movwf	FSR				;
		xorlw	0xC0			; check end of 5 o'clock character

	else

		call	CheckDecrementFSR	; Move pointer
		movwf	FSR				;
		xorlw	0xE3			; check begin of 7 o'clock character

	endif

		btfss 	STATUS,Z		; FSR = 0xE4 ? (or 0xC0)
		goto	ScrollLoop		;

		decfsz	ch_dot_index,f
		goto	RetBank0

		btfss	ch_flags,bDemo_sc; Use the copy of demo mode flag
		btfss	ScrollDateTime
		goto	EPROM_READ

GetNextTimeChar
								; Char index is in EEADR called with Bank2
		movf	EEADR,w			; Get address
		andlw	0x7F
		movwf	digitindex
		btfsc	STATUS,Z
		call	CopyTimeHum		; Copy date, time, temperature and humidity
		movf	digitindex,w
		call	Conv2BCD		; A part of message has 10 characters
		movwf	BCD				; Get table number to high nibble of BCD
		andlw	0x0F
		movwf	digitindex		; Store index

		btfsc	BCD,6			; if table number >=4 end of message
		goto	ScrEnd

		btfsc	BCD,5			; Table 2 and 3
		goto	ScrDayTemp

		btfsc	BCD,4			; Table 1 is Time
		goto	ScrTime

ScrDate							; Table 0 is Date
		sublw	9				; Reverse index
		movwf	digitindex
		call	LoadBCDDate		; Get next char of date
		goto	ScrExit

ScrTime
		call	LoadBCDTime		; Get next char ot time
		goto	ScrExit

ScrDayTemp
		btfss	BCD,4
		goto	ScrDay

ScrTemp							; Table 3 is temperature
		sublw	9				; Reverse index
		movwf	digitindex
		call	LoadBCDTemp		; Get next char on humidity and temerature
		goto	ScrExit

ScrDay							; Table 2 id name of day
		bcf		UseBinFont

		movf	Language_sc,w	; Get language code

		call	DayAddress

		movf	digitindex,w	; Add digitindex
		call	AddBCD16
		movlw	high(DayNames)	; Add address of table
		call	CharPattern		; Get next character
		clrf	PCLATH			; Use page 0
		goto	ScrExit

ScrEnd
		clrw					; End of message
ScrExit
		movwf	EEDATA
		goto	IncEEAdr

EPROM_READ
		bsf		STATUS, RP0		; Bank 3
		bsf		EECON1, RD		; read character from EEProm
		bcf		STATUS ,RP0		; Bank 2
IncEEAdr
		movf	ch_blanking,f
		btfsc	STATUS,Z
		incf	EEADR,f			; inc address for next read
		goto	RetBank0

; Problem of scrolling message with data in it:
; While scrolling the message, data can change - so a part of old data mixed with outher part of new data may be scrolled
; To avoid this date, time, humidity, temperature, language code copied to a buffer
; To use the original routines written for the clock, the buffer is on same address but on an other bank.
; Real time changing data are on Bank0 - buffered are on Bank2

CopyTimeHum
		movlw	Second2			; Copy from Second2 to Humidity
		movwf	FSR
		movlw	Humidity+1-Second2
		movwf	BCD
CopyLoop
		movf	INDF,w			; Get data from Bank0
		bsf		STATUS,IRP
		movwf	INDF			; Write it to Bank2
		bcf		STATUS,IRP
		incf	FSR,f
		decfsz	BCD,f
		goto	CopyLoop
		bcf		STATUS,RP1		; Bank0
		movf	Language,w		; Get Language code
		bsf		STATUS,RP1		; Bank2
		movwf	Language_sc		; Save it
		return

SlideShow
		bsf		STATUS,RP1		; Bank2
		decfsz	ch_slidecnt,f
		goto	RetBank0

		bsf		ch_slidecnt,7

		movf	PictNum_sc,w
		incf	PictNum_sc,f
		andlw	0x0F
		iorlw	0x10
		bcf		STATUS,RP1		; Bank0
		goto	DispPict

PrintDisp
		call	ClearDisplay	; clear display content Clear EEADR
		movf	PictNum,w
		andlw	0x1F
		btfsc	STATUS,Z
		goto	DispZodiac		; Pict0 is Zodiac ring
DispPict
		movwf	BCD_High		; Copy pictnum to BCD_High
		clrf	BCD				; Convert picnum to memory address
		rrf		BCD_High,w
		andlw	0xF
		movwf	BCD_High
		rrf		BCD,f
		movlw	0xD1			; Start at 6'o clock
		goto	DispPict0

DispZodiac
		clrf	BCD_High
		decf	Month,w			; Get month
		movwf	Scratch			; Store as symbol munber
		addlw	low(ZodiacDay)	; Look for change day
		movwf	BCD
		movlw	high(ZodiacDay)
		call	CharPattern		; Get day of change
		clrf	PCLATH			; Use page 0

		subwf	Day,w			; If Day >= Change day  - inc symbol number
		btfsc	STATUS,C
		incf	Scratch,f
		movlw	.12				; Modulo 12
		xorwf	Scratch,w
		btfss	STATUS,Z
		movf	Scratch,w
		call	Mult10			; A symbol has 10 bytes
		movlw	0x20+.25		; Start at top-5

DispPict0
		movwf	Scratch2
		movwf	FSR				; Store start address

DispPict1
		movlw	high(Pict00)
		call	CharPattern		; Get next data
		clrf	PCLATH			; Use page 0

		movwf	INDF			; Store in buffer
		incf	BCD,f			; Move memory pointer

		movlw	0x7F			; Mask bit7 off
		andwf	BCD,w
		xorlw	.120			; Clear pict pointer if >=120
		btfsc	STATUS,Z
		movwf	BCD

		call 	CheckIncrementFSR	; Move pointer
		movwf	FSR				;

		xorwf	Scratch2,w		; Check for end of buffer
		btfss	STATUS,Z
		goto	DispPict1
		goto	RetBank0

;******************************************************************************
;	Some general functions
;******************************************************************************

;- check correct decrement of Display memory pointer
CheckDecrementFSR
		decf	FSR,w
CheckDecrement
		xorlw	0x1F			;
		btfsc	STATUS,Z		;
		movlw	0xEF ^ 0x1F		;
		xorlw	0x9F ^ 0x1F		;
		btfsc	STATUS,Z		;
		movlw	0x47 ^ 0x9F		;
		xorlw	0x9F			;
		return					;

;- check correct increment of Display memory pointer
CheckIncrementFSR
		incf	FSR,w
CheckIncrement
		xorlw	0x48			;
		btfsc	STATUS,Z		;
		movlw	0xA0 ^ 0x48		;
		xorlw	0xF0 ^ 0x48		;
		btfsc	STATUS,Z		;
		movlw	0x20 ^ 0xF0		;
		xorlw	0xF0			;
		return					;

;******************************************************************************
;	I2C RTC functions
;******************************************************************************

InitTime						; Init time from I2C RTC
		movlw	RTC_Second		; Address of Seconds
								; tddH = 0x02
		call	I2CByteReadSetAddr; Read seconds
		call	BcdToBin
		movwf	Second2
		addwf	Second2,f
								; tddH = 0x03

		call	I2CByteRead		; Read minute
		call	BcdToBin
		movwf	Minute
								; tddH = 0x04

		call	I2CByteRead		; Read hour
		call	BcdToBin3F		; Mask format and AM/PM
		movwf	Hour
								; tddH = 0x05

		call	I2CByteRead		; Read day
		call	BcdToBin3F
		movwf	Day
		swapf	tdd,f
		rrf		tdd,f
		rrf		tdd,w
		andlw	0x03
		movwf	dotindex		; Save year bit 1..0
								; tddH = 0x06

		call	I2CByteRead		; Read Month
		andlw	0x1F
		call	BcdToBin
		movwf	Month
		swapf	tdd,f
		rrf		tdd,w
		andlw	0x07
		movwf	WDay
								; tddH = 0x07

		movlw	RTC_Year		; Address of Year
		call	I2CByteReadSetAddr; Read year
		movwf	Year
								; tddH = 0x11

		call	I2CByteRead		; Read year ^ 0xff
		xorlw	0xFF
		xorwf	Year,w
								; tddH = 0x12

		btfss	STATUS,Z
		goto	InitTimeDef		; Year not valid, init with default values

		call	I2CByteRead		; Read Century
		movwf	Century
								; tddH = 0x13

		call	I2CByteRead		; Read flags
		movwf	flags
								; tddH = 0x14

		call	I2CByteRead		; Read DispOffset
		movwf	DispOffset
		call	ValidateOffs
								; tddH = 0x15

		call	I2CByteRead		; Read Scroll speed
		iorlw	0x0F
		movwf	ScrollSpeed
								; tddH = 0x16

		call	I2CByteRead		; Read flags3
		movwf	flags3
								; tddH = 0x17

		call	I2CByteRead		; Read flags4
		andlw	0x30
		movwf	flags4
								; tddH = 0x18

		call	I2CByteRead		; Read language code
		andlw	0x0F
		movwf	Language
								; tddH = 0x19

		call	I2CByteRead		; Read flags5
		andlw	(1<<bBinMode)
		iorwf	RC5_flags,f
								; tddH = 0x1A

		call	I2CByteRead		; Read ch_flags
		bsf		STATUS,RP1		; Bank2
		movwf	ch_flags_RC5
		bcf		STATUS,RP1		; Bank0
								; tddH = 0x1B

		call	I2CByteRead		; Read PictNum
		andlw	0x1F
		movwf	PictNum
								; tddH = 0x1C

		call	I2CByteRead		; Read Bright mode
		andlw	3
		movwf	BrightMode
								; tddH = 0x1D

		call	I2CByteRead		; Read Bright Daytime
		bsf		STATUS,RP1		; Bank2
		movwf	BrightDay
		bcf		STATUS,RP1		; Bank0
;		movwf	Brightness
								; tddH = 0x1E

		call	I2CByteRead		; Read Bright Night
		bsf		STATUS,RP1		; Bank2
		movwf	BrightNight
		bcf		STATUS,RP1		; Bank0
								; tddH = 0x1F

		call	I2CByteRead		; Read Light limit
		movwf	LightLimit
								; tddH = 0x20


		movf	Year,w			; Has the RTC increment Year
		xorwf	dotindex,w
		andlw	0x03
		btfsc	STATUS,Z
		return					; No, return

		incf	Year,f			; Yes, increment Year
		call	TimeCheck		; Will correct Year and Century
		goto	StoreYear

ValidateOffs
		movlw	0x20
		subwf	DispOffset,w
		btfss	STATUS,C
		goto	DifDispOffs
		movlw	0xF0
		subwf	DispOffset,w
		btfsc	STATUS,C
		goto	DifDispOffs
		movlw	0xA0
		subwf	DispOffset,w
		btfsc	STATUS,C
		return
		movlw	0x48
		subwf	DispOffset,w
		btfss	STATUS,C
		return
DifDispOffs
		movlw	DISPLAYOFFSET
		movwf	DispOffset
		return

InitTimeDef
		movlw	.12				;why do clocks always start
		movwf	Hour			;at 12:00 ?
		clrf	Minute
        clrf	Second2
		movlw	.1
		movwf	Day
		clrf	WDay			; 01-01-2001 was monday
		movwf	Month
		movwf	Year
		movlw	.20
		movwf	Century

		bsf		fTimeInv

		movlw	RTC_flags
		movwf	tddH
								; tddH = 0x13

;;1		movlw	(1<<bShowDTime) | (1<<bShowDDate) | (1<<bShowOuter) | (1<<bShowDayName) | (1<<bShowWDay)
		movlw	(1<<bShowDTime) | (1<<bShowDDate) | (1<<bShowOuter) | (1<<bShowWDay)
		movwf	flags			; Init flags
		call	I2CByteWrite	; Store it in RTC
								; tddH = 0x14

								; Display offset
		movlw	DISPLAYOFFSET	; Initial Display Offset for the hardware
		movwf	DispOffset		;
		call	I2CByteWrite	; Store it in RTC
								; tddH = 0x15

		movlw	0xFF			; Scroll spped
		movwf	ScrollSpeed
		call	I2CByteWrite	; Store it in RTC
								; tddH = 0x16

		movlw	(1<<bShowHourHand) |(1<<bShowMinHand) |(1<<bShowSecHand)
		movwf	flags3			; Init flags3
		call	I2CByteWrite	; Store it in RTC
								; tddH = 0x17

		movlw	(1<<bShowTicks)
		movwf	flags4			; Init flags4
		call	I2CByteWrite	; Store it in RTC
								; tddH = 0x18

		movlw	0
		movwf	Language		; Init language
		call	I2CByteWrite	; Store it in RTC
								; tddH = 0x19
 
;		movlw	0
;		iorwf	RC5_flags,f		; Init flags5
		call	I2CByteWrite	; Store it in RTC
								; tddH = 0x1A
;		movlw	0
		bsf		STATUS,RP1		; Bank2
		movwf	ch_flags_RC5	; Init scroll flags
		bcf		STATUS,RP1		; Bank0
		call	I2CByteWrite	; Store it in RTC
								; tddH = 0x1B


;		movlw	0x0				; Pict number
		movwf	PictNum

		call	I2CByteWrite	; Store it in RTC
								; tddH = 0x1C

		movlw	0x03			; Brightness mode
		movwf	BrightMode
		call	I2CByteWrite	; Store it in RTC
								; tddH = 0x1D

		movlw	0xF0			; Brightness for daytime
		bsf		STATUS,RP1		; Bank2
		movwf	BrightDay
		bcf		STATUS,RP1		; Bank0
		call	I2CByteWrite	; Store it in RTC
								; tddH = 0x1E

		movlw	0x20			; Brightness night
		bsf		STATUS,RP1		; Bank2
		movwf	BrightNight
		bcf		STATUS,RP1		; Bank0
		call	I2CByteWrite	; Store it in RTC
								; tddH = 0x1F

		movlw	0x80			; External light level
		movwf	LightLimit
		goto	I2CByteWrite	; Store it in RTC

BcdToBin3F
		andlw	0x3F
BcdToBin						; Uses only numbers < 60
		movwf	FSR
		andlw	0x0F
		btfsc	FSR,4
		addlw	.10
		btfsc	FSR,5
		addlw	.20
		btfsc	FSR,6
		addlw	.40
;		btfsc	FSR,7
;		addlw	.80
		return

StoreTime
		movlw	0x80			; Disable counting command
		call	SendRtcCmd
								; tddH = 0x01

		incf	tddH,f			; Address of seconds
								; tddH = 0x02

		bcf		STATUS,C		; Store Second
		rrf		Second2,w
		call	I2CBcdWrite
								; tddH = 0x03

		movf	Minute,w		; Store Minute
		call	I2CBcdWrite
								; tddH = 0x04

		movf	Hour,w			; Store Hour
		call	I2CBcdWrite
								; tddH = 0x05

		swapf	Year,w			; Store Day and Year bit 1..0
		movwf	tdd
		rlf		tdd,f
		rlf		tdd,f
		movlw	0xC0
		andwf	tdd,f
		movf	Day,w
		call	BCDIorI2CWrite
								; tddH = 0x06

		swapf	WDay,w			; Store Month and Wday
		movwf	tdd
		rlf		tdd,f
		movlw	0xE0
		andwf	tdd,f
		movf	Month,w
		call	BCDIorI2CWrite
								; tddH = 0x07

		movlw	0x00			; Enable counting command
		call	SendRtcCmd
								; tddH = 0x01

StoreYear
		movlw	RTC_Year		; Address Year
		movwf	tddH
								; tddH = 0x10
		movf	Year,w
		call	I2CByteWrite
								; tddH = 0x11
		comf	tdd,w
		call	I2CByteWrite
								; tddH = 0x12
		movf	Century,w
		goto	I2CByteWrite

BCDIorI2CWrite
		call	Conv2BCD
		iorwf	tdd,w
		goto	I2CByteWrite


SendRtcCmd
		clrf	tddH			; tddH = 0x00
		goto	I2CByteWrite

; Only tdd and tddH used in MSSP I2C routines


;*******************Start bit subroutine**************************
;           This routine generates a Start condition
;           (high-to-low transition of SDA while SCL
;           is still high.
;*****************************************************************
I2CBSTART
	    bcf     PIR1,SSPIF          ; Clear SSP interrupt flag
	    bsf     STATUS,RP0          ; Select Bank 01
	    bsf     SSPCON2,SEN         ; Generate Start condition
		goto	I2Cbstop_wait0

;*******************Restart bit subroutine**************************
;           This routine generates a Repeated Start
;           condition (high-to-low transition of SDA
;           while SCL is still high.
;*****************************************************************
;I2CBRESTART
;	    bcf     PIR1,SSPIF          ; Clear SSP interrupt flag
;	    bsf     STATUS,RP0          ; Select Bank 01
;	    bsf     SSPCON2,RSEN        ; Generate Restart condition
;		goto	I2Cbstop_wait0

;*******************Data transmit subroutine**********************
;           This routine transmits the byte of data
;           stored in w to the I2C serial
;           device.
;*****************************************************************
I2CTX
	    bcf		PIR1,SSPIF          ; Clear SSP interrupt flag
    	movwf	SSPBUF              ; Write byte out to device

		goto	I2Cbstop_wait0		; Wait for oparation completed

;*******************Byte read test subroutine*********************
;           This routine tests the byte read feature
;           of the I2C serial device.  It will read
;           1 byte of data at address I2CAddress from the device.
;*****************************************************************

I2CByteReadSetAddr
		movwf	tddH

I2CByteRead
		call	I2CBSTART           ; Generate Start condition
        	                        ; Send control byte
		movlw   RTC_ADDR & 0xFE		; Load control byte for write
    	call    I2CTX               ; Send control byte to device

		movf	tddH,w				; Get address
    	call    I2CTX               ; Send address to device

;    	call    I2CBRESTART         ; Generate Restart condition
	    bcf     PIR1,SSPIF          ; Clear SSP interrupt flag
	    bsf     STATUS,RP0          ; Select Bank 01
	    bsf     SSPCON2,RSEN        ; Generate Restart condition
		call	I2Cbstop_wait0
            	                    ; Send control byte
    	movlw   RTC_ADDR | 0x01		; Load control byte for read
    	call    I2CTX               ; Send control byte to device

        	                        ; Read data byte
    	bcf     PIR1,SSPIF          ; Clear SSP interrupt flag
    	bsf     STATUS,RP0          ; Select Bank 01
    	bsf     SSPCON2,ACKDT       ; Select to send NO ACK bit
    	bsf     SSPCON2,RCEN        ; Initiate reception of byte

		call	I2Cbstop_wait0		; Wait for oparation completed

    	movf    SSPBUF,W            ; Copy byte to WREG
    	bcf     PIR1,SSPIF          ; Clear SSP interrupt flag
    	bsf     STATUS,RP0          ; Select Bank 01
    	bsf     SSPCON2,ACKEN       ; Generate ACK/NO ACK bit

		call	I2Cbstop_wait0		; Wait for oparation completed

		movwf	tdd					; Save data

;*******************Stop bit subroutine***************************
;           This routine generates a Stop condition
;           (low-to-high transition of SDA while SCL
;           is still high.
;*****************************************************************
I2CBSTOP
		incf	tddH,f				; Move address

	    bcf     PIR1,SSPIF          ; Clear SSP interrupt flag
	    bsf     STATUS,RP0          ; Select Bank 01
	    bsf     SSPCON2,PEN         ; Generate Stop condition
I2Cbstop_wait0
	    bcf     STATUS,RP0          ; Select Bank 00
bstop_wait
	    btfss   PIR1,SSPIF          ; Check if operation completed
	ifdef	DebugI2C
		return
	else
	    goto    bstop_wait          ; If not, keep checking
	endif

    	return

;*******************Byte write test subroutine********************
;           This routine tests the byte write feature
;           of the I2C serial device.
;*****************************************************************

I2CBcdWrite
		call	Conv2BCD

I2CByteWrite

		movwf	tdd					; Store data

	    call    I2CBSTART           ; Generate Start condition

	    movlw   RTC_ADDR & 0xFE		; Load control byte for write
    	call    I2CTX               ; Send control byte to device

		movf	tddH,w				; Get address
	    call    I2CTX

		movf	tdd,w				; Get data
    	call    I2CTX               ; Send data byte to device

	   	goto    I2CBSTOP            ; Generate Stop condition


;******************************************************************************
;	Base conversion function for rotation displaying
;******************************************************************************


Word2Bcd5
;Takes hex number in NumH:NumL  Returns decimal in ;TenK:Thou:Hund:Tens:Ones
;written by John Payson

;input

;=A3*16^3 + A2*16^2+ A1*16^1+ A0*16^0 = A3*4096 + A2*256 + A1*16  + A0

NumH            EQU rotation_H  ;A3*16+A2
NumL            EQU rotation_L	;A1*16+A0

;=B4*10^4 + B3*10^3 + B2*10^2 + B1*10^1 + B0*10^0 = B4*10000 + B3*1000 + B2*100 + B1*10 + B0

TenK            EQU rot_bcd+4   ;B4
Thou            EQU rot_bcd+3	;B3
Hund            EQU rot_bcd+2	;B2
Tens            EQU rot_bcd+1	;B1
Ones            EQU rot_bcd		;B0

		bsf		STATUS,RP1	; Bank2
		swapf	NumH,w		;w  = A2*16+A3
        iorlw   0xF0		;w  = A3-16
        movwf   Thou		;B3 = A3-16
        addwf   Thou,f		;B3 = 2*(A3-16) = 2A3 - 32
        addlw   .226		;w  = A3-16 - 30 = A3-46
        movwf   Hund		;B2 = A3-46
        addlw   .50			;w  = A3-46 + 50 = A3+4
        movwf   Ones		;B0 = A3+4

        movf    NumH,w		;w  = A3*16+A2
        andlw   0x0F		;w  = A2
        addwf   Hund,f		;B2 = A3-46 + A2 = A3+A2-46
        addwf   Hund,f		;B2 = A3+A2-46  + A2 = A3+2A2-46
        addwf   Ones,f		;B0 = A3+4 + A2 = A3+A2+4
        addlw   .233		;w  = A2 - 23
        movwf   Tens		;B1 = A2-23
        addwf   Tens,f		;B1 = 2*(A2-23)
        addwf   Tens,f		;B1 = 3*(A2-23) = 3A2-69 (Doh! thanks NG)

        swapf   NumL,w		;w  = A0*16+A1
        andlw   0x0F		;w  = A1
        addwf   Tens,f		;B1 = 3A2-69 + A1 = 3A2+A1-69 range -69...-9
        addwf   Ones,f		;B0 = A3+A2+4 + A1 = A3+A2+A1+4 and Carry = 0 (thanks NG)

        rlf     Tens,f		;B1 = 2*(3A2+A1-69) + C = 6A2+2A1-138 and Carry is now 1 as tens register had to be negitive
        rlf     Ones,f		;B0 = 2*(A3+A2+A1+4) + C = 2A3+2A2+2A1+9 (+9 not +8 due to the carry from prev line, Thanks NG)
        comf    Ones,f		;B0 = ~(2A3+2A2+2A1+9) = -2A3-2A2-2A1-10 (ones complement plus 1 is twos complement. Thanks SD)

;;First two instructions make up negation. So,
;;Ones  = -1 * Ones - 1
;;      = - 2 * (A3 + A2 + A1) - 9 - 1
;;      = - 2 * (A3 + A2 + A1) - 10
        rlf     Ones,f	;B0 = 2*(-2A3-2A2-2A1-10) = -4A3-4A2-4A1-20

        movf    NumL,w		;w  = A1*16+A0
        andlw   0x0F		;w  = A0
        addwf   Ones,f		;B0 = -4A3-4A2-4A1-20 + A0 = A0-4(A3+A2+A1)-20 range -215...-5 Carry=0
        rlf     Thou,f		;B3 = 2*(2A3 - 32) = 4A3 - 64

        movlw   0x07		;w  = 7
        movwf   TenK		;B4 = 7

;B0 = A0-4(A3+A2+A1)-20	;-5...-200
;B1 = 6A2+2A1-138		;-18...-138
;B2 = A3+2A2-46			;-1...-46
;B3 = 4A3-64			;-4...-64
;B4 = 7					;7
; At this point, the original number is
; equal to TenK*10000+Thou*1000+Hund*100+Tens*10+Ones
; if those entities are regarded as two's compliment
; binary.  To be precise, all of them are negative
; except TenK.  Now the number needs to be normal-
; ized, but this can all be done with simple byte
; arithmetic.

        movlw   .10			;w  = 10
Lb1:			;do
        addwf   Ones,f		; B0 += 10
        decf    Tens,f		; B1 -= 1
        btfss   3,0
         goto   Lb1			; while B0 < 0
Lb2:			;do
        addwf   Tens,f		; B1 += 10
        decf    Hund,f		; B2 -= 1
        btfss   3,0
         goto   Lb2			; while B1 < 0
Lb3:			;do
        addwf   Hund,f		; B2 += 10
        decf    Thou,f		; B3 -= 1
        btfss   3,0
         goto   Lb3			; while B2 < 0
Lb4:			;do
        addwf   Thou,f		; B3 += 10
        decf    TenK,f		; B4 -= 1
        btfss   3,0
         goto   Lb4			; while B3 < 0

		movlw	low(rot_bcd)+4
		movwf	FSR
		movlw	4
		movwf	Scratch
		clrf	Scratch2
		bsf		STATUS,IRP

Lb5							; Remove leading spaces
		movf	INDF,w
		iorwf	Scratch2,f
		movlw	0x0F
		btfsc	STATUS,Z
		movwf	INDF
		decf	FSR,f
		decfsz	Scratch,f
		goto	Lb5		

		clrf	STATUS		; Bank0 , IRP0
		bcf		NewRot
        return

		ORG     0xCA0				; These routines are in 1. page: 0x800 -- 0xFFF

ParChar								; Table of parameter characters
		DT	"oslpawfghi"			; Has to have as many characters as many parameter to show
ParBright
		DT	"xtcmdn"
ParEnd

ParamTabl							; Load selected parameter to W, called with Bank0 selected
;0--
		movf	DispOffset,w		; Display offset
		return
;1--
		movf	ScrollSpeed,w		; Scroll speed
		return
;2--
		movf	Language,w			; Language
		return
;3--
		movf	PictNum,w			; Pict number
		return
;4--
		movlw	RemoteAddr			; Address of clock
		return
;5--
		movf	PixelWidth,w		; Width of a pixel
		return
;6--
		movf	flags,w				; flags
		return
;7--
		movf	flags3,w			; flags3
		return
;8--
		movf	flags4,w			; flags4
		return
;9--
		movf	RC5_flags,w			; flags5
		return
;10--
ParamBright
		movf	Scratch,w			; Light
		return
;11--
		movf	LightLimit,w		; Light theshold limit
		return
;12--
		movf	Brightness,w		; Current brightness setting
		return
;13--
		movf	BrightMode,w		; Brightness control mode
		return
;14--
		movf	BCD,w				; Brightness setting daytime
		return
;15--
		movf	Scratch2,w			; Brightness setting night
		return
ParamEnd

	if (ParamBright-ParamTabl)!=2*(ParBright-ParChar)
		error "Parameter table error"
	endif

	if (ParamEnd-ParamTabl)!=2*(ParEnd-ParChar)
		error "Parameter table error"
	endif

	if high(ParChar)!=high(ParamTabl)
		error "Parameter table error"
	endif

	ifdef DebugI2C
		messg	"Don't program this compilation to chip, I2C rutines are debug verisons"
	endif

	END
