;*******************************************************************************
;*                                                                             *
;* Program: LED-DCF-v28.asm                                                    *
;* Author: Chris Deceuninck                                                    *
;* Email: chris@turbokeu.com or chris.deceuninck@skynet.be                     *
;* For PIC16F876 and 6x Common Anode 7-Segment LED displays                    *
;* DCF77 routines initially based on Yves Heilig's DCF77 routines              *
;* Added DD,MM,YY DCF77 decoding                                               *
;* Clock/Calendar routines based on Jaakko Ala-Paavola's Clock routines        *
;*                                                                             *
;* DCF77 LED Clock                                                             *
;* Version 2.8 - 15/01/2006                                                    *
;*                                                                             *
;*******************************************************************************

;*******************************************************************************
;
;	Xtal=4.096MHz
;	Timer0 Prescaler=8
;	ISR=every 2.0msec
;	Interrupt on Timer0 overflow
;
;*******************************************************************************

	LIST	p=16F876A 	; PIC16F876 is the target processor
	#include "P16F876A.INC"	; Include header file
	RADIX	HEX
	EXPAND

	__CONFIG _CP_OFF & _WDT_OFF & _XT_OSC & _LVP_OFF & _BODEN_OFF & _PWRTE_ON

;*******************************************************************************
;*                              - Declarations -                               *
;*******************************************************************************

; DCF77 Variables
SEC_DUR		EQU	0x30	; DCF second duration, used to test absence of 59th sec
BIT_DUR		EQU	0x31	; DCF bit duration
BIT_NBR		EQU	0x32	; DCF bit number (0 to 58)
PARITY		EQU	0x33	; Even Parity check byte
MINDCF		EQU	0x34	; DCF Minutes in packed BCD format: 10's,1's	
HOURDCF		EQU	0x35	; DCF Hours in packed BCD format: 10's,1's
DAYDCF		EQU	0x36	; DCF Days in packed BCD format: 10's,1's
MONTHDCF	EQU	0x37	; DCF Months in packed BCD format: 10's,1's
YEARDCF		EQU	0x38	; DCF Years in packed BCD format: 10's,1's

; Flag bytes
FLAG1		EQU	0x39	; 1st Flag byte
FLAG2		EQU	0x3A	; 2nd Flag byte
FLAG3		EQU	0x3B	; 3rd Flag byte
FLAG4		EQU	0x3C	; 4th Flag byte

; BIN2BCD Conversion Variables
BIN		EQU	0x3D	; Temp register for BIN to BCD conversion
BIN1		EQU	0x3E	; Temp register for BIN to BCD conversion
BIN10		EQU	0x3F	; Temp register for BIN to BCD conversion

; BCD Clock Variables for display output
SEC1		EQU	0x40	; Contains 1's of Seconds
SEC10		EQU	0x41	; Contains 10's of Seconds
MIN1		EQU	0x42	; Contains 1's of Minutes
MIN10		EQU	0x43	; Contains 10's of Minutes
HOUR1		EQU	0x44	; Contains 1's of Hours
HOUR10		EQU	0x45	; Contains 10's of Hours
DAY1		EQU	0x46	; Contains 1's of Days
DAY10		EQU	0x47	; Contains 10's of Days
MONTH1		EQU	0x48	; Contains 1's of Months
MONTH10		EQU	0x49	; Contains 10's of Minutes
YEAR1		EQU	0x4A	; Contains 1's of Years
YEAR10		EQU	0x4B	; Contains 10's of Years
;		EQU	0x4C	; 
LDR		EQU	0x4D	; Contains result from A/D conversion
PWM		EQU	0x4E	; Value to put in CCPR1L register for PWM function
;		EQU	0x4F	; 

; Display Variables
D1		EQU	0x50	; Contains BCD value of Digit 1 to display (most right)
D2		EQU	0x51	; Contains BCD value of Digit 2 to display
D3		EQU	0x52	; Contains BCD value of Digit 3 to display
D4		EQU	0x53	; Contains BCD value of Digit 4 to display
D5		EQU	0x54	; Contains BCD value of Digit 5 to display
D6		EQU	0x55	; Contains BCD value of Digit 6 to display (most left)
DIGIT		EQU	0x59	; Holds digit number to display (1 to 6)

; Counter variables for delays
CNT1		EQU	0x5A	; Counter for 0.5sec Beat timing delay
CNT2		EQU	0x5B	; Counter for Scrolling Routine delay
CNT3		EQU	0x5C	; Counter for Date display timing delay
CNT4		EQU	0x5D	; Counter for hourly Gong timing delay
CTR1		EQU	0x5E	; Delay counter for Main routine
CTR2		EQU	0x5F	; Delay counter for Main routine

; Clock/Calendar Variables
MSEC		EQU	0x60	; 4 millisecs counter (*XD=1sec)
SEC		EQU	0x61	; 24H-Clock Seconds
MIN		EQU	0x62	; 24H-Clock Minutes
HOUR		EQU	0x63	; 24H-Clock Hours
DAY		EQU	0x64	; Calendar Days
MONTH		EQU	0x65	; Calendar Months
YEAR		EQU	0x66	; Calendar Years

; Timer Variables
TIME_ON_H	EQU	0x67	; Time ON Hours
TIME_ON_M	EQU	0x68	; Time ON Minutes
TIME_OFF_H	EQU	0x69	; Time OFF Hours
TIME_OFF_M	EQU	0x6A	; Time OFF Minutes
AUX1		EQU	0x6B	; Used for 16-bit comparisons
AUX2		EQU	0x6C	; Used for 16-bit comparisons

;  Conversion Variables
BCD		EQU	0x6D	; Temp storage for BCD to BIN conversion
BCD1		EQU	0x6E	; Temp storage for BCD to BIN conversion
BCD10		EQU	0x6F	; Temp storage for BCD to BIN conversion

; Common Variables
W_TEMP		EQU	0x70	; W Register storage during Interrupt
STATUS_TEMP	EQU	0x71	; STATUS Register storage during Interrupt
PCLATH_TEMP	EQU	0x72	; PCLATH Register storage during Interrupt
TEMP		EQU	0x73	; Temp storage
OFFSET		EQU	0x74	; Temp storage for Table reads

; Constants (for 4.096MHZ XTAL)
XD		EQU	D'250'	; Xtal divider = D'250' for 4.096MHz Xtal

; FLAG1 bits assignments
M0		EQU	0	; Indicates start of new minute for Time Check
S0		EQU	1	; Indicates start of new minute for DCF_OK
BEATF		EQU	2	; Colon Beat indicator update flag (for 1Hz Beat delay)
TIMERF		EQU 	3	; Time Check status flag (0=OFF, 1=ON)
TDF		EQU	4	; Display Time/Date bit (0=Time, 1=Date)
PBF		EQU	5	; Indicates Push button state change (0=Not changed, 1=changed)
TZ0F		EQU 	6	; Indicates Time Zone ; TZ1F/TZ0F=11=CET
TZ1F		EQU	7	; Indicates Time Zone ; TZ1F/TZ0F=01=CET-1H
										  	  ; TZ1F/TZ0F=10=CET+1H
; FLAG2 bits assignments
ISR		EQU	0	; Flag for DCF77 and Clock update in ISR
RXD		EQU	1	; Indicates reception of a DCF bit (0 or 1)
DCF_BIT		EQU	2	; Value of actual received DCF bit
S59		EQU	3	; Indicates missing 59th pulse of DCF timeframe
CET		EQU	4	; CET Time Zone flag (0= not CET; 1= CET)
DCF_ERR		EQU	5	; DCF Error flag (=0 when Startbit and Parity are OK)
DCF_OK		EQU	6	; DCF frame OK status indicator flag
DCF_ON		EQU	7	; Indicates presence of DCF signal (checked every sec)

; FLAG3 bits assignments
MINF		EQU	0	; Indicates reception of DCF Minute bits
HOURF		EQU	1	; Indicates reception of DCF Hour bits
DAYF		EQU	2	; Indicates reception of DCF Day bits
MONTHF		EQU	3	; Indicates reception of DCF Month bits
YEARF		EQU	4	; Indicates reception of DCF Year bits
DOWF		EQU	5	; Indicates reception of DCF DOW bits
PARF		EQU	6	; Indicates reception of P3 parity bit
LYF		EQU	7	; Leap year flag

; FLAG4 bits assignments
GONGF		EQU	0	; Hourly Gong flag (1=ON, 0=OFF)
SCROLLF		EQU	1	; Flag for scrolling Display routine

; PORTA assignments
LDR_IN 		EQU 	0	; AN0 bit: LDR Analog input
TD		EQU	1	; RA1 bit: "Time/Date" Push Button input
TZ0		EQU 	2	; RA2 bit: "Time Zone0" Jumper input
TZ1		EQU 	3	; RA3 bit: "Time Zone1" Jumper input
DCF_IN		EQU 	4	; RA4 bit: DCF77 signal input
AUTOTD		EQU	5	; RA5 bit: Automatic Time/Date switch input

; PORTB assignments
RB0		EQU	0	; RB0 bit: 4543 D0 bit output
RB1		EQU	1	; RB1 bit: 4543 D1 bit output
RB2		EQU	2	; RB2 bit: 4543 D2 bit output
RB3		EQU	3	; RB3 bit: 4543 D3 bit output
SA_LED		EQU	4	; RB4 bit: Free-running & no DCF signal LED output
DCF_LED		EQU	5	; RB5 bit: DCF77 Beat LED output
BEAT_LED	EQU	6	; RB6 bit: 0.5sec Colon Beat LEDs output
LD		EQU	7	; RB7 bit: Blank output to 4543 LD input

; PORTC assignments
ERR_LED		EQU	0	; RC0 bit: DCF77 Error LED output
AM_PM		EQU	1	; RC2 bit: AM/PM Indicator output
CCP1		EQU	2	; RC2 bit: PWM output to 74HCT238 E3 input
GONG		EQU	3	; RC3 bit: Gong output to SAE800
RC4		EQU	4	; RC4 bit: 74HCT238 A0 bit output
RC5		EQU	5	; RC5 bit: 74HCT238 A1 bit output
RC6		EQU	6	; RC6 bit: 74HCT238 A2 bit output
OK_LED		EQU	7	; RC7 bit: DCF Frame OK LED output

;******************************************************************************
;                                   - Start -                                 *
;******************************************************************************

	ORG	0x00
	GOTO	INIT

	ORG	0x04
	GOTO	INTERRUPT

;*******************************************************************************
;*                                  - Tables -                                 *
;*******************************************************************************

TABLE1 
	ADDWF   PCL,F		; Jump to data, number of days per month+1
	RETLW	0x1E		; Leap-day	30=29	
	RETLW	0x20		; January	32=31
	RETLW	0x1D		; February	29=28
	RETLW	0x20		; Mars		32=31
	RETLW	0x1F		; April		31=30
	RETLW	0x20		; May		32=31
	RETLW	0x1F		; June		31=30
	RETLW	0x20		; July		32=31
	RETLW	0x20		; August	32=31
	RETLW	0x1F		; September	31=30
	RETLW	0x20		; October	32=31
	RETLW	0x1F		; November	31=30
	RETLW	0x20		; December	32=31
		
TABLE1_END
	
; Check to ensure table doesn't cross a page boundary
	IF ( (TABLE1 & 0xFF) >= (TABLE1_END & 0xFF) )
       MESSG   "Warning - User Defined: TABLE1 crosses page boundary in computed jump"
	ENDIF

;------------------------------------------------------------------------------
; Routine to convert a binary value  (0..99) in to a BCD pair. The result is
; returned in W. The lookup is done as a 'calculated GOTO' by modifying PCL
; (the lower eight bits of the program counter). PCL is only eight bits wide
; with the top five bits coming from the PCLATH register.
;------------------------------------------------------------------------------

TABLE2
	MOVWF	PCL
	DT	0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09
	DT	0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19
	DT	0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,0x29
	DT	0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39
	DT	0x40,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49
	DT	0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x59
	DT	0x60,0x61,0x62,0x63,0x64,0x65,0x66,0x67,0x68,0x69
	DT	0x70,0x71,0x72,0x73,0x74,0x75,0x76,0x77,0x78,0x79
	DT	0x80,0x81,0x82,0x83,0x84,0x85,0x86,0x87,0x88,0x89
	DT	0x90,0x91,0x92,0x93,0x94,0x95,0x96,0x97,0x98,0x99

TABLE2_END
	RETLW   0xFF
	
; Check to ensure table doesn't cross a page boundary
	IF ( (TABLE2 & 0xFF) >= (TABLE2_END & 0xFF) )
       MESSG   "Warning - User Defined: TABLE2 crosses page boundary in computed jump"
	ENDIF

;******************************************************************************
;                               - Initialisation -                            *
;******************************************************************************

INIT
	CLRF	STATUS		; Do initialization, select bank 0
	CLRF	INTCON		; Clear int-flags, disable interrupts
	CLRF	PCLATH		; Keep in lower 2 KByte
	CLRF	TMR0		; Clear Timer0
	CLRF	TMR2		; Clear Timer2
	CLRF	TMR1H		; Clear Timer1 upper byte
	CLRF	TMR1L		; Clear Timer1 lower byte
	CLRF	PORTA		; All PORTA output should output low
	CLRF	PORTB		; All PORTB output should output low
	CLRF	PORTC		; All PORTC output should output low
	CLRF	ADCON0		; Clear ADC register
	CLRF	ADRESH		; Clear A/D High result register

	BSF	STATUS,RP0	; Select Bank1

	MOVLW	B'00111111'	; RA0-5 inputs
	MOVWF	TRISA
	MOVLW	B'00000000'	; RB0-7 outputs
	MOVWF	TRISB
	MOVLW	B'00000000'	; RC0-7 outputs
	MOVWF	TRISC

	MOVLW	B'10000010'	; Prescaler 1:8 for Timer0
	MOVWF	OPTION_REG	; Assign Prescaler to TMR0

	MOVLW	H'0E'		; 
	MOVWF	ADCON1		; RA0=AN0, LEFT justified, RA1-5=digital I/O

	CLRF	ADRESL		; Clear A/D Low result register
	
	; Initialize PR2 value
	MOVLW	D'63'		; for 8-bit PWM resolution
	MOVWF	PR2

	BCF	STATUS,RP0	; Select Bank0

	MOVLW	H'81'		; 
	MOVWF	ADCON0		; Fosc/32, select Channel 0 (AN0), A/D is on

	MOVLW	B'00000100'	; Enable TMR2, PostScaler=1, PreScaler=1
	MOVWF	T2CON		; PWM frequency= 16.000KHz

	; Enable PWM on CCP1 and set Duty Cycle
	MOVLW	B'00111100'	; 
	MOVWF	CCP1CON		; PWM ON, CCPCON<5:4>=11 (2 MSB's of 10-bit resolution)
	MOVLW	D'64'		; for 8-bit PWM resolution
	MOVWF	CCPR1L		; Set Duty Cycle to 100%

	; Clear GPR registers (Data RAM) 20-7F (Bank0)
	MOVLW	0x20		; 
	MOVWF	FSR		; Initialize pointer to first RAM location

NEXT
	CLRF	INDF		; Clear INDF register
	INCF	FSR,F		; Increment pointer
	BTFSS	FSR,7		; All done?
	GOTO	NEXT		; No, clear next
	
	; Check TZ0 & TZ1 jumpers and set TZ0F & TZ1F flags accordingly
	BTFSC	PORTA,TZ0	; Read TZ0 jumper, skip if 0
	BSF	FLAG1,TZ0F	; No, set TZ0F flag 
	BTFSC	PORTA,TZ1	; Yes, read TZ1 jumper, skip if 0
	BSF	FLAG1,TZ1F	; No, set TZ1F flag

	; Check for Time Zone CET
	MOVF	FLAG1,W		; Load FLAG1
	ANDLW	B'11000000'	; Mask 2 MSB's from FLAG1
	SUBLW	D'192'		; 
	BTFSC	STATUS,Z	; Timezone=CET? (TZ0F & TZ1F=1), skip if clear
	BSF	FLAG2,CET	; Set CET flag

	; Initialize Time Check values
	MOVLW 	D'22'
	MOVWF	TIME_OFF_H
	MOVLW 	D'59'
	MOVWF	TIME_OFF_M
	MOVLW 	D'03'	
	MOVWF	TIME_ON_H
	MOVLW 	D'01'
	MOVWF	TIME_ON_M

	; Initialize Clock variables (set clock to 12:00:00)
	MOVLW 	D'00'		; Init Seconds
	MOVWF	SEC
	MOVLW 	D'00'		; Init Minutes
	MOVWF	MIN
	MOVLW 	D'12'		; Init Hours 
	MOVWF	HOUR

	; Initialize Calendar variables (set date to 01/01/2000)
	MOVLW 	D'00'		; Init Years 
	MOVWF	YEAR
	MOVLW 	D'01'		; Init Months
	MOVWF	MONTH
	MOVLW 	D'01'		; Init Days
	MOVWF	DAY
		
	; Enable interrupts for Timer0
	MOVLW	B'10100000'
	MOVWF	INTCON		; Set GIE and T0IE

	; Disable interrupts for Timer1 & Timer2
	BSF	STATUS,RP0	; Select bank 1
	BCF	PIE1,TMR1IE 
	BCF	PIE1,TMR2IE 
	BCF	STATUS,RP0	; Select bank 0

;******************************************************************************
;                                - Main Routine -                             *
;******************************************************************************

MAIN
	BTFSC	PORTA,TD		; Check for T/D Push Button pressed (TD=0)
	GOTO	CHECK_AUTOTD		; No, Push Button not pressed
	BTFSC	FLAG1,TDF		; Check TDF flag, skip if clear
	GOTO	TDF_0			; No, clear TDF flag
	BSF	FLAG1,TDF		; Set TDF flag
	GOTO	SCROLL_D		; Continue

TDF_0
	BCF	FLAG1,TDF		; Clear TDF flag

SCROLL_D
	BSF	FLAG4,SCROLLF		; Set SCROLLF flag for display routine
	BTFSS	FLAG1,TDF		; Check TDF flag, skip if set
	GOTO	SCROLL_T		; No, clear PBF flag
	BSF	FLAG1,PBF		; Yes, set PBF flag
	PAGESEL	(SCROLL_DATE)		; Select Program Memory Page of routine
	CALL	SCROLL_DATE		; Scroll Date into display
	BCF	FLAG4,SCROLLF		; Clear SCROLLF flag
	GOTO	CHECK_GONG		; Continue

SCROLL_T
	BCF	FLAG1,PBF		; Clear PBF flag
	PAGESEL	(SCROLL_TIME)		; Select Program Memory Page of routine
	CALL	SCROLL_TIME		; Scroll Time into display
	BCF	FLAG4,SCROLLF		; Clear SCROLLF flag

CHECK_AUTOTD
	BTFSC	PORTA,AUTOTD		; Check for Automatic Time/Date switch status (1=OFF, 0=ON)
	GOTO	CHECK_GONG		; Automatic Time/Date switch not set
	BTFSC	FLAG1,PBF		; Check for PBF flag, skip if clear
	GOTO	CHECK_GONG		; Continue

CHECK_SEC50	; Check for second 50
	MOVLW	D'50'			; 
	SUBWF	SEC,W			; Sec=50?
	BTFSS	STATUS,Z		; Skip if sec=50
	GOTO	CHECK_SEC55		; No, check if sec=55
	BTFSC	FLAG1,TDF		; Skip if TDF=1
	GOTO	CHECK_GONG		; 
	BSF	FLAG1,TDF		; Set TDF flag
	BSF	FLAG4,SCROLLF		; Set SCROLLF flag for display routine
	PAGESEL	(SCROLL_DATE)		; Select Program Memory Page of routine
	CALL	SCROLL_DATE		; Scroll Date into display
	BCF	FLAG4,SCROLLF		; Clear SCROLLF flag

CHECK_SEC55	; Check for second 55
	MOVLW	D'55'			; 
	SUBWF	SEC,W			; Sec=55?
	BTFSS	STATUS,Z		; Skip if sec=55
	GOTO	CHECK_GONG		; No, continue
	BTFSS	FLAG1,TDF		; Skip if TDF=1
	GOTO	CHECK_GONG		; Continue
	BCF	FLAG1,TDF		; Clear TDF flag
	BSF	FLAG4,SCROLLF		; Set SCROLLF flag for display routine
	PAGESEL	(SCROLL_TIME)		; Select Program Memory Page of routine
	CALL	SCROLL_TIME		; Scroll Time into display
	BCF	FLAG4,SCROLLF		; Clear SCROLLF flag

CHECK_GONG
	BTFSS	FLAG4,GONGF		; Check GONGF flag, skip if set
	GOTO	DELAY1			; No, continue
	MOVLW	D'10'			; 
	SUBWF	CNT4,W			; 
	BTFSC	STATUS,Z		; CNT4=10 (=20msec)
	GOTO	GONG_OFF		; Yes, 20msec have passed, clear Gong output
	BSF	PORTC,GONG		; No, set Gong output
	GOTO	DELAY1			; Continue

GONG_OFF
	BCF	FLAG4,GONGF		; Clear GONGF flag
	BCF	PORTC,GONG		; Clear GONG output

DELAY1	; Creates a 1.5sec delay
	MOVLW	D'00'			; 
	SUBWF	CTR1,W			; 
	BTFSS	STATUS,Z		; CTR1 overflow?
	GOTO	MAIN			; No, loop

	INCF	CTR2,F			; Yes, check CTR2
	MOVLW	D'00'			; 
	SUBWF	CTR2,W			; 
	BTFSS	STATUS,Z		; CTR2 overflow?
	GOTO	MAIN			; No, loop       
	CLRF	CTR2			; Yes, clear CTR2

	BSF	ADCON0,GO		; Start A/D conversion

POLL
	BTFSC	ADCON0,GO		; Check GO/DONE bit, skip if clear
	GOTO	POLL			; No, continue GO/DONE bit polling
	MOVF	ADRESH,W		; Yes, read 8 MSB's from conversion result
	MOVWF	CCPR1L			; Store into PWM Duty Cycle register
	GOTO	MAIN 			; Loop      

;******************************************************************************
;                 - Interrupt Service Routine (every 2.0ms) -                 *
;******************************************************************************
	
INTERRUPT				; Interrupt Service Routine
PUSH
	MOVWF	W_TEMP			; Copy W to W_TEMP register
	SWAPF	STATUS,W		; Swap status to save into W
	CLRF	STATUS			; Bank0, regardless of current bank
	MOVWF	STATUS_TEMP		; Save status to STATUS_TEMP register
	MOVF	PCLATH,W		; Copy PCLATH into W
	MOVWF	PCLATH_TEMP		; Save PCLATH to PCLATH_TEMP 
	CLRF	PCLATH			; Page 0, regardless of current page

;	CLRWDT				; Clear Watchdog Timer

	INCF	CTR1,F			; Increment CTR1 (delay for Main Routine)
	INCF	CNT2,F			; Increment CNT2 (delay for Scrolling Routine)
	INCF	CNT3,F			; Increment CNT3 (delay for Date Display Routine)
	INCF	CNT4,F			; Increment CNT4 (delay for hourly gong pulse)

	; DCF77 and Clock routines are called every 2 interrupts (every 4msec)
	; Display routine is called every interrupt (every 2msec)
	BTFSS	FLAG2,ISR		; Check ISR flag, skip if set
	GOTO	PASS_0			; No, ISR=0, Update DCF, Clock & Display
	BCF	FLAG2,ISR		; Yes, ISR=1, clear ISR flag for next pass
	GOTO	PASS_1			; Only update Display

PASS_0
	BSF	FLAG2,ISR		; Set ISR flag for next pass
	PAGESEL	(DCF77)			; Select Program Memory Page of routine
	CALL	DCF77			; Call DCF77 decode routine
	PAGESEL	(CLOCK)			; Select Program Memory Page of routine
	CALL	CLOCK			; Update Clock routine

PASS_1
	PAGESEL	(DISPLAY)		; Select Program Memory Page of routine
	CALL	DISPLAY			; Update Display routine

	CLRF	INTCON			; Clear INTCON Register (& clear TMR0 Overflow bit)
	BSF	INTCON,5		; Enable Timer0 Interrupt
POP
	MOVF	PCLATH_TEMP,W		; Restore PCLATH
	MOVWF	PCLATH			; Store W into PCLATH
	SWAPF	STATUS_TEMP,W		; Swap STATUS_TEMP into W
	MOVWF	STATUS			; Store W into STATUS register
	SWAPF	W_TEMP,F		; Swap W_TEMP
	SWAPF	W_TEMP,W		; Swap W_TEMP into W
	RETFIE				; Return & enable interrupts

;******************************************************************************
;               - DCF77 Signal Decoding Routine (every 4.0ms) -               *
;******************************************************************************

DCF77	; If overflow on SEC_DUR (255=1024msec) we are at sec 59, or no DCF signal
	INCFSZ	SEC_DUR,F		; Increment SEC_DUR, skip if zero (0 to 255)
	GOTO	RA4			; No, test state of RA4 input (DCF77 signal)

SEC59	; 59th sec?
	MOVLW	D'58'			; 
	SUBWF	BIT_NBR,W		; BIT_NBR=58?
	BTFSS	STATUS,Z		; Skip if BIT_NBR=58
	GOTO	SEC255			; Yes, DCF frame completely received
	BTFSS	FLAG2,DCF_ERR		; No, skip if DCF_ERR=1
	BSF	FLAG2,S59		; We now are at second 59

SEC255
	MOVLW	H'FF'			; 
	MOVWF	BIT_NBR			; Set BIT_NBR to 255 (next number will be 0)
	BCF	FLAG2,DCF_ERR		; Clear DCF_ERR flag
	BCF	FLAG2,RXD		; Clear RXD flag
	BCF	FLAG2,DCF_ON		; No DCF pulse, clear DCF_ON flag
	RETURN				; Exit routine (program also exits here when no DCF77 signal)

RA4	; Test RA4 for DCF pulse
	MOVF	PORTA,W			; Read PORTA bits
	MOVWF	TEMP			; Store for later

	; Uncomment following line if the inverted DCF output is used on the DCF receiver
	BTFSS	TEMP,DCF_IN		; Check bit RA4, skip if set (negative pulses)
	; Uncomment following line if the non-inverted DCF output is used on the DCF receiver
;	BTFSC	TEMP,DCF_IN		; Check bit RA4, skip if clear (positive pulses)

	GOTO	RA4_1			; No, check bit RXD
	BTFSS	FLAG2,RXD		; Yes, check bit RXD, skip if set
	RETURN				; Exit routine
	BCF	FLAG2,RXD		; Yes, set RXD=0 (no receive bit)

	MOVLW	D'32'			; 
	SUBWF	BIT_DUR,W		; BIT_DUR=32?
	BTFSS	STATUS,C		; Skip if BIT_DUR>32
	GOTO	INF_17			; No, check BIT_DUR<17
	MOVLW	D'38'			; Yes, BIT_DUR>29
	SUBWF	BIT_DUR,W		; BIT_DUR=38?
	BTFSS	STATUS,C		; Skip if BIT_DUR>38
	GOTO	ERR_DCF			; No, BIT_DUR<38
	MOVLW	D'58'			; Yes, BIT_DUR>38
	SUBWF	BIT_DUR,W		; BIT_DUR=58?
	BTFSC	STATUS,C		; Skip if BIT_DUR<58
	GOTO 	ERR_DCF			; Yes, DCF bit is out of range
	BSF	FLAG2,DCF_BIT		; No, then DCF_BIT=1
	GOTO	NEXT_SEC		; Next DCF_BIT

INF_17
	MOVLW	D'17'			; 
	SUBWF	BIT_DUR,W		; BIT_DUR=17?
	BTFSC	STATUS,C		; Skip if BIT_DUR<17
	GOTO	BIT_0			; No, then DCF_BIT=0

ERR_DCF
	BSF	FLAG2,DCF_ERR		; Set DCF77 Error flag
	BCF	FLAG2,DCF_OK		; Yes, set DCFF flag
	RETURN				; Exit routine

BIT_0
	BCF	FLAG2,DCF_BIT		; Clear DCF bit

NEXT_SEC
	INCF	BIT_NBR,F		; Increment BIT_NBR
	PAGESEL	(DCF_DECODE)		; Select Program Memory Page of routine
	CALL	DCF_DECODE		; Fill DCF registers with DCF bits and check parity bit
	RETURN				; Exit routine

RA4_1
	BTFSS	FLAG2,RXD		; Skip if RXD=1
	GOTO	RXD_1			; No, set RXD=1 (DCF receive bit) 
	INCF	BIT_DUR,F		; Yes, increment BIT_DUR
	GOTO	TEST_59			; Test if S59 flag is set

RXD_1
	BSF	FLAG2,RXD		; Set DCF receive bit flag
	BSF	FLAG2,DCF_ON		; Set DCF_ON flag (DCF pulse present)
	CLRF	BIT_DUR			; Clear BIT_DUR
	CLRF	SEC_DUR			; Clear SEC_DUR

TEST_59
	BTFSS	FLAG2,S59		; Skip if S59=1
	RETURN				; Exit routine

TEST_DCF_ERR
	BTFSC	FLAG2,DCF_ERR		; Yes, check DCF Error flag, skip if clear
	GOTO	DCF_NOK			; No, DCF Error, exit routine
	CLRF	MSEC			; Clear MSEC counter to sync clock secs with DCF secs
	BSF	FLAG2,DCF_OK		; Set DCF_OK flag (DCF Frame OK)

	; Check for Timezone=CET -> no Timer check + always do CLOCKSYNC
	BTFSS	FLAG2,CET		; Check CET flag, skip if set
	GOTO	NOT_CET			; Timezone is not CET
	GOTO	_CET			; Timezone=CET

_CET
	BCF	FLAG2,S59		; Clear S59
	BSF	FLAG1,S0		; Set Second 0 flag
	PAGESEL	(CET_SYNC)		; Select Program Memory Page of routine
	CALL	CET_SYNC		; CET & DCF frame OK, synchronize Clock & Calendar
	RETURN				; Exit routine

NOT_CET
	BCF	FLAG2,S59		; Clear S59
	BSF	FLAG1,S0		; Set Second 0 flag
	BTFSC	FLAG1,TIMERF		; DCF frame OK, check for TIMERF flag, skip if clear
	GOTO	FREERUN			; No, TIMERF=1 (free-running)

	PAGESEL	(CLOCKSYNC)		; Select Program Memory Page of routine
	CALL	CLOCKSYNC		; Yes, TIMERF=0 & DCF frame OK, synchronize Clock & Calendar
	RETURN				; Exit routine

DCF_NOK
	BCF	FLAG2,DCF_OK		; Clear DCF_OK flag (DCF Frame not OK)
	RETURN				; Exit routine

FREERUN
	BSF	FLAG1,S0		; Set Second 0 flag
	RETURN				; No, TIMERF=1 (free-running), exit routine

;*******************************************************************************
;*               - DCF77 Data Decoding Routine + Parity Check -                *
;*******************************************************************************

DCF_DECODE
	MOVLW	D'20'			; Check for DCF Start bit (bit 20)
	SUBWF	BIT_NBR,W		; BIT_NBR=20?
	BTFSS	STATUS,Z		; Skip if BIT_NBR=20 (Startbit)
	GOTO	MIN_1			; No, DCF Minutes reception

	BTFSS	FLAG2,DCF_BIT		; Yes, skip if DCF_BIT=1
	GOTO	SET_DCF_ERR		; No, set DCF_ERR flag
	BSF	FLAG3,MINF		; Yes, DCF Minutes reception
	BCF	FLAG3,HOURF		; Clear HOURF flag
	BCF	FLAG3,DAYF		; Clear DAYF flag
	BCF	FLAG3,MONTHF		; Clear MONTHF flag
	BCF	FLAG3,YEARF		; Clear YEARF flag
	BCF	FLAG3,PARF		; Clear PARF flag
	CLRF	PARITY			; Clear PARITY
	CLRF	MINDCF			; Clear DCF minutes
	CLRF	HOURDCF			; Clear DCF hours
	CLRF	DAYDCF			; Clear DCF days
	CLRF	MONTHDCF		; Clear DCF months
	CLRF	YEARDCF			; Clear DCF years
	RETURN				; Exit routine

MIN_1		; Start of HH/MM decoding
	BTFSS	FLAG3,MINF		; Check MINF flag, skip if set
	GOTO	HOUR_1			; No, DCF Hours reception
	BTFSS	FLAG2,DCF_BIT		; Yes, skip if DCF_BIT=1
	GOTO	C_01			; No, clear Carry bit (DCF bit=0)
	BSF	STATUS,C		; Yes, set Carry bit (DCF bit=1)
	INCF	PARITY,F		; Increment PARITY
	GOTO 	RRF_MIN			; And fill bit into MINDCF

C_01
	BCF	STATUS,C		; Clear Carry bit

RRF_MIN		; Fill minutes (bit 21 to 27 = 7 bits + P1)
	RRF	MINDCF,F		; Rotate Right through Carry of MINDCF
	MOVLW	D'28'			; 
	SUBWF	BIT_NBR,W		; BIT_NBR=28?
	BTFSS	STATUS,Z		; Skip if BIT_NBR=28 (End of Minutes reception, now P1 parity bit)
	RETURN				; No, exit routine
	RRF	PARITY,F		; Rotate Right through Carry of PARITY
	BTFSC	STATUS,C		; Skip if PARITY is even
	GOTO	SET_DCF_ERR		; Otherwise set DCF_ERR flag
	BCF	FLAG3,MINF		; Clear DCF Minutes receive flag
	BSF	FLAG3,HOURF		; Set DCF Hours receive flag
	CLRF	PARITY			; Clear PARITY
	RETURN				; Exit routine

HOUR_1
	BTFSS	FLAG3,HOURF		; Check HOURF flag, skip if set
	GOTO	DAY_1			; No, DCF Days reception
	BTFSS	FLAG2,DCF_BIT		; Yes, skip if DCF_BIT=1
	GOTO	C_02			; Clear Carry bit
	BSF	STATUS,C		; Set Carry bit
	INCF	PARITY,F		; Increment PARITY
	GOTO 	RRF_HOUR		; And fill bit into HOURDCF

C_02
	BCF	STATUS,C		; Clear Carry bit

RRF_HOUR	; Fill hours (bit 29 to 34 = 6 bits + P2)
	RRF	HOURDCF,F		; Rotate Right through Carry of HOURDCF
	MOVLW	D'35'			; 
	SUBWF	BIT_NBR,W		; BIT_NBR=35?
	BTFSS	STATUS,Z		; Skip if BIT_NBR=35 (End of Fours reception, now P2 parity bit)
	RETURN				; No, exit routine
	RRF	PARITY,F		; Rotate Right through Carry of PARITY
	BTFSC	STATUS,C		; Skip if PARITY is even
	GOTO	SET_DCF_ERR		; Otherwise set DCF_ERR flag
	RRF	HOURDCF,F		; No, Rotate Right through Carry of HOURDCF
					; Position of bits is now correct
	BCF	FLAG3,HOURF		; Clear DCF Hours receive flag
	BSF	FLAG3,DAYF		; Set DCF Days receive flag
	CLRF	PARITY			; Clear Parity
	RETURN				; Exit routine
		; End of HH/MM decoding

DAY_1		; Start of DD/MM/YY decoding
	BTFSS	FLAG3,DAYF		; Check DAYF flag, skip if set
	GOTO	DOW_1			; No, DCF DOW reception
	BTFSS	FLAG2,DCF_BIT		; Yes, skip if DCF_BIT=1
	GOTO	C_03			; No, clear Carry bit
	BSF	STATUS,C		; Yes, set Carry bit
	INCF	PARITY,F		; Increment PARITY
	GOTO	RRF_DAY			; And fill bit into DAYDCF

C_03
	BCF	STATUS,C		; Clear Carry bit

RRF_DAY		; Fill days (bit 36 to 41 = 6 bits)
	RRF	DAYDCF,F		; Rotate Right through Carry of DAYDCF
	MOVLW	D'41'			; 
	SUBWF	BIT_NBR,W		; BIT_NBR=41? (End of Days reception, now start of DOW)
	BTFSS	STATUS,Z		; Skip if BIT_NBR=41
	RETURN				; No, exit routine
	RRF	DAYDCF,F		; Yes, Rotate Right once more
	RRF	DAYDCF,F		; Yes, Rotate Right once more, position of bits is now correct
	BCF	FLAG3,DAYF		; Clear DCF Days receive flag
	BSF	FLAG3,DOWF		; Set DCF DOW receive flag
	RETURN				; Exit routine

DOW_1		; Only used to calculate Parity
	BTFSS	FLAG3,DOWF		; Check DOWF flag, skip if set
	GOTO	MONTH_1			; No, DCF Months reception
	BTFSS	FLAG2,DCF_BIT		; Yes, skip if DCF_BIT=1
	GOTO	C_04			; No, Clear Carry bit
	BSF	STATUS,C		; Yes, set Carry bit
	INCF	PARITY,F		; Increment PARITY
	GOTO	CNT_DOW			; Count DOW bits (bits are only used for parity checking)

C_04
	BCF	STATUS,C		; Clear Carry bit

CNT_DOW		; Count Day of Week bits (bit 42 to 44 = 3 bits)
	MOVLW	D'44'			; 
	SUBWF	BIT_NBR,W		; BIT_NBR=44? (End of Day of Week reception)
	BTFSS	STATUS,Z		; Skip if BIT_NBR=44
	RETURN				; No, exit routine
	BCF	FLAG3,DOWF		; Clear DCF Days of Week receive flag
	BSF	FLAG3,MONTHF		; Set DCF Months receive flag
	RETURN				; Exit routine

MONTH_1
	BTFSS	FLAG3,MONTHF		; Check MONTHF flag, skip if set
	GOTO	YEAR_1			; No, DCF Years reception
	BTFSS	FLAG2,DCF_BIT		; Yes, skip if DCF_BIT=1
	GOTO	C_05			; No, clear Carry bit
	BSF	STATUS,C		; Yes, set Carry bit
	INCF	PARITY,F		; Increment PARITY
	GOTO	RRF_MONTH		; And fill bit into MONTHDCF

C_05
	BCF	STATUS,C		; Clear Carry bit

RRF_MONTH	; Fill Months (bit 45 to 49 = 5 bits)
	RRF	MONTHDCF,F		; Rotate Right through Carry of MONTHDCF
	MOVLW	D'49'			; 
	SUBWF	BIT_NBR,W		; BIT_NBR=49? (End of Months reception, now start of Year)
	BTFSS	STATUS,Z		; Skip if BIT_NBR=49
	RETURN				; No, exit routine
	RRF	MONTHDCF,F		; Rotate Right once more
	RRF	MONTHDCF,F		; Rotate Right once more again
	RRF	MONTHDCF,F		; And again..., position of bits is now correct
	BCF	FLAG3,MONTHF		; Clear DCF Months receive flag
	BSF	FLAG3,YEARF		; Set DCF Years receive flag
	RETURN				; Exit routine

YEAR_1
	BTFSS	FLAG3,YEARF		; Check YEARF flag, skip if set
	GOTO	PAR_1			; No, DCF P3 parity reception
	BTFSS	FLAG2,DCF_BIT		; Yes, skip if DCF_BIT=1
	GOTO	C_06			; No, clear Carry bit
	BSF	STATUS,C		; Yes, set Carry bit
	INCF	PARITY,F		; Increment PARITY
	GOTO	RRF_YEAR		; And fill bit into YEARDCF

C_06
	BCF	STATUS,C		; Clear Carry bit

RRF_YEAR	; Fill Years (bit 50 to 57 = 8 bits)
	RRF	YEARDCF,F		; Rotate Right through Carry of YEARDCF
	MOVLW	D'57'			;
	SUBWF	BIT_NBR,W		; BIT_NBR=57?
	BTFSS	STATUS,Z		; Skip if BIT_NBR=57
	RETURN				; Exit routine
	BCF	FLAG3,YEARF		; Clear DCF Years receive flag
	BSF	FLAG3,PARF		; Set DCF P3 parity bit receive flag
	RETURN				; Exit routine

PAR_1
	BTFSS	FLAG3,PARF		; Check PARF flag, skip if set
	RETURN				; Exit routine
	BTFSS	FLAG2,DCF_BIT		; Yes, skip if DCF_BIT=1
	GOTO	C_07			; No, clear Carry bit
	BSF	STATUS,C		; Yes, set Carry bit
	INCF	PARITY,F		; Increment PARITY
	GOTO	RRF_PAR			; And check parity

C_07
	BCF	STATUS,C		; Clear Carry bit

RRF_PAR
	RRF	PARITY,F		; Rotate Right through Carry of PARITY
	BTFSC	STATUS,C		; Skip if Parity is Even	
	GOTO	SET_DCF_ERR		; No, set DCF_ERR flag
	BCF	FLAG3,PARF		; Yes, clear DCF Years receive flag
	CLRF	PARITY			; Clear Parity
	RETURN				; Exit routine

SET_DCF_ERR
	BSF		FLAG2,DCF_ERR	; Set DCF Error flag
	BCF		FLAG2,DCF_OK	; Clear DCF_OK flag
	RETURN				; Exit routine

;******************************************************************************
;              - Clock & Calendar Synchronization (every 4.0ms) -             *
;******************************************************************************

CLOCKSYNC	; After every good DCF77 frame, every minute at sec=00
	; Check for Timezone=CET-1H -> Substract 1H from DCF Hour
	MOVF	FLAG1,W			; Load FLAG1
	ANDLW	B'11000000'		; Mask 2 MSB's from FLAG1
	SUBLW	D'64'			; 
	BTFSC	STATUS,Z		; Timezone=CET-1H? (TZ0F=1), skip if clear
	GOTO	CET_MINUS1		; Timezone=CET-1

	; Check for Timezone=CET+1H
	MOVF	FLAG1,W			; Load FLAG1
	ANDLW	B'11000000'		; Mask 2 MSB's from FLAG1
	SUBLW	D'128'			; 
	BTFSC	STATUS,Z		; Timezone=CET+1H? (TZ1F=1), skip if clear
	GOTO	CET_PLUS1		; Timezone=CET+1
	GOTO	CET_SYNC		; Else Timezone=CET

CET_MINUS1	; Timezone=CET-1H -> Substract 1H from DCF Hour
	CLRF	SEC			; Clear Seconds
	CLRF	SEC1			; Clear Second 1's
	CLRF	SEC10			; Clear Second 10's

	MOVLW	H'0F'			; (0000.1111)
	ANDWF	HOURDCF,W		; Mask lower nibble
	MOVWF	BCD1			; Store result in BCD1
	MOVLW	H'03'			; (0000.0011)
	SWAPF	HOURDCF,F		; Swap nibbles
	ANDWF	HOURDCF,W		; Mask 2 lower bits
	MOVWF	BCD10			; Store result in BCD10
	PAGESEL	(BCD2BIN)		; Select Program Memory Page of routine
	CALL 	BCD2BIN			; Convert to binary
	MOVWF	HOUR			; Store in Clock Hours
	MOVLW	1
	SUBWF	HOUR,F			; Substract 1 from HOUR
	GOTO	COMMON			; Continue Sync routine

CET_PLUS1	; Timezone=CET+1H -> Add 1H to DCF Hour
	CLRF	SEC			; Clear Seconds
	CLRF	SEC1			; Clear Second 1's
	CLRF	SEC10			; Clear Second 10's

	MOVLW	H'0F'			; (0000.1111)
	ANDWF	HOURDCF,W		; Mask lower nibble
	MOVWF	BCD1			; Store result in BCD1
	MOVLW	H'03'			; (0000.0011)
	SWAPF	HOURDCF,F		; Swap nibbles
	ANDWF	HOURDCF,W		; Mask 2 lower bits
	MOVWF	BCD10			; Store result in BCD10
	PAGESEL	(BCD2BIN)		; Select Program Memory Page of routine
	CALL 	BCD2BIN			; Convert to binary
	ADDLW	1			; Add 1 to HOUR
	MOVWF	HOUR			; Store in Clock Hours
	GOTO	COMMON			; Continue Sync routine

CET_SYNC	; After every good DCF77 frame, every minute at sec=00
	CLRF	SEC			; Clear Seconds
	CLRF	SEC1			; Clear Second 1's
	CLRF	SEC10			; Clear Second 10's

	MOVLW	H'0F'			; (0000.1111)
	ANDWF	HOURDCF,W		; Mask lower nibble
	MOVWF	BCD1			; Store result in BCD1
	MOVLW	H'03'			; (0000.0011)
	SWAPF	HOURDCF,F		; Swap nibbles
	ANDWF	HOURDCF,W		; Mask 2 lower bits
	MOVWF	BCD10			; Store result in BCD10
	PAGESEL	(BCD2BIN)		; Select Program Memory Page of routine
	CALL 	BCD2BIN			; Convert to binary
	MOVWF	HOUR			; Store in Clock Hours

COMMON
	MOVLW	H'0F'			; (0000.1111)
	ANDWF	MINDCF,W		; Mask lower nibble
	MOVWF	BCD1			; Store result in BCD1
	MOVLW	H'07'			; (0000.0111)
	SWAPF	MINDCF,F		; Swap nibbles
	ANDWF	MINDCF,W		; Mask 3 lower bits
	MOVWF	BCD10			; Store result in BCD10
	PAGESEL	(BCD2BIN)		; Select Program Memory Page of routine
	CALL 	BCD2BIN			; Convert to binary
	MOVWF	MIN			; Store in Clock Minutes
	
		; Calendar Sync
	MOVLW	H'0F'			; (0000.1111)
	ANDWF	DAYDCF,W		; Mask lower nibble
	MOVWF	BCD1			; Store result in BCD1
	MOVLW	H'03'			; (0000.0011)
	SWAPF	DAYDCF,F		; Swap nibbles
	ANDWF	DAYDCF,W		; Mask 2 lower bits
	MOVWF	BCD10			; Store result in BCD10
	PAGESEL	(BCD2BIN)		; Select Program Memory Page of routine
	CALL 	BCD2BIN			; Convert to binary
	MOVWF	DAY			; Store in Calendar Days

	MOVLW	H'0F'			; (0000.1111)
	ANDWF	MONTHDCF,W		; Mask lower nibble
	MOVWF	BCD1			; Store result in BCD1
	MOVLW	H'01'			; (0000.0001)
	SWAPF	MONTHDCF,F		; Swap nibbles
	ANDWF	MONTHDCF,W		; Mask lower bit
	MOVWF	BCD10			; Store result in BCD10
	PAGESEL	(BCD2BIN)		; Select Program Memory Page of routine
	CALL 	BCD2BIN			; Convert to binary
	MOVWF	MONTH			; Store in Calendar Months
	
	MOVLW	H'0F'			; (0000.1111)
	ANDWF	YEARDCF,W		; Mask lower nibble
	MOVWF	BCD1			; Store result in BCD1
	MOVLW	H'0F'			; (0000.1111)
	SWAPF	YEARDCF,F		; Swap nibbles
	ANDWF	YEARDCF,W		; Mask 4 lower bits
	MOVWF	BCD10			; Store result in BCD10
	PAGESEL	(BCD2BIN)		; Select Program Memory Page of routine
	CALL 	BCD2BIN			; Convert to binary
	MOVWF	YEAR			; Store in Calendar Years
	RETURN				; Exit routine

;*******************************************************************************
;*         - Standalone 24H Clock and Calendar Routine (every 4msec) -         *
;*******************************************************************************

CLOCK	; 4msec Routine (when msec=250=>1sec)
	INCF	MSEC,F			; MSEC++;
	MOVF	MSEC,W			; ACCU = MSEC;
	SUBLW	XD			; if ((ACCU-XD) != 0)
	BTFSS	STATUS,Z		; return;
	RETURN				; else {
	CLRF	MSEC			; MSEC = 0;

	; Clock Routine
	BSF	FLAG1,BEATF		; Set BEATF flag to indicate start of new second
	INCF	SEC,F			; SEC++;
	MOVF	SEC,W			; ACCU = SEC;
	SUBLW	0x3C			; if ((ACCU-60) != 0)
	BTFSS	STATUS,Z		; return;
	RETURN				; else {
	CLRF	SEC			; SEC = 0
	BSF	FLAG1,M0		; Set M0 flag to indicate start of new min for Timer check
	INCF	MIN,F			; MIN++;
	MOVF	MIN,W			; ACCU = MIN;
	SUBLW	0x3C			; if ((ACCU-60) != 0)
	BTFSS	STATUS,Z		; return;
	RETURN				; else {
	CLRF	MIN			; MIN = 0;
	INCF	HOUR,F			; HOUR++;

	BSF	FLAG4,GONGF		; Set Hourly Gong flag
	CLRF	CNT4			; Clear CNT4 counter

	MOVF	HOUR,W			; ACCU = HOUR;
	SUBLW	0x18			; if ((ACCU-24) != 0)
	BTFSS	STATUS,Z		; return;
	RETURN				; else {
	CLRF	HOUR			; HOUR = 0;

CALENDAR
	; Check for leap year
	MOVF	YEAR,W			; ACCU = year;
	ANDLW	0x03			; if ((ACCU & 00000011) == 0)
	BTFSC	STATUS,Z		; {
	BSF	FLAG3,LYF		; lyf = 1;}

	; Calendar Routine
	INCF	DAY,F			; day++;
	MOVF	MONTH,W			; ACCU = month;
	SUBLW	0x02			; 
	BTFSS	STATUS,Z		; if ((ACCU - 2) == 0)
	GOTO	_NOLEAP			; {
	BTFSS	FLAG3,LYF		; if (leap_year)
	GOTO	_NOLEAP			; }else
	ANDLW	0x00			; ACCU = 0;
	GOTO	_CONTINUE		; Continue

_NOLEAP
	MOVF	MONTH,W			; ACCU = month;

_CONTINUE
	PAGESEL	(TABLE1)		; Select Program Memory Page of routine
	CALL	TABLE1			; ACCU = days[ACCU];
	SUBWF	DAY,W			; if ((day-ACCU) != 0)
	BTFSS	STATUS,Z		; return;
	RETURN				; else {
	MOVLW	0x01			; ACCU = 1;
	MOVWF	DAY			; day = ACCU;
	INCF	MONTH,F			; month++;
	MOVF	MONTH,W			; ACCU = month;
	SUBLW	0x0D			; if ((ACCU-13) != 0)
	BTFSS	STATUS,Z		; return;
	RETURN				; else {
	MOVLW	0x01			; ACCU = 1;
	MOVWF	MONTH			; month = ACCU;
	INCF	YEAR,F			; year++;
	BCF	FLAG3,LYF		; lyf = 0;
	MOVF	YEAR,W			; ACCU = year;
	ANDLW	0x03			; if ((ACCU & 00000011) == 0)
	BTFSC	STATUS,Z		; {
	BSF	FLAG3,LYF		; lyf = 1;}
	RETURN				; else {

;******************************************************************************
;         - Copy Clock registers to Display registers (every 2.0ms) -         *
;******************************************************************************

TIME
	MOVF	SEC,W			; Load SEC into W
	MOVWF	BIN			; Store in BIN
	PAGESEL	(BIN2BCD)		; Select Program Memory Page of routine
	CALL	BIN2BCD			; Convert to BCD
	MOVF	BIN1,W			; Load BIN1 into W
	MOVWF	SEC1			; Store into SEC1
	MOVF	BIN10,W			; Load BIN10 into W
	MOVWF	SEC10			; Store into SEC10

	MOVF	MIN,W			; Load MIN into W
	MOVWF	BIN			; Store in BIN
	PAGESEL	(BIN2BCD)		; Select Program Memory Page of routine
	CALL	BIN2BCD			; Convert to BCD
	MOVF	BIN1,W			; Load BIN1 into W
	MOVWF	MIN1			; Store into MIN1
	MOVF	BIN10,W			; Load BIN10 into W
	MOVWF	MIN10			; Store into MIN10

	MOVF	HOUR,W			; Load HOUR into W
	MOVWF	BIN			; Store in BIN
	PAGESEL	(BIN2BCD)		; Select Program Memory Page of routine
	CALL	BIN2BCD			; Convert to BCD
	MOVF	BIN1,W			; Load BIN1 into W
	MOVWF	HOUR1			; Store into HOUR1
	MOVF	BIN10,W			; Load BIN10 into W
	MOVWF	HOUR10			; Store into HOUR10

	BTFSC	FLAG4,SCROLLF		; Check SCROLLF flag, skip if clear
	RETURN				; Exit routine

	MOVF	SEC1,W			; Load SEC1 into W
	MOVWF	D1			; And copy to D1
	MOVF	SEC10,W			; Load SEC10 into W
	MOVWF	D2			; And copy to D2

	MOVF	MIN1,W			; Load MIN1 into W
	MOVWF	D3			; And copy to D3
	MOVF	MIN10,W			; Load MIN10 into W
	MOVWF	D4			; And copy to D4

	MOVF	HOUR1,W			; Load HOUR1 into W
	MOVWF	D5			; And copy to D5
	MOVF	HOUR10,W		; Load HOUR10 into W
	MOVWF	D6			; And copy to D6
	RETURN				; Exit routine

;******************************************************************************
;        - Convert Calendar registers to BCD registers (every 2.0ms) -        *
;******************************************************************************

DATE
	MOVF	YEAR,W			; Load YEAR into W
	MOVWF	BIN			; Store in BIN
	PAGESEL	(BIN2BCD)		; Select Program Memory Page of routine
	CALL	BIN2BCD			; Convert to BCD
	MOVF	BIN1,W			; Load BIN1 into W
	MOVWF	YEAR1			; Store into YEAR1
	MOVF	BIN10,W			; Load BIN10 into W
	MOVWF	YEAR10			; Store into YEAR10

	MOVF	MONTH,W			; Load MONTH into W
	MOVWF	BIN			; Store in BIN
	PAGESEL	(BIN2BCD)		; Select Program Memory Page of routine
	CALL	BIN2BCD			; Convert to BCD
	MOVF	BIN1,W			; Load BIN1 into W
	MOVWF	MONTH1			; Store into MONTH1
	MOVF	BIN10,W			; Load BIN10 into W
	MOVWF	MONTH10			; Store into MONTH10

	MOVF	DAY,W			; Load DAY into W
	MOVWF	BIN			; Store in BIN
	PAGESEL	(BIN2BCD)		; Select Program Memory Page of routine
	CALL	BIN2BCD			; Convert to BCD
	MOVF	BIN1,W			; Load BIN1 into W
	MOVWF	DAY1			; Store into DAY1
	MOVF	BIN10,W			; Load BIN10 into W
	MOVWF	DAY10			; Store into DAY10

	BTFSC	FLAG4,SCROLLF		; Check SCROLLF flag, skip if clear
	RETURN				; Exit routine

	MOVF	YEAR1,W			; Load YEAR1 into W
	MOVWF	D1			; And copy to D1
	MOVF	YEAR10,W		; Load YEAR10 into W
	MOVWF	D2			; And copy to D2

	MOVF	MONTH1,W		; Load MONTH1 into W
	MOVWF	D3			; And copy to D3
	MOVF	MONTH10,W		; Load MONTH10 into W
	MOVWF	D4			; And copy to D4

	MOVF	DAY1,W			; Load DAY1 into W
	MOVWF	D5			; And copy to D5
	MOVF	DAY10,W			; Load DAY10 into W
	MOVWF	D6			; And copy to D6
	RETURN				; Exit routine

;******************************************************************************
;       - Display BCD values of D1 to D6 to LED Display and check LEDs -      *
;******************************************************************************

DISPLAY		; every 2.0ms = 500Hz/digit, = 83.2Hz refresh rate for the 6 digits
	BTFSS	FLAG1,TDF		; Check TDF=1, skip if set
	GOTO	DATE_OFF		; No, display Time
	
	PAGESEL	(DATE)			; Select Program Memory Page of routine
	CALL	DATE			; Copy Date to display registers
	PAGESEL	(MPX)			; Select Program Memory Page of routine
	CALL	MPX			; Display Date
	RETURN				; Exit routine

DATE_OFF
	PAGESEL	(TIME)			; Select Program Memory Page of routine
	CALL	TIME			; Copy Time to display registers
	PAGESEL	(MPX)			; Select Program Memory Page of routine
	CALL	MPX			; Display Time
	RETURN				; Exit routine

MPX
	MOVLW	D'6'			; 
	SUBWF	DIGIT,W			; DIGIT=6?
	BTFSC	STATUS,Z		; Skip if DIGIT is different from 6
	CLRF	DIGIT			; Yes, DIGIT=6, clear digit number

	; Disable latch to 7-segment display just before sending new value to 4543
	BCF	PORTB,LD		; Disable 4543 Latch

	; Output digit value to 4543
	MOVF	PORTB,W			; Read PORTB bits
	ANDLW	B'11110000'		; Mask bit 4 to 7
	MOVWF	TEMP			; Save it for later
	MOVF	DIGIT,W			; Recall digit number (0 to 5)
	ADDLW	H'50'			; Point at register to display (D1 to D6)
	MOVWF	FSR			; Use as pointer
	MOVF	INDF,W			; Get value of register pointed to into W
	XORWF	TEMP,W			; Exclusive OR with TEMP
	MOVWF	PORTB			; BCD output to 4543

	INCF	DIGIT,F			; Next digit number (0 to 5 for 4543, 1 to 6 for 74HCT238)

	; Select digit to lit
	MOVF	PORTC,W			; Read PORTC bits
	ANDLW	B'10001111'		; Mask bit 0 to 4, bit7
	MOVWF	TEMP			; Save it for later
	SWAPF	DIGIT,W			; Swap lower and higher nibble of DIGIT into W
	XORWF	TEMP,W          	; Exclusive OR with TEMP
	MOVWF	PORTC			; BCD output to 74HCT238

	; Latch digit to 7-segment display
	BSF	PORTB,LD		; Enable 4543 Latch

	; TDF flag check to lit Colons continuously when date display
	BTFSS	FLAG1,TDF		; Check TDF, skip if set
	GOTO	NEW_SEC			; No, TDF=0, Time Display
	BTFSS	PORTB,BEAT_LED		; Yes, skip if BEAT_LED=1
	BSF	PORTB,BEAT_LED		; TDF=1, Date Display, lit Colons
	GOTO	TEST_RXD		; Skip Colon Beat	

NEW_SEC		; Check Start of new second
	BTFSS	FLAG1,BEATF		; Skip if BEATF=1
	GOTO	N_SEC			; No, BEATF=0
	BTFSS	PORTB,BEAT_LED		; Yes, skip if BEAT_LED=1
	BSF	PORTB,BEAT_LED		; No, BEAT_LED=0	

BEAT		; Colons 0.5sec beat routine
	INCF	CNT1,F			; Increment CNT1 (0 to 150 for 4msec ISR)
	MOVLW	D'150'			; 
	SUBWF	CNT1,W			; CNT1=125?
	BTFSS	STATUS,Z		; Skip if CNT1=125
	GOTO	TEST_RXD		; No, exit BEAT routine
	BCF	FLAG1,BEATF		; Clear BEATF flag
	BTFSC	PORTB,BEAT_LED		; Yes, skip if BEAT_LED=0
	BCF	PORTB,BEAT_LED		; Clear BEAT_LED

N_SEC
	CLRF	CNT1			; Clear CNT1

TEST_RXD	; Lit DCF Beat LED every 0.1 or 0.2msec DCF pulse
	BTFSS	FLAG2,RXD		; Skip if RXD=1
	GOTO	B1_0			; No, no DCF reception, check DCF Beat LED
	BTFSS	PORTB,DCF_LED		; Yes, skip if DCF_LED=1
	GOTO	B1_1			; No, set DCF_BEAT=1	
	GOTO	TIMER			; Yes, DCF_LED is already 0

B1_0
	BTFSC	PORTB,DCF_LED		; Skip if DCF_LED=0
	BCF	PORTB,DCF_LED		; No, DCF_LED=1
	GOTO	TIMER			; Yes, DCF_LED is already 0

B1_1
	BTFSS	PORTB,DCF_LED		; Skip if DCF_LED=1
	BSF	PORTB,DCF_LED		; No, lit DCF Beat LED

TIMER		; Check start of new minute and check Timer
	BTFSS	FLAG1,M0		; Skip if M0=1 (every minute)
	GOTO	DCF_ONLINE		; No, M0=0

	BTFSC	FLAG2,CET		; Check CET flag, skip if clear
	GOTO	DCF_ONLINE		; No, timezone=CET, no free-running Clock needed

	PAGESEL	(CHECK_TIME)
	CALL	CHECK_TIME		; Yes, check for Time=22:59 or 03:01

DCF_OFFLINE
	BTFSS	FLAG1,TIMERF		; Check for TIMERF flag, skip if set
	GOTO	DCF_ONLINE		; No, continue
	BTFSS	PORTB,SA_LED		; Skip if SA_LED=1
	BSF	PORTB,SA_LED		; Yes, lit SA_LED
	GOTO	TEST_DCF		; Continue

DCF_ONLINE
	BTFSC	FLAG1,TIMERF		; Check for TIMERF flag, skip if clear
	GOTO	TEST_DCF		; No, do not clear SA_LED

	BTFSS	FLAG2,DCF_ON		; Check for DCF pulses present
	GOTO	DCF_OFF			; No, continue
	BTFSC	PORTB,SA_LED		; Skip if SA_LED=0
	BCF	PORTB,SA_LED		; Yes, clear SA_LED
	GOTO	TEST_DCF		; Continue

DCF_OFF
	BTFSS	PORTB,SA_LED		; Skip if SA_LED=1
	BSF	PORTB,SA_LED		; No, SA_LED=0

TEST_DCF
	BTFSS	FLAG2,DCF_ERR		; Skip if DCF error
	GOTO	DCFOK			; No, clear DCF Error LED
	BTFSS	PORTC,ERR_LED		; Yes, skip if ERR_LED=1
	BSF	PORTC,ERR_LED		; Yes, lit DCF Error LED (ERR_LED=1 if DCF_ERR=1)
	BCF	PORTC,OK_LED		; Clear OK_LED
	RETURN				; Exit routine

DCFOK
	BTFSC	PORTC,ERR_LED		; Check ERR_LED, skip if 0
	BCF	PORTC,ERR_LED		; ERR_LED=0 if DCF_ERR=0

	BTFSS	FLAG2,DCF_OK		; Check DCF_OK flag, skip if set
	GOTO	NOT_OK			; No, clear OK_LED

	BTFSC	FLAG2,S59		; Check for 59th sec flag, skip if clear
	GOTO	NOT_OK			; Yes, clear OK_LED

	BCF	FLAG1,S0		; 
	BTFSS	PORTC,OK_LED		; Check OK_LED, skip if already 1
	BSF	PORTC,OK_LED		; Lit OK_LED
	RETURN				; Exit routine

NOT_OK
	BTFSC	PORTC,OK_LED		; Check OK_LED, skip if already 0
	BCF	PORTC,OK_LED		; Clear OK_LED
	RETURN				; Exit routine

;******************************************************************************
;                         - Check Time Subroutine -                           *
;******************************************************************************

CHECK_TIME	; If CLOCK_HHMM < TIME_OFF_HHMM Then BSF AUX1,1
	BCF	FLAG1,M0	; Clear M0 flag

	MOVF	TIME_OFF_M,W
	SUBWF	MIN,W
	MOVF	TIME_OFF_H,W
	BTFSS	STATUS,C
	INCFSZ	TIME_OFF_H,W
	SUBWF	HOUR,W
	BTFSC	STATUS,C
	GOTO	ENDIF1

THEN1
	BSF	AUX1,0
	GOTO	CHK2

ENDIF1
	BCF	AUX1,0

CHK2		; If TIME_ON_HHMM <= CLOCK_HHMM Then BSF AUX2,1
	MOVF	TIME_ON_M,W
	SUBWF	MIN,W
	MOVF	HOUR,W
	BTFSS	STATUS,C
	INCFSZ	TIME_ON_H,W
	SUBWF	HOUR,W
	BTFSC	STATUS,C
	GOTO	THEN2
	GOTO	ENDIF2

THEN2
	BSF	AUX2,0
	GOTO	END_CHK

ENDIF2
	BCF	AUX2,0

END_CHK		; If AUX1 AND AUX2 = 1 Then BSF FLAG1,TIMERFElse BCF FLAG1,TIMERF
	MOVF	AUX2,W
	ANDWF	AUX1,W
	BTFSC	STATUS,Z
	GOTO	TIMER_OFF

TIMER_ON
	BCF	FLAG1,TIMERF
	RETURN

TIMER_OFF
	BSF	FLAG1,TIMERF
	RETURN				; Exit routine

;******************************************************************************
;               - Scroll Clock registers to Display registers -               *
;******************************************************************************

SCROLL_TIME
	MOVLW	0x0F			; Write 0x0F to all digits
	MOVWF	D1
	MOVWF	D2
	MOVWF	D3
	MOVWF	D4
	MOVWF	D5	
	MOVWF	D6			; 4543 blanks all digits if input > 0x09		

	PAGESEL (DELAY2)		; Select Program Memory Page of routine
	CALL	DELAY2
	PAGESEL (DELAY2)		; Select Program Memory Page of routine
	CALL	DELAY2

	MOVF	HOUR10,W
	MOVWF	D1
	PAGESEL (DELAY2)		; Select Program Memory Page of routine
	CALL	DELAY2

	MOVF	HOUR10,W
	MOVWF	D2	
	MOVF	HOUR1,W
	MOVWF	D1
	PAGESEL (DELAY2)		; Select Program Memory Page of routine
	CALL	DELAY2

	MOVF	HOUR10,W
	MOVWF	D3
	MOVF	HOUR1,W
	MOVWF	D2
	MOVF	MIN10,W
	MOVWF	D1
	PAGESEL (DELAY2)		; Select Program Memory Page of routine
	CALL	DELAY2

	MOVF	HOUR10,W
	MOVWF	D4
	MOVF	HOUR1,W
	MOVWF	D3
	MOVF	MIN10,W
	MOVWF	D2
	MOVF	MIN1,W
	MOVWF	D1
	PAGESEL (DELAY2)		; Select Program Memory Page of routine
	CALL	DELAY2

	MOVF	HOUR10,W
	MOVWF	D5
	MOVF	HOUR1,W
	MOVWF	D4
	MOVF	MIN10,W
	MOVWF	D3
	MOVF	MIN1,W
	MOVWF	D2
	MOVF	SEC10,W
	MOVWF	D1
	PAGESEL (DELAY2)		; Select Program Memory Page of routine
	CALL	DELAY2

	MOVF	HOUR10,W
	MOVWF	D6
	MOVF	HOUR1,W
	MOVWF	D5
	MOVF	MIN10,W
	MOVWF	D4
	MOVF	MIN1,W
	MOVWF	D3
	MOVF	SEC10,W
	MOVWF	D2
	MOVF	SEC1,W
	MOVWF	D1
	RETURN

;******************************************************************************
;              - Scroll Calendar registers to Display registers -             *
;******************************************************************************

SCROLL_DATE
	MOVLW	0x0F			; Write 0x0F to all digits
	MOVWF	D1
	MOVWF	D2
	MOVWF	D3
	MOVWF	D4
	MOVWF	D5	
	MOVWF	D6			; 4543 blanks all digits if input>0x09		

	PAGESEL (DELAY2)		; Select Program Memory Page of routine
	CALL	DELAY2
	PAGESEL (DELAY2)		; Select Program Memory Page of routine
	CALL	DELAY2

	MOVF	DAY10,W
	MOVWF	D1
	PAGESEL (DELAY2)		; Select Program Memory Page of routine
	CALL	DELAY2

	MOVF	DAY10,W
	MOVWF	D2
	MOVF	DAY1,W
	MOVWF	D1
	PAGESEL (DELAY2)		; Select Program Memory Page of routine
	CALL	DELAY2

	MOVF	DAY10,W
	MOVWF	D3
	MOVF	DAY1,W
	MOVWF	D2
	MOVF	MONTH10,W
	MOVWF	D1
	PAGESEL (DELAY2)		; Select Program Memory Page of routine
	CALL	DELAY2

	MOVF	DAY10,W
	MOVWF	D4
	MOVF	DAY1,W
	MOVWF	D3
	MOVF	MONTH10,W
	MOVWF	D2
	MOVF	MONTH1,W
	MOVWF	D1
	PAGESEL (DELAY2)		; Select Program Memory Page of routine
	CALL	DELAY2

	MOVF	DAY10,W
	MOVWF	D5
	MOVF	DAY1,W
	MOVWF	D4
	MOVF	MONTH10,W
	MOVWF	D3
	MOVF	MONTH1,W
	MOVWF	D2
	MOVF	YEAR10,W
	MOVWF	D1
	PAGESEL (DELAY2)		; Select Program Memory Page of routine
	CALL	DELAY2

	MOVF	DAY10,W
	MOVWF	D6
	MOVF	DAY1,W
	MOVWF	D5
	MOVF	MONTH10,W
	MOVWF	D4
	MOVF	MONTH1,W
	MOVWF	D3
	MOVF	YEAR10,W
	MOVWF	D2
	MOVF	YEAR1,W
	MOVWF	D1
	RETURN

;******************************************************************************
;                   - Delay Routine for Scrolling Routines -                  *
;******************************************************************************

DELAY2	; Delay for Scrolling Routines
	CLRF	CNT2			; Clear CNT2

LOOP
	MOVLW	D'35'			; Creates delay of 70msec
	SUBWF	CNT2,W			; CNT1=35?
	BTFSS	STATUS,Z		; Skip if CNT1=35
	GOTO	LOOP			; Loop
	RETURN				; Exit routine

;*******************************************************************************
;*           - Binary Coded Decimal to Binary Conversion Routine -             *
;*              Values must be in BCD1 and BCD10, result is in W               *
;*******************************************************************************

BCD2BIN
	MOVF    BCD1,W			; 
	BTFSC   BCD10,0			; 
	ADDLW   D'10'			; 
	BTFSC   BCD10,1			; 
	ADDLW   D'20'			; 
	BTFSC   BCD10,2			; 
	ADDLW   D'40'			; 
	BTFSC   BCD10,3			; 
	ADDLW   D'80'			; 
	RETURN				; Exit routine

;******************************************************************************
;*                - Binary to BCD ASCII Conversion Routine -                  *
;*         Value to convert must be in BIN, results in BIN1 and BIN10         *
;******************************************************************************

BIN2BCD
	MOVLW	LOW TABLE2		; Get lower 8bits of table address
	ADDLW	1			; Add 1 to offset (offset=PCL+1)
	ADDWF	BIN,W			; Add offset
	MOVWF	OFFSET			; Store in OFFSET
	MOVLW	HIGH TABLE2		; Get higher 5 bits of table address
	BTFSC	STATUS,C		; Page crossed?
	ADDLW	1			; Yes, then increment high address
	MOVWF	PCLATH			; Store high address in PCLATH
	MOVF	OFFSET,W		; Load computed offset 
	CALL	TABLE2			; Convert to BCD
	MOVWF	BCD			; Store in BCD register
	SWAPF	BCD,W			; Swap nibbles
	ANDLW	0x0F			; Mask lower nibble (1's)
	MOVWF	BIN10			; Store in BIN10 register
	MOVF	BCD,W			; Load BCD in W
	ANDLW	0x0F			; mask higher nibble (10's)
	MOVWF	BIN1			; Store in BIN1 register
	RETURN				; Exit routine

	END
