#include <msp430.h>
#include "i2c.h"

volatile signed char byteCtr;
volatile unsigned char stop_condition;       // set to UCTXSTT for RESTART condition
unsigned char *TI_receive_field;
unsigned char *TI_transmit_field;

//------------------------------------------------------------------------------
// void TI_USCI_I2C_init(unsigned char slave_address, 
//                       unsigned char prescale)
//
// This function initializes the USCI module for master-receive operation. 
//
// IN:   unsigned char slave_address   =>  Slave Address
//       unsigned char prescale        =>  SCL clock adjustment 
//-----------------------------------------------------------------------------
void TI_USCI_I2C_init(unsigned char slave_address, 
                      unsigned char prescale) {
  P1SEL2 |= BIT7 + BIT6;                 // Assign I2C pins to USCI
  P1SEL |= BIT7 + BIT6;                 // Assign I2C pins to USCI
  UCB0CTL1 = UCSWRST;                         // Enable SW reset
  UCB0CTL0 = UCMST + UCMODE_3 + UCSYNC;       // I2C Master, synchronous mode
  UCB0CTL1 = UCSSEL_2 + UCSWRST;              // Use SMCLK, keep SW reset
  UCB0BR0 = prescale;                         // set prescaler
  UCB0BR1 = 0;
  UCB0I2CSA = slave_address;                  // set slave address
  UCB0CTL1 &= ~UCSWRST;                       // Clear SW reset, resume operation
  UCB0I2CIE = UCNACKIE;
  IE2 |= UCB0RXIE | UCB0TXIE;                 // Enable interrupts
  stop_condition=UCTXSTP;
}

void TI_USCI_I2C_off() {
 // P1SEL2 &= ~(BIT7 + BIT6);              // Unassign I2C pins to USCI
  UCB0I2CIE = 0;
  IE2 &= ~(UCB0RXIE | UCB0TXIE);              // Disable interrupts
}

//------------------------------------------------------------------------------
// void TI_USCI_I2C_receive(unsigned char byteCount, unsigned char *field)
//
// This function is used to start an I2C commuincation in master-receiver mode. 
//
// IN:   unsigned char byteCount  =>  number of bytes that should be read
//       unsigned char *field     =>  array variable used to store received data
//       void *recieve_done()     =>  callback when recieve completes
//------------------------------------------------------------------------------
void TI_USCI_I2C_receive(unsigned char byteCount,
                         unsigned char *field) {
                         
/* I2C Master Receiver Mode

After initialization, master receiver mode is initiated by writing the desired
slave address to the UCBxI2CSA register, selecting the size of the slave address
with the UCSLA10 bit, clearing UCTR for receiver mode, and setting UCTXSTT
to generate a START condition. The USCI module checks if the bus is available,
generates the START condition, and transmits the slave address. As soon as the
slave acknowledges the address the UCTXSTT bit is cleared.

After the acknowledge of the address from the slave the first data byte from
the slave is received and acknowledged and the UCBxRXIFG flag is set. Data is
received from the slave ss long as UCTXSTP or UCTXSTT is not set. If UCBxRXBUF
is not read the master holds the bus during reception of the last data bit and
until the UCBxRXBUF is read. If the slave does not acknowledge the transmitted
address the not-acknowledge interrupt flag UCNACKIFG is set. The master must
react with either a STOP condition or a repeated START condition.

Setting the UCTXSTP bit will generate a STOP condition. After setting UCTXSTP,
a NACK followed by a STOPcondition is generated after reception of the data from
the slave, or immediately if the USCI module is currently waiting for UCBxRXBUF
to be read.

If a master wants to receive a single byte only, the UCTXSTP bit must be set
while the byte is being received. For this case, the UCTXSTT may be polled
to determine when it is cleared:

BIS.B #UCTXSTT,&UCBOCTL1 ;Transmit START cond.
POLL_STT BIT.B #UCTXSTT,&UCBOCTL1 ;Poll UCTXSTT bit
JC POLL_STT ;When cleared,
BIS.B #UCTXSTP,&UCB0CTL1 ;transmit STOP cond.

Setting UCTXSTT will generate a repeated START condition. In this case, UCTR may be set or cleared to configure transmitter or receiver, and a different slave address may be written into UCBxI2CSA if desired.

*/
  TI_receive_field = field;
  if ( byteCount == 1 ){
    byteCtr = 0 ;
    __disable_interrupt();
    UCB0CTL1 |= UCTXSTT;                      // I2C start condition
    while (UCB0CTL1 & UCTXSTT);               // Start condition sent?
    UCB0CTL1 |= stop_condition;               // I2C stop condition
    stop_condition = UCTXSTP;                 // set stop on next transfer
    __enable_interrupt();
  } else if ( byteCount > 1 ) {
    byteCtr = byteCount - 2 ;
    UCB0CTL1 |= UCTXSTT;                      // I2C start condition
  } else
    while(1);                                  // illegal parameter
}

//------------------------------------------------------------------------------
// void TI_USCI_I2C_transmit(unsigned char byteCount, unsigned char *field)
//
// This function is used to start an I2C commuincation in master-transmit mode. 
//
// IN:   unsigned char byteCount  =>  number of bytes that should be transmitted
//       unsigned char *field     =>  array variable. Its content will be sent.
//------------------------------------------------------------------------------
void TI_USCI_I2C_transmit(unsigned char byteCount, unsigned char *field){
  TI_transmit_field = field;
  byteCtr = byteCount;
  UCB0CTL1 |= UCTR + UCTXSTT;                 // I2C TX, start condition
}



void TI_USCI_I2C_restart(void) { /* set restart condition on the next transfer*/
  stop_condition = UCTXSTT;
}



//------------------------------------------------------------------------------
// unsigned char TI_USCI_I2C_slave_present(unsigned char slave_address)
//
// This function is used to look for a slave address on the I2C bus.  
//
// IN:   unsigned char slave_address  =>  Slave Address
// OUT:  unsigned char                =>  0: address was not found, 
//                                        1: address found
//------------------------------------------------------------------------------
unsigned char TI_USCI_I2C_slave_present(unsigned char slave_address){
  unsigned char ie2_bak, slaveadr_bak, ucb0i2cie, returnValue;
  ucb0i2cie = UCB0I2CIE;                      // restore old UCB0I2CIE
  ie2_bak = IE2;                              // store IE2 register
  slaveadr_bak = UCB0I2CSA;                   // store old slave address
  UCB0I2CIE &= ~ UCNACKIE;                    // no NACK interrupt
  UCB0I2CSA = slave_address;                  // set slave address
  IE2 &= ~(UCB0TXIE + UCB0RXIE);              // no RX or TX interrupts
  __disable_interrupt();
  UCB0CTL1 |= UCTR + UCTXSTT + UCTXSTP;       // I2C TX, start condition
  while (UCB0CTL1 & UCTXSTP);                 // wait for STOP condition
  
  returnValue = !(UCB0STAT & UCNACKIFG);
  __enable_interrupt();
  IE2 = ie2_bak;                              // restore IE2
  UCB0I2CSA = slaveadr_bak;                   // restore old slave address
  UCB0I2CIE = ucb0i2cie;                      // restore old UCB0CTL1
  return returnValue;                         // return whether or not 
                                              // a NACK occured
}


void TI_USCI_I2C_read(unsigned char *f, unsigned char byteCount, unsigned char *field)
{
	//__disable_interrupt();
	__enable_interrupt();
	stop_condition = UCTXSTT;
	TI_transmit_field = f;
  byteCtr = 1;
  UCB0CTL1 |= UCTR + UCTXSTT;                 // I2C TX, start condition
  //	while(UCB0CTL1 & UCTXSTT); // poll the start bit
  	
  	while(stop_condition == UCTXSTT);
  //	while(!(IFG2 & UCB0TXIFG));
  	//UCB0TXBUF = 0xAA;
  //	IFG2 &= ~UCB0TXIFG;
  	//while(!(IFG2 & UCB0TXIFG));
  	
 // 	UCB0CTL1 |= UCTXSTT; // repeat start
  //	while(UCB0CTL1 & UCTXSTT);
	
	
	// __enable_interrupt();
	TI_receive_field = field;
	if ( byteCount == 1 ){
    byteCtr = 0 ;
    __disable_interrupt();
    //UCB0CTL1 |= UCTXSTT;                      // I2C start condition
    while (UCB0CTL1 & UCTXSTT);               // Start condition sent?
    UCB0CTL1 |= stop_condition;               // I2C stop condition
    stop_condition = UCTXSTP;                 // set stop on next transfer
    while (UCB0CTL1 & UCTXSTP);
    // clear int
    IFG2 &= ~UCB0RXIFG;
    __enable_interrupt();
  } else if ( byteCount > 1 ) {
    byteCtr = byteCount - 2 ;
   // UCB0CTL1 |= UCTXSTT;                      // I2C start condition
  } else
    while(1);                                  // illegal parameter
}

//------------------------------------------------------------------------------
// unsigned char TI_USCI_I2C_notready()
//
// This function is used to check if there is commuincation in progress. 
//
// OUT:  unsigned char  =>  0: I2C bus is idle, 
//                          1: communication is in progress
//------------------------------------------------------------------------------
unsigned char TI_USCI_I2C_notready(){
  return (UCB0STAT & UCBBUSY);
}



/* A0/B0 I2C status change shared vector: UCALIFG, UCNACKIFG, ICSTTIFG, UCSTPIFG flags */
__attribute__((interrupt((USCIAB0RX_VECTOR)))) 
void TI_USCI_I2C_status_change_isr(void) {
  if (UCB0STAT & UCNACKIFG) {            // send STOP if slave sends NACK
    UCB0CTL1 |= UCTXSTP;
    UCB0STAT &= ~UCNACKIFG;
  }

}

/* To A0/B0 I2C transmit/recieve shared vector: UCB0RXIFG, UCB0TXIFG flags*/
__attribute__((interrupt((USCIAB0TX_VECTOR)))) 
void TI_USCI_I2C_rx_tx_isr(void) {
  if (IFG2 & UCB0RXIFG) {
    if (byteCtr == 0) {
      UCB0CTL1 |= stop_condition;            // I2C stop condition
      stop_condition = UCTXSTP;
    }  
    *TI_receive_field = UCB0RXBUF;
    TI_receive_field++;
    byteCtr--;
  }
  else
  if (IFG2 & UCB0TXIFG) {
    if (byteCtr == 0){
      UCB0CTL1 &= ~UCTR;
      UCB0CTL1 |= stop_condition;             // I2C stop condition
      stop_condition = UCTXSTP;
      
      IFG2 &= ~UCB0TXIFG;                     // Clear USCI_B0 TX int flag
      //rdy = 1;
    }
    else {
      UCB0TXBUF = *TI_transmit_field;
      TI_transmit_field++;
    }
    byteCtr--;
  }
}





