// Dutchtronix AVR Oscilloscope Clock
//
//  Copyright  2010 Johannes P.M. de Rie
//
//  All Rights Reserved
//
//  This file is part of the Dutchtronix Oscilloscope Clock Distribution.
//  Use, modification, and re-distribution is permitted subject to the
//  terms in the file named "LICENSE.TXT", which contains the full text
//  of the legal notices and should always accompany this Distribution.
//
//  This software is provided "AS IS" with NO WARRANTY OF ANY KIND.
//
//  This notice (including the copyright and warranty disclaimer)
//  must be included in all copies or derivations of this software.
//
// Firmware for Dutchtronix AVR Oscilloscope Clock
//
#include <avr/io.h>
#include <stdlib.h>
#include <string.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <util/delay.h>

#include "./ClkConfig.h"
#include "./ClkData.h"
#include "./ClkISR.h"

//
// Output frequency (using 24 bit accumulator) :
//
//	f = deltaPhase * fClock/2^24

#if SRAMTABLES
//
//   fClock is in this case the CPU clock divided by the
//	number of cycles to output the data (12 cycles ->1666666.6)
//
//	f = ADDER0/ADDER1/ADDER2 * (1666666.6/16777216)
//
//	f = ADDER0/ADDER1/ADDER2 * 0.099341
//
//	fMax (theoretical) = 0.5 * fClock
//	Observed decent frequency:  KHz (adder value 0x)
//
#else
//
//   fClock is in this case the CPU clock divided by the
//	number of cycles to output the data (13 cycles ->1538461.5)
//
//	f = ADDER0/ADDER1/ADDER2 * (1538461.5/16777216)
//
//	f = ADDER0/ADDER1/ADDER2 * 0.0916994
//
//	fMax (theoretical) = 0.5 * fClock
//	Observed decent frequency: 200 KHz (adder value 0x002147AF)
;
#endif

#if SRAMTABLES
// computed as 1000 / .099341
#define OneKHZ 	10066
#else
// computed as 1000 / .0916994
#define OneKHZ 	10905
#endif

extern void SpecialWait4UART(void);
extern void Special_UARTPfu08(byte b);
extern void UsrSendByte(char ch);
extern byte SineTbl[] PROGMEM;

extern void up_one(void);
extern void up_hundred(void);
extern void up_tenthousand(void);
extern void down_one(void);
extern void down_hundred(void);
extern void down_tenthousand(void);
extern unsigned long GetAdder(void);
extern void SetAdder(unsigned long l);

void StartMiniDDS(void);
void StopMiniDDS(void);
pByte CopyWaveTable(PGM_P pSrc);
void SendAdderValue(void);
void ReadAdderValue(void);

void StartMiniDDS(void)
{
	SpecialWait4UART();			//wait for I/O to complete
	cli();
	DisableINT0();					//Stop INT0 interrupts
	DisableTC0();					//Stop TC0 Timer Interrupts
	DisableTC1();					//Stop TC1 Timer Interrupts	Check may not be running
	DisableTC2();					//Stop TC2 Timer Interrupts
#if !ASCIILINK
	UARTSetBaudVal(4);				//DDSControl.exe expects 115,200
#endif
	fMiniDDS = TRUE;				//Set MiniDDS mode
	sei();
}
//
// Stopping MiniDDS is not trivial since we don't want to poll
//
// Suggested solution: From the serial input function, when an exit
//	command is detected, PATCH the code in wavegen to stop the loop
//	and execute StopMiniDDS
//
void StopMiniDDS(void)
{
//
// while no patch solution, just reboot
//
	InitiateSysReset();
}
pByte SetWaveTable(char ch)
{
#if SRAMTABLES
//
// Create the various Waveform Tables in the SRAM buffer
// All but SineTbl are generated on the fly.
// ch is expected to be in the range '1' to '4'. If not,
// the buffer is set to 0xfe.
//
	pByte pDst = WaveTbl;			//destination
	byte cnt = 0, v;				//default copy 256 bytes
	pByte pSrc = (pByte)SineTbl;	//for Sine
	v = 0xfe;						//for Triangle
	do {
		if (ch == '1') {			//Sine Table
			v = pgm_read_byte(pSrc++);
		} else if (ch == '2') {		//Sawtooth Table
			v = cnt;
		} else if (ch == '3') {		//Triangle Table
			if (cnt < 128) {
				v += 2;
			} else {
				if (cnt == 128) v = 0x01;
				v -= 2;
			}
		} else if (ch == '4') {		//Square Table
			if (cnt < 128) {
				v = 0;
			} else {
				v = 0xff;
			}
		}
		*pDst++ = v;
	} while (++cnt);				//until overflow
	return WaveTbl;

#else
	if (ch == '1') {			//Sine Table
		return SineTbl;
	} else if (ch == '2') {		//Sawtooth Table
		return SawtoothTbl;
	} else if (ch == '3') {		//Triangle Table
		return TrianleTbl;
	} else if (ch == '4') {		//Square Table
		return SquareTbl;
	}
}
#endif
}
//	
// send the current adder value (frequency) to the PC
// as a 5 byte sequence :
// 'F' folowed by a 32 bit phase accumulator value
//	  or 4 hex values in ASCIILINK
//
void SendAdderValue(void)
{
	union {
		unsigned long l;
		unsigned char b[4];			//Little Endian
	} adder;
	adder.l = GetAdder();

	_delay_ms(10);
#if ASCIILINK
	UsrSendByte('F');				// flag
	Special_UARTPfu08(0);					// zero byte for 32-bit compatibility
	Special_UARTPfu08(adder.b[2]);			// high adder
	Special_UARTPfu08(adder.b[1]);			// mid adder
	Special_UARTPfu08(adder.b[0]);			// low adder
	UsrSendByte(CR);				// terminator
	UsrSendByte(LF);				// terminator
#else
	UsrSendByte('F');				// flag
	UsrSendByte(0);				// zero byte for 32-bit compatibility
	UsrSendByte(adder.b[2]);		// high adder
	UsrSendByte(adder.b[1]);		// mid adder
	UsrSendByte(adder.b[0]);		// low adder
	UsrSendByte(LF);				// terminator
#endif
}


#if ASCIILINK
byte getnibble(void)
{
	char ch = UARTReceiveByte();
	if (ch < '0') {
		return 0;
	}
	ch -= '0';
	if (ch > 9) {
		ch -= 7;
	}
	return ch;
}
//
// read 6 hex characters (24-bit value) from the serial link
// no error checking
//
void ReadAdderValue(void)
{
	union {
		unsigned long l;
		unsigned char b[4];				//Little Endian
	} adder;
	adder.b[3] = 0;
	adder.b[2] = (getnibble() << 4) | getnibble();	//MSB
	adder.b[1] = (getnibble() << 4) | getnibble();	//Middle SB
	adder.b[0] = (getnibble() << 4) | getnibble();	//LSB
	SetAdder(adder.l);
}	


#else
// 
// read in 4 bytes from the serial link
//
void ReadAdderValue(void)
{
	union {
		unsigned long l;
		unsigned char b[4];				//Little Endian
	} adder;

	(void)UARTReceiveByte();			// read and ignore bits 32..24
	adder.b[3] = 0;
	adder.b[2] = UARTReceiveByte();		// read bits 23..16: MSB
	adder.b[1] = UARTReceiveByte();		// read bits 15..8: MIDDLE SB
	adder.b[0] = UARTReceiveByte();		// read bits 7..0: LSB
	SetAdder(adder.l);
}
#endif
//
// Called from Timer1 ISR, therefore all "Call Used" registers have been saved.
//
void FuncGenProcessInput(void)
{
	DisableTC2();						//no need for TC2 anymore
	Flags.InputActive = TRUE;			//stop enabling TC1 when input arrives
	sei();								//allow more serial input
//
// use UARTReceiveByte() to get serial input.
// use UsrSendByte(ch) to send serial output.
// These calls may block
//

//
// Check if the push button switch S2 is closed. Reboot if it is
//
	if ((SWPin & _BV(SW2Bit)) == 0) {		//Low means push button pressed
		InitiateSysReset();
	}

	byte ch = UARTReceiveByte();
#if ASCIILINK
	UsrSendByte(ch);
#endif
	// DDSControl.exe (Jesper Hansen) protocol
	if (ch == '-') {
		up_one();
	} else if (ch == 'd') {
		up_hundred();
	} else if (ch == 'D') {
		up_tenthousand();
	} else if (ch == '+') {
		down_one();
	} else if (ch == 'u') {
		down_hundred();
	} else if (ch == 'U') {
		down_tenthousand();
	} else if (ch == 's') {
		ReadAdderValue();
#if ASCIILINK
	} else if (ch == 'q') {
		StopMiniDDS();
#endif
	} else {
		(void)SetWaveTable(ch);
	}
//
// always reply with the current adder value
//
	SendAdderValue();
	while (!IsUARTOutReady()) ;			//wait for send to complete
	cli();
	Flags.InputActive =  FALSE;			//start enabling TC2 when input arrives
#if !SRAMTABLES
	// HOW TO MODIFY Z????. Correct value in CurWaveTblPtr.
	// return to TIMER1_COMPA_vect, which has saved Z on the stack, which will be
	// restored to the saved value.
	// Z doesn't change when SRAMTABLES==TRUE, so not an issue there.
#endif
}

byte SineTbl[256] PROGMEM = {			// 256 step sinewave table
		0x80,0x83,0x86,0x89,0x8c,0x8f,0x92,0x95,0x98,0x9c,0x9f,0xa2,0xa5,0xa8,0xab,0xae,
		0xb0,0xb3,0xb6,0xb9,0xbc,0xbf,0xc1,0xc4,0xc7,0xc9,0xcc,0xce,0xd1,0xd3,0xd5,0xd8,
		0xda,0xdc,0xde,0xe0,0xe2,0xe4,0xe6,0xe8,0xea,0xec,0xed,0xef,0xf0,0xf2,0xf3,0xf5,
		0xf6,0xf7,0xf8,0xf9,0xfa,0xfb,0xfc,0xfc,0xfd,0xfe,0xfe,0xff,0xff,0xff,0xff,0xff,
		0xff,0xff,0xff,0xff,0xff,0xff,0xfe,0xfe,0xfd,0xfc,0xfc,0xfb,0xfa,0xf9,0xf8,0xf7,
		0xf6,0xf5,0xf3,0xf2,0xf0,0xef,0xed,0xec,0xea,0xe8,0xe6,0xe4,0xe2,0xe0,0xde,0xdc,
		0xda,0xd8,0xd5,0xd3,0xd1,0xce,0xcc,0xc9,0xc7,0xc4,0xc1,0xbf,0xbc,0xb9,0xb6,0xb3,
		0xb0,0xae,0xab,0xa8,0xa5,0xa2,0x9f,0x9c,0x98,0x95,0x92,0x8f,0x8c,0x89,0x86,0x83,
		0x80,0x7c,0x79,0x76,0x73,0x70,0x6d,0x6a,0x67,0x63,0x60,0x5d,0x5a,0x57,0x54,0x51,
		0x4f,0x4c,0x49,0x46,0x43,0x40,0x3e,0x3b,0x38,0x36,0x33,0x31,0x2e,0x2c,0x2a,0x27,
		0x25,0x23,0x21,0x1f,0x1d,0x1b,0x19,0x17,0x15,0x13,0x12,0x10,0x0f,0x0d,0x0c,0x0a,
		0x09,0x08,0x07,0x06,0x05,0x04,0x03,0x03,0x02,0x01,0x01,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x02,0x03,0x03,0x04,0x05,0x06,0x07,0x08,
		0x09,0x0a,0x0c,0x0d,0x0f,0x10,0x12,0x13,0x15,0x17,0x19,0x1b,0x1d,0x1f,0x21,0x23,
		0x25,0x27,0x2a,0x2c,0x2e,0x31,0x33,0x36,0x38,0x3b,0x3e,0x40,0x43,0x46,0x49,0x4c,
		0x4f,0x51,0x54,0x57,0x5a,0x5d,0x60,0x63,0x67,0x6a,0x6d,0x70,0x73,0x76,0x79,0x7c
};

#if !SRAMTABLES	//These tables are generated on the fly if SRAMTABLES
byte TriangleTbl[256] PROGMEM = {		// 256 step trianglewave table
		0x00,0x02,0x04,0x06,0x08,0x0a,0x0c,0x0e,0x10,0x12,0x14,0x16,0x18,0x1a,0x1c,0x1e,
		0x20,0x22,0x24,0x26,0x28,0x2a,0x2c,0x2e,0x30,0x32,0x34,0x36,0x38,0x3a,0x3c,0x3e,
		0x40,0x42,0x44,0x46,0x48,0x4a,0x4c,0x4e,0x50,0x52,0x54,0x56,0x58,0x5a,0x5c,0x5e,
		0x60,0x62,0x64,0x66,0x68,0x6a,0x6c,0x6e,0x70,0x72,0x74,0x76,0x78,0x7a,0x7c,0x7e,
		0x80,0x82,0x84,0x86,0x88,0x8a,0x8c,0x8e,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,
		0xa0,0xa2,0xa4,0xa6,0xa8,0xaa,0xac,0xae,0xb0,0xb2,0xb4,0xb6,0xb8,0xba,0xbc,0xbe,
		0xc0,0xc2,0xc4,0xc6,0xc8,0xca,0xcc,0xce,0xd0,0xd2,0xd4,0xd6,0xd8,0xda,0xdc,0xde,
		0xe0,0xe2,0xe4,0xe6,0xe8,0xea,0xec,0xee,0xf0,0xf2,0xf4,0xf6,0xf8,0xfa,0xfc,0xfe,
		0xff,0xfd,0xfb,0xf9,0xf7,0xf5,0xf3,0xf1,0xef,0xef,0xeb,0xe9,0xe7,0xe5,0xe3,0xe1,
		0xdf,0xdd,0xdb,0xd9,0xd7,0xd5,0xd3,0xd1,0xcf,0xcf,0xcb,0xc9,0xc7,0xc5,0xc3,0xc1,
		0xbf,0xbd,0xbb,0xb9,0xb7,0xb5,0xb3,0xb1,0xaf,0xaf,0xab,0xa9,0xa7,0xa5,0xa3,0xa1,
		0x9f,0x9d,0x9b,0x99,0x97,0x95,0x93,0x91,0x8f,0x8f,0x8b,0x89,0x87,0x85,0x83,0x81,
		0x7f,0x7d,0x7b,0x79,0x77,0x75,0x73,0x71,0x6f,0x6f,0x6b,0x69,0x67,0x65,0x63,0x61,
		0x5f,0x5d,0x5b,0x59,0x57,0x55,0x53,0x51,0x4f,0x4f,0x4b,0x49,0x47,0x45,0x43,0x41,
		0x3f,0x3d,0x3b,0x39,0x37,0x35,0x33,0x31,0x2f,0x2f,0x2b,0x29,0x27,0x25,0x23,0x21,
		0x1f,0x1d,0x1b,0x19,0x17,0x15,0x13,0x11,0x0f,0x0f,0x0b,0x09,0x07,0x05,0x03,0x01
};

byte SawtoothTbl[256] PROGMEM = {		// 256 step sawtoothwave table
		0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,
		0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1a,0x1b,0x1c,0x1d,0x1e,0x1f,
		0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,0x29,0x2a,0x2b,0x2c,0x2d,0x2e,0x2f,
		0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,
		0x40,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4a,0x4b,0x4c,0x4d,0x4e,0x4f,
		0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5a,0x5b,0x5c,0x5d,0x5e,0x5f,
		0x60,0x61,0x62,0x63,0x64,0x65,0x66,0x67,0x68,0x69,0x6a,0x6b,0x6c,0x6d,0x6e,0x6f,
		0x70,0x71,0x72,0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x7a,0x7b,0x7c,0x7d,0x7e,0x7f,
		0x80,0x81,0x82,0x83,0x84,0x85,0x86,0x87,0x88,0x89,0x8a,0x8b,0x8c,0x8d,0x8e,0x8f,
		0x90,0x91,0x92,0x93,0x94,0x95,0x96,0x97,0x98,0x99,0x9a,0x9b,0x9c,0x9d,0x9e,0x9f,
		0xa0,0xa1,0xa2,0xa3,0xa4,0xa5,0xa6,0xa7,0xa8,0xa9,0xaa,0xab,0xac,0xad,0xae,0xaf,
		0xb0,0xb1,0xb2,0xb3,0xb4,0xb5,0xb6,0xb7,0xb8,0xb9,0xba,0xbb,0xbc,0xbd,0xbe,0xbf,
		0xc0,0xc1,0xc2,0xc3,0xc4,0xc5,0xc6,0xc7,0xc8,0xc9,0xca,0xcb,0xcc,0xcd,0xce,0xcf,
		0xd0,0xd1,0xd2,0xd3,0xd4,0xd5,0xd6,0xd7,0xd8,0xd9,0xda,0xdb,0xdc,0xdd,0xde,0xdf,
		0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,0xe9,0xea,0xeb,0xec,0xed,0xee,0xef,
		0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff
};

byte SquareTbl[256] PROGMEM = {		// 256 step squarewave table
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
		0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
		0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
		0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
		0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
		0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
		0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
		0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff
};
#endif
