#include "Arduino.h"
#include "Core.h"
#include "Utils.h"
#include "DTC.h"
#include "ConfEditor.h";
#include "SimpleRotatingActuator.h"


prog_uchar nodeDescription[][55] PROGMEM = {
	"DMN-EDC Software Version", // 0
	"Engine RPM", // 1
	"Injection Advance", // 2
	"Engine temperature",  // 3
	"Fuel temperature", // 4
	"Air temperature", // 5
	"Boost pressure", // 6
	"Heartbeat", // 7
	"Injection trigger threshold voltage", // 8
	"Battery Voltage", // 9
	"Running State", // 10 
	"Fuel trim enable", // 11
	"", // 12
	"", // 13
	"", // 14
	"", // 15
	"", // 16
	"TPS: signal min limit", // 17
	"TPS: signal max limit", // 18
	"TPS: safety bits (0=off, 1=idleSw, 2=dual)", // 19
	"MAP: signal min limit (at 100kPa)", // 20
	"MAP: signal max limit", // 21
	"MAP: sensor max pressure", // 22
	"RPM: engine max RPM (control-map scale)", // 23
	"RPM: DSP (0=direct/1=avg/2=2xteeth)", // 24
	"Probe: signal out (0=RPM/1=injSig/2=freqConv)",  // 25
	"Probe: freq conv ratio (1.00=100)", // 26
	"Injection: fuel cut when engine is stopped", // 27
	"Injection: advance (0=off/1=open/2=closed)",  // 28
	"Injection: automatic balance", // 29
	"Idle: adjusting enabled", // 30
	"Idle: speed target", // 31
	"Idle: PID Control Kp", // 32
	"Idle: PID Control Ki", // 33
	"Idle: PID Control Kd", // 34
	"Idle: PID Control Speed", // 35
	"Idle: PID Control Bias", // 36
	"Idle: PID Control Max Fuel", // 37
	"Idle: PID Control Min Fuel", // 38
	"QA Feedback: signal min limit", // 39
	"QA Feedback: signal max limit", // 40
	"QA Feedback: HDK reference enabled", // 41
	"QA servo: min PWM cycle (x/1024)", // 42
	"QA servo: max PWM cycle (x/1024)",// 43
	"QA servo: setPoint", // 44
	"Generic variable for debug", // 45
	"QA Debug: setPoint diff / FB Jitter", // 46
	"QA PID Control: Kp factor", // 47
	"QA PID Control: Ki factor", // 48
	"QA PID Control: Kd factor", // 49
	"QA PID Control: Speed", // 50
	"QA PID Control: Bias", // 51
	"Boost Control (0=off/1=open/2=closed/3/4)", // 52
	"Boost Control: PID Speed", // 53
	"Boost Control: PID Kp factor", // 54
	"Boost Control: PID Ki factor", // 55
	"Boost Control: PID Kd factor", // 55
	"Boost Control: PID Control Bias", // 57
	"Boost Control: PID Range", // 58
	"Boost Control: Current Pressure", // 59	
	"Boost Control: Target Pressure", // 60
	"Advance Control: PID Kp factor", // 61
	"Advance Control: PID Ki factor", // 62
	"SimpleRotatingActuator: min position", // 63
	"SimpleRotatingActuator: max position", // 64
	"SimpleRotatingActuator: inverse operation", // 65

	""
};


Core core;

Core::Core() {
	// Initialize core parameters
	// (file_id in EEPROM, initial_value, min, max, increment_step, bindedRawValue, bindedActualValue,isLocked?,description of this value (max 45 chars!))

	node[nodeSoftwareVersion] = (nodeStruct) {0x0001,VERSION_NUMBER,0x0103,9999,1,valueEngineRPMMin,valueEngineRPMMax,NODE_PROPERTY_LOCKED,VALUE_INT};  
	node[nodeEngineRPM] =       (nodeStruct) {0x1001,0,0,0,1,valueEngineRPMFiltered,valueEngineRPMJitter, NODE_PROPERTY_LOCKED,VALUE_INT};
	node[nodeEngineTiming] =    (nodeStruct) {0x1002,0,0,0,1,valueEngineTimingActual,valueNone, NODE_PROPERTY_LOCKED,VALUE_INJECTION_TIMING};  
	node[nodeTempEngine] =      (nodeStruct) {0x1003,87+64,255,0,1,valueTempEngine,valueNone, NODE_PROPERTY_EDITABLE,VALUE_CELSIUS};  
	node[nodeTempFuel] =        (nodeStruct) {0x1004,45+64,0,255,1,valueTempFuel,valueNone, NODE_PROPERTY_EDITABLE,VALUE_CELSIUS};  
	node[nodeTempAir] =         (nodeStruct) {0x1005,45+64, 0,255,1,valueTempAir,valueNone, NODE_PROPERTY_EDITABLE,VALUE_CELSIUS};  
	node[nodePressure] =        (nodeStruct) {0x1006,0,0,0,1,valueBoostPressure,valueNone, NODE_PROPERTY_LOCKED,VALUE_KPA};  
	node[nodeHeartBeat] =       (nodeStruct) {0x1007,0,1,3,1,valueEngineTimingDiff,valueEngineRPMDurationBetweenTeeths, NODE_PROPERTY_EDITABLE,VALUE_INT};      
	node[nodeInjectionThresholdVoltage] = 
	                            (nodeStruct) {0x1000,0,0,1,1,valueInjectionThresholdVoltage,valueNone, NODE_PROPERTY_LOCKED,VALUE_VOLTAGE};     
	node[nodeBatteryVoltage] =  (nodeStruct) {0x1008,0,0,1,1,valueBatteryVoltage,valueNone, NODE_PROPERTY_LOCKED,VALUE_BATTERY_VOLTAGE};     
	node[nodeRunMode] =         (nodeStruct) {0x1009,0,0,1,1,valueRunMode,valueNone, NODE_PROPERTY_LOCKED,VALUE_INT};     
	node[nodeFuelTrim] =        (nodeStruct) {0x100A,0,0,1,1,valueNone,valueFuelTrim, NODE_PROPERTY_EDITABLE,VALUE_BOOLEAN};
	node[nodeFree1] =           (nodeStruct) {0x100B,0,0,1,1,valueNone,valueNone, NODE_PROPERTY_HIDDEN,VALUE_INT};     
	node[nodeFree2] =           (nodeStruct) {0x100C,0,0,1,1,valueNone,valueNone, NODE_PROPERTY_HIDDEN,VALUE_INT};     
	node[nodeFree3] =           (nodeStruct) {0x100D,0,0,1,1,valueNone,valueNone, NODE_PROPERTY_HIDDEN,VALUE_INT};     
	node[nodeFree4] =           (nodeStruct) {0x100E,0,0,1,1,valueNone,valueNone, NODE_PROPERTY_HIDDEN,VALUE_INT};     
	node[nodeFree5] =           (nodeStruct) {0x100F,0,0,1,1,valueNone,valueNone, NODE_PROPERTY_HIDDEN,VALUE_INT};     

	node[nodeTPSMin] = 	        (nodeStruct) {0x1010,108,25,1000,1,valueTPSRaw,valueTPSActual,NODE_PROPERTY_EDITABLE,VALUE_VOLTAGE};  
	node[nodeTPSMax] =          (nodeStruct) {0x1011,534,25,1000,1,valueTPSRaw,valueTPSActual,NODE_PROPERTY_EDITABLE,VALUE_VOLTAGE};
	node[nodeTPSSafetyBits] =   (nodeStruct) {0x1012,0,0,255,1,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};  
	node[nodeFuelCutAtStall] =  (nodeStruct) {0x1013,1,0,1,1,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_BOOLEAN};  
	node[nodeQAInjectorBalance] = 
	                            (nodeStruct) {0x1015,0,0,1,1,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_BOOLEAN};  
	node[nodeTimingMethod] =    (nodeStruct) {0x1014,0,0,2,1,valueNone,valueTimingPIDAmount,NODE_PROPERTY_EDITABLE,VALUE_INT};  
	                            
	node[nodeProbeSignalOutput] = 
	                            (nodeStruct) {0x1016,0,0,10,1,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};  
	node[nodeFreqConvRatio] =   (nodeStruct) {0x1017,100,1,1000,1,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};  

	node[nodeIdleAdjusting] =   (nodeStruct) {0x1018,0,0,1,1,valueNone,valueIdlePIDCorrection,NODE_PROPERTY_EDITABLE,VALUE_BOOLEAN};  
	node[nodeIdleSpeedTarget] = (nodeStruct) {0x1019,830,350,1600,1,valueEngineRPMFiltered,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};  
	node[nodeIdleKp] =          (nodeStruct) {0x101A,41,1,300,1,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};  
	node[nodeIdleKi] =          (nodeStruct) {0x101B,1,0,300,1,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};  
	node[nodeIdleKd] =          (nodeStruct) {0x1032,13,0,300,1,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};  
	node[nodeIdlePIDSpeed] =    (nodeStruct) {0x1035,15,1,140,1,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};  	
	node[nodeIdlePIDBias] =     (nodeStruct) {0x1036,33,1,200,1,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};  
	node[nodeIdleMaxFuel] =     (nodeStruct) {0x1033,520,0,1024,5,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};  
	node[nodeIdleMinFuel] =     (nodeStruct) {0x1034,140,0,1024,5,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};  

	node[nodeRPMDSP] =          (nodeStruct) {0x101C,0,0,3,1,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};  

	node[nodeQAFeedbackMin] =   (nodeStruct) {0x101D,205,1,1022,1,valueQAfeedbackActual,valueQAfeedbackRaw,NODE_PROPERTY_EDITABLE,VALUE_VOLTAGE};
	node[nodeQAFeedbackMax] =   (nodeStruct) {0x101E,881,1,1023,1,valueQAfeedbackActual,valueQAfeedbackRaw,NODE_PROPERTY_EDITABLE,VALUE_VOLTAGE};
	node[nodeQAReferenceEnabled] = 
								(nodeStruct) {0x101F,0,0,1,1,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_BOOLEAN};  
	node[nodeQASetPoint] =      (nodeStruct) {0x1020,0,0,1023,1,valueNone,valueQAfeedbackSetpoint,NODE_PROPERTY_LOCKED,VALUE_INT};  
	node[nodeQAMinPWM] =        (nodeStruct) {0x1021,100,1,700,1,valueNone,valueQAPWMActual,NODE_PROPERTY_EDITABLE,VALUE_INT};      
	node[nodeQAMaxPWM] =        (nodeStruct) {0x1022,650,1,800,1,valueNone,valueQAPWMActual,NODE_PROPERTY_EDITABLE,VALUE_INT};     
	node[nodeMAPMin] =          (nodeStruct) {0x1023,138,25,1000,1,valueMAPRaw,valueMAPActual,NODE_PROPERTY_EDITABLE,VALUE_VOLTAGE};  
	node[nodeMAPMax] =          (nodeStruct) {0x1024,435,25,1000,1,valueMAPRaw,valueMAPActual,NODE_PROPERTY_EDITABLE,VALUE_VOLTAGE};
	node[nodeMAPkPa] =          (nodeStruct) {0x1025,250,100,1000,5,valueBoostPressure,valueNone,NODE_PROPERTY_EDITABLE,VALUE_KPA};  
	node[nodeControlMapScaleRPM] = 
	                            (nodeStruct) {0x1026,5000,100,10000,100,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};  
	node[nodeGenericDebugValue] = 
	                            (nodeStruct) {0x1027,0,0,1023,16,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_KPA};  
	node[nodeQADebugJitter] =   (nodeStruct) {0x1028,0,0,1024,1,valueNone,valueQAJitter,NODE_PROPERTY_LOCKED,VALUE_INT};  
	node[nodeQAPIDKp] =         (nodeStruct) {0x1029,80,0,1000,1,valueNone,valueQAPIDPparam,NODE_PROPERTY_EDITABLE,VALUE_INT};  
	node[nodeQAPIDKi] =         (nodeStruct) {0x102A,3,0,1000,1,valueNone,valueQAPIDIparam,NODE_PROPERTY_EDITABLE,VALUE_INT};  
	node[nodeQAPIDKd] =         (nodeStruct) {0x102B,3,0,1000,1,valueNone,valueQAPIDDparam,NODE_PROPERTY_EDITABLE,VALUE_INT};  
	node[nodeQAPIDSpeed] =      (nodeStruct) {0x102C,25,1,128,1,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};     
	node[nodeQAPIDBias] =       (nodeStruct) {0x102D,55,1,200,5,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};      

	node[nodeBoostAdjusting] =  (nodeStruct) {0x102E,0,0,4,1,valueN75DutyCycleBaseForPid,valueBoostPIDCorrection,NODE_PROPERTY_EDITABLE,VALUE_INT};      
	node[nodeBoostSpeed] =      (nodeStruct) {0x102F,5,1,200,10,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};      
	node[nodeBoostKp] =         (nodeStruct) {0x1030,5,1,2000,1,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};      
	node[nodeBoostKi] =         (nodeStruct) {0x1031,5,1,2000,1,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};      
	node[nodeBoostKd] =         (nodeStruct) {0x1032,0,0,2000,1,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};  
	node[nodeBoostBias] =    	(nodeStruct) {0x1038,100,1,200,1,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};  
	node[nodeBoostPIDRange] =   (nodeStruct) {0x1039,20,1,200,1,valueNone,valueBoostPIDCorrection,NODE_PROPERTY_EDITABLE,VALUE_INT};  
	node[nodeBoostTargetPressure] =   (nodeStruct) {0x0,0,1,200,1,valueBoostTarget,valueNone,NODE_PROPERTY_LOCKED,VALUE_KPA};  
	node[nodeBoostActualPressure] =   (nodeStruct) {0x0,0,1,200,1,valueBoostPressure,valueNone,NODE_PROPERTY_LOCKED,VALUE_KPA};  

	node[nodeTimingKp] =         (nodeStruct) {0x103A,10,1,200,1,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};      
	node[nodeTimingKi] =         (nodeStruct) {0x103B,2,1,200,1,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_INT};   

	node[nodeSRAMinPos] =         (nodeStruct) {0x103C,300,50,950,5,valueSRAPosition,valueSRATarget,NODE_PROPERTY_EDITABLE,VALUE_INT};      
	node[nodeSRAMaxPos] =         (nodeStruct) {0x103D,700,50,950,5,valueSRAPosition,valueSRATarget,NODE_PROPERTY_EDITABLE,VALUE_INT};  
	node[nodeSRAInverseOperation] =         (nodeStruct) {0x103E,0,0,1,1,valueNone,valueNone,NODE_PROPERTY_EDITABLE,VALUE_BOOLEAN};  

	currentNode = LIST_RESET;


	/* MAP header <map-id>, 0xf0, .... 
	Map id is used for saving/loading maps from eeprom, use unique id
	*/ 
   
	static unsigned char fuelMap[] = {
	  0xF0,0xF0,'M','2','D',
	  0x8,0x6,MAP_AXIS_RPM,MAP_AXIS_TPS,MAP_AXIS_INJECTED_FUEL,
 		180,	80,     35,     0,      0,      0,      0,      0,
		180,    98,    98,    37,     0,      0,      0,      0,
	    180,    109,    109,    79,    37,     10,      0,      0,
	    180,    109,    123,    134,    143,    87,     10,     0,
        180,    109,    131,    151,    156,    156,    77,     0,
        180,    109,    150,    156,    156,    164,    164,    0,      
	  0,0,0,0,0                // lastX,lastY,lastRet,lastRet 10bit (2 bytes)
	 };
	static unsigned char boostMap[] = {
	  0x01,0xF0,'M','2','D',
	  0x8,0x6,MAP_AXIS_RPM,MAP_AXIS_KPA,MAP_AXIS_INJECTED_FUEL,
	  0,0,0,0,     0,0,0,0,
	  31,31,31,31, 31,31,31,0,
	  53,47,65,65, 75,75,75,0,
	  30,30,78,78, 85,85,85,0,	  
	  0,0,0,0, 0,0,0,0,
	  0,0,0,0, 0,0,0,0,
	  0,0,0,0,0                // lastX,lastY,lastRet,lastRet 10bit (2 bytes)
	};

	static unsigned char maxFuelMap[] = {
	  0x0F,0xF0,'M','2','D',
	  0x8,0x6,MAP_AXIS_RPM,MAP_AXIS_KPA,MAP_AXIS_INJECTED_FUEL,
	  250,250,250,250, 250,250,250,250,
	  250,250,250,250, 250,250,250,250,
	  250,250,250,250, 250,250,250,250,
	  250,250,250,250, 250,250,250,250,
	  250,250,250,250, 250,250,250,250,
	  250,250,250,250, 250,250,250,250,
	  0,0,0,0,0                // lastX,lastY,lastRet,lastRet 10bit (2 bytes)
	};

	static unsigned char openLoopAdvanceMap[] = {
		0x02,0xF0,'M','2','D',
		0x6,0x6,MAP_AXIS_RPM,MAP_AXIS_INJECTED_FUEL,MAP_AXIS_DUTY_CYCLE,
		255,255,255, 255,255,255, 
		255,255,255, 255,255,210, 
		255,255,255, 255,180,180, 
		255,255,255,    190,140,0, 
		255,255,255,    100,80,0, 
		100,100,100,    50,50,0, 
		0,0,0,0,0                // lastX,lastY,lastRet,lastRet 10bit (2 bytes)
	};

	static unsigned char closedLoopAdvanceMap[] = {
		0x03,0xF0,'M','2','D',
		0x6,0x6,MAP_AXIS_RPM,MAP_AXIS_INJECTED_FUEL,MAP_AXIS_INJECTION_TIMING,
		0,0,0, 0,0,0, 
		0,0,0, 0,0,0, 
		0,0,0, 0,0,0,  

		0,0,0, 0,0,0, 
		0,0,0, 0,0,0, 
		0,0,0, 0,0,0,  
		0,0,0,0,0                // lastX,lastY,lastRet,lastRet 10bit (2 bytes)
	};

	static unsigned char idleMap[] = {
		0x04,0xF0,'M','2','D',
		0x8,0x4,MAP_AXIS_IDLERPM,MAP_AXIS_CELSIUS,MAP_AXIS_INJECTED_FUEL,
		255,255,255,114, 80,60,60,50,
		190,130,90,80, 70,60,50,40,		
		190,120,90,80, 70,60,50,40,
		190,110,90,80, 70,60,50,40,
		0,0,0,0,0                // lastX,lastY,lastRet,lastRet 10bit (2 bytes)
	};

	static unsigned char turboControlMap[] = {
		0x05,0xF0,'M','2','D',
		0x6,0x6,MAP_AXIS_RPM,MAP_AXIS_INJECTED_FUEL,MAP_AXIS_DUTY_CYCLE,
		201,227,210,201,171,158,
		201,227,208,192,169,158,	
		201,182,198,180,162,158,
		201,201,195,166,153,140,
		201,182,118,151,151,28,
		201,182,156,103,59,28,		
		0,0,0,0,0                // lastX,lastY,lastRet,lastRet 10bit (2 bytes)
	};

	static unsigned char turboTargetPressureMap[] = {
		0x06,0xF0,'M','2','D',
		0x6,0x4,MAP_AXIS_RPM,MAP_AXIS_TPS,MAP_AXIS_KPA,
		0,0,0,0,0,0,
		0,0,0,10,20,100,	
		0,0,10,30,100,128,
		100,100,100,100,100,100,
		0,0,0,0,0                // lastX,lastY,lastRet,lastRet 10bit (2 bytes)
	};

	static unsigned char glowPeriodMap[] = {
		0x07,0xF0,'M','1','D',
		0x6,0x1,MAP_AXIS_CELSIUS,MAP_AXIS_NONE,MAP_AXIS_SECONDS,
		30,30,20,10,0,0,
		0,0,0,0,0                // lastX,lastY,lastRet,lastRet 10bit (2 bytes)
	};

	static unsigned char engineTempSensorMap[] = {
		0x08,0xF0,'M','1','D',
		0x6,0x1,MAP_AXIS_VOLTAGE,MAP_AXIS_NONE,MAP_AXIS_CELSIUS,
		30,20,9,0,0,0,
		0,0,0,0,0                // lastX,lastY,lastRet,lastRet 10bit (2 bytes)
	};

	static unsigned char fuelTempSensorMap[] = {
		0x09,0xF0,'M','1','D',
		0x6,0x1,MAP_AXIS_VOLTAGE,MAP_AXIS_NONE,MAP_AXIS_CELSIUS,
		30,30,20,10,0,0,
		0,0,0,0,0                // lastX,lastY,lastRet,lastRet 10bit (2 bytes)
	};

	static unsigned char airTempSensorMap[] = {
		0x0A,0xF0,'M','1','D',
		0x6,0x1,MAP_AXIS_VOLTAGE,MAP_AXIS_NONE,MAP_AXIS_CELSIUS,
		30,30,20,10,0,0,
		0,0,0,0,0                // lastX,lastY,lastRet,lastRet 10bit (2 bytes)
	};

	static unsigned char fuelTrimMap[] = {
		0x0B,0xF0,'M','1','D',
		0x5,0x5,MAP_AXIS_INJECTED_FUEL,MAP_AXIS_FUEL_TRIM_AMOUNT,MAP_AXIS_INJECTED_FUEL,
		0,64,128,172,235,
		0,64,128,182,245,
		0,64,128,192,255,
		0,64,128,202,255,
		0,64,128,212,255,		
		0,0,0,0,0                // lastX,lastY,lastRet,lastRet 10bit (2 bytes)
	};

	static unsigned char fuelTrimFuelTemp[] = {
		0x0C,0xF0,'M','1','D',
		0x8,0x1,MAP_AXIS_CELSIUS,MAP_AXIS_NONE,MAP_AXIS_FUEL_TRIM_AMOUNT,
		128,128,128,128,128,128,128,128,
		0,0,0,0,0                // lastX,lastY,lastRet,lastRet 10bit (2 bytes)
	};
	
	static unsigned char fuelTrimAirTemp[] = {
		0x0D,0xF0,'M','1','D',
		0x8,0x1,MAP_AXIS_CELSIUS,MAP_AXIS_NONE,MAP_AXIS_FUEL_TRIM_AMOUNT,
		128,128,128,128,128,128,128,128,
		0,0,0,0,0                // lastX,lastY,lastRet,lastRet 10bit (2 bytes)
	};

	static unsigned char fuelTrimEngineTemp[] = {
		0x0E,0xF0,'M','1','D',
		0x8,0x1,MAP_AXIS_CELSIUS,MAP_AXIS_NONE,MAP_AXIS_FUEL_TRIM_AMOUNT,
		128,128,128,128,128,128,128,128,
		0,0,0,0,0                // lastX,lastY,lastRet,lastRet 10bit (2 bytes)
	};	

	static unsigned char testMap[] = {
		0x10,0xF0,'M','2','D',
		0x5,0x5,MAP_AXIS_RAW,MAP_AXIS_RAW,MAP_AXIS_RAW,
		0,  10, 20,  30,40,
		50, 60, 70,  80,90,
		100,110,120,130,140,
		150,160,170,180,190,
		200,210,220,230,240,
		0,0,0,0,0                // lastX,lastY,lastRet,lastRet 10bit (2 bytes)
	};

	static unsigned char actuatorTension[] = {
		0x11,0xF0,'M','1','D',
		0x8,0x1,MAP_AXIS_RAW,MAP_AXIS_NONE,MAP_AXIS_RAW,
		255,215,192,143,90,55,23,0,
		0,0,0,0,0                // lastX,lastY,lastRet,lastRet 10bit (2 bytes)
	};						

	mapNames[Core::mapIdxFuelMap] = "FuelMap";
	mapNames[Core::mapIdxBoostMap] = "BoostMap (enrichment)";
	mapNames[Core::mapIdxMaxFuelMap] = "MaxFuelMap (torque limiter)";
	mapNames[Core::mapIdxIdleMap] = "IdleMap (IQ when starting & idling)";
	mapNames[Core::mapIdxOpenLoopAdvanceMap] = "Open Loop advance";
	mapNames[Core::mapIdxClosedLoopAdvanceMap] = "Closed Loop advance";
	mapNames[Core::mapIdxTurboControlMap] = "Turbo actuator duty cycle base map";
	mapNames[Core::mapIdxTurboTargetPressureMap] = "Turbo actuator target pressure";
	mapNames[Core::mapIdxGlowPeriodMap] = "Glow period";
	mapNames[Core::mapIdxEngineTempSensorMap] = "Engine temp. sensor calibration";
	mapNames[Core::mapIdxFuelTempSensorMap] = "Fuel temp. sensor calibration";
	mapNames[Core::mapIdxAirTempSensorMap] = "Intake air temp. sensor calibration";
	mapNames[Core::mapIdxFuelTrimMap] = "Fuel Trim Apply Map";
	mapNames[Core::mapIdxFuelTrimFuelTemp] = "Fuel Trim amount vs. Fuel Temp.";
	mapNames[Core::mapIdxFuelTrimAirTemp] = "Fuel Trim amount vs. Air Temp.";
	mapNames[Core::mapIdxFuelTrimEngineTemp] = "Fuel Trim amount vs. Engine Temp";
	mapNames[Core::mapIdxTestMap] = "Test Map";
	mapNames[Core::mapIdxActuatorTension] = "N75 Actuator Tension";

	maps[Core::mapIdxFuelMap] = (unsigned char*)&fuelMap;
	maps[Core::mapIdxIdleMap] = (unsigned char*)&idleMap;
	maps[Core::mapIdxBoostMap] = (unsigned char*)&boostMap;
	maps[Core::mapIdxMaxFuelMap] = (unsigned char*)&maxFuelMap;
	maps[Core::mapIdxOpenLoopAdvanceMap] = (unsigned char*)&openLoopAdvanceMap;
	maps[Core::mapIdxClosedLoopAdvanceMap] = (unsigned char*)&closedLoopAdvanceMap;
	maps[Core::mapIdxTurboControlMap] = (unsigned char*)&turboControlMap;
	maps[Core::mapIdxTurboTargetPressureMap] = (unsigned char*)&turboTargetPressureMap;
	maps[Core::mapIdxGlowPeriodMap] = (unsigned char*)&glowPeriodMap;
	maps[Core::mapIdxEngineTempSensorMap] = (unsigned char*)&engineTempSensorMap;
	maps[Core::mapIdxFuelTempSensorMap] = (unsigned char*)&fuelTempSensorMap;
	maps[Core::mapIdxAirTempSensorMap] = (unsigned char*)&airTempSensorMap;
	maps[Core::mapIdxFuelTrimMap] = (unsigned char*)&fuelTrimMap;
	maps[Core::mapIdxFuelTrimFuelTemp] = (unsigned char*)&fuelTrimFuelTemp;
	maps[Core::mapIdxFuelTrimAirTemp] = (unsigned char*)&fuelTrimAirTemp;
	maps[Core::mapIdxFuelTrimEngineTemp] = (unsigned char*)&fuelTrimEngineTemp;
	maps[Core::mapIdxTestMap] = (unsigned char*)&testMap;
	maps[Core::mapIdxActuatorTension] = (unsigned char*)&actuatorTension;


	numberOfMaps=18;
}

/*
New EEPROM Structure 

0000 ROOT ID 4B
0004 file id 2B  // chunk 1
0006 Size 2B
0008 Data (length=size)
000X file id     // chunk n
000X+2 Size
000X+4 Data (length=size)
...
0??? file id=FFFF  // EOF

3584-4096 -> Reserved for DTC


item_id = 0x10xx control values
item_id = 0xf0xx control maps
*/


void Core::save() {
	SimpleRotatingActuator::disable();

	int ofs = CONFIGURATION_EEPROM_OFFSET;    
	int size;
	// write header
	EEPROMwriteData(ofs,CONFIGURATION_FILE_4BYTE_ID,4);
	ofs += 4;
	// Write nodes 
	unsigned char idx;
	for (unsigned char idx = 0;idx<NODE_MAX;idx++) {
		EEPROMwriteData(ofs,(char*)&node[idx].fileId,2); // File id 
		ofs += 2;				
		size = 2;
		EEPROMwriteData(ofs,(char*)&size,2);					// Size
		ofs += 2;				
		EEPROMwriteData(ofs,(char*)&node[idx].value,2);	// Data
		ofs += 2;		
	}
	// Write maps
	for (unsigned char idx = 0;idx<numberOfMaps;idx++) {
		char x = core.maps[idx][5];
		char y = core.maps[idx][6];
		
		size = x*y+15;

		EEPROMwriteData(ofs,(char*)core.maps[idx],2); // File id 
		ofs += 2;				
		EEPROMwriteData(ofs,(char*)&size,2);				// Size
		ofs += 2;				
		EEPROMwriteData(ofs,(char*)core.maps[idx],15+x*y);    // Data
		ofs += size;

	}
	// FILE ID for EOF
	size = 0xFFFF;
	EEPROMwriteData(ofs,(char*)&size,2);

	SimpleRotatingActuator::enable();	
}


bool Core::load() {
	unsigned char c1=0,c2=0;
	int ofs = CONFIGURATION_EEPROM_OFFSET;  
	int fileId;
	int size;  
	int value;
	char buf[7];
	EEPROMreadData(ofs,buf,4);
	buf[4] = 0;
	ofs += 4;

	if (strcmp(buf,CONFIGURATION_FILE_4BYTE_ID) == 0) {
		do {
			EEPROMreadData(ofs,(char*)&fileId,2);
			ofs += 2;
			EEPROMreadData(ofs,(char*)&size,2);
			ofs += 2;

			switch ((fileId & 0xFF00)) {
				case 0x1000:
					if (size == 2) {
						// Core item, length should be 2

						for (unsigned char idx = 0;idx<NODE_MAX;idx++) {
							if (node[idx].fileId == fileId) {
								EEPROMreadData(ofs,(char*)&value,size);
								c1++;
								node[idx].value = value;
							}
						}
					} else {
						dtc.setError(DTC_CONFIGURATION_MISMATCH);
					}

					break;
				case 0xF000:
					// Map item, check matching length/header before loading 
					// EEPROMreadData(ofs,(char*)&buf,7);

					for (unsigned char idx = 0;idx<numberOfMaps;idx++) {
						int mapId;
						mapId = (int)core.maps[idx][1]*256+core.maps[idx][0];

						if (mapId == fileId) {
							if (size == (core.maps[idx][5]*core.maps[idx][6]+15)) {
								c2++;
								EEPROMreadData(ofs,(char*)core.maps[idx],size);
							} else {
								dtc.setError(DTC_CONFIGURATION_MISMATCH);
							}
						}
					}
					break;
			}

			ofs += size;
		} while (fileId != 0xFFFF);
		Serial.print(c1);
		Serial.print(" nodes, ");			
		Serial.print(c2);
		Serial.print(" maps..");		

		return true;
	} else {
		// file id not detected, do not load
		dtc.setError(DTC_CONFIGURATION_ERROR);
	}
	return false;
}

void Core::save_old() {
	// TODO new format
	// <id><len><data><cksum>
	// <id><len><data><cksum> ...
	// 
	// conf saved as node<->value pairs
	// tables "as it is" (check allocated size in preinitalized header vs. stored size)
	
	int ofs = CONFIGURATION_EEPROM_OFFSET;    
	// write header
	EEPROMwriteData(ofs,CONFIGURATION_FILE_4BYTE_ID,4);
	ofs += 4;
	// write size
	int size = sizeof(node);
	EEPROMwriteData(ofs,(char*)&size,2);
	ofs += 2;
	// write contents - core nodes
	EEPROMwriteData(ofs,(char*)&node,size);
	ofs += size;  

	// write map data
	for (char i=0;i<numberOfMaps;i++) {
		char x = core.maps[i][3];
		char y = core.maps[i][4];
		EEPROMwriteData(ofs,(char*)core.maps[i],13+x*y);
		ofs += 13+x*y;
	}
	// write contents - map data -- todo better structure!!
	//EEPROMwriteData(ofs,(char*)&basicFuelMap,sizeof(basicFuelMap));
	
	/* ****
	EEPROMwriteData(ofs,(char*)core.maps[0],13+8*6);
	ofs += 13+8*6;
	
	EEPROMwriteData(ofs,(char*)core.maps[1],13+8*4);
	ofs += 13+8*4;

	**** */

	//EEPROMwriteData(ofs,(char*)core.maps[2],13+5*1);
	//ofs += 13+5*1;
	
	confeditor.setSystemStatusMessage("Saved");
}

bool Core::load_old() {
	int ofs = CONFIGURATION_EEPROM_OFFSET;    
	char buf[5];
	EEPROMreadData(ofs,buf,4);
	buf[4] = 0;
	ofs += 4;
	if (strcmp(buf,CONFIGURATION_FILE_4BYTE_ID) == 0) {
		// read record length
		int len;
		EEPROMreadData(ofs,(char*)&len,2);
		ofs += 2;
		if (len == sizeof(node)) {
			EEPROMreadData(ofs,(char*)&node,sizeof(node));
			ofs += len;

			for (char i=0;i<numberOfMaps;i++) {
				char x = core.maps[i][3];
				char y = core.maps[i][4];
				EEPROMreadData(ofs,(char*)core.maps[i],13+x*y);
				ofs += 13+x*y;				
			}
			/*
			// TODO add header check before loading table
			EEPROMreadData(ofs,(char*)core.maps[0],13+8*6);
			ofs += 13+8*6;

			EEPROMreadData(ofs,(char*)core.maps[1],13+8*4);
			ofs += 13+8*4;
			*/

			//EEPROMreadData(ofs,(char*)core.maps[2],13+5*1);
			//ofs += 13+5*1;

			return true;
		} else {
			// Load id stores core fits current table
			if (len<sizeof(node)) {
				EEPROMreadData(ofs,(char*)&node,sizeof(node));
				confeditor.setSystemStatusMessage("Conf. truncated?");
			}
			dtc.setError(DTC_CONFIGURATION_ERROR);   
			return false;         
		}
	} else {
		// file id not detected, do not load
		dtc.setError(DTC_CONFIGURATION_ERROR);
	}
	return false;
}

void Core::setCurrentNode(int start) {
	if (start<=NODE_MAX) { 
		currentNode = start;
	} else {
		currentNode = LIST_RESET;
	}
}

int Core::seekNextNode() {
	if (currentNode+1<=NODE_MAX) { 
		currentNode++;
	} else {
		currentNode = LIST_RESET;
	}
	return currentNode;
}

nodeStruct* Core::getNextNode() {
	seekNextNode();
	return getNodeData();
}

nodeStruct* Core::getNodeData() {
	if (currentNode == LIST_RESET)
		return NULL;
	return &node[currentNode];
}

void Core::incValue() {
	setValue(node[currentNode].value+node[currentNode].step);
}

void Core::decValue() {
	setValue(node[currentNode].value-node[currentNode].step);
}

void Core::setValue(int value) {
	if (node[currentNode].properties != NODE_PROPERTY_LOCKED) {
		if (value < node[currentNode].min) 
			value = node[currentNode].min;
		if (value > node[currentNode].max) 
			value = node[currentNode].max;
	
		node[currentNode].value = value;
	}
}
