/*
(c) 2012-2013 Juho Pesonen, dmn@qla.fi

Program layout:

+ EDCmain (sets interrupt services and handles user interface communication in main loop)
  - refreshQuantityAdjuster is called periodically, handles control of the quantity adjuster
  - refreshFastSensors is called periodically, sets fuel amount based on current engine parameters
  - refreshSlowSensors is called periodically, reads TPS, MAP and other sensor values. Also set updates output ports.
  - rpmTrigger is called when rpm interrupt is high (actual calculation is based on timer1 counter value )
  - needleTrigger is called when injection needle interrupt is high (injection offset is timer 1 counter value after rpm trigger)

  + QuantityAdjuster (PID control of QA)
  + Core (contains configurable items, maps and also storage for sensor & other processing data)
  + ConfEditor (logic for text mode configuration)
  + DTC (non-volatile storage of diagnostic errors)
  + PID (general purpose PID routine)
  + Utils (map lookup & interpolation routines, some ui routines as well)
  + defines.h PIN mapping and other low level data


TODO:
- timer2: based freq converter (pwm or one shot) for tacho etc..
- overboost DTC + safe mode
- Boost map scaler -> separate map -> better drivability


idle PID 33/2/1   (2-stable when engine load differs, 33 for rpm hunting to settle)
qa   PID 54/7/1


qa debugVal 
	0 - as it is
	1 - median filtering
	2 - read pos when trigger happens

 */

#include <EEPROM.h>
#include "ConfEditor.h"
#include "Core.h"
#include "QuantityAdjuster.h"
#include "TimerThree.h"
#include "utils.h"
#include "DTC.h"
#include "defines.h"
#include "MemoryFree.h"
#include "PID.h"
#include "TachoOut.h"
// #include "SimpleRotatingActuator.h"

// Using Timer1 to provide accurate(?) engine speed 

static inline void rpmTimerSetup() __attribute__((always_inline));
static inline void rpmTimerEnable() __attribute__((always_inline));
static inline void rpmTimerDisable() __attribute__((always_inline));

//((16000000/64)/65536)*60

#define RPMTIMER_COUNTER TCNT1

// TODO Set scaler to /16 when running and to /64 when starting !! (http://www.engblaze.com/microcontroller-tutorial-avr-and-arduino-timer-interrupts/)

// ((((16000000/64)/65535)*60/5                         16000000L             4000 ~ 780rpm
#define RPMTIMER_DURATION_TO_RPM(x) ((unsigned long)(60*F_CPU/64/NUMBER_OF_CYLINDERS)/((unsigned long)x))  // 250 000hz frequency (max. 65535 ticks per teeth ~45rpm)
#define RPMTIMER_MIN_DURATON 400 // 7500rpm

volatile int rpmStore[NUMBER_OF_CYLINDERS*2]; 

// VP37 Quantity adjuster module
static QuantityAdjuster adjuster;

// Tacho frequency conversion
TachoOut tacho;

// Periodic routines called by 2500Hz (or less, set divider to 2 or greater)
#define interruptHandlerMax 8

volatile struct interruptHandlerStruct { 
	void (*handler)();
	unsigned char divider;
} interruptHandlerArray[interruptHandlerMax];

volatile unsigned char shortTicks=0;

// Called when Timer3 overflow occurs. Then calls handler routines according to their divider value
void mainInterruptHandler() {
	// Enable nested interrupts to not miss (or wrongly) calculate RPM signal 
	sei();    
	for (unsigned char i=0;i<interruptHandlerMax;i++) {
		if (interruptHandlerArray[i].handler != NULL 
			&& (interruptHandlerArray[i].divider == 0
				|| (shortTicks % interruptHandlerArray[i].divider == 0))) {
			interruptHandlerArray[i].handler();
		}
	}
	shortTicks++;
	cli();
} 

void refreshQuantityAdjuster() {
	adjuster.update();
}

static inline void rpmTimerSetup()  {
 	cli();
 	TCCR1A = 0; 
 	TCCR1B = 0; 
 	TCNT1 = 0;
 	OCR1A = 0xffff;
  
 	TIMSK1 |= (1 << OCIE1A); // enable timer compare interrupt
 	sei();
}

static inline void rpmTimerEnable() {
 	cli();
 	TCNT1 = 0;
 	TCCR1B = 11; // WGM12 = 8 +CS11 = 2+CS10 = 1
 	sei();
}

static void rpmTimerDisable() {
 	cli();
 	TCCR1A = 0;
 	TCCR1B = 0;
 	sei();
}

ISR(TIMER1_COMPA_vect) 
{	
	// Timer has waited rotation for too long, declare engine as stopped
	rpmTimerDisable();
	core.controls[Core::valueEngineRPM] = 0;
	core.node[Core::nodeEngineRPM].value = 0;
	core.controls[Core::valueEngineRPMFiltered] = 0;   
}

/*
	Handles slow sensors input / output 
	- Temperatures
	- TPS position
	- Boost and Injection Advance PID control
*/

static unsigned char tpsAvg[AVG_TABLE_SIZE_TPS];
static unsigned char tpsAvgIdx;

volatile static char calls=0;
void refreshSlowSensors() {
	calls++;
	if (calls>1)
		dtc.setError(DTC_TRAP_5);
	// read slow changing sensors
	int value;
	int scaledValue;
	int tps1,tps2;

	// Engine TEMP
	value = analogRead(PIN_ENGINETEMP);
	scaledValue = mapLookUp(core.maps[Core::mapIdxEngineTempSensorMap], value / 4, 0);		
	if (value > ANALOG_INPUT_HIGH_STATE_LIMIT) {
		// generate error only if map is configured and sensor reading is not present
		if (scaledValue > 0)
			dtc.setError(DTC_ENGINE_TEMP_UNCONNECTED);
		scaledValue = core.node[Core::nodeTempEngine].value; // use configuration setpoint value sensor's failback substitute value
	} 

	core.controls[Core::valueTempEngine]=scaledValue;
	
	// Fuel TEMP
	value = analogRead(PIN_FUELTEMP);
	scaledValue = mapLookUp(core.maps[Core::mapIdxFuelTempSensorMap], value / 4, 0);		
	if (value > ANALOG_INPUT_HIGH_STATE_LIMIT) {
		// generate error only if map is configured and sensor reading is not present
		if (scaledValue > 0)
			dtc.setError(DTC_FUEL_TEMP_UNCONNECTED);
		scaledValue = core.node[Core::nodeTempFuel].value; // use configuration setpoint value sensor's failback substitute value
	}
	core.controls[Core::valueTempFuel]=scaledValue;

	// Air TEMP
	value = analogRead(PIN_ENGINETEMP);
	scaledValue = mapLookUp(core.maps[Core::mapIdxAirTempSensorMap], value / 4, 0);		
	if (value > ANALOG_INPUT_HIGH_STATE_LIMIT) {
		// generate error only if map is configured and sensor reading is not present
		if (scaledValue > 0)
			dtc.setError(DTC_AIR_TEMP_UNCONNECTED);
		scaledValue = core.node[Core::nodeTempAir].value; // use configuration setpoint value sensor's failback substitute value
	}
	core.controls[Core::valueTempAir]=scaledValue;

	// Throttle position sensor(s) - Common VW sensor type does have dual potentiometer
	// and swiches for idle and full gas (later one not used)
	value = analogRead(PIN_TPS);
	core.controls[Core::valueTPSRaw]=value;

	// Check TPS it is connected, otherwise apply limp mode amount (about 15%) 
	if (value > ANALOG_INPUT_HIGH_STATE_LIMIT) {
	 	dtc.setError(DTC_TPS_UNCONNECTED);
		core.controls[Core::valueTPSActual] = TPS_LIMP_MODE_AMOUNT; 
	} else {
		scaledValue = mapValues(core.controls[Core::valueTPSRaw],
						   core.node[Core::nodeTPSMin].value,
						   core.node[Core::nodeTPSMax].value);
		/*
		tpsAvg[tpsAvgIdx] = scaledValue;
		tpsAvgIdx++;
		if (tpsAvgIdx>AVG_TABLE_SIZE_TPS) {
			tpsAvgIdx=0;
		}
		unsigned int tpsSum=0;
		for (char i=0; i<AVG_TABLE_SIZE_TPS; i++) {
			tpsSum += tpsAvg[i];
		}
		scaledValue = tpsSum / AVG_TABLE_SIZE_TPS;
		*/
		core.controls[Core::valueTPSActual] = scaledValue; 

	}

	/* use dual TPS sensors, if enabled */
	if ((core.node[Core::nodeTPSSafetyBits].value & TPS_SAFETY_BITS_DUAL) == TPS_SAFETY_BITS_DUAL) {

		value = analogRead(PIN_TPS2);
		if (value > ANALOG_INPUT_HIGH_STATE_LIMIT) {
		 	dtc.setError(DTC_TPS2_UNCONNECTED);
		}
		// second TPS is inverse
		scaledValue = 255-mapValues(value,
						   core.node[Core::nodeTPSMin].value,
						   core.node[Core::nodeTPSMax].value);

		// Set error if error margin is too high
		if (abs(scaledValue-value)>DUAL_TPS_ALLOWED_ERROR_MARGIN) {
		 	dtc.setError(DTC_TPS_UNPLAUSIBLE);			
		}	
		// If DTC for unplausible tps is active on current run, do not trust TPS until next run or runtime dtc reset
		if (dtc.isErrorActive(DTC_TPS_UNPLAUSIBLE)) {
			core.controls[Core::valueTPSActual] = TPS_LIMP_MODE_AMOUNT; 
		} else {
			// Final TPS
			core.controls[Core::valueTPSActual] = (core.controls[Core::valueTPSActual]+scaledValue)/2;
		}
	}

	/* override for gas pedal idle position */
	if ((core.node[Core::nodeTPSSafetyBits].value & TPS_SAFETY_BITS_IDLESW) == TPS_SAFETY_BITS_IDLESW) {
		value = digitalRead(PIN_TPS_IDLE);
		if (!value) {
			core.controls[Core::valueTPSActual] = 0;
		}
	}

	value = analogRead(PIN_MAP);
  	core.controls[Core::valueMAPRaw] = value; 
  	static int mapFailCount = 0;
	if (value > ANALOG_INPUT_HIGH_STATE_LIMIT) {
		if (mapFailCount > SENSOR_FAIL_COUNT) {
		 	dtc.setError(DTC_MAP_UNCONNECTED);
		 	// Map failback is zero kPa
			core.controls[Core::valueBoostPressure] = 0; 
		} else {
			mapFailCount++;	
		}
	} else {
		core.controls[Core::valueBoostPressure] = mapValues(core.controls[Core::valueMAPRaw],
						   core.node[Core::nodeMAPMin].value,
						   core.node[Core::nodeMAPMax].value);
		mapFailCount = 0;
	}

 	//core.controls[Core::valueTempEngine] = analogRead(PIN_ENGINETEMP);
	core.controls[Core::valueInjectionThresholdVoltage] = analogRead(PIN_INJECTION_THRESHOLD_VOLTAGE);  	
	core.controls[Core::valueBatteryVoltage] = analogRead(PIN_BATTERY_VOLTAGE);  

	// Log adjuster accuracy for debugging
	core.controls[Core::valueQAJitter] = adjuster.accuracy; 
	core.node[Core::nodeQADebugJitter].value = adjuster.setPointMax - adjuster.setPointMin;
	adjuster.setPointMax=0;
	adjuster.setPointMin=2000;
	
	// Calculate advance based on avarage 
	if (core.controls[Core::valueEngineTimingDiff] && core.controls[Core::valueEngineRPMDurationBetweenTeeths] ) {
		// 2018 .. 3544    t=49  r=411
		unsigned int timerTicksPerDegree = core.controls[Core::valueEngineRPMDurationBetweenTeeths] / (360/NUMBER_OF_CYLINDERS);
		// Calculate relative position, using *10 fixed point for internal presentation 
		unsigned int advanceRelative = ((unsigned long)core.controls[Core::valueEngineTimingDiff]*(unsigned long)10)/(unsigned long)timerTicksPerDegree;
		// Apply BTDC Mark correction
		int advanceAbsolute = BTDC_MARK-advanceRelative;

		core.controls[Core::valueEngineTimingActual] = (int)advanceAbsolute;
	}

  	//doBoostControl();
	//doTimingControl();
 	//doUserStuff();

	// Enable glow plugs
	digitalWrite(PIN_GLOW_RELAY,core.controls[Core::valueOutputGlow]?HIGH:LOW);
	calls--;
}

/*
	Calculates injection pump advance, controls N108 duty cycle which affects pump internal pressure and thus advance piston position
	- Operating modes 
		0 = off
		1 = open loop control, duty cycle is read from table (injFuel x RPM)
		2 = closed loop control, PID control from target table
*/
void doTimingControl() {
	if (core.controls[Core::valueOutputTestMode] != 0 || core.node[Core::nodeTimingMethod].value == 0) {	
		analogWrite(PIN_TIMING_PWMCONTROL,core.controls[Core::valueEngineTimingDutyCycle]);	

		return;
	}

	if (core.node[Core::nodeTimingMethod].value == 2) {
		/* closed loop mode */
		

		core.controls[Core::valueEngineTimingTarget] = mapLookUp(core.maps[Core::mapIdxClosedLoopAdvanceMap],
												core.controls[Core::valueRPM8bit],
												core.controls[Core::valueFuelAmount8bit]);	

		static int timingKd = 0, timingSpeed=16, timingMin=N108_MIN_DUTY_CYCLE, timingMax=N108_MAX_DUTY_CYCLE;
		static PID timingPidControl(
			(int*)&core.node[Core::nodeTimingKp].value,
			(int*)&core.node[Core::nodeTimingKi].value,
			&timingKd,
			&timingMin,
			&timingMax,
			&timingSpeed,
			NULL,
			(int*)&core.controls[Core::valueEngineTimingActual], // Current timing
			(int*)&core.controls[Core::valueTimingPIDAmount] // output DC
		);
		
		timingPidControl.setPosition(core.controls[Core::valueEngineTimingTarget]);
		timingPidControl.calculate();

		core.controls[Core::valueEngineTimingDutyCycle] = core.controls[Core::valueTimingPIDAmount];

		// for ui
		core.node[Core::nodeEngineTiming].value = core.controls[Core::valueEngineTimingTarget];
	} else {
		/* open loop mode */
		core.controls[Core::valueEngineTimingDutyCycle] = mapLookUp(core.maps[Core::mapIdxOpenLoopAdvanceMap],
													core.controls[Core::valueRPM8bit],
													core.controls[Core::valueFuelAmount8bit]);			
	}

	analogWrite(PIN_TIMING_PWMCONTROL,core.controls[Core::valueEngineTimingDutyCycle]);	
}

/*
	Calculates boost control parameters, controls N75 duty cycle which actuates Turbo VNT vanes
	- Operating modes 
		0 = off
		1 = open loop control, N75 duty cycle is read from table (injFuel x RPM)
		2 = closed loop control, N75 duty cycle is read from table + small amount of PID is applied
*/

void doBoostControl() {
/*	static SimpleRotatingActuator SRA((int*)&core.node[Core::nodeSRAMinPos].value,
		(int*)&core.node[Core::nodeSRAMaxPos].value,
		(int*)&core.node[Core::nodeSRAInverseOperation].value);
*/
	if (core.controls[Core::valueOutputTestMode] != 0 || core.node[Core::nodeBoostAdjusting].value == 0) {	
//		SRA.setPosition(core.controls[Core::valueN75DutyCycle]);
//		SRA.calculate();
		analogWrite(PIN_N75,core.controls[Core::valueN75DutyCycle]);
		return;
	}
	static int boostMin=0, boostMax=255;
	static PID boostPidControl(
			(int*)&core.node[Core::nodeBoostKp].value,
			(int*)&core.node[Core::nodeBoostKi].value,
			(int*)&core.node[Core::nodeBoostKd].value,
			&boostMin,
			&boostMax,
			(int*)&core.node[Core::nodeBoostSpeed].value,
			(int*)&core.node[Core::nodeBoostBias].value,						
			(int*)&core.controls[Core::valueBoostPressure], // input
			(int*)&core.controls[Core::valueBoostPIDCorrection] // output
		);


	/* Open loop control */


	/* Look up for requested boost level (RPM x TPS) */
	core.controls[Core::valueBoostTarget] = mapLookUp(core.maps[Core::mapIdxTurboTargetPressureMap],
												core.controls[Core::valueRPM8bit],
												core.controls[Core::valueTPSActual]);	

	/* TODO: add check for overboost situations and set up DTC and safe mode */
	
	core.controls[Core::valueN75DutyCycleBaseForPid] =  mapLookUp(core.maps[Core::mapIdxTurboControlMap],
												core.controls[Core::valueRPM8bit],
												core.controls[Core::valueFuelAmount8bit]);	


	// for ui
	core.node[Core::nodePressure].value = core.controls[Core::valueBoostTarget];

	long int idx,tmp1,tmp2;
	unsigned char out;
	core.controls[Core::valueBoostActuatorClipReason] = BOOST_OK;

	switch (core.node[Core::nodeBoostAdjusting].value) {
		case 2:
			/* Closed loop control 
			- Duty cycle table predefines max opening of vanes
			- Small amount of pid correction is subtracted of max opening position
			*/

			// Saturate PID control min/max according to fixed duty cycle table	
			// boostMin = 0;//-core.node[Core::nodeBoostPIDRange].value;
			// Fixed DC is the maximum opening
			// boostMax = core.node[Core::nodeBoostPIDRange].value;
	
			boostPidControl.setPosition(core.controls[Core::valueBoostTarget]);
			boostPidControl.calculate();

		 	core.controls[Core::valueN75DutyCycleBaseForPid] = core.controls[Core::valueN75DutyCycle];

		 	if (core.controls[Core::valueBoostPIDCorrection]>core.controls[Core::valueN75DutyCycleBaseForPid]) {
		 		core.controls[Core::valueN75DutyCycle] = core.controls[Core::valueN75DutyCycleBaseForPid];
		 		core.controls[Core::valueBoostActuatorClipReason] = BOOST_MAX_CLIP;
		 	} else if (core.controls[Core::valueBoostPIDCorrection]<(core.controls[Core::valueN75DutyCycleBaseForPid]-core.node[Core::nodeBoostPIDRange].value)) {
					core.controls[Core::valueN75DutyCycle] = core.controls[Core::valueN75DutyCycleBaseForPid]-core.node[Core::nodeBoostPIDRange].value;
		 			core.controls[Core::valueBoostActuatorClipReason] = BOOST_MIN_CLIP;				
		 		} else {
					core.controls[Core::valueN75DutyCycle] = core.controls[Core::valueBoostPIDCorrection];
				}

			break;
		case 3:		
			// Classic VNT-LDA style 
			//- Vanes max opening is limited by duty cycle table
			//- Vanes opening method is predefined curve of currentPressure/targetPressure, which is clipped by max opening values
			//
		    idx = (((long int)core.controls[Core::valueBoostPressure]*100/(long int)core.controls[Core::valueBoostTarget])*256)/100;

			if (idx<0) {
				idx = 0;
			}
			if (idx>255) {
				idx = 255;
			}
		   	out = mapLookUp(core.maps[Core::mapIdxActuatorTension],idx,0); 
		   	/*
		   	if (out > core.controls[Core::valueN75DutyCycle]) {
				core.controls[Core::valueN75DutyCycle] = out;
		   	} 
		   	*/
		  	core.controls[Core::valueN75DutyCycleBaseForPid] = core.controls[Core::valueN75DutyCycle];
/*
		 	if (out>core.controls[Core::valueN75DutyCycleBaseForPid]) {
		 		core.controls[Core::valueN75DutyCycle] = core.controls[Core::valueN75DutyCycleBaseForPid];
		 		core.controls[Core::valueBoostActuatorClipReason] = BOOST_MAX_CLIP;		 		
		 	} else if (out<(core.controls[Core::valueN75DutyCycleBaseForPid]-core.node[Core::nodeBoostPIDRange].value)) {
					core.controls[Core::valueN75DutyCycle] = core.controls[Core::valueN75DutyCycleBaseForPid]-core.node[Core::nodeBoostPIDRange].value;
		 			core.controls[Core::valueBoostActuatorClipReason] = BOOST_MIN_CLIP;				
		 		} else {
					core.controls[Core::valueN75DutyCycle] = out;
				}
			*/
			core.controls[Core::valueN75DutyCycle] = out;
			break;
		case 4:
			// Classic VNT-LDA style, with a bit different approach 
			//- Vanes max opening is limited by duty cycle table
			//- Vanes opening method is predefined curve of currentPressure/targetPressure, curve is scaled between
			//  0<-->max. opening, instead of clipping
			//
		    idx = (((long int)core.controls[Core::valueBoostPressure]*100/(long int)core.controls[Core::valueBoostTarget])*256)/100;

			if (idx<0) {
				idx = 0;
			}
			if (idx>255) {
				idx = 255;
			}
			idx = idx;

		   	out = 255-mapLookUp(core.maps[Core::mapIdxActuatorTension],idx,0); 
		   	
		   	// scale value
			int joo = map(out,0,255,0,core.controls[Core::valueN75DutyCycleBaseForPid]);   		   	
			core.controls[Core::valueN75DutyCycle] = 255-joo;
			break;
	}

	static unsigned int idleRunCycleCount = 0;


	/* Open vanes when idling */
	if (core.controls[Core::valueTPSActual] == 0 && core.controls[Core::valueEngineRPMFiltered]<1200) {
		idleRunCycleCount++;
		/* quick test: VNT autoclean */ 
		if (VNT_OPEN_DELAY_ON_IDLE != 0 && idleRunCycleCount > VNT_OPEN_DELAY_ON_IDLE ) {
			core.controls[Core::valueN75DutyCycle] = 255;
		}
		if (VNT_OPEN_DELAY_ON_IDLE != 0 && idleRunCycleCount > VNT_OPEN_DELAY_ON_IDLE*10 ) {
			core.controls[Core::valueN75DutyCycle] = 0;
		}
		if (VNT_OPEN_DELAY_ON_IDLE != 0 && idleRunCycleCount > VNT_OPEN_DELAY_ON_IDLE*11 ) {
			idleRunCycleCount = 0;
		}

	} 
	if (core.controls[Core::valueEngineRPMFiltered]>1200 || core.controls[Core::valueTPSActual] > 0) {
		idleRunCycleCount = 0;
	}

/*
	SRA.calculate();	
	SRA.setPosition(core.controls[Core::valueN75DutyCycle]);
*/

	analogWrite(PIN_N75,core.controls[Core::valueN75DutyCycle]);

}

void doUserStuff() {
	/* Add your code to here */
}

unsigned char safetyStop = 0;
/*
	Handles fast sensors input / output 
	- Quantity Adjuster Fuel amount
	- Idling PID control
*/
void refreshFastSensors() {
	//static     PID(int *p,int *i,int *d,int *minOutput,int *maxOutput,int *speed,int *input,int *output);
	static int idleMinFuel,idleMaxFuel;
	static PID idlePidControl(
			(int*)&core.node[Core::nodeIdleKp].value,
			(int*)&core.node[Core::nodeIdleKi].value,
			(int*)&core.node[Core::nodeIdleKd].value,
			(int*)&idleMinFuel, //(int*)&core.node[Core::nodeIdleMinFuel].value,
			(int*)&idleMaxFuel, // (int*)&core.node[Core::nodeIdleMaxFuel].value,
			(int*)&core.node[Core::nodeIdlePIDSpeed].value,
			(int*)&core.node[Core::nodeIdlePIDBias].value,			
			(int*)&core.controls[Core::valueEngineRPMFiltered],
			(int*)&core.controls[Core::valueIdlePIDCorrection]
		);

	int fuelAmount;
	static int idleBaseFuelAmount;
	static unsigned int runCount;

	if (core.controls[Core::valueEngineRPMFiltered] == 0 && core.node[Core::nodeFuelCutAtStall].value == 1) {
		if (runCount > FAST_START_DELAY)
			safetyStop = true;
		if (runCount<255)
			runCount++;
		core.controls[Core::valueRunMode]=0; // Stopped  		
	}

	if (safetyStop) {
		// When engine is not running, do not inject anything to prevent runaway situation when RPM signal is missing and engine is running
		fuelAmount = 0;
		core.controls[Core::valueRunMode]=0; // Stopped  
	} else { 
		safetyStop = false;
		unsigned int rpm;

		cli();
		rpm = core.controls[Core::valueEngineRPMFiltered];
		sei();
		int rpmCorrected = mapValues(rpm,0,core.node[Core::nodeControlMapScaleRPM].value);   

		core.controls[Core::valueRPM8bit] = rpmCorrected;
/*
		if (rpm<350) {
			core.controls[Core::valueRunMode] = 1; // starting 
		} else {
			if (core.controls[Core::valueRunMode]<98)
				core.controls[Core::valueRunMode]++;
		}*/

		if (core.controls[Core::valueRunMode] == 0) {
			// Saturate PID control before start
			core.controls[Core::valueIdlePIDCorrection] = core.node[Core::nodeIdleMinFuel].value*2;
		}
	
		// Use or switch to PID idle control if gas is not pressed and RPM below threshold. When PID is activated, it is switched of only when gas is pressed (no RPM threshold check)
		if ((/*core.controls[Core::valueRunMode] == 2 ||*/ rpm<(core.node[Core::nodeIdleSpeedTarget].value+400)) && core.controls[Core::valueTPSActual] == 0) {
			// Use idle & cold start map.

			int rpmIdle = mapValues(rpm,0,1024);

			if (core.node[Core::nodeIdleAdjusting].value && rpm>(core.node[Core::nodeIdleSpeedTarget].value-220)) {
				// TODO fix valueRunMode!!
				fuelAmount = mapLookUp10bit(core.maps[Core::mapIdxIdleMap],
										rpmIdle,
										core.controls[Core::valueTPSActual]);	
				idleMinFuel = core.node[Core::nodeIdleMinFuel].value; // JOOSE
				idleMaxFuel = core.node[Core::nodeIdleMaxFuel].value;
				core.controls[Core::valueRunMode] = 2; // pid idle 
				idlePidControl.setPosition(core.node[Core::nodeIdleSpeedTarget].value);
				idlePidControl.calculate();
				fuelAmount += core.controls[Core::valueIdlePIDCorrection];

			} else {
				fuelAmount = mapLookUp10bit(core.maps[Core::mapIdxIdleMap],
										rpmIdle,
										core.controls[Core::valueTPSActual]);
				idleBaseFuelAmount = fuelAmount;
				core.controls[Core::valueRunMode] = 1; // starting 

				// Sets up the integral part of PID calculation
				// idlePidControl.setPositionHint(fuelAmount);
			}					 
										
		} else {
			// Return upscaled interpolated value of range 0..1023 according to fuel map data (0..255)
			core.controls[Core::valueRunMode]=100; // Engine running 

			// Base injection quantity 
			// TODO::: generate fuelMapLookup which uses upscaled fuel trim values before final interpolation

			core.controls[Core::valueFuelBaseAmount] = mapLookUp10bit(core.maps[Core::mapIdxFuelMap],
													rpmCorrected,
													core.controls[Core::valueTPSActual]);

			/* Limit enrichment amount to by REQUESTED boost level (turbo control table), not the actual */
			unsigned char boostRequlated = core.controls[Core::valueBoostPressure];

			if (boostRequlated > core.controls[Core::valueBoostTarget])
				boostRequlated = core.controls[Core::valueBoostTarget];

			// Enrichment based on boost, amount is TPS% * fuel enrichment map value to smooth apply of enrichment
			core.controls[Core::valueFuelEnrichmentAmount] = 
										((unsigned long)(mapLookUp10bit(core.maps[Core::mapIdxBoostMap],
																		rpmCorrected,
																		boostRequlated)) 
										*(unsigned  long)(core.controls[Core::valueTPSActual])
										/ (unsigned long)256);

			// Smoke limit map (TODO: add wide band lambda support and auto-map feature)
			core.controls[Core::valueFuelLimitAmount] = mapLookUp10bit(core.maps[Core::mapIdxMaxFuelMap],
										rpmCorrected,
										core.controls[Core::valueBoostPressure]);	

			fuelAmount = core.controls[Core::valueFuelBaseAmount];

			if (fuelAmount) {
				// Enrichment amount is TPS% * fuel enrichment map value to smooth apply of enrichment
				fuelAmount += core.controls[Core::valueFuelEnrichmentAmount];
			}

			if (fuelAmount>core.controls[Core::valueFuelLimitAmount])	
				fuelAmount = core.controls[Core::valueFuelLimitAmount];		

		}
	}
	core.controls[Core::valueFuelAmount] = fuelAmount;	
	core.controls[Core::valueFuelAmount8bit] = fuelAmount/4;
	adjuster.setPosition(fuelAmount);
}

volatile unsigned long injectionBegin;
volatile char unplausibleRpm = 0;
volatile char unplausibleNeedleLiftSensor = 0;
volatile unsigned long rpmMin;
volatile unsigned long rpmMax;
volatile int skipNextTeeth = 4;
volatile unsigned int rpmDuration;
volatile char probeEnabled = 0;
volatile unsigned char skipInjectionTrigger;

void rpmTrigger() { 
 	static unsigned long int previosTeethDuration = 0;  
 	unsigned int rpmCalculated;     
    cli();
 	if (skipNextTeeth > 0) {
  		skipNextTeeth--;
  		sei();
  		return;
  	}     
  	rpmDuration = RPMTIMER_COUNTER;
  	rpmTimerEnable();

  	if (rpmDuration == 0xffff)
  		rpmCalculated=0;
  	//sei();

  	/* combine duration from last two teeths */ 
  	unsigned long int t=(rpmDuration+previosTeethDuration)/2;
	rpmCalculated = RPMTIMER_DURATION_TO_RPM(t);

	// Log unplausible signals. 
	if (rpmDuration<RPMTIMER_MIN_DURATON) {
		skipNextTeeth=1; // do not trust next teeth, in case of error
 		unplausibleRpm = 1;
 		sei();
		return;
	} 
	previosTeethDuration = rpmDuration;	
	core.node[Core::nodeEngineRPM].value = rpmCalculated;
	
//  	cli();
 	core.controls[Core::valueEngineRPM] = rpmCalculated;
 //	sei();

 	if (core.controls[Core::valueEngineRPMDurationBetweenTeeths]<32768) {
		core.controls[Core::valueEngineRPMDurationBetweenTeeths] = (core.controls[Core::valueEngineRPMDurationBetweenTeeths]+rpmDuration)/2; 
 	} else {
		core.controls[Core::valueEngineRPMDurationBetweenTeeths] = rpmDuration;

 	}
 	
	core.controls[Core::valueEngineRPMDurationBetweenTeeths] = (core.controls[Core::valueEngineRPMDurationBetweenTeeths]+rpmDuration)/2; 


	// TODO, combine two teeths to one value
   	core.controls[Core::valueCurrentTeeth]++;
   	if (core.controls[Core::valueCurrentTeeth]>=NUMBER_OF_CYLINDERS)
   		core.controls[Core::valueCurrentTeeth] = 0;
	rpmStore[core.controls[Core::valueCurrentTeeth]] = rpmCalculated;
	
	unsigned long int avg = rpmCalculated;

	// Always force per teeth calculation when starting to maintain better engine control, when critical point (700rpm) is reached
	// RPM can be avaraged

	if (avg>600) {
		unsigned char teeth=rpmStore[core.controls[Core::valueCurrentTeeth]];

		switch (core.node[Core::nodeRPMDSP].value) {
			case 2: // rpm based on last two teeths
				avg = rpmStore[teeth];
				if (teeth == 0) {
					teeth = (NUMBER_OF_CYLINDERS*2)-1;
				} else {
					teeth--;
				}
				avg += rpmStore[teeth];
				avg = avg / 2;
			break;
			case 1: // return avarage rpm per last full turn
				avg = 0;
				for (unsigned char i=0;i<NUMBER_OF_CYLINDERS;i++) {
	  				avg += rpmStore[i];
				}

				avg = avg/NUMBER_OF_CYLINDERS;
			break;
		}
	}
	//cli();
	core.controls[Core::valueEngineRPMFiltered] = avg;
//	sei();
	if (avg>rpmMax) {
		rpmMax = avg;
		core.controls[Core::valueEngineRPMMax] = avg;
	} else if (avg<rpmMin) {
		rpmMin = avg;
	 	core.controls[Core::valueEngineRPMMin] = avg;
	}
	
	if (core.node[Core::nodeProbeSignalOutput].value == 0) {
		// probe.hit();
		digitalWrite(PIN_RPM_OUT, HIGH); 
		probeEnabled = 1;
	}
	attachInterrupt(1, needleTrigger, RISING);
	// Skip first interrupt (interrupt pending flag is not cleared by attachIntterrupt?? )
	skipInjectionTrigger = 2; 
	injectionBegin = 0;

	sei();
	safetyStop=false;
}

void needleTrigger() { 
	cli();
	skipInjectionTrigger--;

	if (skipInjectionTrigger) {
		sei();
		return;
	}
	injectionBegin = RPMTIMER_COUNTER; 
	detachInterrupt(1); // disable interrupt handler for a while

	if (core.controls[Core::valueEngineTimingDiff]<32768) {
		core.controls[Core::valueEngineTimingDiff] = (core.controls[Core::valueEngineTimingDiff]+injectionBegin)/2; 
	} else {
		core.controls[Core::valueEngineTimingDiff] = injectionBegin;
	}
	core.node[Core::nodeHeartBeat].value++;

	sei();
}

prog_uchar main_pressAKeyString[] PROGMEM = " bytes free.\r\n\r\nPress a key for configuration interface ...";

void setup() {
	pinMode(13, OUTPUT);
	pinMode(PIN_QA_FEEDBACK, INPUT);
	pinMode(PIN_TPS, INPUT_PULLUP);
	pinMode(PIN_TPS2, INPUT_PULLUP);
	pinMode(PIN_TPS_IDLE, INPUT_PULLUP);
	pinMode(PIN_INJECTION_THRESHOLD_VOLTAGE,INPUT);
	pinMode(PIN_BATTERY_VOLTAGE,INPUT);	
	pinMode(PIN_RPM,INPUT_PULLUP);
	pinMode(PIN_NEEDLELIFTSENSOR,INPUT_PULLUP);
	pinMode(PIN_QA_FEEDBACK,INPUT_PULLUP);
	pinMode(PIN_MAP,INPUT_PULLUP);
	pinMode(PIN_ENGINETEMP,INPUT_PULLUP);
	pinMode(PIN_AIRTEMP,INPUT_PULLUP);
	pinMode(PIN_FUELTEMP,INPUT_PULLUP);
	pinMode(PIN_QA_PWMCONTROL, OUTPUT);
	pinMode(PIN_GLOW_RELAY, OUTPUT);
	pinMode(PIN_RPM_OUT, OUTPUT);
	pinMode(PIN_TIMING_PWMCONTROL,OUTPUT);
	pinMode(PIN_N75,OUTPUT);
	pinMode(PIN_RPM_OUT,OUTPUT);

	pinMode(PIN_SRA_FEEDBACK,INPUT);

	pinMode(PIN_SRA_OUTPUT1,OUTPUT);
	pinMode(PIN_SRA_OUTPUT2,OUTPUT);
	pinMode(PIN_SRA_ENABLE,OUTPUT);

	tacho.init();
	/*
	// Setting ADC prescaler to /16 (http://www.arduino.cc/cgi-bin/yabb2/YaBB.pl?num=1208715493/11 )
	sbi(ADCSRA,ADPS2) ;
	cbi(ADCSRA,ADPS1) ;
	cbi(ADCSRA,ADPS0) ;
	*/
	Serial.begin(115200);
	ansiClearScreen();
	
	interruptHandlerArray[2].handler=refreshFastSensors; 
	interruptHandlerArray[2].divider=4;
	interruptHandlerArray[3].handler=refreshSlowSensors; 
	interruptHandlerArray[3].divider=8;
	
	attachInterrupt(0, rpmTrigger, RISING);  // Interrupt 0 -- PIN2 -- LM1815 gated output 
	attachInterrupt(1, needleTrigger, RISING);  // From voltage comparator 
	
	// Timer info - see: http://sobisource.com/?p=195

	// Timer 4 base freq -- timing solenoid 0x03 - 500Hz, 0x04 - 125Hz
	TCCR4B = (TCCR4B & 0xF8) | 0x04; // 490Hz was 0x04
	
	// Reference pulse for testing RPM signalling
	//TCCR4B = (TCCR4B & 0xF8) | 0x05;
	//analogWrite(PIN_REFPULSE,128); // TCCR4B - timer 4
	
	Serial.print("BOOT: ");
	if (core.load()) {
		char *msg="OK";
		confeditor.setSystemStatusMessage(msg);
	} else {
		char *msg="Conf. error";
		Serial.print(msg);
		confeditor.setSystemStatusMessage(msg);
	}

	setupQATimers();
	rpmTimerSetup(); 
	rpmTimerEnable();

	Serial.print("... ");	
	Serial.print(freeMemory());
	Serial.print(fetchFromFlash(main_pressAKeyString));
}

void setupQATimers() {
	// 2000 = 500ticks -// was 500
	Timer3.initialize(4000); // in microseconds, also sets PWM base frequency for "mega" pins 5,2,3 // 2000 = old default
	adjuster.initialize();

	Timer3.attachInterrupt(mainInterruptHandler,0);
	interruptHandlerArray[0].handler=refreshQuantityAdjuster; 
	interruptHandlerArray[0].divider=2; 
}

int i;
#define BUFFER_SIZE 64
char buffer[BUFFER_SIZE];
char lastKey;
long time;

boolean confChanged = 0;
char loopCount = 0;


void loop() {


	boolean printValues = 0;
	boolean ignoreSleep = false;
//	static unsigned char testMapX = 0;
//	static unsigned char testMapY = 0;

	doBoostControl();
	doTimingControl();

	//tacho.setRpm(core.controls[Core::valueEngineRPM]);
	/*if (core.controls[Core::valueEngineRPM]>350) {
		SimpleRotatingActuator::enable();
	} else {
		SimpleRotatingActuator::disable();
	}*/

/*	testMapX++;
	if (testMapX%4)
		testMapY++;
	mapLookUp(core.maps[Core::mapIdxTestMap],testMapX,testMapY);
*/
	if (loopCount == 60) {
		// Log some "short term" differencies
		core.controls[Core::valueEngineRPMJitter]=rpmMax-rpmMin;
		rpmMin = core.controls[Core::valueEngineRPM];
		rpmMax = core.controls[Core::valueEngineRPM];
		loopCount=0;    
	}   
	loopCount++;
	
	lastKey = 0;
	// Read incoming command from serial interface (USB)
	if (Serial.available()>0) {
		int i=0;    
		char c = Serial.read();
		if (c == 27) {
			c = Serial.read();
			if (c == '[') {
				c = Serial.read();
				// Cursor to "vi" nodes
				switch (c) { 
					case 'A':
						lastKey = KEY_UP;
						break;
					case 'B':
						lastKey = KEY_DOWN;
						break;
					case 'C':
						lastKey = KEY_RIGHT;
						break;
					case 'D':
						lastKey = KEY_LEFT;
						break;
				}
			}
			confeditor.handleInput(lastKey);
		} else {
			// Special commands
			switch (c) {
/*                case 2: // STX
					ignoreSleep = 1;
					i=0;
					int node;
					int val;
					node=-1;
					val=-1;
					while (1) {
						if (Serial.available()) {
							c = Serial.read();
							if (c==3) // ETX
								break;
							if (c==':') {
								buffer[i]=0;           
								node = atoi(buffer);
								i=0;
							} else if (c==';') {
								buffer[i]=0;           
								val = atoi(buffer);
								i=0;
							} else if (i<BUFFER_SIZE-1) {
								buffer[i]=c;
								i++;
							}
						}
					}
					buffer[i]=0;
					int cksum;
					cksum=atoi(buffer);    
									  
					if (cksum == 66 && node == 255) {
						core.save();
					}       
					if (cksum == 66 && node > 0 && node <= Core::KEY_MAX) {
						core.setCurrentNode(node);
						core.setValue(val);
						confChanged=1;
						//if (node == Core::nodeQAPWMBase || node == Core::nodeQATimerInterval)
						  //§  setupQATimers();
					}
					break;
*/
				default:
					confeditor.handleInput(c);
			}
		}
	}

	/*
	if (printValues) {
		// print one packet of internal status (RPM,TPS, QA SetPoint, QA Position)
		Serial.write(2); // stx
		Serial.print("0,"); // Type=0 packet
		Serial.print(core.controls[Core::valueEngineRPM],DEC); // RPM
		Serial.write(',');
		Serial.print(core.controls[Core::valueTPSActual],DEC);
		Serial.write(',');
		Serial.print(core.controls[Core::valueQAfeedbackSetpoint],DEC);
		Serial.write(',');
		Serial.print(core.controls[Core::valueQAfeedbackRaw],DEC);
		Serial.write(',');
		Serial.print(core.controls[Core::valueQAPWMActual],DEC);
		Serial.write(',');
		Serial.print((int)adjuster.p,DEC); 
		Serial.write(',');
		Serial.print((int)adjuster.i,DEC); 
		Serial.write(',');
		Serial.print((int)adjuster.d,DEC);        
		Serial.write(',');
		Serial.print(core.controls[Core::valueEngineTimingActual],DEC);    
		Serial.write(3); // etx;
		printValues = false;
	}*/
	
	confeditor.refresh();

	if (!ignoreSleep)
		delay(1000/60); 
	
	// Handle errors (generated by interrupt service)
	if (unplausibleRpm) {
	 	dtc.setError(DTC_RPM_UNPLAUSIBLE_SIGNAL);
		unplausibleRpm = 0;
		// TODO: switch over backup signal (needlelift) after nnn failing signals  
	}
	if (unplausibleNeedleLiftSensor) {
		//dtc.setError(DTC_NEEDLESENSOR_UNPLAUSIBLE_SIGNAL);
		unplausibleNeedleLiftSensor = 0;
	}

	// Saves any DTC codes generated .. 
	dtc.save();

	if (probeEnabled) {
		digitalWrite(PIN_RPM_OUT, LOW); 
		probeEnabled = 0;
	}

}


/*
 Useful data :-)
 
 http://arduino.cc/en/Hacking/PinMapping2560
 
 Phase correct mode?
 
 Most Arduino boards have two external interrupts: numbers 0 (on digital pin 2) and 1 (on digital pin 3). 
 The Arduino Mega has an additional four: numbers 2 (pin 21), 3 (pin 20), 4 (pin 19), and 5 (pin 18).
 
 Mega timers: http://sobisource.com/?p=195
 
 */

