;Tutorial 6_7
;A2D with LCD display and 24C256
;Nigel Goodwin 2002

	LIST	p=16F628		;tell assembler what chip we are using
	include "P16F628.inc"		;include the defaults for the chip
	ERRORLEVEL	0,	-302	;suppress bank selection messages
	__config 0x3D18			;sets the configuration settings (oscillator type etc.)




		cblock	0x20			;start of general purpose registers
			count			;used in looping routines
			count1			;used in delay routine
			counta			;used in delay routine
			countb			;used in delay routine
			LoX
			Bit_Cntr
			Cmd_Byte
			Dev_Byte
			Timer_H
			Flags
			Flags2
			tmp1			;temporary storage
			tmp2
			tmp3

			templcd			;temp store for 4 bit mode
			templcd2
			lcdtmp
	
			Adr_Lo			; EEPROM memory address to be accessed
			Adr_Hi
			Adr2_Lo			; EEPROM memory read address pointer
			Adr2_Hi
			Chip_Adr
			DAT_VAL
			_N
			N
			InputByte		; byte read from EEPROM is stored in this register
			Data_Page		; EEPROM page, 0-7
			OutputByte		; used for holding byte to be output to EEPROM
			I2Cflags		; flag bit register
			buf			; buffer for sequential write and read
			buf1
			buf2
			buf3

        		NumL			;Binary inputs for decimal convert routine
	        	NumH	

        		TenK			;Decimal outputs from convert routine
	        	Thou	
        		Hund	
	        	Tens	
        		Ones

			temp_w			;used in interrupt routine
			temp_stat

			second_cnt
			time_cnt
			time_var
			time_edit
			temp_table
			EEPROM_Addr
			n1L
			n1H
			n2L
			n2H
			select
		endc

LCD_PORT	Equ	PORTA
LCD_TRIS	Equ	TRISA
LCD_RS		Equ	0x04			;LCD handshake lines
LCD_RW		Equ	0x06
LCD_E		Equ	0x07

I2C_PORT	Equ	PORTB
I2C_TRIS	Equ	TRISB

ErrFlag		Equ	0x00
StartFlag	Equ	0x01			;flags used for received bit
Z_		Equ	0x02

Stop		Equ	0x04			;set by keys
Run		Equ	0x05			;set for running
Lo		Equ	0x06			;set for normal addressing
Hi		Equ	0x07			;set for I2C extended addressing


#define		SDA 	7
#define		SCL	6

SW1		Equ	0			;set constants for the switches
SW2		Equ	1
SW3		Equ	2
SW4		Equ	3

Chip_Read	Equ	0x01
EEPROM_Adr	Equ	0xA0
A2D_Adr		Equ	0x90

Buf_Size	Equ	0x04			;buffer 4 bytes long

		org	0x0000
		goto	Start

            	org     0x0004			;interrupt service routine
		movwf 	temp_w
		swapf 	STATUS,		W
		bcf 	STATUS,		RP0
		movwf 	temp_stat
		btfsc 	INTCON,		T0IF
		goto 	Timer
Continue	swapf 	temp_stat,	W
		movwf 	STATUS
		swapf 	temp_w,		F
		swapf 	temp_w,		W
            	retfie

;TABLES - moved to start of page to avoid paging problems,
;a table must not cross a 256 byte boundary.
HEX_Table  	addwf   PCL       , f
            	retlw   0x30
            	retlw   0x31
            	retlw   0x32
            	retlw   0x33
            	retlw   0x34
            	retlw   0x35
            	retlw   0x36
            	retlw   0x37
            	retlw   0x38
            	retlw   0x39
            	retlw   0x41
            	retlw   0x42
            	retlw   0x43
            	retlw   0x44
            	retlw   0x45
            	retlw   0x46


Time1		andlw	0x07			;make sure only 8 possible values
		movwf	time_edit		;so we stay in table
		addwf   PCL       , f
            	retlw   D'1'			;100mS
            	retlw   D'2'
            	retlw   D'5'
            	retlw   D'10'			;1 second
            	retlw   D'20'
            	retlw   D'50'
            	retlw   D'100'
            	retlw   D'200'			;20 seconds


;end of tables

Timer:
		bcf 	INTCON,		T0IF
		movlw 	-D'118' 
		movwf 	TMR0
		decfsz 	second_cnt,	F
		goto	Continue
		movlw 	D'13'
		movwf 	second_cnt
		decfsz 	time_cnt,	F
		goto	Continue
		movf	time_var,	w
		movwf 	time_cnt
		bsf	Flags,		Run
		goto	Continue

Start		movlw	0x07
		movwf	CMCON			;turn comparators off (make it like a 16F84)

Initialise	clrf	count
		clrf	PORTA
		clrf	PORTB	
		bsf	Flags,		Stop


SetPorts	bsf 	STATUS,		RP0	;select bank 1
		movlw	0x00			;make all LCD pins outputs
		movwf	LCD_TRIS
		movlw	b'11111111'		;make all I2C port pins inputs
		movwf	I2C_TRIS
		bcf 	STATUS,		RP0	;select bank 0

		call	LCD_Init		;setup LCD module
		movlw	0x00
		movwf	Adr_Hi
		movwf	Adr_Lo			;zero memory counters
		movwf	Adr2_Hi
		movwf	Adr2_Lo			;zero memory counters
		movwf	EEPROM_Addr		;set EEPROM_Addr


Set_Timer	movlw	D'13'
		movwf	second_cnt
		bsf 	STATUS,		RP0 	;switch to Bank 1
		movlw	B'10000101'
		movwf	OPTION_REG		;prescaler /64
		bcf 	STATUS,		RP0 	;switch to Bank 0
		movlw 	-D'126' 		;count up 125 to rollover
		movwf 	TMR0
		movlw 	B'10100000' 		;enable timer interrupt
		movwf 	INTCON

		call	EE_Read
		movwf	time_edit		;set initial time from internal EEPROM
		call	Time1
		movwf	time_var
		movwf 	time_cnt

		movlw	0x02			;select string 2
		call	Print_String
		call	Wait_Key
Repeat		call	Menu
		goto	Repeat

;*************** Main menu ****************

Menu		clrf	select			;select first option
Next_Menu	movf	select,		w
		addlw	0x07			;add offset to select correct string
		call	Print_String
		movlw	0x06			;select string 6
		call	Print_String

No_Key1		call	ChkKeys
		addwf   PCL       , f		;add W to program counter
		goto	No_Key1			;and jump depending on key press (this is no key)
		goto	End_Select		;key1 pressed
		goto	Dec_Select		;key2 pressed
		goto	Inc_Select		;key3 pressed
		goto	End_Select		;key4 pressed

Inc_Select	incf	select,		f
		movf	select,		w
		andlw	0x03
		movwf	select
		goto	Next_Menu

Dec_Select	decf	select,		f
		movf	select,		w
		andlw	0x03
		movwf	select
		goto	Next_Menu

End_Select	movf	select,		w	;move select to W
		addwf   PCL,		f	;add W to program counter
		goto	Main			;and jump depending on value of select
		goto	Show_Data
		goto	Set_Time
		goto	Clr_Memory

;**************** Write Memory Routine ******************

Main
		call	Read_A2D
		movlw	0x0B
		call	LCD_Line1W
		movf	Adr_Hi,		w
		movwf	NumH
		movf	Adr_Lo,		w
		movwf	NumL
		call	Show_16Bit
		call	Show_Buf

		call	Jump1
		btfsc	Flags,		Stop
		goto	Main
		btfss	Flags,		Run
		goto	Main			;only store if Run flag set
		call	Write_Mem
		call	Incr_Mem
		bcf	Flags,		Run	;turn flag back off
		goto	Main

Jump1		call	ChkKeys			;returns 0-4 (0 = no key)
		addwf   PCL       , f		;add W to program counter
		return				;and jump depending on key press (this is no key)
		goto	Set_Time		;key1 pressed
		goto	Run_Stop		;key2 pressed
		goto	Run_Go			;key3 pressed
		goto	Show_Data		;key4 pressed

Run_Stop	bsf	Flags,		Stop
		return

Run_Go		movlw 	D'13'			;reset timer counters
		movwf 	second_cnt
		movf	time_var,	w
		movwf 	time_cnt
		movlw 	-D'126' 		;reset TMR0
		movwf 	TMR0
		movlw 	B'10100000' 		;enable timer interrupt
		movwf 	INTCON
		bcf	Flags,		Run
		bcf	Flags,		Stop
		return

;**************** Read Memory Routine ******************

Show_Data	clrf	Adr_Hi
		clrf	Adr_Lo
		movlw	0x03			;show 'Reading:'
		call	Print_String
Next_Read	call	Read_Mem
		movlw	0x0B
		call	LCD_Line1W
		movf	Adr_Hi,		w
		movwf	NumH
		movf	Adr_Lo,		w
		movwf	NumL
		call	Show_16Bit
		call	Show_Buf
No_Key2		call	ChkKeys
		addwf   PCL       , f		;add W to program counter
		goto	No_Key2			;and jump depending on key press (this is no key)
		goto	End_Read		;key1 pressed
		goto	Dec_Read		;key2 pressed
		goto	Inc_Read		;key3 pressed
		goto	End_Read		;key4 pressed

Inc_Read	call	Incr_Mem
		goto	Next_Read
Dec_Read	call	Decr_Mem
		goto	Next_Read

End_Read	movlw	0x00			;select string 0
		call	Print_String
		return

Zero_Count	clrf	Adr_Hi
		clrf	Adr_Lo
		return

;**************** Set Time Routine ******************

Set_Time	movlw	0x01			;select string 1
		call	Print_String
		movlw	0x06			;select string 6
		call	Print_String
		call	Run_Stop		;stop counting
		call	Zero_Count		;and zero count
Next_Time	call	Show_Time
		call	ChkKeys			;returns 0-4 (0 = no key)
		addwf   PCL       , f		;add W to program counter
		goto	Next_Time		;and jump depending on key press (this is no key)
		goto	End_Time		;key1 pressed
		goto	Decrement		;key2 pressed
		goto	Increment		;key3 pressed
		goto	End_Time		;key4 pressed		

Increment
		incf	time_edit,	f
		movf	time_edit,	w
		call	Time1
		movwf	time_var
		goto	Next_Time

Decrement
		decf	time_edit,	f
		movf	time_edit,	w
		call	Time1
		movwf	time_var
		goto	Next_Time

End_Time	movf	time_edit,	w
		call	EE_Write
		movlw	0x00			;select first string
		call	Print_String
		return

;**************** Clear Memory Routine ******************

Clr_Memory	
		clrf	Adr_Lo			;set address to zero
		clrf	Adr_Hi
		clrf	buf
		clrf	buf1			;and zero buffers
		clrf	buf2
		clrf	buf3
		movlw	0x0B			;print string 11
		call	Print_String
Next_Clr	movf	Adr_Lo,		w
		movwf	n1L
		movf	Adr_Hi,		w
		movwf	n1H
		movwf	n1H
		movlw	D'4'
		movwf	n2L
		clrf	n2H
		call	Add16
		movf	n1L,		w
		movwf	Adr_Lo
		movf	n1H,		w
		movwf	Adr_Hi
		btfsc	Adr_Hi,		7	;if greater than 32K,
		goto	Done_Clr		;we've cleared it all
		call	Write_Mem
		goto	Next_Clr
Done_Clr	movlw	0x0C			;print string 12
		call	Print_String
		call	Wait_Key
		return



Show_Time	movlw	0x09
		call	LCD_Line1W
		movf	time_var,	w
		call	Show_8Bit
		movlw	'0'
		call	LCD_Char
		movlw	'0'
		call	LCD_Char
		movlw	'm'
		call	LCD_Char
		movlw	'S'
		call	LCD_Char
		return

Print_String	movwf	count1			;save string number (0 is first string)
		clrf	count			;set counter register to zero
NextLine	movf	count1,		w
		xorlw	0x00			;is it a zero?
		btfsc	STATUS, 	Z
		goto	NextChar		;if so we've reached required string
		movlw	0x07
		movwf	PCLATH
		movf	count, 		w	;put counter value in W
		call	Strings			;get a character from the text table
		clrf	PCLATH
		xorlw	0x00			;is it a zero?
		btfsc	STATUS, 	Z
		decf	count1,		f
		incf	count, 		f
		goto	NextLine
		
NextChar					;display selected string
		movlw	0x07			;set PCLATH for strings in last page of memory
		movwf	PCLATH
		movf	count, 		w	;put counter value in W
		call	Strings			;get a character from the text table
		clrf	PCLATH			;reset PCLATH
		movwf	temp_table
		xorlw	0xFF			;is it an 0xFF? (command character)
		btfsc	STATUS, 	Z
		goto	Command			;if so, process command
		movf	temp_table,	w
		xorlw	0x00			;is it a zero?
		btfsc	STATUS, 	Z
		return				;exit from routine
		call	LCD_Char
		incf	count, 		f
		goto	NextChar

Command		incf	count,		f	;process command
		movlw	0x07
		movwf	PCLATH
		movf	count, 		w
		call	Strings			;get a command from the table
		clrf	PCLATH
		call	LCD_Cmd			;execute the command
		incf	count,		f	;step to next character
		goto	NextChar

;internal EEPROM routines
						; return EEPROM_Addr in W
EE_Read		bsf 	STATUS,	RP0 		; Bank 1
		movlw 	EEPROM_Addr
		movwf 	EEADR			; Address to read
		bsf 	EECON1,	RD 		; EE Read
		movf 	EEDATA,	W 		; W = EEDATA
		bcf 	STATUS,	RP0 		; Bank 0
		return

EE_Write					; write W to EEPROM_Addr
		bsf 	STATUS,	RP0 		; Bank 1
		bsf	EECON1,	WREN 		; Enable write
		movwf	EEDATA			; set EEPROM data
		movlw	EEPROM_Addr
		movwf	EEADR			; set EEPROM address
		movlw	0x55
		movwf	EECON2 			; Write 55h
		movlw	0xAA
		movwf	EECON2 			; Write AAh
		bsf	EECON1,	WR 		; Set WR bit
						; begin write
		bcf 	STATUS,	RP0 		; Bank 0

		btfss	PIR1,	EEIF		; wait for write to complete.
		goto	$-1
		bcf	PIR1,	EEIF		; and clear the 'write complete' flag
		bsf 	STATUS,	RP0 		; Bank 1
		bcf	EECON1,	WREN 		; Disable write
		bcf 	STATUS,	RP0 		; Bank 0
		return

;end of internal EEPROM routines

;Start of A2D routines

Read_A2D
		movlw	0x00
		movwf	Data_Page		;set A2D page
		movlw	A2D_Adr
		movwf	Chip_Adr		;set for A2D access
		bcf	Flags,		Hi	;do not use extended addresing	
		bcf	Flags,		Lo	;use no addressing at all
		movlw	0x04			;set auto-increment
		call	Write_I2C	
		call	Seq_Read_I2C
		return

Write_Mem
		movlw	0x00
		movwf	Data_Page		;set EEPROM page
		movlw	EEPROM_Adr
		movwf	Chip_Adr		;set for EEPROM access
		bsf	Flags,		Hi	;do use extended addresing	
		bsf	Flags,		Lo	;do use full addressing
		call	Seq_Write_I2C
		return

Read_Mem	movlw	0x00
		movwf	Data_Page		;set EEPROM page
		movlw	EEPROM_Adr
		movwf	Chip_Adr		;set for EEPROM access
		bsf	Flags,		Hi	;do use extended addresing	
		bsf	Flags,		Lo	;do use full addressing
		call	Seq_Read_I2C
		return

Incr_Mem
		movf	Adr_Lo,		w
		movwf	n1L
		movf	Adr_Hi,		w
		movwf	n1H
		movlw	D'4'
		movwf	n2L
		clrf	n2H
		call	Add16
		movf	n1L,		w
		movwf	Adr_Lo
		movf	n1H,		w
		movwf	Adr_Hi
		btfsc	Adr_Hi,		7	;if greater than 32K,
		goto	Decr_Mem		;decrement again
		return

Decr_Mem
		movf	Adr_Lo,		w
		movwf	n1L
		movf	Adr_Hi,		w
		movwf	n1H
		movlw	D'4'
		movwf	n2L
		clrf	n2H
		call	Sub16
		movf	n1L,		w
		movwf	Adr_Lo
		movf	n1H,		w
		movwf	Adr_Hi
		btfsc	Adr_Hi,		7	;if less than zero,
		goto	Incr_Mem		;increment again
		return

;**************** 16 bit add and subtract ****************

Sub16        	comf	n2H,		f	; Take two's complement
             	comf	n2L,		f	; of n2. If zero, n2L
             	incf	n2L,		f
            	btfsc	STATUS,		Z	; overflowed, so carry 
             	incf	n2H,		f	; into n2H.
Add16        	movf	n2L,		w	; Add the LSBs. If carry,
             	addwf	n1L,		f
             	btfsc	STATUS,		C	; increment MSB of sum. 
             	incf	n1H,		f                  
             	movf	n2H,		w	; Add the MSBs  
             	addwf	n1H,		f
             	return 

;**************** 16 bit compare ****************

; Upon return, a code in w will indicate the outcome of the comparison. 
; If n1 > n2 then w = 1. If n1 < n2 then w =2. If n1 = n2 then w = 0. 
Comp16       	bcf 	Flags,		Z_	; Clear aux Z bit.
             	movf 	n2L,		w 	; w = n2L
             	subwf 	n1L,		0	; w = n1L-n2L <Microchip instruction>
             	btfsc 	STATUS,		Z	; Copy Z bit to Z_. 
             	bsf 	Flags,		Z_             
             	movf 	n2H,		w 	; If n1L-n2L underflowed, 
             	btfsc 	STATUS,		C 	; then increment n2H (same
             	goto 	Comp16_cont
             	incf 	n2H,		w 	; as decrementing n1H).
             	btfsc 	STATUS,		Z 	; If n2H overflows, n1 must be < n2
             	retlw 	D'2'			; so return with 2 in w.
Comp16_cont  	subwf 	n1H,		0	; w = n1H-n2H  <Microchip instruction>
             	btfss 	STATUS,		C	; If n1H underflows, n1<n2 retw 2 ; so return with 2 in w. jnb Z_,:cont2 ; By now we're suren1'>=n2. 
             	btfsc 	STATUS,		Z	; If both z and Z_ are set, 
             	retlw 	D'0'			; both subtractions resulted in 0, so
Comp16_cont2 	retlw 	D'1'			; n1=n2. Otherwise n1>n2.

; End of A2D routines

Show_8Bit
		movwf	NumL
		clrf	NumH
		call	Convert
		movf	Hund,	w
		call	LCD_CharD
		movf	Tens,	w
		call	LCD_CharD
		movf	Ones,	w
		call	LCD_CharD
		return

Show_16Bit
		call	Convert
		movf	TenK,	w
		call	LCD_CharD
		movf	Thou,	w
		call	LCD_CharD
		movf	Hund,	w
		call	LCD_CharD
		movf	Tens,	w
		call	LCD_CharD
		movf	Ones,	w
		call	LCD_CharD
		return

Show_Buf
		call	LCD_Line2
		movf	buf,	w
		call	Show_8Bit
		movlw	' '
		call	LCD_Char
		movf	buf1,	w
		call	Show_8Bit
		movlw	' '
		call	LCD_Char
		movf	buf2,	w
		call	Show_8Bit
		movlw	' '
		call	LCD_Char
		movf	buf3,	w
		call	Show_8Bit
		movlw	' '
		call	LCD_Char
		call	LCD_Line1
		return

; Start of key routines

Wait_Key	call	ChkKeys
		xorlw	0x00			;is it a zero?
		btfsc	STATUS, 	Z
		goto	Wait_Key		;wait until key pressed
		return

ChkKeys		btfss	I2C_PORT,	SW1
		goto	Switch1
		btfss	I2C_PORT,	SW2
		goto	Switch2
		btfss	I2C_PORT,	SW3
		goto	Switch3
		btfss	I2C_PORT,	SW4
		goto	Switch4
		retlw	0x00				;return 0, no key pressed

Switch1		call	Delay50				;give switch time to stop bouncing
		btfsc	I2C_PORT,	SW1		;check it's still pressed
		retlw	0x00				;return is not
		

Sw1On		call	Delay50
		btfsc	I2C_PORT,	SW1		;wait until button is released
		retlw	0x01				;return 0x01
		goto	Sw1On	

Switch2		call	Delay50				;give switch time to stop bouncing
		btfsc	I2C_PORT,	SW2		;check it's still pressed
		retlw	0x00				;return is not
		

Sw2On		call	Delay50
		btfsc	I2C_PORT,	SW2		;wait until button is released
		retlw	0x02				;return 0x02
		goto	Sw2On

Switch3		call	Delay50				;give switch time to stop bouncing
		btfsc	I2C_PORT,	SW3		;check it's still pressed
		retlw	0x00				;return is not
		

Sw3On		call	Delay50
		btfsc	I2C_PORT,	SW3		;wait until button is released
		retlw	0x03				;return 0x03
		goto	Sw3On

Switch4		call	Delay50				;give switch time to stop bouncing
		btfsc	I2C_PORT,	SW4		;check it's still pressed
		retlw	0x00				;return is not
		

Sw4On		call	Delay50
		btfsc	I2C_PORT,	SW4		;wait until button is released
		retlw	0x04				;return 0x04
		goto	Sw4On



; End of key routines

;Start of I2C routines

Write_I2C					; write W register to address Adr_Lo
		movwf	DAT_VAL			; save W
		call	I2C_Start
		call	I2C_Set_Write

; If ACK error display 'e'

		btfsc	I2Cflags, 0
		call	Error_Routine

		btfsc	Flags,		Hi	; skip high address if flag not set
        	call	I2C_Hi_Adr		; used for extended addressing (24L64/5)

		btfsc	Flags,		Lo
		call	I2C_Lo_Adr

; If ACK error display 'f'

		btfsc	I2Cflags, 0
		call	Error_Routine1

        	movf	DAT_VAL, 	W	; send the actual data
        	movwf	OutputByte
        	call	I2C_Out
        	call	I2C_NAK

; If ACK error display 'g'

		btfsc	I2Cflags, 0
		call	Error_Routine2

        	call	I2C_Stop 
		call	WaitForWrite
;		call	Delay10			; 10ms delay max. required for EEPROM write cycle
        	return

Read_I2C					; reads data at location specified in Adr_Lo (& Adr_Hi for 24LC64)
						; and page specified in Data_Page
						; returns result in W
		call 	I2C_Start

		call	I2C_Set_Write

; If ACK error display 'e'

		btfsc	I2Cflags, 0
		call	Error_Routine

		btfsc	Flags,		Hi	; skip high address if flag not set
		call	I2C_Hi_Adr		; used for extended addressing (24LC64/5)

		btfsc	Flags,		Lo
		call	I2C_Lo_Adr

; If ACK error display 'f'

		btfsc	I2Cflags, 0
		call	Error_Routine1

        	call 	I2C_Start		; note there is no STOP
		call	I2C_Set_Read

; If ACK error display 'g'

		btfsc	I2Cflags, 0
		call	Error_Routine2

        	call 	I2C_Read		; fetch the byte
        	call 	I2C_Send_NAK		; send no acknowledgement

        	call 	I2C_Stop
		movf 	InputByte, 	W	; return the byte in W
        	return

Seq_Write_I2C					; write buffer to address Adr_Lo
		movwf	DAT_VAL			; save W
		call	I2C_Start
		call	I2C_Set_Write

; If ACK error display 'e'

		btfsc	I2Cflags, 0
		call	Error_Routine

		btfsc	Flags,		Hi	; skip high address if flag not set
        	call	I2C_Hi_Adr		; used for extendd addressing (24L64/5)

		btfsc	Flags,		Lo
		call	I2C_Lo_Adr

; If ACK error display 'f'

		btfsc	I2Cflags, 	0
		call	Error_Routine1

		movlw 	buf			; now write Buff_Size bytes to EEPROM
		movwf 	FSR		
		movlw 	Buf_Size
		movwf 	N
SEQ_WRITE_1:
		movf 	INDF, 		w	; fetch the data byte from the buffer
        	movwf	OutputByte
        	call	I2C_Out
        	call 	I2C_NAK

; If ACK error display 'g'

		btfsc	I2Cflags, 0
		call	Error_Routine2

		incf 	FSR, 		f
		decfsz 	N, 		f
		goto 	SEQ_WRITE_1

        	call	I2C_Stop 
		call	WaitForWrite
;		call	Delay10			; 10ms delay max. required for EEPROM write cycle
        	return


Seq_Read_I2C					; reads data at location specified in Adr_Lo (& Adr_Hi for 24LC64)
						; and page specified in Data_Page
						; returns 4 byte result in buf
		call 	I2C_Start
		call	I2C_Set_Write

; If ACK error display 'e'

		btfsc	I2Cflags, 0
		call	Error_Routine

		btfsc	Flags,		Hi	; skip high address if flag not set
		call	I2C_Hi_Adr		; used for extended addressing (24LC64/5)

		btfsc	Flags,		Lo
		call	I2C_Lo_Adr

; If ACK error display 'f'

		btfsc	I2Cflags, 0
		call	Error_Routine1

        	call 	I2C_Start		; note there is no STOP
		call	I2C_Set_Read

; If ACK error display 'g'

		btfsc	I2Cflags, 0
		call	Error_Routine2

		movlw 	buf			; now sequentially read bytes from EEPROM
		movwf 	FSR		
		movlw 	Buf_Size
		movwf 	N
SEQ_READ_1:
		call 	I2C_Read		; fetch each byte from EEPROM
		movf 	InputByte,	w
		movwf 	INDF			; and save in data buffer
		incf 	FSR,		f
		decfsz 	N, 		f
		goto 	SEQ_READ_2		; not done
		goto 	SEQ_READ_3

SEQ_READ_2:
		call 	I2C_ACK			; if not done, send an ACK and continue
		goto 	SEQ_READ_1
SEQ_READ_3:

        	call 	I2C_Send_NAK		; send no acknowledgement

        	call 	I2C_Stop
		movf 	InputByte, 	w	; return the byte in W
        	return

; The following routines are low level I2C routines applicable to most
; interfaces with I2C devices.

I2C_Read					; read byte on i2c bus
		clrf 	InputByte
		movlw 	0x08
		movwf 	_N			; set index to 8	
		call	HIGH_SDA		; be sure SDA is configured as input
In_Bit
		call 	HIGH_SCL		; clock high
		btfss 	I2C_PORT,	SDA	; test SDA bit
		goto	In_Zero
		goto	In_One

In_Zero
		bcf 	STATUS, 	C	; clear carry
		rlf 	InputByte, 	f	; i_byte = i_byte << 1 | 0
		goto 	Cont_In

In_One
		bsf	STATUS, 	C	; set carry
		rlf 	InputByte, 	f

Cont_In
		call	LOW_SCL			; bring clock low
		decfsz 	_N, 	F		; decrement index
		goto 	In_Bit
		return

I2C_Out:					; send w register on I2C bus
        	movwf 	OutputByte
		movlw 	0x08
		movwf 	_N
Out_Bit:
		bcf 	STATUS,		C	; clear carry
		rlf 	OutputByte, 	f	; left shift, most sig bit is now in carry
		btfss 	STATUS, 	C	; if one, send a one
		goto 	Out_Zero
		goto 	Out_One

Out_Zero:
		call 	LOW_SDA			; SDA at zero
		call 	Clock_Pulse	
		call 	HIGH_SDA
		goto 	Out_Cont

Out_One:
		call 	HIGH_SDA		; SDA at logic one
		call 	Clock_Pulse
Out_Cont:
		decfsz 	_N,		F	; decrement index
		goto 	Out_Bit
		return	

;;;;;;
		
I2C_NAK:					; bring SDA high and clock
		call 	HIGH_SDA
		call 	HIGH_SCL

		clrf	count			; wait for ACK
WaitForACK	incf	count, f		; increase timeout counter each time ACK is not received
		btfsc	STATUS, Z
		goto	No_ACK_Rec
		btfsc	I2C_PORT,	SDA	; test pin. If clear, EEPROM is pulling SDA low for ACK
		goto	WaitForACK		; ...otherwise, continue to wait
		bcf	I2Cflags, 0		; clear flag bit (ACK received)
		call 	LOW_SCL
		return

I2C_Send_NAK:					; bring SDA high and clock
		call 	HIGH_SDA
		call 	Clock_Pulse
		return

WaitForWrite					; poll ACK for write timing
		call 	I2C_Start
		call	I2C_Set_Write
		btfsc	I2Cflags, 0
		goto	WaitForWrite
		return

;------ No ACK received from slave (must use "return" from here)
;; Typically, set a flag bit to indicate failed write and check for it upon return.
No_ACK_Rec
		bsf	I2Cflags, 0		; set flag bit
		return	

;------ No ACK received from slave.  This is the error handler.
Error_Routine
; Output error message, etc. here
		movlw	'e'
		call	LCD_Char
		return				; returns to INITIAL calling routine

Error_Routine1
; Output error message, etc. here
		movlw	'f'
		call	LCD_Char
		return

Error_Routine2
; Output error message, etc. here
		movlw	'g'
		call	LCD_Char
		return

I2C_ACK:
		call 	LOW_SDA
		call 	Clock_Pulse
		return

I2C_Start:				
		call 	LOW_SCL
		call 	HIGH_SDA
		call 	HIGH_SCL
		call 	LOW_SDA			; bring SDA low while SCL is high
		call 	LOW_SCL
		return

I2C_Stop:
		call 	LOW_SCL
		call 	LOW_SDA
		call 	HIGH_SCL
		call 	HIGH_SDA		; bring SDA high while SCL is high
		call 	LOW_SCL
		return

I2C_Set_Write	rlf 	Data_Page, 	W	; shift device page
		andlw	b'00001110'		; AND to make sure in range
        	iorwf 	Chip_Adr,	W
        	call 	I2C_Out
        	call 	I2C_NAK
		return

I2C_Set_Read	rlf 	Data_Page, 	W	; shift device page
		andlw	b'00001110'		; AND to make sure in range
        	iorwf 	Chip_Adr,	W
		iorlw	Chip_Read		; add 1 for read
        	call 	I2C_Out
        	call 	I2C_NAK
		return

I2C_Lo_Adr      movf 	Adr_Lo, 	W	; send low byte of address
        	call 	I2C_Out
        	call 	I2C_NAK
		return

I2C_Hi_Adr      movf 	Adr_Hi, 	W	; send high byte of address
        	call 	I2C_Out
        	call 	I2C_NAK
		return

Clock_Pulse:					; SCL momentarily to logic one
		call 	HIGH_SCL
		call 	LOW_SCL
		return		

HIGH_SDA:					; high impedance by making SDA an input
		bsf 	STATUS, 	RP0	; bank 1
		bsf 	I2C_TRIS, 	SDA	; make SDA pin an input
		bcf 	STATUS, 	RP0	; back to bank 0
		return

LOW_SDA:
		bcf 	I2C_PORT, 	SDA	
		bsf 	STATUS, 	RP0	; bank 1
		bcf 	I2C_TRIS, 	SDA	; make SDA pin an output
		bcf 	STATUS, 	RP0	; back to bank 0
		return

HIGH_SCL:
		bsf 	STATUS, 	RP0	; bank 1
		bsf 	I2C_TRIS, 	SCL	; make SCL pin an input
		bcf 	STATUS, 	RP0	; back to bank 0
		return

LOW_SCL:
		bcf	I2C_PORT, 	SCL
		bsf 	STATUS, 	RP0	; bank 1
		bcf 	I2C_TRIS, 	SCL	; make SCL pin an output
		bcf 	STATUS, 	RP0	; back to bank 0
		return

; End of I2C routines

;LCD routines

;Initialise LCD
LCD_Init	call 	LCD_Busy		;wait for LCD to settle

		movlw	0x20			;Set 4 bit mode
		call	LCD_Cmd

		movlw	0x28			;Set display shift
		call	LCD_Cmd

		movlw	0x06			;Set display character mode
		call	LCD_Cmd

		movlw	0x0c			;Set display on/off and cursor command
		call	LCD_Cmd			;Set cursor off

		call	LCD_Clr			;clear display

		retlw	0x00

; command set routine
LCD_Cmd		movwf	templcd
		swapf	templcd,	w	;send upper nibble
		andlw	0x0f			;clear upper 4 bits of W
		movwf	LCD_PORT
		bcf	LCD_PORT, LCD_RS	;RS line to 1
		call	Pulse_e			;Pulse the E line high

		movf	templcd,	w	;send lower nibble
		andlw	0x0f			;clear upper 4 bits of W
		movwf	LCD_PORT
		bcf	LCD_PORT, LCD_RS	;RS line to 1
		call	Pulse_e			;Pulse the E line high
		call 	LCD_Busy
		retlw	0x00

LCD_CharD	addlw	0x30			;add 0x30 to convert to ASCII
LCD_Char	movwf	templcd
		swapf	templcd,	w	;send upper nibble
		andlw	0x0f			;clear upper 4 bits of W
		movwf	LCD_PORT
		bsf	LCD_PORT, LCD_RS	;RS line to 1
		call	Pulse_e			;Pulse the E line high

		movf	templcd,	w	;send lower nibble
		andlw	0x0f			;clear upper 4 bits of W
		movwf	LCD_PORT
		bsf	LCD_PORT, LCD_RS	;RS line to 1
		call	Pulse_e			;Pulse the E line high
		call 	LCD_Busy
		retlw	0x00

LCD_Line1	movlw	0x80			;move to 1st row, first column
		call	LCD_Cmd
		retlw	0x00

LCD_Line2	movlw	0xc0			;move to 2nd row, first column
		call	LCD_Cmd
		retlw	0x00

LCD_Line1W	addlw	0x80			;move to 1st row, column W
		call	LCD_Cmd
		retlw	0x00

LCD_Line2W	addlw	0xc0			;move to 2nd row, column W
		call	LCD_Cmd
		retlw	0x00

LCD_CurOn	movlw	0x0d			;Set display on/off and cursor command
		call	LCD_Cmd
		retlw	0x00

LCD_CurOff	movlw	0x0c			;Set display on/off and cursor command
		call	LCD_Cmd
		retlw	0x00

LCD_Clr		movlw	0x01			;Clear display
		call	LCD_Cmd
		retlw	0x00

LCD_HEX		movwf	lcdtmp
		swapf	lcdtmp,	w
		andlw	0x0f
		call	HEX_Table
		call	LCD_Char
		movf	lcdtmp, w
		andlw	0x0f
		call	HEX_Table
		call	LCD_Char
		retlw	0x00

Pulse_e		bsf	LCD_PORT, LCD_E
		nop
		bcf	LCD_PORT, LCD_E
		retlw	0x00

LCD_Busy
		bsf	STATUS,	RP0		;set bank 1
		movlw	0x0f			;set Port for input
		movwf	LCD_TRIS
		bcf	STATUS,	RP0		;set bank 0
		bcf	LCD_PORT, LCD_RS	;set LCD for command mode
		bsf	LCD_PORT, LCD_RW	;setup to read busy flag
		bsf	LCD_PORT, LCD_E
		swapf	LCD_PORT, w		;read upper nibble (busy flag)
		bcf	LCD_PORT, LCD_E		
		movwf	templcd2 
		bsf	LCD_PORT, LCD_E		;dummy read of lower nibble
		bcf	LCD_PORT, LCD_E
		btfsc	templcd2, 7		;check busy flag, high = busy
		goto	LCD_Busy		;if busy check again
		bcf	LCD_PORT, LCD_RW
		bsf	STATUS,	RP0		;set bank 1
		movlw	0x00			;set Port for output
		movwf	LCD_TRIS
		bcf	STATUS,	RP0		;set bank 0
		return

;end of LCD routines


;Delay routines

Long_Delay
		call	Delay255
		call	Delay255
		call	Delay255
		call	Delay255
		return

Delay255	movlw	0xff		;delay 255 mS
		goto	d0
Delay100	movlw	d'100'		;delay 100mS
		goto	d0
Delay50		movlw	d'50'		;delay 50mS
		goto	d0
Delay20		movlw	d'20'		;delay 20mS
		goto	d0
Delay10		movlw	d'10'		;delay 10mS
		goto	d0
Delay1		movlw	d'1'		;delay 1mS
		goto	d0
Delay5		movlw	0x05		;delay 5.000 ms (4 MHz clock)
d0		movwf	count1
d1		movlw	0xC7
		movwf	counta
		movlw	0x01
		movwf	countb
Delay_0		decfsz	counta, f
		goto	$+2
		decfsz	countb, f
		goto	Delay_0

		decfsz	count1	,f
		goto	d1
		retlw	0x00

;end of Delay routines

;This routine downloaded from http://www.piclist.com
Convert:                        ; Takes number in NumH:NumL
                                ; Returns decimal in
                                ; TenK:Thou:Hund:Tens:Ones
        swapf   NumH, w
	iorlw	B'11110000'
        movwf   Thou
        addwf   Thou,f
        addlw   0XE2
        movwf   Hund
        addlw   0X32
        movwf   Ones

        movf    NumH,w
        andlw   0X0F
        addwf   Hund,f
        addwf   Hund,f
        addwf   Ones,f
        addlw   0XE9
        movwf   Tens
        addwf   Tens,f
        addwf   Tens,f

        swapf   NumL,w
        andlw   0X0F
        addwf   Tens,f
        addwf   Ones,f

        rlf     Tens,f
        rlf     Ones,f
        comf    Ones,f
        rlf     Ones,f

        movf    NumL,w
        andlw   0X0F
        addwf   Ones,f
        rlf     Thou,f

        movlw   0X07
        movwf   TenK

                    ; At this point, the original number is
                    ; equal to
                    ; TenK*10000+Thou*1000+Hund*100+Tens*10+Ones
                    ; if those entities are regarded as two's
                    ; complement binary.  To be precise, all of
                    ; them are negative except TenK.  Now the number
                    ; needs to be normalized, but this can all be
                    ; done with simple byte arithmetic.

        movlw   0X0A                             ; Ten
Lb1:
        addwf   Ones,f
        decf    Tens,f
        btfss   3,0
        goto   Lb1
Lb2:
        addwf   Tens,f
        decf    Hund,f
        btfss   3,0
        goto   Lb2
Lb3:
        addwf   Hund,f
        decf    Thou,f
        btfss   3,0
        goto   Lb3
Lb4:
        addwf   Thou,f
        decf    TenK,f
        btfss   3,0
        goto   Lb4

        retlw	0x00

	org	0x0700				;set to last page of memory
Strings  	addwf   PCL       , f

		retlw	0xFF			;string 0
		retlw	0x01			;clear screen
            	retlw   'W'
            	retlw   'r'
            	retlw   'i'
            	retlw   't'
            	retlw   'i'
            	retlw   'n'
		retlw	'g'
            	retlw   ':'
            	retlw   0x00

		retlw	0xFF			;string 1
		retlw	0x01			;clear screen
            	retlw   'S'
            	retlw   'a'
            	retlw   'm'
            	retlw   'p'
            	retlw   'l'
            	retlw   'e'
            	retlw   ':'
            	retlw   0x00

            	retlw   '*'			;string 2
            	retlw   ' '
            	retlw   'P'
            	retlw   'I'
            	retlw   'C'
            	retlw   ' '
            	retlw   'T'
            	retlw   'u'
            	retlw   't'
            	retlw   'o'
            	retlw   'r'
            	retlw   'i'
            	retlw   'a'
            	retlw   'l'
            	retlw   ' '
            	retlw   '*'
		retlw	0xFF
		retlw	0xC0			;goto line two
            	retlw   '*'
            	retlw   ' '
            	retlw   'D'
            	retlw   'a'
            	retlw   't'
            	retlw   'a'
            	retlw   ' '
            	retlw   'L'
            	retlw   'o'
            	retlw   'g'
            	retlw   'g'
            	retlw   'e'
            	retlw   'r'
            	retlw   '.'
            	retlw   ' '
            	retlw   '*'
            	retlw   0x00

		retlw	0xFF			;string 3
		retlw	0x01			;clear screen
            	retlw   'R'
            	retlw   'e'
            	retlw   'a'
            	retlw   'd'
            	retlw   'i'
            	retlw   'n'
		retlw	'g'
            	retlw   ':'
            	retlw   0x00

		retlw	0xFF			;string 4
		retlw	0x01			;clear screen
            	retlw   'A'
            	retlw   'r'
            	retlw   'e'
            	retlw   ' '
            	retlw   'y'
            	retlw   'o'
		retlw	'u'
            	retlw   ' '
            	retlw   's'
            	retlw   'u'
            	retlw   'r'
		retlw	'e'
            	retlw   '?'
		retlw	0x00

		retlw	0xFF			;string 5
		retlw	0xC0			;goto line two, pos 0
            	retlw   'N'
            	retlw   'o'
            	retlw   '!'
		retlw	0xFF
		retlw	0xCB			;goto line two, pos 11
            	retlw   'Y'
            	retlw   'e'
            	retlw   's'
            	retlw   0x00

		retlw	0xFF			;string 6
		retlw	0xC0			;goto line two
            	retlw   'M'
            	retlw   'e'
            	retlw   'n'
            	retlw   'u'
            	retlw   ' '
		retlw	' '
            	retlw   '-'
		retlw	' '
            	retlw   ' '
            	retlw   ' '
            	retlw   '+'
		retlw	' '
            	retlw   ' '
            	retlw   'S'
		retlw	'e'
            	retlw   't'
		retlw	0x00

		retlw	0xFF			;string 7
		retlw	0x01			;clear screen
            	retlw   'W'
            	retlw   'r'
            	retlw   'i'
            	retlw   't'
            	retlw   'e'
            	retlw   ' '
		retlw	'M'
            	retlw   'e'
            	retlw   'm'
            	retlw   'o'
            	retlw   'r'
            	retlw   'y'
		retlw	'?'
		retlw	0x00

		retlw	0xFF			;string 8
		retlw	0x01			;clear screen
            	retlw   'R'
            	retlw   'e'
            	retlw   'a'
            	retlw   'd'
            	retlw   ' '
		retlw	'M'
            	retlw   'e'
            	retlw   'm'
            	retlw   'o'
            	retlw   'r'
            	retlw   'y'
		retlw	'?'
		retlw	0x00

		retlw	0xFF			;string 9
		retlw	0x01			;clear screen
		retlw	'S'
            	retlw   'e'
            	retlw   't'
            	retlw   ' '
		retlw	's'
            	retlw   'a'
            	retlw   'm'
            	retlw   'p'
            	retlw   'l'
            	retlw   'e'
            	retlw   ' '
		retlw	't'
            	retlw   'i'
            	retlw   'm'
            	retlw   'e'
		retlw	'?'
		retlw	0x00


		retlw	0xFF			;string 10
		retlw	0x01			;clear screen
            	retlw   'C'
            	retlw   'l'
            	retlw   'e'
            	retlw   'a'
            	retlw   'r'
            	retlw   ' '
		retlw	'M'
            	retlw   'e'
            	retlw   'm'
            	retlw   'o'
            	retlw   'r'
            	retlw   'y'
		retlw	'?'
		retlw	0x00

		retlw	0xFF			;string 11
		retlw	0x01			;clear screen
            	retlw   'C'
            	retlw   'l'
            	retlw   'e'
            	retlw   'a'
            	retlw   'r'
            	retlw   'i'
		retlw	'n'
            	retlw   'g'
            	retlw   '!'
		retlw	0x00

		retlw	0xFF			;string 12
		retlw	0x01			;clear screen
		retlw	'M'
            	retlw   'e'
            	retlw   'm'
            	retlw   'o'
            	retlw   'r'
            	retlw   'y'
		retlw	' '
            	retlw   'C'
            	retlw   'l'
            	retlw   'e'
            	retlw   'a'
            	retlw   'r'
            	retlw   'e'
		retlw	'd'
		retlw	'.'
		retlw	0x00

		end