#include <TinyWireS.h>
#include <Adafruit_NeoPixel.h>

/*
For Attiny85: 16MHz with PLL source (fuse settings)

Pinout:

               ----------
       RESET - | 1    8 | - VCC 
NeoPixel OUT - | 2    7 | - I2C SCL
               | 3    6 |
         GND - | 4    5 | - I2C SDA
               ----------
                Attiny85
                
I2C address: 0x04

I2C commands:

   1. 2.                 Description
---------------------------------------------
  'C'                    Clear data
  'L' XX                 Set number of pixels (0-128)
  'S'                    Show pixels (sends data to neopixel)
  'B' XX                 Set brightness of the NeoPixel (0-255)
  'P' PP RR GG BB        Set pixel color (PP pixel num,
                            RR - red, GG - green, BB - blue)
                            
Used libraries:
  NeoPixel:  git clone https://github.com/adafruit/Adafruit_NeoPixel.git
  TinyWireS: git clone https://github.com/rambo/TinyWire.git

Example:
  RainbowI2C in the example folder for Arduino Nano/UNO
  
*/
 
#define PIN 3
#define I2C_SLAVE_ADDRESS 0x04
 
// Parameter 1 = number of pixels in strip
// Parameter 2 = pin number (most are valid)
// Parameter 3 = pixel type flags, add together as needed:
//   NEO_KHZ800  800 KHz bitstream (most NeoPixel products w/WS2812 LEDs)
//   NEO_KHZ400  400 KHz (classic 'v1' (not v2) FLORA pixels, WS2811 drivers)
//   NEO_GRB     Pixels are wired for GRB bitstream (most NeoPixel products)
//   NEO_RGB     Pixels are wired for RGB bitstream (v1 FLORA pixels, not v2)
Adafruit_NeoPixel strip = Adafruit_NeoPixel(16, PIN, NEO_GRB + NEO_KHZ800);
 
//Handles receiving i2c data.
void receiveEvent(uint8_t howMany)
{
  uint8_t buffer[16];
      
  if ( howMany < 1 || howMany > sizeof(buffer) )
  {   // insane number
    return;
  }
      
  uint8_t p = 0;
  while(howMany--)
    buffer[p++] = TinyWireS.receive();
      
  if( p == 0 )
    return;
  
  switch( buffer[0] )
  {
    case 'C':
      strip.clear();
      break;
    case 'B':
      strip.setBrightness(buffer[1]);
      break;
    case 'S':
      strip.show();
      break;
    case 'L':
      strip.updateLength(buffer[1]);
      break;
    case 'P':
      {
        uint8_t pixel = buffer[1];
        uint8_t i = 2;
        while( i + 3 <= p )
        {
          strip.setPixelColor(pixel++, buffer[i], buffer[i+1], buffer[i+2]);
          i += 3;
        }
      }
      break;
  }
}
 
void setup() {
  strip.begin();
  strip.show(); // Initialize all pixels to 'off'

  TinyWireS.begin(I2C_SLAVE_ADDRESS);
  TinyWireS.onReceive(receiveEvent);
}
 
void loop() {
  TinyWireS_stop_check();
}

