// bufferelt UART be/ki driver by Roberto Benjami v0.8
// rvid hasznlati tmutat az "uart.h"-ban

#ifdef   __18CXX
#include <p18cxxx.h>
#elif    __XC
#include <xc.h>
#elif    defined(__PICC18__)
#include <htc.h>
#endif

#include "uart.h"

// I/O lbak adatirnyai
#define TRIS_(p, m)            TRIS ## p ## bits.TRIS ## p ## m
// C18
#if defined(__18CXX)
#define IOIN(x)                x(TRIS_) = 1
#define IOOUT(x)               x(TRIS_) = 0
// HTC18, XC
#else
#define IOIN(x)                TRIS_(x) = 1
#define IOOUT(x)               TRIS_(x) = 0
#endif

// vtel buffer tmbje s vltozi
#if URXBUFFERSIZE != 0
#ifdef  URXBUFFERADDR
#ifdef   __18CXX
#pragma udata UARTRX_BUF = URXBUFFERADDR
char uRxBuffer[URXBUFFERSIZE];          // vteli buffer
#pragma udata
#else // XC, HTC
char uRxBuffer[URXBUFFERSIZE] @URXBUFFERADDR; // vteli buffer
#endif
#else // #ifdef URXBUFFERADDR
char uRxBuffer[URXBUFFERSIZE];          // vteli buffer
#endif // #ifdef URXBUFFERADDR
volatile char uRxBufferOverFlow = 0;
volatile unsigned char  uRxStart = 0, uRxLength = 0;
#if URXBUFFERSIZE == 256
volatile unsigned char uRxBufFree = 0xFF;
#else
volatile unsigned char uRxBufFree = URXBUFFERSIZE;
#endif
#endif // #if URXBUFFERSIZE != 0

// ads buffer tmbje s vltozi
#if UTXBUFFERSIZE != 0
#ifdef  UTXBUFFERADDR
#ifdef   __18CXX
#pragma udata UARTTX_BUF = UTXBUFFERADDR
char uTxBuffer[UTXBUFFERSIZE];          // ads buffer
#pragma udata
#else // XC, HTC
char uTxBuffer[UTXBUFFERSIZE] @UTXBUFFERADDR; // ads buffer
#endif
#else // #ifdef UTXBUFFERADDR
char uTxBuffer[UTXBUFFERSIZE];          // ads buffer
#endif // #ifdef UTXBUFFERADDR
volatile char uTxBufferOverFlow = 0;
volatile unsigned char  uTxStart = 0, uTxLength = 0;
#if UTXBUFFERSIZE == 256
volatile unsigned char uTxBufFree = 0xFF;
#else
volatile unsigned char uTxBufFree = UTXBUFFERSIZE;
#endif
#endif // #if UTXBUFFERSIZE != 0

//=============================================================================
void UartInit(void)
{
  #define BAUDDIV (SystemClock/BAUDRATE)

// buta 8 bites sebessgbelltsi lehet?sggel elltott UART
#ifdef   UBRG8BIT
#if BAUDDIV < (256*16)
  #define BRG_DIV         16
  #define CLKSEL           1
  #elif BAUDDIV < (256*64)
  #define BRG_DIV         64
  #define CLKSEL           0
  #else
  #error "PIC18: ez a BAUDRATE nem elllthat"
  #endif // BAUDDIV

  #define BAUDRATEREG ((SystemClock+((BRG_DIV/2)*BAUDRATE))/BRG_DIV/BAUDRATE-1)
  SPBRG = BAUDRATEREG;                  // als byte;
  TXSTAbits.BRGH = CLKSEL;              // lo/hi speed
  #if URXBUFFERSIZE != 0
  IOIN(UARTRX);                         // RX bemenet
  #endif
  #if UTXBUFFERSIZE != 0
  IOOUT(UARTTX);                        // TX kimenet
  #endif
  RCSTAbits.SPEN = 1;                   // soros port eng
  TXSTAbits.SYNC = 0;                   // aszinkron md
#else
  
// okosabb 16 bites sebessgbelltsi lehet?sggel elltott UART
#if BAUDDIV < (65536*4)
  #define BRG_DIV          4
  #define CLKSEL           1
  #elif BAUDDIV < (65536*16)
  #define BRG_DIV         16
  #define CLKSEL           0
  #else
  #error "PIC18: ez a BAUDRATE nem elllthat"
  #endif // BAUDDIV

  #define BAUDRATEREG ((SystemClock+((BRG_DIV/2)*BAUDRATE))/BRG_DIV/BAUDRATE-1)
  SPBRG = BAUDRATEREG;                  // als byte;
  SPBRGH = BAUDRATEREG >> 8;            // fels byte
  TXSTAbits.BRGH = CLKSEL;              // lo/hi speed
  BAUDCONbits.BRG16 = 1;                // 16 bites
  #if URXBUFFERSIZE != 0
  IOIN(UARTRX);                         // RX bemenet
  #endif
  #if UTXBUFFERSIZE != 0
  IOOUT(UARTTX);                        // TX kimenet
  #endif
  RCSTAbits.SPEN = 1;                   // soros port eng
  TXSTAbits.SYNC = 0;                   // aszinkron md
#endif 

  #if (RXPR18 == -1 && TXPR18 != -1) || (TXPR18 == -1 && RXPR18 != -1)
  #error "Hiba: priorits nlkli hasznlat esetn az RX s a TX-nek is annak kell lennie!"
  #endif

  #if defined(UARTRXINTMODE) && (URXBUFFERSIZE != 0)
  // vteli prioritsok
  #if RXPR18 == -1
  // nem kell semmit belltani
  #elif RXPR18 == 0
  IPR1bits.RCIP = 0;                    // alacsony priorits
  #elif RXPR18 == 1
  IPR1bits.RCIP = 1;                    // magas priorits
  #endif
  #endif // #ifdef UARTRXINTMODE
  
  #if defined(UARTTXINTMODE) && (UTXBUFFERSIZE != 0)
  // ads prioritsok
  #if TXPR18 == -1
  // nem kell semmit belltani
  #elif TXPR18 == 0
  IPR1bits.TXIP = 0;                    // alacsony priorits
  #elif TXPR18 == 1
  IPR1bits.TXIP = 1;                    // magas priorits
  #endif
  #endif // #ifdef UARTRXINTMODE

  #if URXBUFFERSIZE != 0
  RCSTAbits.CREN = 1;                   // vtel engedlyezs
  #ifdef UARTRXINTMODE
  PIE1bits.RCIE  = 1;                   // vtel IRQ eng 
  #endif
  #endif

  #if UTXBUFFERSIZE != 0
  TXSTAbits.TXEN = 1;                   // ads engedlyezs
  // (ads megszakts engedlyezst csak forgalmazskor kell megtenni, ha mr TXREG nem rhat)
  #endif
  
  // ha megszakts md engedlyezve van, akkor a perifria IRQ-t is engedlyezzk
  #if defined(UARTRXINTMODE) || defined(UARTTXINTMODE)
  #if RXPR18 == -1
  RCONbits.IPEN = 0;                    // ktszint IRQ tilts
  #else  // #if RXPR18 == -1
  RCONbits.IPEN = 1;                    // ktszint IRQ eng
  #endif // #if RXPR18 == -1
  INTCONbits.PEIE = 1;                  // perifria IRQ eng
  #ifdef UARTINITGLOBALIRQENG
  INTCONbits.GIEL = 1;                  // globlis IRQ eng
  INTCONbits.GIEH = 1;                  // globlis IRQ eng
  #endif // #ifdef UARTINITGLOBALIRQENG
  #endif // #if defined(UARTRXINTMODE) || defined(UARTTXINTMODE)
}

//=============================================================================
// Rx pufferbl egy byte-ot kiolvas, trldik is a pufferbl
#if URXBUFFERSIZE != 0
unsigned char UartRx(void)
{
  unsigned char ch;
  ch = 0;
  #ifdef UARTRXINTMODE
  PIE1bits.RCIE = 0;                    // megszakts tilts (hogy ne vltozhasson meg kzben a puffermutat)
  #endif
  if(uRxLength)                         // van mg adat a pufferben ?
  {
    ch = uRxBuffer[uRxStart++];
    #if URXBUFFERSIZE != 256
    uRxStart &= (URXBUFFERSIZE - 1);
    #endif
    uRxLength--; uRxBufFree++;
  }
  #ifdef UARTRXINTMODE
  PIE1bits.RCIE = 1;                    // megszakts engedlyezs
  #endif
  return ch;
}
#endif

//=============================================================================
// Rx pufferbl gy olvas ki egy byte-ot hogy az nem trldik a pufferbl
#if URXBUFFERSIZE != 0
unsigned char UartRxTop(void)
{
  return uRxBuffer[uRxStart];
}
#endif

//=============================================================================
#if URXBUFFERSIZE != 0
void UartRxProcess(void)
{
  unsigned char c;
  c = RCREG;                            // RCIF is trldik itt
  if(RCSTAbits.FERR | RCSTAbits.OERR)
  { // vtel tilts / engedlyezssel hibareset
    RCSTAbits.CREN = 0;
    RCSTAbits.CREN = 1;
  }
  else
  {
    if (uRxBufFree)                     // van mg hely a pufferben ?
    {   
      #if URXBUFFERSIZE == 256
      uRxBuffer[uRxStart + uRxLength] = c; // egy karakter megy a pufferbe
      #else
      uRxBuffer[(uRxStart + uRxLength) & (URXBUFFERSIZE - 1)] = c; // egy karakter megy a pufferbe
      #endif
      uRxLength++; uRxBufFree--;
    }
    else
      uRxBufferOverFlow = 1;
  }
}
#endif

//=============================================================================
#if (URXBUFFERSIZE != 0) && defined(UARTDEBUG)
void UartRxWrite(unsigned char ch)
{
  if(uRxBufFree)                        // van mg hely a pufferben ?
  {   
    #if URXBUFFERSIZE == 256
    uRxBuffer[uRxStart + uRxLength] = ch; // egy karakter megy a pufferbe
    #else
    uRxBuffer[(uRxStart + uRxLength) & (URXBUFFERSIZE - 1)] = ch; // egy karakter megy a pufferbe
    #endif
    uRxLength++; uRxBufFree--;
  }
}
#endif

//=============================================================================
#if UTXBUFFERSIZE != 0
void UartTx(unsigned char ch)
{
  if (!uTxLength && PIR1bits.TXIF)      // buffer res, s TXREG rht
  {
    TXREG = ch;                         // TXIF is automatikusan trldik itt
  }
  else
  {
    #ifdef UARTTXINTMODE
    PIE1bits.TXIE = 0;                  // TX IRQ tilts (hogy a megszakts ne nyljon hozz a puffermutatkhoz)
    #endif
    if(uTxBufFree)                      // van mg hely a pufferben ?
    {   
      #if UTXBUFFERSIZE == 256
      uTxBuffer[uTxStart + uTxLength] = ch; // egy karakter megy a pufferbe
      #else
      uTxBuffer[(uTxStart + uTxLength) & (UTXBUFFERSIZE - 1)] = ch; // egy karakter megy a pufferbe
      #endif
      uTxLength++; uTxBufFree--;
    }
    else
      uTxBufferOverFlow = 1;            // itt jelezzk hogy buffertlcsorduls miatt adatveszts trtnt
    #ifdef UARTTXINTMODE
    PIE1bits.TXIE = 1;                  // TX IRQ engedlyezs
    #endif
  }
}
#endif

//=============================================================================
#if UTXBUFFERSIZE != 0
void UartTxProcess(void)
{
  if(!uTxLength)                        // ha mr nincs mit adni
    #ifdef UARTTXINTMODE
    PIE1bits.TXIE = 0;                  // TX IRQ tilts
    #else
    return;
	#endif
  else
  {
    TXREG = uTxBuffer[uTxStart++];      // TXREG = adat mehet
    #if UTXBUFFERSIZE != 256
    uTxStart &= (UTXBUFFERSIZE - 1);
    #endif
    uTxLength--; uTxBufFree++;
  }
}
#endif
