#include <math.h>
#include "bmp085.h"
#include "c_bmp085.h"
#include "i_bmp085.h"
#include "w_bmp085.h"
#include "bmp085_conversion.h"

#include "iic_master.h"
#include "mal.h"
#include "k_stdtype.h"
#include "task.h"

#define BMP085_ADDRESS 0b1110111
#ifndef BMP085_USE_INTEGER_VERSION
	#if (BMP085_OSS == 0)
			#define BMP085_OSS_CALC 1.0 //should be 2^OSS
	#endif
	#if (BMP085_OSS == 1)
		#define BMP085_OSS_CALC 2.0 //should be 2^OSS
	#endif
	#if (BMP085_OSS == 2)
		#define BMP085_OSS_CALC 4.0 //should be 2^OSS
	#endif
	#if (BMP085_OSS == 3)
		#define BMP085_OSS_CALC 8.0 //should be 2^OSS
	#endif
#endif

typedef enum _Bmp085States {
	Bmp085State_Calibration_1_Start,
	Bmp085State_Calibration_1,
	Bmp085State_Calibration_2_Start,
	Bmp085State_Calibration_2,
	Bmp085State_Calibration_3_Start,
	Bmp085State_Calibration_3,
	Bmp085State_Calibration_4_Start,
	Bmp085State_Calibration_4,
	Bmp085State_Calibration_5_Start,
	Bmp085State_Calibration_5,
	Bmp085State_Calibration_6_Start,
	Bmp085State_Calibration_6,
	Bmp085State_Calibration_7_Start,
	Bmp085State_Calibration_7,
	Bmp085State_Calibration_8_Start,
	Bmp085State_Calibration_8,
	Bmp085State_Calibration_9_Start,
	Bmp085State_Calibration_9,
	Bmp085State_Calibration_10_Start,
	Bmp085State_Calibration_10,
	Bmp085State_Calibration_11_Start,
	Bmp085State_Calibration_11,
	Bmp085State_Idle,
	Bmp085State_StartTemp,
	Bmp085State_ReadTemp,
	Bmp085State_ReadTemp_Value,
	Bmp085State_StartPressure,
	Bmp085State_ReadPressure,
	Bmp085State_ReadPressure_Value,
	Bmp085State_CalculateTemp,
	Bmp085State_CalculatePressure,
} Bmp085States;

#ifndef BMP085_USE_INTEGER_VERSION
	float x1 = 0.0;
	float x2 = 0.0;
	float b5 = 0.0;
	float b6 = 0.0;
	float x3 = 0.0;
	float b3 = 0.0;
	float p = 0.0;
	float b4 = 0.0;
	float b7 = 0.0;

	float ac1 = 408.0;
	float ac2 = -72.0;
	float ac3 = -14383.0;
	float ac4 = 32741.0;
	float ac5 = 32757.0;
	float ac6 = 23153.0;
	float b1 = 6194.0;
	float b2 = 4.0;
	float mb = -32768.0;
	float mc = -8711.0;
	float md = 2868.0;
#else
	//float x1 = 0.0;
	//float x2 = 0.0;
	sint32 b5 = 0.0;
	//float b6 = 0.0;
	//float x3 = 0.0;
	//float b3 = 0.0;
	//float p = 0.0;
	//float b4 = 0.0;
	//float b7 = 0.0;

	const uint8 OSS = BMP085_OSS;  // Oversampling Setting
	
	sint16 ac1 = 408.0;
	sint16 ac2 = -72.0;
	sint16 ac3 = -14383.0;
	uint16 ac4 = 32741.0;
	uint16 ac5 = 32757.0;
	uint16 ac6 = 23153.0;
	sint16 b1 = 6194.0;
	sint16 b2 = 4.0;
	sint16 mb = -32768.0;
	sint16 mc = -8711.0;
	sint16 md = 2868.0;
#endif

Bmp085States bmp085States = Bmp085State_Idle;
#ifndef BMP085_USE_INTEGER_VERSION
	float UT = 27898.0;
	float UP = 23843.0;
#else
	uint16 UT = 27898.0;
	uint32 UP = 23843.0;
#endif
float resultTemp = 0.0;
float resultPressure = 0.0;
float resultAltitude = 0.0;

uint8 bmp085ReadStart(unsigned char address);
uint8 bmp085Read(sint8 *value);

uint8 bmp085ReadIntStart(unsigned char address);
uint8 bmp085ReadInt(sint16 *value);

uint8 bmp085_start_convert_pressure(void);
uint8 bmp085_start_convert_temp(void);
sint32 bmp085_read_temp(void);
sint32 bmp085_read_pressure(void);
#ifndef BMP085_USE_INTEGER_VERSION
	void bmp085_convert_temp(float ut, float *temp);
	void bmp085_convert_pressure(float up, float* pressure);
#else
	sint16 bmp085GetTemperature(uint16 ut);
	sint32 bmp085GetPressure(uint32 up);
#endif
float convertPressureToAltitude(float sealevelPressure, float pressure);
uint32 doBmp085TaskNr = 0;
unsigned char bmp085TempData[4] = {0x00, 0x00, 0x00, 0x00};
unsigned char bmp085TempData2[4] = {0x00, 0x00, 0x00, 0x00};

void doBmp085(void);

void init_bmp085(void) {
	bmp085States = Bmp085State_Calibration_1_Start;
	doBmp085TaskNr = addTask(doBmp085, 50, 1, "doBmp085");
}

void do_bmp085(void) {
}

void doBmp085(void) {
	switch (bmp085States) {
		case Bmp085State_Calibration_1_Start : {
			if (bmp085ReadIntStart(0xAA) == 1) {
				bmp085States = Bmp085State_Calibration_1;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}
		case Bmp085State_Calibration_1 : {
			sint16 value = 0;
			if (bmp085ReadInt(&value) == 1) {
				ac1 = value;
				scheduleTask(doBmp085TaskNr);
				bmp085States = Bmp085State_Calibration_2_Start;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}
		case Bmp085State_Calibration_2_Start : {
			if (bmp085ReadIntStart(0xAC) == 1) {
				bmp085States = Bmp085State_Calibration_2;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}
		case Bmp085State_Calibration_2 : {
			sint16 value = 0;
			if (bmp085ReadInt(&value) == 1) {
				ac2 = value;
				scheduleTask(doBmp085TaskNr);
				bmp085States = Bmp085State_Calibration_3_Start;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}
		case Bmp085State_Calibration_3_Start : {
			if (bmp085ReadIntStart(0xAE) == 1) {
				bmp085States = Bmp085State_Calibration_3;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}
		case Bmp085State_Calibration_3 : {
			sint16 value = 0;
			if (bmp085ReadInt(&value) == 1) {
				ac3 = value;
				scheduleTask(doBmp085TaskNr);
				bmp085States = Bmp085State_Calibration_4_Start;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}
		case Bmp085State_Calibration_4_Start : {
			if (bmp085ReadIntStart(0xB0) == 1) {
				bmp085States = Bmp085State_Calibration_4;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}
		case Bmp085State_Calibration_4 : {
			sint16 value = 0;
			if (bmp085ReadInt(&value) == 1) {
				ac4 = (uint16)value;
				scheduleTask(doBmp085TaskNr);
				bmp085States = Bmp085State_Calibration_5_Start;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}		
		case Bmp085State_Calibration_5_Start : {
			if (bmp085ReadIntStart(0xB2) == 1) {
				bmp085States = Bmp085State_Calibration_5;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}
		case Bmp085State_Calibration_5 : {
			sint16 value = 0;
			if (bmp085ReadInt(&value) == 1) {
				ac5 = (uint16)value;
				scheduleTask(doBmp085TaskNr);
				bmp085States = Bmp085State_Calibration_6_Start;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}		
		case Bmp085State_Calibration_6_Start : {
			if (bmp085ReadIntStart(0xB4) == 1) {
				bmp085States = Bmp085State_Calibration_6;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}
		case Bmp085State_Calibration_6 : {
			sint16 value = 0;
			if (bmp085ReadInt(&value) == 1) {
				ac6 = (uint16)value;
				scheduleTask(doBmp085TaskNr);
				bmp085States = Bmp085State_Calibration_7_Start;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}	
		case Bmp085State_Calibration_7_Start : {
			if (bmp085ReadIntStart(0xB6) == 1) {
				bmp085States = Bmp085State_Calibration_7;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}
		case Bmp085State_Calibration_7 : {
			sint16 value = 0;
			if (bmp085ReadInt(&value) == 1) {
				b1 = (sint16)value;
				scheduleTask(doBmp085TaskNr);
				bmp085States = Bmp085State_Calibration_8_Start;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}	
		case Bmp085State_Calibration_8_Start : {
			if (bmp085ReadIntStart(0xB8) == 1) {
				bmp085States = Bmp085State_Calibration_8;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}
		case Bmp085State_Calibration_8 : {
			sint16 value = 0;
			if (bmp085ReadInt(&value) == 1) {
				b2 = (sint16)value;
				scheduleTask(doBmp085TaskNr);
				bmp085States = Bmp085State_Calibration_9_Start;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}	
		case Bmp085State_Calibration_9_Start : {
			if (bmp085ReadIntStart(0xBA) == 1) {
				bmp085States = Bmp085State_Calibration_9;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}
		case Bmp085State_Calibration_9 : {
			sint16 value = 0;
			if (bmp085ReadInt(&value) == 1) {
				mb = (sint16)value;
				scheduleTask(doBmp085TaskNr);
				bmp085States = Bmp085State_Calibration_10_Start;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}			
		case Bmp085State_Calibration_10_Start : {
			if (bmp085ReadIntStart(0xBC) == 1) {
				bmp085States = Bmp085State_Calibration_10;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}
		case Bmp085State_Calibration_10 : {
			sint16 value = 0;
			if (bmp085ReadInt(&value) == 1) {
				mc = (sint16)value;
				scheduleTask(doBmp085TaskNr);
				bmp085States = Bmp085State_Calibration_11_Start;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}			
		case Bmp085State_Calibration_11_Start : {
			if (bmp085ReadIntStart(0xBE) == 1) {
				bmp085States = Bmp085State_Calibration_11;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}
		case Bmp085State_Calibration_11 : {
			sint16 value = 0;
			if (bmp085ReadInt(&value) == 1) {
				md = (sint16)value;
				scheduleTask(doBmp085TaskNr);
				bmp085States = Bmp085State_Idle;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}			
		case Bmp085State_Idle : {
			bmp085States = Bmp085State_StartTemp;
			break;
		}
		case Bmp085State_StartTemp : {
			if (bmp085_start_convert_temp() == 1) {
				bmp085States = Bmp085State_ReadTemp;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}
		case Bmp085State_ReadTemp : {
			unsigned char address = 0xF6;
			if (writeIIC(BMP085_ADDRESS, &address, 1, 0) == ACK) {
				bmp085States = Bmp085State_ReadTemp_Value;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}
		case Bmp085State_ReadTemp_Value : {
			sint32 result = 0;
			if (readIIC(BMP085_ADDRESS, bmp085TempData, 2) == ACK) {
					result = bmp085TempData[0];
					result <<= 8;
					result += bmp085TempData[1];
					#ifndef BMP085_USE_INTEGER_VERSION
						UT = (float)(sint32)result;
					#else
						UT = result;
					#endif
					bmp085States = Bmp085State_StartPressure;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}
		case Bmp085State_StartPressure : {
			if (bmp085_start_convert_pressure() == 1) {
				bmp085States = Bmp085State_ReadPressure;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}
		case Bmp085State_ReadPressure : {
			unsigned char address = 0xF6;
			if (writeIIC(BMP085_ADDRESS, &address, 1, 0) == ACK) {
				bmp085States = Bmp085State_ReadPressure_Value;
			} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}
		case Bmp085State_ReadPressure_Value : {
			sint32 result = 0;
			if (readIIC(BMP085_ADDRESS, bmp085TempData, 3) == ACK) {
					result = bmp085TempData[0];
					result <<= 8;
					result += bmp085TempData[1];
					result <<= 8;
					result += bmp085TempData[2];
					result >>= (8 - BMP085_OSS);
					#ifndef BMP085_USE_INTEGER_VERSION
						UP = (float)(sint32)result;
					#else
						UP = result;
					#endif
					bmp085States = Bmp085State_CalculateTemp;
				} else {
				bmp085States = Bmp085State_Calibration_1_Start;
			}
			break;
		}
		case Bmp085State_CalculateTemp : {
			/*UT = 27898.0;
			UP = 23843.0;
			ac6 = 23153.0;
			ac5 = 32757.0;
			mb = -32768.0;
			mc = -8711.0;
			md = 2868.0;
			b1 = 6190.0;
			b2 = 4.0;
			ac3 = -14383.0;
			ac2 = -72.0;
			ac1 = 408.0;
			ac4 = 32741.0;*/
			#ifndef BMP085_USE_INTEGER_VERSION
				bmp085_convert_temp(UT, &resultTemp);
			#else
				resultTemp = bmp085GetTemperature(UT);
				resultTemp /= 10.0;
			#endif
			bmp085States = Bmp085State_CalculatePressure;
			break;
		}
		case Bmp085State_CalculatePressure : {
			float seaLevelPressure;
			#ifndef BMP085_USE_INTEGER_VERSION
				bmp085_convert_pressure(UP, &resultPressure);
			#else
				resultPressure = bmp085GetPressure(UP);
			#endif
			resultAltitude = convertPressureToAltitude(BMP085_AVERAGESEELEVEL_PRESSURE, resultPressure);
			/*seaLevelPressure = calculateSeeLevelPressure(resultPressure, 325.0); //Leonberg height
			resultAltitude = convertPressureToAltitude(seaLevelPressure, resultPressure);*/
			bmp085States = Bmp085State_Idle;
			break;
		}
		default : {
			bmp085States = Bmp085State_Idle;
			break;
		}
	}
}

void isr_bmp085_1ms(void) {
}

uint8 bmp085ReadStart(unsigned char address) {
	uint8 result = 0;
	if (writeIIC(BMP085_ADDRESS, &address, 1, 0) == ACK) {
		result = 1;
	}
	return result;
}

uint8 bmp085Read(sint8 *value) {
	uint8 result = 0;
	if (value != NULL) {
		if (readIIC(BMP085_ADDRESS, bmp085TempData, 1) == ACK) {
			*value = bmp085TempData[0];
			result = 1;
		}
	}
	return result;
}

uint8 bmp085ReadIntStart(unsigned char address) {
	uint8 result = 0;
	if (writeIIC(BMP085_ADDRESS, &address, 1, 0) == ACK) {
		result = 1;
	}
	return result;
}

uint8 bmp085ReadInt(sint16 *value) {
	uint8 result = 0;
	if (value != NULL) {
		if (readIIC(BMP085_ADDRESS, bmp085TempData, 2) == ACK) {
			sint16 bmp085TempData2 = 0;
			bmp085TempData2 = bmp085TempData[0];
			bmp085TempData2 <<= 8;
			bmp085TempData2 += bmp085TempData[1];
			*value = bmp085TempData2;
			result = 1;
		}
	}
	return result;
}

uint8 bmp085_start_convert_pressure(void) {
	uint8 result = 0;
	bmp085TempData[0] = 0xF4;
	bmp085TempData[1] = 0x34 + (BMP085_OSS << 6);
	if (writeIIC(BMP085_ADDRESS, bmp085TempData, 2, 0) == ACK) {
		result = 1;
	}
	return result;
}

uint8 bmp085_start_convert_temp(void) {
	uint8 result = 0;
	bmp085TempData[0] = 0xF4;
	bmp085TempData[1] = 0x2E;
	if (writeIIC(BMP085_ADDRESS, bmp085TempData, 2, 0) == ACK) {
		result = 1;
	}
	return result;
}

#ifndef BMP085_USE_INTEGER_VERSION
	void bmp085_convert_temp(float ut, float *temp) {
		x1 = ((ut - ac6) * ac5) / 32768.0;
		x2 = ( mc * 2048.0) / (x1 + md);
		b5 = x1 + x2;
		//printf(" T x1: %ld   x2: %ld   b5: %ld\r\n", x1, x2, b5);
		*temp = (b5 + 8.0) / 16.0 / 10.0;
	}

	void bmp085_convert_pressure(float up, float* pressure) {
		// Calculate the true pressure.
		b6 = b5 - 4000.0;
		x1 = (b2 * (b6 * b6 / 4096.0)) / 2048.0;
		x2 = ac2 * b6 / 2048.0;
		x3 = x1 + x2;
		b3 = (((ac1 * 4.0 + x3) * BMP085_OSS_CALC) + 2.0) / 4.0;
		x1 = ac3 * b6 / 8192.0;
		x2 = (b1 * (b6 * b6 / 4096.0)) / 65536.0;
		x3 = ((x1 + x2) + 2.0) / 4.0;
		b4 = (ac4 * (x3 + 32768.0)) / 32768.0;
		b7 = (up - b3) * (50000.0 / BMP085_OSS_CALC);
		if (b7 > 0.0) {
			p = (b7 * 2.0) / b4;
		} else {
			p = (b7 / b4) / 2.0;
		}
		x1 = (p / 256.0) * (p / 256.0);
		x1 = (x1 * 3038.0) / 65536.0;
		x2 = (-7357.0 * p) / 65536.0;
		*pressure = p + ((x1 + x2 + 3791.0) / 16.0);
	}
#else
	sint16 bmp085GetTemperature(uint16 ut) {
		sint32 x1, x2;
		x1 = (((sint32)ut - (sint32)ac6)*(sint32)ac5) >> 15;
		x2 = ((sint32)mc << 11)/(x1 + md);
		b5 = x1 + x2;
		return ((b5 + 8)>>4);  
	}

	sint32 bmp085GetPressure(uint32 up) {
		sint32 x1, x2, x3, b3, b6, p;
		uint32 b4, b7;

		b6 = b5 - 4000;
		// Calculate B3
		x1 = (b2 * (b6 * b6)>>12)>>11;
		x2 = (ac2 * b6)>>11;
		x3 = x1 + x2;
		b3 = (((((sint32)ac1)*4 + x3)<<OSS) + 2)>>2;

		// Calculate B4
		x1 = (ac3 * b6)>>13;
		x2 = (b1 * ((b6 * b6)>>12))>>16;
		x3 = ((x1 + x2) + 2)>>2;
		b4 = (ac4 * (uint32)(x3 + 32768))>>15;

		b7 = ((uint32)(up - b3) * (50000>>OSS));
		if (b7 < 0x80000000)
		p = (b7<<1)/b4;
		else
		p = (b7/b4)<<1;

		x1 = (p>>8) * (p>>8);
		x1 = (x1 * 3038)>>16;
		x2 = (-7357 * p)>>16;
		p += (x1 + x2 + 3791)>>4;

		return p;
	}
#endif

float getBMP085Temp(void) {
	float result = 0.0;
	result = (float)resultTemp;
	return result;
}

float getBMP085Pressure(void) {
	float result = 0.0;
	result = (float)resultPressure;
	return result;
}

float getBMP085Altitude(void) {
	float result = 0.0;
	result = (float)resultAltitude;
	return result;
}
