#include "ringBuffer.h"
#include "c_ringBuffer.h"
#include <string.h>
#include "mal.h"

int ringBuffer_initBuffer(RingBuffer *ringBuffer, unsigned char *buffer, unsigned int bufferSize) {
	int result = -1;
	if ((ringBuffer != NULL) && (buffer != NULL) && (bufferSize > 1)) {
		ringBuffer->buffer = buffer;
		ringBuffer->bufferSize = bufferSize;
		ringBuffer->bufferRead = buffer;
		ringBuffer->bufferWrite = buffer;
		ringBuffer->bufferMaximumFillness = 0;
		ringBuffer->bufferFillness = 0;
		memset(ringBuffer->buffer, 0x00 , ringBuffer->bufferSize);
		result = 0;
	}
	return result;
}

unsigned int ringBuffer_getFillness(RingBuffer *ringBuffer) {
	unsigned int result = 0;
	if (ringBuffer != NULL) {
		int delta = 0;
		lock_isr();
		if (ringBuffer->bufferRead > ringBuffer->bufferWrite) {
			delta = (int)ringBuffer->bufferRead - (int)ringBuffer->bufferWrite;
			delta = ringBuffer->bufferSize - delta;
		} else {
			delta = (int)ringBuffer->bufferWrite - (int)ringBuffer->bufferRead;
		}		
		unlock_isr();
		if (delta != 0) {
			result = delta;
		}
	}
	return result;
}

int ringBuffer_addItem(RingBuffer *ringBuffer, unsigned char data) {
	int result = -1;
	if ((ringBuffer != NULL) && (ringBuffer->bufferWrite != NULL)) {
		*ringBuffer->bufferWrite = data;
		ringBuffer->bufferWrite++;
		
		#ifdef RINGBUFFER_USE_FILLNESS
			lock_isr();
			ringBuffer->bufferFillness++;
			if (ringBuffer->bufferMaximumFillness < ringBuffer->bufferFillness) {
				ringBuffer->bufferMaximumFillness = ringBuffer->bufferFillness;
			}
			unlock_isr();
		#endif

		if (ringBuffer->buffer + ringBuffer->bufferSize == ringBuffer->bufferWrite) {
			ringBuffer->bufferWrite = ringBuffer->buffer;
		}
		result = 0;
	}
	return result;
}

unsigned int ringBuffer_getItem(RingBuffer *ringBuffer, unsigned char *data) {
	unsigned int result = 0;
	if ((ringBuffer != NULL) && (data != NULL) && (ringBuffer->bufferRead != NULL)) {
		result = ringBuffer_getFillness(ringBuffer);
		if (result != 0) {
			*data = *ringBuffer->bufferRead;
			*ringBuffer->bufferRead = 0x00;
			ringBuffer->bufferRead++;
			
			#ifdef RINGBUFFER_USE_FILLNESS
				lock_isr();
				ringBuffer->bufferFillness--;
				unlock_isr();
			#endif

			if (ringBuffer->buffer + ringBuffer->bufferSize == ringBuffer->bufferRead) {
				ringBuffer->bufferRead = ringBuffer->buffer;
			}
		}
	}
	return result;
}

unsigned int ringBuffer_readItem(RingBuffer *ringBuffer, unsigned char *data, unsigned int item) {
	unsigned int result = 0;
	if ((ringBuffer != NULL) && (data != NULL) && (ringBuffer->bufferRead != NULL)) {
		result = ringBuffer_getFillness(ringBuffer);
		if (result != 0) {
			if ((unsigned int)(ringBuffer->bufferRead - ringBuffer->buffer + item) > (unsigned int)ringBuffer->bufferSize) {
				*data = *((ringBuffer->bufferRead + item - ringBuffer->bufferSize) );
			} else {
				*data = *((ringBuffer->bufferRead + item) );
			}
		}
	}
	return result;
}

#ifdef RINGBUFFER_USE_BUFFER_INTERFACE

int ringBuffer_addBuffer(RingBuffer *ringBuffer, unsigned char *data, unsigned int len) {
	int result = -1;
	unsigned int i = 0;
	for (i = 0; i < len; i++) {
		result = ringBuffer_addItem(ringBuffer, data[i]);
	}
	return result;
}

unsigned int ringBuffer_getBuffer(RingBuffer *ringBuffer, unsigned char *data, unsigned int len) {
	unsigned int result = 0;
	unsigned int i = 0;
	for (i = 0; i < len; i++) {
		result = ringBuffer_getItem(ringBuffer, data + i);
	}
	return result;
}

#endif

void ringBuffer_tryReadjustBuffer(RingBuffer *ringBuffer) {
	lock_isr();
	if (ringBuffer != NULL) {
		if (ringBuffer->bufferRead == ringBuffer->bufferWrite) {
			ringBuffer->bufferRead = ringBuffer->buffer;
			ringBuffer->bufferWrite = ringBuffer->buffer;
		}
	}
	unlock_isr();
}
