#include "eep_manager.h"
#include "c_eep_manager.h"
#include "i_eep_manager.h"
#include "w_eep_manager.h"

#include "eeprom.h"
#include "k_stdtype.h"
#include <stdio.h>
#include <string.h>

#define EEPMANAGER_ADDR_OFFSET	0x10
#define EEPMANAGER_ADDR_CRC_OFFSET 1
#define EEPMANAGER_ADDR_CRC_ADDR (EEPROM_START_ADDR + EEPMANAGER_ADDR_CRC_OFFSET)

#define EEPMANAGER_ADDR_VERSION_OFFSET 2
#define EEPMANAGER_ADDR_VERSION_ADDR (EEPROM_START_ADDR + EEPMANAGER_ADDR_VERSION_OFFSET)

#define EEPMANAGER_ADDR_DATA_START	(EEPROM_START_ADDR + EEPMANAGER_ADDR_OFFSET)
#define EEPMANAGER_ADDR_DATA_END	(EEPROM_START_ADDR + EEPROM_SIZE)
//	extern void write_eeprom(unsigned char adr,unsigned char data);
//	extern unsigned char read_eeprom(unsigned char adr);
uint32 eepManagerFindAddr(EepManager_Item item);
uint8 eepManagerReadVar(EepManager_Item item, uint32 size, void * data);
uint8 eepManagerWriteVar(EepManager_Item item, uint32 size, const void * data);
uint8 eepManagerCalculateCrc(void);
void eepManagerValid(void);
void eepManagerEmpty(void);
uint8 eepManagerCheckCrc(void);
void eepmanagerUpdateCrc(void);

void init_eepmanager(void) {
	eepManagerReadAll();
}

void do_eepmanager(void) {
}

void isr_eepmanager1ms(void) {
}

void eepManagerReadAll(void) {
	uint8 firstByte = read_eeprom(EEPROM_START_ADDR);
	if (firstByte == 0xFF) {
		eepManagerEmpty();
	} else if (firstByte == 0xAA) {
		if (eepManagerCheckCrc()) {
			uint8 versionCheckResult = 0;
			#ifndef EEPROM_VERSION
				#error Missing EEPROM_VERSION define in config file
			#endif
			#if (EEPROM_VERSION == 0xFF)
				versionCheckResult = 1;
			#else
				uint8 versionByte = read_eeprom(EEPMANAGER_ADDR_VERSION_ADDR);
				if (versionByte == EEPROM_VERSION) {
					versionCheckResult = 1;
				}
			#endif
			if (versionCheckResult) {
				eepManagerValid();
			} else {
				write_eeprom(EEPROM_START_ADDR, 0xFF);
				eepManagerEmpty();
			}
		} else {
			write_eeprom(EEPROM_START_ADDR, 0xFF);
			eepManagerEmpty();
		}
	} else {
		write_eeprom(EEPROM_START_ADDR, 0xFF);
		eepManagerEmpty();
	}
}

uint8 eepManagerCheckCrc(void) {
	uint8 result = 0;
	uint8 crcCalculated = eepManagerCalculateCrc();
	uint8 crcRead = read_eeprom(EEPMANAGER_ADDR_CRC_ADDR);
	if (crcCalculated == crcRead) {
		result = 1;
	}
	return result;
}

uint8 eepManagerCalculateCrc(void) {
	uint32 i = 0;
	uint8 crc = 0;
	for ( i = EEPMANAGER_ADDR_DATA_START; i < EEPMANAGER_ADDR_DATA_END; i++) {
		uint8 data = read_eeprom(i);
		crc ^= data;
	}
	return crc;
}

void eepmanagerUpdateCrc(void) {
	uint8 crcCalculated = eepManagerCalculateCrc();
	write_eeprom(EEPMANAGER_ADDR_CRC_ADDR, crcCalculated);
}

void eepManagerValid(void) {
	//eeprom is valid
	uint32 i = 0;
	for (i = EepManager_Items_FirstItem + 1; i < EepManager_Items_LastItem; i++) {
		uint32 size = eepManager_ItemTable[i - 1].size;
		void *variablePtr = eepManager_ItemTable[i - 1].variablePtr;
		if (variablePtr != NULL) {
			eepManagerReadVar(i, size, variablePtr);
		} else {
			//do nothing
		}
	}
}

void eepManagerEmpty(void) {
	uint32 i = 0;
	for (i = EepManager_Items_FirstItem + 1; i < EepManager_Items_LastItem; i++) {
		uint32 size = eepManager_ItemTable[i - 1].size;
		void *variablePtr = eepManager_ItemTable[i - 1].variablePtr;
		const void *defaultPtr = eepManager_ItemTable[i - 1].defaultPtr;
		if ((defaultPtr != NULL) && (variablePtr != NULL)) {
			eepManagerWriteVar(i, size, defaultPtr);
			memcpy(variablePtr, defaultPtr, size);
		} else {
		}
	}
	write_eeprom(EEPMANAGER_ADDR_VERSION_ADDR, EEPROM_VERSION);
	eepmanagerUpdateCrc();
	write_eeprom(EEPROM_START_ADDR, 0xAA);
}

void eepManagerWriteAll(void) {
	uint32 i = 0;
	write_eeprom(EEPROM_START_ADDR, 0xFF);
	for (i = EepManager_Items_FirstItem + 1; i < EepManager_Items_LastItem; i++) {
		uint32 size = eepManager_ItemTable[i - 1].size;
		void *variablePtr = eepManager_ItemTable[i - 1].variablePtr;
		if (variablePtr != NULL) {
			eepManagerWriteVar(i, size, variablePtr);
		} else {
		}
	}
	eepmanagerUpdateCrc();
	write_eeprom(EEPROM_START_ADDR, 0xAA);
}

void eepManagerWriteAllDefault(void) {
	eepManagerEmpty();
}

uint8 eepManagerReadVar(EepManager_Item item, uint32 size, void * data) {
	uint8 result = 0;
	if (data != NULL) {
		uint32 addr = eepManagerFindAddr(item);
		if (size == eepManager_ItemTable[item - 1].size) {
			uint32 i = 0;
			for (i = 0; i < size; i++) {
				((unsigned char * )data)[i] = read_eeprom(addr + i);
			}
			result = 1;
		}
	}
	return result;
}

uint8 eepManagerWriteVar(EepManager_Item item, uint32 size, const void * data) {
	uint8 result = 0;
	if (data != NULL) {
		uint32 addr = eepManagerFindAddr(item);
		if (size == eepManager_ItemTable[item - 1].size) {
			uint32 i = 0;
			for (i = 0; i < size; i++) {
				uint8 dataTemp = read_eeprom(addr + i);
				uint8 defaultDataTemp = ((unsigned char * )data)[i];
				if (dataTemp != defaultDataTemp) {
					write_eeprom(addr + i, defaultDataTemp);
				}
			}
			result = 1;
		}
	}
	return result;
}

uint32 eepManagerFindAddr(EepManager_Item item) {
	uint32 i = 0;
	uint32 result = EEPROM_START_ADDR + EEPMANAGER_ADDR_OFFSET;
	for (i = EepManager_Items_FirstItem; i < item - 1; i++) {
		result += eepManager_ItemTable[i].size;
	}
	return result;
}
