//#include "mal.h"
#include "nrf24l01.h"
#include "c_nrf24l01.h"
#include "i_nrf24l01.h"
#include "w_nrf24l01.h"
#include "int_nrf24l01.h"
#include "nrf24l01_low.h"
#include "nrf24l01_medium.h"
#include "nrf24l01_nrf_radio.h"
#include "nrf24l01_radio.h"

#include "tmr.h"
#include "rijndael.h"
#include "aes_helper.h"

#include "task.h"

typedef enum _NRF24L01_States {
	NRF24L01_State_Init,
	NRF24L01_State_Init_1,
	NRF24L01_State_Init_2,
	NRF24L01_State_Init_3,
	NRF24L01_State_Idle,
} NRF24L01_States;

unsigned char do_nrf24l01_isr = 0;
NRF24L01_States NRF24L01_State = NRF24L01_State_Idle;
Timer nrf24l01_timer;
TaskId doNrf24L01HandlerTaskNr = 0;

void doNrf24L01RxHandler(uint8 pipe_number);
void doNrf24L01TxHandler(void);
void doNrf24L01Handler(void);
void radio_rxhandler(uint8 pipe_number, radiopacket_t *payload);
unsigned char nrf24l01_assemblyPacket(radiopacket_t *payload, unsigned char type);
unsigned char nrf24l01_disassemblyPacket(radiopacket_t *payload);

void init_nrf24l01(void) {
	init_nrf24l01_medium();
	init_nrf24l01_nrfradio();
	init_nrf24l01_radio();

	removeTimer(&nrf24l01_timer);
	addTimer(&nrf24l01_timer);

	CE_LAT = 0;// disable radio during config
	NRF_PORT_SYNC();
	
	CE_TRIS = 0;
	NRF_PORT_SYNC();

	CSN_LAT = 1;
	NRF_PORT_SYNC();
	CSN_TRIS = 0;	
	NRF_PORT_SYNC();

	IRQ_TRIS = 1;
	NRF_PORT_SYNC();

	NRF24L01_State = NRF24L01_State_Init;
	
	doNrf24L01HandlerTaskNr = addTask(doNrf24L01Handler, 100, 1, "doNrf24L01Handler");
	scheduleTask(doNrf24L01HandlerTaskNr);
}

void do_nrf24l01(void) {
	#ifdef NRF_POLLING
		#ifdef NRF_POLLING_IN_MAIN_CONTEXT
			if ((IRQ_PORT == 0) && (do_nrf24l01_isr == 0)) {
				do_nrf24l01_isr = 1;
				scheduleTask(doNrf24L01HandlerTaskNr);
			}
		#endif
	#endif
}

void doNrf24L01Handler(void) {
	switch (NRF24L01_State) {
		case NRF24L01_State_Init : {
			CE_LAT = 0;// disable radio during config
			NRF_PORT_SYNC();
			writeTimer(&nrf24l01_timer, 11);
			NRF24L01_State = NRF24L01_State_Init_1;
			break;
		}
		case NRF24L01_State_Init_1 : {
			if (readTimer(&nrf24l01_timer) == 0) {
				// Configure the radio registers that are not application-dependent.
				nrf24l01_configure_registers();
				writeTimer(&nrf24l01_timer, 2);
				NRF24L01_State = NRF24L01_State_Init_2;
			}
			break;
		}
		case NRF24L01_State_Init_2 : {
			if (readTimer(&nrf24l01_timer) == 0) {
				// enable radio as a receiver
				CE_LAT = 1;
				NRF_PORT_SYNC();
				NRF24L01_State = NRF24L01_State_Init_3;
			}
			break;
		}
		case NRF24L01_State_Init_3 : {
			uint8 address[5] = {0xA1, 0xA2, 0xA3, 0xA4, NRF_ADDRESSLSB};
			nrf24l01_ConfigureChannel(CHANNEL_CFG, RADIO_2MBPS, RADIO_HIGHEST_POWER, address);
			changeActivationCycleTask(doNrf24L01HandlerTaskNr, 1);
			NRF24L01_State = NRF24L01_State_Idle;
			break;
		}
		case NRF24L01_State_Idle : {
		
			doNrf24L01TxHandler();
			#ifdef NRF_POLLING
				#ifndef NRF_POLLING_IN_MAIN_CONTEXT
					if (IRQ_PORT == 0) {
						do_nrf24l01_isr = 1;
					}
				#endif
			#endif
			if (do_nrf24l01_isr) {
				do_nrf24l01_isr = 0;
				{
					REG_NRF24L01_nrfStatus nrfStatus = nrf24l01_get_nrfStatus();
					if (nrfStatus.RX_DR) {
						uint8 pipe_number = nrfStatus.RX_P_NO;
						doNrf24L01RxHandler(pipe_number);
					}
					if (nrfStatus.MAX_RT) {
						Radio_Transmit_isFinished();
					}
					if (nrfStatus.TX_DS) {
						Radio_Transmit_isFinished();
					}
					if (nrfStatus.TX_FULL) {
						send_instruction(FLUSH_TX, NULL, NULL, 0);
					}
					{
						REG_NRF24L01 regTemp;
						regTemp.nrfStatus = nrfStatus;
						set_register(NRF_REGISTERS_nrfStatus, &regTemp.value, 1);
					}					
				}
			}
			break;
		}
		default : {
			break;
		}
	}
}

void doNrf24L01TxHandler(void) {
	if (get_rxtx_mode() == NRF24L01_RX_MODE) {
		if (isCharInBufferNrfRadioTx() != 0) {
			radiopacket_t payload;
			unsigned char cnt = 0;
			cnt = nrf24l01_assemblyPacket(&payload, RADIO_PACKET_TYPE_NRF_CONTROL);
			if (cnt != 0) {
				Radio_Transmit(&payload);
			}
		} else if (isCharInBufferRadioTx() != 0) {
			radiopacket_t payload;
			unsigned char cnt = 0;
			cnt = nrf24l01_assemblyPacket(&payload, RADIO_PACKET_TYPE_SIO_DATA);
			if (cnt != 0) {
				Radio_Transmit(&payload);
			}
		}
	} else {
		if (Radio_Transmit_isTimeout() == RADIO_TX_TIMEOUT) {
		}
	}
}

void doNrf24L01RxHandler(uint8 pipe_number) {
	RADIO_RX_nrfStatus result;
	radiopacket_t payload;
	send_instruction(R_RX_PL_WID, NULL, &payload.length, 1);
	result = Radio_Receive(&payload);
	if ((result == RADIO_RX_MORE_PACKETS) || (result == RADIO_RX_SUCCESS)) {
		radio_rxhandler(pipe_number, &payload);				
		NRF24L01_NOTIFY_RX_PACKET();
	}
	if (result == RADIO_RX_MORE_PACKETS) {
		scheduleTask(doNrf24L01HandlerTaskNr);
	}
}

void radio_rxhandler(uint8 pipe_number, radiopacket_t *payload) {
	if (pipe_number == RADIO_PIPE_0) {
		nrf24l01_disassemblyPacket(payload);
	}
}

unsigned char nrf24l01_disassemblyPacket(radiopacket_t *payload) {
	unsigned char cnt = 0;
	if (payload != NULL) {
		uint8 i = 0;
		{
			radiopacket_payload_t payload_out;
			unsigned int sizeOut = 0;
			if (payload->length == 32) {
				aes_decrypt(payload->payload.data, payload->length, payload_out.data, &sizeOut);
				switch (payload_out.structure.type) {
					case RADIO_PACKET_TYPE_NRF_CONTROL: {//NRF Control
						for (i = 0; i < payload_out.structure.cnt; i++) {
							putCharNrfRadioRx(payload_out.structure.data[i]);
						}
						break;
					}
					case RADIO_PACKET_TYPE_SIO_DATA: {//SIO
						for (i = 0; i < payload_out.structure.cnt; i++) {
							putCharRadioRx(payload_out.structure.data[i]);
						}
						break;
					}
					default : {
						break;
					}
				}
			} else {
				return;
			}
		}
	}
	return cnt;
}

unsigned char nrf24l01_assemblyPacket(radiopacket_t *payload, unsigned char type) {
	unsigned char cnt = 0;
	if (payload != NULL) {
		unsigned char packetSize = sizeof(payload->payload.structure.data) / sizeof(*payload->payload.structure.data);
		memset(payload->payload.data, 0xFF, (sizeof(payload->payload.data) / sizeof(*payload->payload.data)) );
		for (cnt = 0; cnt < packetSize; cnt++) {
			int dataTemp = -1;
			switch (type) {
				case RADIO_PACKET_TYPE_NRF_CONTROL: {//NRF Control
					dataTemp = getCharNrfRadioTx();
					break;
				}
				case RADIO_PACKET_TYPE_SIO_DATA: {//SIO
					dataTemp = getCharRadioTx();
					break;
				}
				default : {
					break;
				}
			}
			if (dataTemp != -1) {
				payload->payload.structure.data[cnt] = (unsigned char)dataTemp; //Length is the first byte
			} else {
				break;
			}
		}
		if (cnt != 0) {
			RADIO_TX_nrfStatus result;
			payload->length = cnt;
			{
				unsigned char dataOut[32];
				unsigned int sizeOut = 0;
				uint32 globalTimeValue = getGlobalTime();
				payload->payload.structure.cnt = cnt;
				payload->payload.structure.type = type;
				payload->payload.structure.time = globalTimeValue;
				aes_encrypt(payload->payload.data, (sizeof(payload->payload.data) / sizeof(*payload->payload.data)), dataOut, &sizeOut);
				memcpy(payload->payload.data, dataOut, (sizeof(dataOut) / sizeof(*dataOut)) );
				payload->length = (sizeof(dataOut) / sizeof(*dataOut));
			}
		}
	}
	return cnt;
}

void isr_nrf24l01_1ms(void) {
	
}

void isr_nrf24l01(void) {
	#ifndef NRF_POLLING
		do_nrf24l01_isr = 1;
	#endif
}
