#include <stdio.h>
#include <string.h>
#include "ds18b20_multi.h"
#include "c_ds18b20_multi.h"
#include "i_ds18b20_multi.h"
#include "w_ds18b20_multi.h"

#include "k_stdtype.h"
#include "task.h"
#include "onewire.h"
#include "onewiresearch.h"
#include "ds18b20.h"

typedef enum _DS18B20_MULTI_STATES {
	DS18B20_MULTI_STATES_SEARCH_INIT = 0,
	DS18B20_MULTI_STATES_SEARCH = 1,
	DS18B20_MULTI_STATES_SEARCH_FINISH = 2,	
	DS18B20_MULTI_STATES_MEASURING = 3,	
} DS18B20_MULTI_STATES;

unsigned char temperatureValid[NUMBER_OF_DS18B20];
OW_device ds18b20_devices_multi[NUMBER_OF_DS18B20];
uint8 deviceFound[NUMBER_OF_DS18B20];
float temperature[NUMBER_OF_DS18B20];
uint8 currentMeasurementItem = 0;
uint8 restartCounter = 0;
uint8 restartFlag = 0;

DS18B20_MULTI_STATES ds18b20_states = DS18B20_MULTI_STATES_SEARCH_INIT;
TaskId task_ds18b20_multi_TaskId = -1;
void task_ds18b20_multi(void);
void stepMeasurementItem(void);

void init_ds18b20_multi(void) {
	ds18b20_states = DS18B20_MULTI_STATES_SEARCH_INIT;
	removeTask(task_ds18b20_multi_TaskId);
	task_ds18b20_multi_TaskId = addTask(task_ds18b20_multi, 100, 1, (const char *)"task_ds18b20_multi");
}

void task_ds18b20_multi(void) {
	switch (ds18b20_states) {
		case DS18B20_MULTI_STATES_SEARCH_INIT : {
			uint8 i = 0;
			for (i = 0; i < NUMBER_OF_DS18B20; i++) {
				temperatureValid[i] = 0;
				memset(ds18b20_devices_multi[i].byte, 0x00, 8);
				deviceFound[i] = 0;
				temperature[i] = 0.0;
			}
			currentMeasurementItem = 0;
			set_OW_addressing_mode(OW_SKIP_ROM_ADDRESS);
			restartCounter = 0;
			ds18b20_states = DS18B20_MULTI_STATES_SEARCH;
			scheduleTask(task_ds18b20_multi_TaskId);
			break;
		}
		case DS18B20_MULTI_STATES_SEARCH : {
			uint8 deviceToSearch = 0;
			uint8 i = 0;
			uint8 restartSearch = 0;
			for (i = 0; i < NUMBER_OF_DS18B20; i++) {
				deviceToSearch = i;
				if (deviceFound[i] == 0) {
					break;
				}
			}
			if (deviceToSearch == 0) {
				restartSearch = 1;
			}
			deviceFound[deviceToSearch] = onewiresearch_getDevice(&ds18b20_devices_multi[deviceToSearch], restartSearch);
			if (deviceFound[deviceToSearch] == 0) {
				#ifndef THREAT_NUMBER_OF_DS18B20_SOFT
					ds18b20_states = DS18B20_MULTI_STATES_SEARCH_INIT;
				#else
					//There are less sensors than initial configured
					ds18b20_states = DS18B20_MULTI_STATES_SEARCH_FINISH;
					scheduleTask(task_ds18b20_multi_TaskId);
				#endif
			} else {
				if (deviceToSearch == (NUMBER_OF_DS18B20 - 1)) {
					ds18b20_states = DS18B20_MULTI_STATES_SEARCH_FINISH;
					scheduleTask(task_ds18b20_multi_TaskId);
				}
			}
			break;
		}
		case DS18B20_MULTI_STATES_SEARCH_FINISH : {
			uint8 i = 0;
			set_OW_addressing_mode(OW_MATCH_ROM_ADDRESS);
			for (i = 0; i < NUMBER_OF_DS18B20; i++) {
				ds18b20_init(ds18b20_devices_multi[i]);	
			}
			ds18b20_states = DS18B20_MULTI_STATES_MEASURING;
			scheduleTask(task_ds18b20_multi_TaskId);
			break;
		}
		case DS18B20_MULTI_STATES_MEASURING : {
			int result = 0;
			if (deviceFound[currentMeasurementItem] != 0) {
				if (restartCounter != 0) {
					restartCounter--;
					result = ds18b20_get_temperature_nb_restart(ds18b20_devices_multi[currentMeasurementItem], &temperature[currentMeasurementItem], restartFlag);
					restartFlag = 0;
					if (result == 0) {
						//busy
					} else if (result == 1) {
						temperatureValid[currentMeasurementItem] = 1;
						stepMeasurementItem();
						//ok
					} else {
						temperatureValid[currentMeasurementItem] = 0;
						stepMeasurementItem();
						//error
					}	
				} else {
					stepMeasurementItem();
				}
			} else {
				stepMeasurementItem();
			}
			break;
		}
		default : {
			ds18b20_states = DS18B20_MULTI_STATES_SEARCH_INIT;
			break;
		}
	}
}

void stepMeasurementItem(void) {
	restartCounter = 50;
	restartFlag = 1;
	currentMeasurementItem++;
	if (currentMeasurementItem >= NUMBER_OF_DS18B20) {
		currentMeasurementItem = 0;
	}
}

void isr_ds18b20_multi1ms(void) {
}

float getTemperature(uint8 sensor) {
	float result = 0.0;
	if (sensor < NUMBER_OF_DS18B20) {
		result = temperature[sensor];
	}
	return result;
}

uint8 getTemperatureValidity(uint8 sensor, float *temperature, uint8 *validity) {
	uint8 result = 0;
	if ((sensor < NUMBER_OF_DS18B20) && (temperature != NULL) && (validity != NULL)) {
		*temperature = temperature[sensor];
		*validity = temperatureValid[sensor];
		result = 1;
	}
	return result;
}
