#include <math.h>
#include "SHT1x_calc.h"

/*
*********************************************************************************************************
*                         				CALCULATE RH AND TEMPERATURE
*
* Description :	This function returns temperature and relative humidity in floating point values
* Arguments   : float '*p_humidity'		pointer to humidity
*				float '* p_temperature' pointer to temperature
*				
* Returns     : none
* Notes		  : humidity in 12 bit raw data and temperature in 14 bit raw data passed to this
*				function, where at the same ram location, floating point values of temperature and
*				relative humidity value will be returned. 
*				Reference: an Keil C example for 8051 from Sensirion at www.sensirion.com
*********************************************************************************************************
*/	
char calc_sht1x(float *p_humidity ,float *p_temperature) {
	char result = 1;
// calculates temperature [C] and humidity [%RH] 
// input :  humi [Ticks] (12 bit) 
//          temp [Ticks] (14 bit)
// output:  humi [%RH]
//          temp [deg Cel]
	float rh = *p_humidity; // rh:      Humidity [Ticks] 12 Bit 
	float t = *p_temperature; // t:       Temperature [Ticks] 14 Bit
	float rh_lin = 0.0; // rh_lin:  Humidity linear
	float rh_true = 0.0; // rh_true: Temperature compensated humidity
	float t_C = 0.0; // t_C   :  Temperature [C]

	t_C = t * 0.01 - 40.0; //calc. temperature from ticks to [deg Cel]	
	rh_lin = 0.0405 * rh - 0.0000028 * rh * rh - 4.0; //calc. humidity from ticks to [%RH]
	rh_true=(t_C - 25.0) * (0.01 + 0.00008 * rh) + rh_lin; //calc. temperature compensated humidity [%RH]
	if (rh_true > 100.0) {
		rh_true = 100.0; //cut if the value is outside of
		result = 0;
	}
	if (rh_true < 0.1) {
		rh_true = 0.1; //the physical possible range
		result = 0;
	}

	*p_temperature = t_C;               				//return temperature [deg Cel]
	*p_humidity = rh_true;              				//return humidity[%RH]
	return result;
}

float SHT_Calc_Dewpoint(float h, float t) { 
//--------------------------------------------------------------------   
// calculates dew point
// input:   humidity [%RH], temperature [C]
// output:  dew point [C]
	float logEx = 0.0;
	float dew_point = 0.0;
	logEx = 0.66077 + 7.5 * t / (237.3 + t) + (log10(h) - 2.0);
	dew_point = (logEx - 0.66077) * 237.3 / (0.66077 + 7.5 - logEx);
	return dew_point;
}
