#include <string.h>
#include "sio.h"
#include "w_sio.h"
#include "c_sio.h"
#include "i_sio.h"
#include "e_sio.h"

#include "mal.h"
#include "ringbuffer.h"

#ifdef USE_RS232_USART1
	#include "serial_usart1.h"
	#define USE_RS232_LOCAL
#endif
#ifdef USE_RS232_USART2
	#include "serial_usart2.h"
	#define USE_RS232_LOCAL
#endif
#ifdef USE_USB
	#include "serial_usb.h"
#endif
#ifdef USE_RADIO_NRF
	#include "nrf24l01_radio.h"
#endif
#ifdef USE_ESP8266_SLAVE
	#include "esp8266_radio.h"
#endif
#ifdef USE_GSM_SMS_SLAVE
	#include "gsm_sms_radio.h"
#endif
#ifdef USE_I2C_SLAVE
	#include "iic_slave.h"
#endif
#ifdef USE_CAN
	#include "can_slave.h"
	#include "i_can_sio.h"
	#include "w_can_sio.h"
#endif
#ifdef USE_TEST
	#include "sio_test.h"
#endif

#if defined(USE_RS232_LOCAL) || defined(USE_I2C_SLAVE) || defined(USE_CAN) || defined(USE_RADIO_NRF) || defined(USE_USB) || defined(USE_ESP8266_SLAVE) || defined(USE_GSM_SMS_SLAVE) || defined(USE_TEST)
	//Ok since at least one is defined
#else
	#error One communication medium must be defined (USE_RS232_USART1, USE_RS232_USART2, USE_CAN or USE_I2C_SLAVE, USE_GSM_SMS_SLAVE, USE_RADIO_NRF) in c_sio.h
#endif

uint8 doPacket = 0;
uint8 doSio1ms = 0;
RingBuffer myRingBuffer_sio_rx;
uint8 myRingBuffer_sio_rx_buffer[SIO_BUFFER_CNT];
uint8 sioCurrentMessageBuffer[SIO_BUFFER_CNT];
uint32 sio_crc32 = 0;

void sioProcessNewData(RingBuffer *sioRingBuffer, int ch);
uint8 isSioPacket(RingBuffer *sioRingBuffer, unsigned int *startCnt, unsigned int *length);
void processSioPacket(uint8 *data, uint16 length);
uint32 sio_calulate_crc32(RingBuffer *sioRingBuffer, uint16 start, uint16 length);

void init_sio(void) {
	ringBuffer_initBuffer(&myRingBuffer_sio_rx, myRingBuffer_sio_rx_buffer, SIO_BUFFER_CNT);

	#ifdef USE_CAN
		init_do_can_sio();
	#endif
}

void isr_communication(void) {
	doPacket = 1;

	#ifdef USE_CAN
		do_can_sio_read = 1;
	#endif
}

void isr_sio1ms(void) {
	#ifdef USE_TEST
		extern void isr_sio_test_1ms(void);
		isr_sio_test_1ms();
	#endif
	#ifdef USE_CAN
		isr_can_sio1ms();
	#endif
	{
		doSio1ms = 1;
	}
}

void do_sio(void) {
	#ifdef USE_CAN
		do_can_sio();
	#endif
	if (doSio1ms) {
		doSio1ms = 0;
		doPacket = 1;
	}
	
	if (doPacket) {
		doPacket = 0;
		while (1) {
			int ch = -1;
			ch = getCharWrapper();
			if (ch != -1) {
				int temp = ch; //added because of MCC18 compiler swaps data bug
				sioProcessNewData(&myRingBuffer_sio_rx, temp);
			} else {
				break;
			}
		}
	}
}

void sioProcessNewData(RingBuffer *sioRingBuffer, int ch) {
	if ((ch != -1) && (sioRingBuffer != NULL)) {
		if (ringBuffer_addItem(sioRingBuffer, (unsigned char)ch) == 0) {
			if (ch == CMD_END) {
				unsigned int startCnt = 0;
				unsigned int length = 0;
				if (isSioPacket(sioRingBuffer, &startCnt, &length)) {
					unsigned int i = 0;
					unsigned char tempCh = 0;
					for (i = 0; i < startCnt; i++) {//drop characters from front of the message (possible some garbage or a previous corrupt message
						ringBuffer_getItem(sioRingBuffer, &tempCh);
					}
					{//Copy actual message to the working buffer
						for (i = 0; i < length; i++) {
							ringBuffer_getItem(sioRingBuffer, &tempCh);
							sioCurrentMessageBuffer[i] = tempCh;
						}
						processSioPacket(sioCurrentMessageBuffer, length);
					}
					ringBuffer_initBuffer(sioRingBuffer, myRingBuffer_sio_rx_buffer, SIO_BUFFER_CNT);
				}
			}
		}
	}
}

void processSioPacket(uint8 *data, uint16 length) {
	uint8 start = data[PACKET_START_POS];
	uint16 address = (data[PACKET_ADDRESSH_POS] << 8) + data[PACKET_ADDRESSL_POS];
	uint16 command = (data[PACKET_COMMANDH_POS] << 8) + data[PACKET_COMMANDL_POS];
	uint8 *payload = data + PACKET_PAYLOAD_POS;
	uint16 payloadLengthPlaus = length - MINIMUM_PACKET_SIZE;
	uint16 payloadLength = (data[PACKET_LENGTHH_POS] << 8) + data[PACKET_LENGTHL_POS];

	if (address == 0xFFFF) {//Answer packet, need to root to PC
		uint16 x = 0;
		for (x = 0; x < length; x++) {
			putCharMaster_Wrapper(data[x]);
		}
	} else if (address != SIO_ADDRESS) {//Command packet, need to root to Slave
		uint16 x = 0;
		for (x = 0; x < length; x++) {
			putCharSlave_Wrapper(data[x]);
		}
	} else {//Command packet, for me
		switch (command) {
			case CMD_VERSIONID : {
				RdVersionID();
				break;
			}
			case CMD_VERSIONSTR : {
				RdVersionStr();
				break;
			}
			case CMD_RDPARAMBUF : {
				RdParamBuf(payload, payloadLength);
				break;
			}
			case CMD_WRPARAMBUF : {
				WrParamBuf(payload, payloadLength);
				break;
			}
			case CMD_RDPARAM : {
				RdParam(payload, payloadLength);
				break;
			}
			case CMD_WRPARAM : {
				WrParam(payload, payloadLength);
				break;
			}
			case CMD_RDPARAMW : {
				RdParamW(payload, payloadLength);
				break;
			}
			case CMD_WRPARAMW : {
				WrparamW(payload, payloadLength);
				break;
			}
			case CMD_RDPARAML : {
				RdParamL(payload, payloadLength);
				break;
			}
			case CMD_WRPARAML : {
				WrParamL(payload, payloadLength);
				break;
			}
			case CMD_RDPARAMF : {
				RdParamF(payload, payloadLength);
				break;
			}
			case CMD_WRPARAMF : {
				WrParamF(payload, payloadLength);
				break;
			}
			case CMD_RESET : {
				mal_reset();
				break;
			}
			case CMD_GETSEED : {
				GetSeed(payload, payloadLength);
				break;
			}
			case CMD_CHECKKEY : {
				CheckKey(payload, payloadLength);
				break;
			}
			case CMD_GETVALIDID : {
				GetValidId(payload, payloadLength);
				break;
			}
			default : {
				break;
			}
		}
	}
}

uint8 isSioPacket(RingBuffer *sioRingBuffer, unsigned int *startCnt, unsigned int *length) {
	uint8 result = 0;
	if ((sioRingBuffer != NULL) && (startCnt != NULL) && (length != NULL)) {
		uint16 bytesInBufferTemp = ringBuffer_getFillness(sioRingBuffer);
		uint16 i = 0;
		uint16 j = 0;
		uint16 k = 0;
		unsigned char tempCh = 0;
		if (bytesInBufferTemp >= MINIMUM_PACKET_SIZE) {
			for (i = 0; i < bytesInBufferTemp; i++) {
				unsigned char Start = 0;
				ringBuffer_readItem(sioRingBuffer, &Start, i + PACKET_START_POS);
				if (Start == CMD_START) {//Search for start of a packet
					j = bytesInBufferTemp;
					for (k = i; k < bytesInBufferTemp; k++) {
						j--;
						ringBuffer_readItem(sioRingBuffer, &tempCh, j); //PACKET_STOP_POS
						if (tempCh == CMD_END) {//Search for end of a packet
							//Now check if packet is valid
							unsigned int lengthTemp = j - i + 1;
							if (lengthTemp >= MINIMUM_PACKET_SIZE) {	//Minimum packet size is 14 byte.
								uint16 expectedLength = 0;
								unsigned char tempChH = 0;
								unsigned char tempChL = 0;
								ringBuffer_readItem(sioRingBuffer, &tempChH, i + PACKET_LENGTHH_POS);
								ringBuffer_readItem(sioRingBuffer, &tempChL, i + PACKET_LENGTHL_POS);
								expectedLength = tempChH;
								expectedLength <<= tempChH;
								expectedLength += tempChL;
								expectedLength += MINIMUM_PACKET_SIZE; //because of command and crc
								if (expectedLength == lengthTemp) {
									unsigned char tempChHH = 0;
									unsigned char tempChHL = 0;
									unsigned char tempChLH = 0;
									unsigned char tempChLL = 0;
									uint32 crc_calculated = 0;
									uint32 crc_received = 0;

									ringBuffer_readItem(sioRingBuffer, &tempChHH, i + PACKET_CRCHH_POS);
									ringBuffer_readItem(sioRingBuffer, &tempChHL, i + PACKET_CRCHL_POS);
									ringBuffer_readItem(sioRingBuffer, &tempChLH, i + PACKET_CRCLH_POS);
									ringBuffer_readItem(sioRingBuffer, &tempChLL, i + PACKET_CRCLL_POS);
									crc_received = tempChHH;
									crc_received <<= 8;
									crc_received += tempChHL;
									crc_received <<= 8;
									crc_received += tempChLH;
									crc_received <<= 8;
									crc_received += tempChLL;
									
									sio_calulate_crc32_start();
									{
										unsigned int crcItem = 0;
										unsigned char chItem = 0;
										for (crcItem = 0; crcItem < lengthTemp; crcItem++) {
											if ((crcItem >= (PACKET_CRCHH_POS)) && (crcItem <= (PACKET_CRCLL_POS))) {
												chItem = 0;
											} else {
												ringBuffer_readItem(sioRingBuffer, &chItem, i + crcItem);
											}
											sio_calulate_crc32_add(chItem);
										}
									}
									crc_calculated = sio_calulate_crc32_get();
									
									if (crc_calculated == crc_received) {
										*startCnt = i;
										*length = lengthTemp;
										result = 1;
										break;
									}
								}
							}
						}
					}
				}
				if (result) {
					break;
				}
			}
		}
	}
	return result;
}

void sio_calulate_crc32_start(void) {
	sio_crc32 = 0xCAFECAFE;
}

void sio_calulate_crc32_add(unsigned char tempCh) {
	uint16 j = 0;
	uint32 byte = 0;
	uint32 mask = 0;
	sio_crc32 += tempCh;

	byte = tempCh;            // Get next byte.
	sio_crc32 = sio_crc32 ^ byte;
	for (j = 0; j < 8; j++) {    // Do eight times.
		mask = -(sio_crc32 & 1);
		sio_crc32 = (sio_crc32 >> 1) ^ (0xEDB88320 & mask);
	}	
}

uint32 sio_calulate_crc32_get(void) {
	uint32 result = 0xCAFECAFE;
	sio_crc32 = ~sio_crc32;
	result = sio_crc32;
	return result;
}
