#include <string.h>
#include "temperature.h"
#include "c_temperature.h"
#include "i_temperature.h"
#include "w_temperature.h"

#include "task.h"

#ifdef USE_PT100
	#include "adcint.h"
	#include "pt100_conversion.h"
#endif

#ifdef USE_DS18B20
	#include "ds18b20.h"
#endif

#ifdef USE_SHT11
	#include "SHT1x.h"
#endif

unsigned char thermostat_temperature_valid = 0;
unsigned char thermostat_temperature_validExternal = 0;
float temperature = 0.0;
float temperatureExternal = 0.0;

#ifdef USE_PT100
	float pt100Temp = 0.0;
	float convertAdcToTemp(unsigned int adcValue);
	float currentSource = 0.000258; //uA
	float voltageAmp = 64.9; //A
	float voltageReference = 4.094; //V
	float conversionValue = 0.0;
	void doPT100(void);
#endif

#ifdef USE_DS18B20
	float ds18b20Temp = 0.0;
	unsigned char ds18b20_temperature_valid = 0;
	OW_device thermostat_ow_device;
	void doOneWire250ms(void);
	void temperatureSingleShoot(void);
#endif

#ifdef USE_SHT11
	float sht11Temp = 0.0;
	void doSHT11(void);
#endif

TemperatureSource temperatureSource = TemperatureSource_DS18B20;
TemperatureSource temperatureSourceExternal = TemperatureSource_DS18B20;

void doTemperatureRead(void);

void init_temperature(void) {
	#ifdef USE_PT100
		addInitTask(calculateConversionValue);
		addTask(doPT100, 10, 1);
	#endif
	addTask(doTemperatureRead, 250, 1, "doTemperatureRead");
	#ifdef USE_DS18B20
		addSingleShootTask(temperatureSingleShoot, "temperatureSingleShoot");
	#endif
	#ifdef USE_SHT11
		addTask(doSHT11, 250, 1, "doSHT11");
	#endif
}

void do_temperature(void) {
}

#ifdef USE_SHT11
	void doSHT11(void) {
		sht11Temp = getSHTTemp();
	}
#endif

#ifdef USE_PT100
	void doPT100(void) {
		static float tempFilter = 0.0;
		static unsigned int cnt = 0;
		unsigned int value = 0;
		getADCValue(0, &value);
		tempFilter += convertAdcToTemp(value);
		cnt++;
		if (cnt >= 100) {
			cnt = 0;
			pt100Temp = tempFilter / 100.0;
			tempFilter = 0.0;
		}
	}	
#endif


#ifdef USE_DS18B20
	void doOneWire250ms(void) {
		int result = 0;
		result = ds18b20_get_temperature_nb(thermostat_ow_device, &ds18b20Temp);
		if (result == 0) {
			//busy
			ds18b20_temperature_valid = 1;
		} else if (result == 1) {
			//ok
			ds18b20_temperature_valid = 1;
		} else {
			//error
			ds18b20_temperature_valid = 0;
		}
	}
#endif

#ifdef USE_DS18B20
	void temperatureSingleShoot(void) {
			thermostat_ow_device.value.family = 0x10; //TODO
			thermostat_ow_device.value.serial_number[0] = 0x01;
			thermostat_ow_device.value.serial_number[1] = 0x01; 
			thermostat_ow_device.value.serial_number[2] = 0x01;
			thermostat_ow_device.value.serial_number[3] = 0x01;
			thermostat_ow_device.value.serial_number[4] = 0x01;
			thermostat_ow_device.value.serial_number[5] = 0x01;
			thermostat_ow_device.value.crc = 0x01;
			ds18b20_init(thermostat_ow_device);
			addTask(doOneWire250ms, 250, 1, "doOneWire250ms");
	}
#endif

void doTemperatureRead(void) {
	switch (temperatureSource) {
		#ifdef USE_PT100
			case TemperatureSource_PT100 : {
					temperature = pt100Temp;
					thermostat_temperature_valid = 1;
				break;
			}
		#endif
		#ifdef USE_DS18B20
			case TemperatureSource_DS18B20 : {
					temperature = ds18b20Temp;
					thermostat_temperature_valid = ds18b20_temperature_valid;
				break;
			}
		#endif
		#ifdef USE_SHT11
			case TemperatureSource_SHT11 : {
					temperature = sht11Temp;
					thermostat_temperature_valid = 1;
				break;
			}
		#endif
		default : {
			temperature = 0.0;
			thermostat_temperature_valid = 0;
			break;
		}
	}
	switch (temperatureSourceExternal) {
		#ifdef USE_PT100
			case TemperatureSource_PT100 : {
					temperatureExternal = pt100Temp;
					thermostat_temperature_validExternal = 1;
				break;
			}
		#endif
		#ifdef USE_DS18B20
			case TemperatureSource_DS18B20 : {
					temperatureExternal = ds18b20Temp;
					thermostat_temperature_validExternal = ds18b20_temperature_valid;
				break;
			}
		#endif
		#ifdef USE_SHT11
			case TemperatureSource_SHT11 : {
					temperatureExternal = sht11Temp;
					thermostat_temperature_validExternal = 1;
				break;
			}
		#endif
		default : {
			temperatureExternal = 0.0;
			thermostat_temperature_validExternal = 0;
			break;
		}
	}
}

void isr_temperature1ms(void) {
}

#ifdef USE_PT100
	float convertAdcToTemp(unsigned int adcValue) {
		float result = 0;
		float resistance = 0.0;
	
		resistance = adcValue;
		resistance *= conversionValue;
	
		result = convertRtoT(resistance);
		return result;
	}
#endif

#ifdef USE_PT100
	void calculateConversionValue(void) {
			conversionValue = voltageReference;
			conversionValue /= 4094.0;
			conversionValue /= voltageAmp;
			conversionValue /= currentSource;
	}
#endif

void setSource(TemperatureSource newValue) {
	temperatureSource = newValue;
}

void setSourceExternal(TemperatureSource newValue) {
	temperatureSourceExternal = newValue;
}

TemperatureSource getSource(void) {
	return temperatureSource;
}

TemperatureSource getSourceExternal(void) {
	return temperatureSourceExternal;
}

int getTemperature(float *temp) {
	int result = -1;
	if (temp != NULL) {
		*temp = temperature;
		result = thermostat_temperature_valid;
	}
	return result;
}

int getTemperatureExternal(float *temp) {
	int result = -1;
	if (temp != NULL) {
		*temp = temperatureExternal;
		result = thermostat_temperature_valid;
	}
	return result;
}
