#include "nrf24l01.h"
#include "c_nrf24l01.h"
#include "i_nrf24l01.h"
#include "w_nrf24l01.h"
#include "int_nrf24l01.h"
#include "nrf24l01_low.h"
#include "nrf24l01_medium.h"
#include "nrf24l01_radio.h"

Timer nrf24l01_tx_timer;
#ifdef NRF24L01_USE_BUFFERED_RXTX_MODE
	NRF24L01_RxTxMode nrf24l01_RxTxMode_buffered = NRF24L01_RX_MODE;
#endif

void init_nrf24l01_medium(void) {
	removeTimer(&nrf24l01_tx_timer);
	addTimer(&nrf24l01_tx_timer);
}

void Radio_Transmit(radiopacket_t* payload) {
	// disable the radio while writing to the Tx FIFO.
	CE_LAT = 0;
	NRF_PORT_SYNC();

	set_tx_mode();

	// transfer the packet to the radio's Tx FIFO for transmission
	send_instruction(W_TX_PAYLOAD, payload->payload.data, NULL, payload->length);

	// start the transmission.
	CE_LAT = 1;
	NRF_PORT_SYNC();

	writeTimer(&nrf24l01_tx_timer, NRF_TX_TIMEOUT);
}

RADIO_TX_nrfStatus Radio_Transmit_isFinished(void) {
	RADIO_TX_nrfStatus result = RADIO_TX_BUSY;

	REG_NRF24L01_nrfStatus nrfStatus = nrf24l01_get_nrfStatus();
	if (nrfStatus.TX_DS) {
		set_rx_mode();
		result = RADIO_TX_SUCCESS;
	} else if (nrfStatus.MAX_RT) {
		set_rx_mode();
		result = RADIO_TX_MAX_RT;
	}
	return result;
}

RADIO_TX_nrfStatus Radio_Transmit_isTimeout(void) {
	RADIO_TX_nrfStatus result = RADIO_TX_BUSY;

	if (readTimer(&nrf24l01_tx_timer) == 0) {
		set_rx_mode();
		result = RADIO_TX_TIMEOUT;
	}
	return result;
}

RADIO_RX_nrfStatus Radio_Receive(radiopacket_t* payload) {
	RADIO_RX_nrfStatus result = RADIO_RX_FIFO_EMPTY;
	REG_NRF24L01_nrfStatus nrfStatus;
	
	CE_LAT = 0;
	NRF_PORT_SYNC();
	nrfStatus = nrf24l01_get_nrfStatus();

	if (nrfStatus.RX_P_NO == RADIO_PIPE_EMPTY) {
		result = RADIO_RX_FIFO_EMPTY;
	} else {
		// Move the data payload into the local
		send_instruction(R_RX_PAYLOAD, NULL, payload->payload.data, payload->length);
		nrfStatus = nrf24l01_get_nrfStatus();
		if (nrfStatus.RX_P_NO != RADIO_PIPE_EMPTY) {
			result = RADIO_RX_MORE_PACKETS;
		} else {
			result = RADIO_RX_SUCCESS;
		}
	}
	CE_LAT = 1;
	NRF_PORT_SYNC();
	return result;
}

// default transmitter address is 0xe7e7e7e7e7.
void Radio_Set_Tx_Addr(uint8* address) {
	set_register(NRF_REGISTERS_TX_ADDR, address, ADDRESS_LENGTH);
}

NRF24L01_RxTxMode get_rxtx_mode(void) {
	NRF24L01_RxTxMode result = NRF24L01_TX_MODE;
	#ifdef NRF24L01_USE_BUFFERED_RXTX_MODE
		result = nrf24l01_RxTxMode_buffered;
	#else
		REG_NRF24L01 config;
		get_register(NRF_REGISTERS_CONFIG, &config.value, 1);
		if ((config.config.PRIM_RX) == 1) {
			result = NRF24L01_RX_MODE;
		} else {
			result = NRF24L01_TX_MODE;
		}
	#endif
	return result;
}

/**
 * Switch the radio to receive mode.  If the radio is already in receive mode, this does nothing.
 */
void set_rx_mode(void) {
	REG_NRF24L01 config;
	get_register(NRF_REGISTERS_CONFIG, &config.value, 1);
	if ((config.config.PRIM_RX) == 0) {
		config.config.PRIM_RX = 1;
		set_register(NRF_REGISTERS_CONFIG, &config.value, 1);
		// the radio takes 130 us to power up the receiver.
		nrf24l01_delay_65us();
		nrf24l01_delay_65us();
		#ifdef NRF24L01_USE_BUFFERED_RXTX_MODE
			nrf24l01_RxTxMode_buffered = NRF24L01_RX_MODE;
		#endif
	}
}

/**
 * Switch the radio to transmit mode.  If the radio is already in transmit mode, this does nothing.
 */
void set_tx_mode(void) {
	REG_NRF24L01 config;
	get_register(NRF_REGISTERS_CONFIG, &config.value, 1);
	if ((config.config.PRIM_RX) != 0) {
		config.config.PRIM_RX = 0;
		set_register(NRF_REGISTERS_CONFIG, &config.value, 1);
		// The radio takes 130 us to power up the transmitter
		// You can delete this if you're sending large packets (I'm thinking > 25 bytes, but I'm not sure) because it
		// sending the bytes over SPI can take this long.
		nrf24l01_delay_65us();
		nrf24l01_delay_65us();
		#ifdef NRF24L01_USE_BUFFERED_RXTX_MODE
			nrf24l01_RxTxMode_buffered = NRF24L01_TX_MODE;
		#endif
	}
}

/**
 * Configure radio defaults and turn on the radio in receive mode.
 * This configures the radio to its max-power, max-packet-header-length settings.  If you want to reduce power consumption
 * or increase on-air payload bandwidth, you'll have to change the config.
 */
void nrf24l01_configure_registers(void) {
	{
		unsigned char value = 0x73;
		set_register(ACTIVATE, &value, 1);
	}

	{
		unsigned char value = 0;
		set_register(NRF_REGISTERS_EN_RXADDR, &value, 1);
	}
	{
		unsigned char value = 0;
		REG_NRF24L01 feature;
		get_register(NRF_REGISTERS_FEATURE, &value, 1);
		feature.value = value;
		feature.feature.EN_DPL = 1;
		set_register(NRF_REGISTERS_FEATURE, &feature.value, 1);
	}
	{
		REG_NRF24L01 en_aa;
		get_register(NRF_REGISTERS_EN_AA, &en_aa.value, 1);	
		en_aa.en_aa.ENAA_P0 = 0;
		en_aa.en_aa.ENAA_P1 = 0;
		en_aa.en_aa.ENAA_P2 = 0;
		en_aa.en_aa.ENAA_P3 = 0;
		en_aa.en_aa.ENAA_P4 = 0;
		en_aa.en_aa.ENAA_P5 = 0;
		set_register(NRF_REGISTERS_EN_AA, &en_aa.value, 1);
	}
	{
		REG_NRF24L01 dynpd;
		get_register(NRF_REGISTERS_DYNPD, &dynpd.value, 1);
		dynpd.dynpd.DPL_P0 = 1;
		dynpd.dynpd.DPL_P1 = 1;
		dynpd.dynpd.DPL_P2 = 1;
		dynpd.dynpd.DPL_P3 = 1;
		dynpd.dynpd.DPL_P4 = 1;
		dynpd.dynpd.DPL_P5 = 1;
		set_register(NRF_REGISTERS_DYNPD, &dynpd.value, 1);
	}
	
	{
		REG_NRF24L01 setup_aw;
		// set address width to 5 bytes.
		setup_aw.value = 0;
		setup_aw.setup_aw.AW = ADDRESS_LENGTH - 2;
		set_register(NRF_REGISTERS_SETUP_AW, &setup_aw.value, 1);
	}

	{
		REG_NRF24L01 setup_retr;
		setup_retr.value = 0;
		// set Enhanced Shockburst retry to every 586 us, up to 5 times.  If packet collisions are a problem even with AA enabled,
		// then consider changing the retry delay to be different on the different stations so that they do not keep colliding on each retry.
		setup_retr.setup_retr.ARD = 0x01;
		setup_retr.setup_retr.ARC = 0x05;
		set_register(NRF_REGISTERS_SETUP_RETR, &setup_retr.value, 1);
	}

	{
		uint8 value = CHANNEL_CFG; // Set to use 2.4 GHz channel 110.
		set_register(NRF_REGISTERS_RF_CH, &value, 1);
	}

	#ifdef NRF24L01P
		{
			REG_NRF24L01 rf_setup;
			rf_setup.value = 0;
			// Set radio to 2 Mbps and high power.
			rf_setup.rf_setup.RF_PWR = 0b11;
			rf_setup.rf_setup.RF_DR_HIGH = 1;
			rf_setup.rf_setup.PLL_LOCK = 0;
			rf_setup.rf_setup.RF_DR_LOW = 0;
			rf_setup.rf_setup.CONT_WAVE = 0;
			set_register(NRF_REGISTERS_RF_SETUP, &rf_setup.value, 1);
		}

	#else
		{
			REG_NRF24L01 rf_setup;
			rf_setup.value = 0;
			// Set radio to 2 Mbps and high power.  Leave LNA_HCURR at its default.
			rf_setup.rf_setup.LNA_HCURR = 1;
			rf_setup.rf_setup.RF_PWR = 0b11;
			rf_setup.rf_setup.RF_DR = 1;
			rf_setup.rf_setup.PLL_LOCK = 0;
			set_register(NRF_REGISTERS_RF_SETUP, &rf_setup.value, 1);
		}
	#endif
	{
		REG_NRF24L01 config;
		config.value = 0;
		// Enable 2-byte CRC and power up in receive mode.
		config.config.EN_CRC = 1;
		config.config.CRCO = 1;
		config.config.PWR_UP = 1;
		config.config.PRIM_RX = 1;
		set_register(NRF_REGISTERS_CONFIG, &config.value, 1);
		#ifdef NRF24L01_USE_BUFFERED_RXTX_MODE
			nrf24l01_RxTxMode_buffered = NRF24L01_RX_MODE;
		#endif
	}
	set_rx_mode();
	{
		REG_NRF24L01 nrfStatus;
		// clear the interrupt flags in case the radio's still asserting an old unhandled interrupt
		nrfStatus.value = 0;
		nrfStatus.nrfStatus.RX_DR = 1;
		nrfStatus.nrfStatus.TX_DS = 1;
		nrfStatus.nrfStatus.MAX_RT = 1;
		set_register(NRF_REGISTERS_nrfStatus, &nrfStatus.value, 1);
	}

	// flush the FIFOs in case there are old data in them.
	send_instruction(FLUSH_RX, NULL, NULL, 0);
	send_instruction(FLUSH_TX, NULL, NULL, 0);
}

void Radio_Configure(RADIO_DATA_RATE dr, RADIO_TX_POWER power) {
#ifdef NRF24L01P
	REG_NRF24L01 value;

	if (power < RADIO_LOWEST_POWER || power > RADIO_HIGHEST_POWER || dr < RADIO_1MBPS || dr > RADIO_250KPS) return;

	// set the data rate and power bits in the RF_SETUP register
	get_register(NRF_REGISTERS_RF_SETUP, &value.value, 1);

	value.rf_setup.RF_PWR = power;
	value.rf_setup.RF_DR_HIGH = dr & 0x01;
	value.rf_setup.RF_DR_LOW = (dr >> 1) & 0x01;
	set_register(NRF_REGISTERS_RF_SETUP, &value.value, 1);
#else
	REG_NRF24L01 value;

	if (power < RADIO_LOWEST_POWER || power > RADIO_HIGHEST_POWER || dr < RADIO_1MBPS || dr > RADIO_2MBPS) return;

	// set the data rate and power bits in the RF_SETUP register
	get_register(NRF_REGISTERS_RF_SETUP, &value.value, 1);

	value.rf_setup.RF_PWR = power;
	value.rf_setup.RF_DR = dr;
	set_register(NRF_REGISTERS_RF_SETUP, &value.value, 1);
#endif
}

void Radio_Configure_Rx(RADIO_PIPE pipe, uint8* address, ON_OFF enable) {
	uint8 payload_width = 32;
	if ((payload_width < 1) || (payload_width > 32) || (pipe < RADIO_PIPE_0) || (pipe > RADIO_PIPE_5)) return;

	// Set the address.  We set this stuff even if the pipe is being disabled, because for example the transmitter
	// needs pipe 0 to have the same address as the Tx address for auto-ack to work, even if pipe 0 is disabled.
	if (pipe > RADIO_PIPE_1) {
		set_register(NRF_REGISTERS_RX_ADDR_P0 + pipe, address, 1);
	} else {
		set_register(NRF_REGISTERS_RX_ADDR_P0 + pipe, address, ADDRESS_LENGTH);
	}

	// Set auto-ack.
	{
		REG_NRF24L01 en_aa;
		uint8 use_aa = 1;
		en_aa.value = 0;

		get_register(NRF_REGISTERS_EN_AA, &en_aa.value, 1);
		
		if (use_aa) {
			en_aa.value |= (1 << pipe);
		} else {
			en_aa.value &= ~(1 << pipe);
		}
		set_register(NRF_REGISTERS_EN_AA, &en_aa.value, 1);
	}
	{
		uint8 value = 0;
		// Set the pipe's payload width.  If the pipe is being disabled, then the payload width is set to 0.
		value = enable ? payload_width : 0;
		set_register(NRF_REGISTERS_RX_PW_P0 + pipe, &value, 1);

		// Enable or disable the pipe.
		get_register(NRF_REGISTERS_EN_RXADDR, &value, 1);
		if (enable) {
			value |= (1 << pipe);
		} else {
			value &= ~(1 << pipe);
		}
		set_register(NRF_REGISTERS_EN_RXADDR, &value, 1);
	}
}

REG_NRF24L01_nrfStatus nrf24l01_get_nrfStatus(void) {
	REG_NRF24L01_nrfStatus nrfStatus;
	nrfStatus = get_register(NRF24L01_NOP, NULL, 0);
	return nrfStatus;
}

void nrf24l01_ConfigureChannel(uint8 Channel, RADIO_DATA_RATE dr, RADIO_TX_POWER power, uint8* address) {
	{
		uint8 value = Channel; //2400 + RF_CH MHz
		set_register(NRF_REGISTERS_RF_CH, &value, 1);
	}
	Radio_Configure_Rx(RADIO_PIPE_0, address, ENABLE);
	Radio_Configure(dr, power);
	Radio_Set_Tx_Addr(address);
}

void nrf24l01_ConfigureAddress(uint8* address) {
	Radio_Configure_Rx(RADIO_PIPE_0, address, ENABLE);
	Radio_Set_Tx_Addr(address);
}
