#include <string.h>
#include "onewiresearch.h"
#include "onewire.h"
#include "onewirecrc.h"

// global search state
uint8 ROM_NO[8] = {0,0,0,0,0,0,0,0};
uint16 LastDiscrepancy = 0;
uint16 LastFamilyDiscrepancy = 0;
uint16 LastDeviceFlag = 0;
uint8 crc8 = 0;

uint16 OWSearch(void);

uint8 onewiresearch_getDevice(OW_device *device, uint8 start) {
	uint8 result = 0;
	if (device != NULL) {
		uint16 searchResult = 0;
		if (start) {
			LastDiscrepancy = 0;
			LastDeviceFlag = 0;
			LastFamilyDiscrepancy = 0;
			searchResult = OWSearch();
		} else {
			searchResult = OWSearch();
		}
		if (searchResult) {
			memcpy(device->byte, (const uint8*)ROM_NO, 8);
			result = 1;
		}
	}
	return result;
}

//--------------------------------------------------------------------------
// Perform the 1-Wire Search Algorithm on the 1-Wire bus using the existing
// search state.
// Return 1  : device found, ROM number in ROM_NO buffer
//        0 : device not found, end of search
//
uint16 OWSearch(void) {
	uint16 id_bit_number = 0;
	uint16 last_zero = 0;
	uint16 rom_byte_number = 0;
	uint16 search_result = 0;
	uint16 id_bit = 0;
	uint16 cmp_id_bit = 0;
	uint8 rom_byte_mask = 0;
	uint8 search_direction = 0;

	// initialize for search
	id_bit_number = 1;
	last_zero = 0;
	rom_byte_number = 0;
	rom_byte_mask = 1;
	search_result = 0;
	crc8 = 0;
	// if the last call was not the last one
	if (!LastDeviceFlag) {
		// 1-Wire reset
		if (OW_reset_pulse()) {
			// reset the search
			LastDiscrepancy = 0;
			LastDeviceFlag = 0;
			LastFamilyDiscrepancy = 0;
			return 0;
		}

		// issue the search command 
		OW_write_byte(0xF0);  

		// loop to do the search
		do {
			// read a bit and its complement
			id_bit = OW_read_bit();
			cmp_id_bit = OW_read_bit();

			// check for no devices on 1-wire
			if ((id_bit == 1) && (cmp_id_bit == 1)) {
				break;
			} else {
				// all devices coupled have 0 or 1
				if (id_bit != cmp_id_bit) {
					search_direction = id_bit;  // bit write value for search
				} else {
					// if this discrepancy if before the Last Discrepancy
					// on a previous next then pick the same as last time
					if (id_bit_number < LastDiscrepancy) {
						search_direction = ((ROM_NO[rom_byte_number] & rom_byte_mask) > 0);
					} else {
						// if equal to last pick 1, if not then pick 0
						search_direction = (id_bit_number == LastDiscrepancy);
					}
					// if 0 was picked then record its position in LastZero
					if (search_direction == 0)
					{
						last_zero = id_bit_number;

						// check for Last discrepancy in family
						if (last_zero < 9)
						LastFamilyDiscrepancy = last_zero;
					}
				}

				// set or clear the bit in the ROM byte rom_byte_number
				// with mask rom_byte_mask
				if (search_direction == 1) {
					ROM_NO[rom_byte_number] |= rom_byte_mask;
				} else {
					ROM_NO[rom_byte_number] &= ~rom_byte_mask;
				}
				// serial number search direction write bit
				OW_write_bit(search_direction);

				// increment the byte counter id_bit_number
				// and shift the mask rom_byte_mask
				id_bit_number++;
				rom_byte_mask <<= 1;

				// if the mask is 0 then go to new SerialNum byte rom_byte_number and reset mask
				if (rom_byte_mask == 0) {
					//docrc8(ROM_NO[rom_byte_number]);  // accumulate the CRC
					rom_byte_number++;
					rom_byte_mask = 1;
				}
			}
		} while(rom_byte_number < 8);  // loop until through all ROM bytes 0-7

		{
			uint8 myCRC = 0;
			myCRC =  calculateOneWireCRC(ROM_NO, (sizeof(ROM_NO) / sizeof(*ROM_NO)));
			crc8 = myCRC;
		}


		// if the search was successful then
		if ((crc8 == 0) && (id_bit_number == 65)) {
		//if (!((id_bit_number < 65) || (crc8 != 0))) {
			// search successful so set LastDiscrepancy,LastDeviceFlag,search_result
			LastDiscrepancy = last_zero;

			// check for last device
			if (LastDiscrepancy == 0) {
				LastDeviceFlag = 1;
			}

			search_result = 1;
		}
	}

	// if no device found then reset counters so next 'search' will be like a first
	/*if (!search_result || !ROM_NO[0]) {
		LastDiscrepancy = 0;
		LastDeviceFlag = 0;
		LastFamilyDiscrepancy = 0;
		search_result = 0;
	}*/

	return search_result;
}

//--------------------------------------------------------------------------
// Verify the device with the ROM number in ROM_NO buffer is present.
// Return 1  : device verified present
//        0 : device not present
//
uint16 OWVerify(void) {
	uint8 rom_backup[8];
	uint16 i = 0;
	uint16 rslt = 0;
	uint16 ld_backup = 0;
	uint16 ldf_backup = 0;
	uint16 lfd_backup = 0;

	// keep a backup copy of the current state
	for (i = 0; i < 8; i++) {
		rom_backup[i] = ROM_NO[i];
	}
	ld_backup = LastDiscrepancy;
	ldf_backup = LastDeviceFlag;
	lfd_backup = LastFamilyDiscrepancy;

	// set search to find the same device
	LastDiscrepancy = 64;
	LastDeviceFlag = 0;

	if (OWSearch()) {
		// check if same device found
		rslt = 1;
		for (i = 0; i < 8; i++) {
			if (rom_backup[i] != ROM_NO[i]) {
				rslt = 0;
				break;
			}
		}
	} else {
		rslt = 0;
	}

	// restore the search state 
	for (i = 0; i < 8; i++) {
		ROM_NO[i] = rom_backup[i];
	}
	LastDiscrepancy = ld_backup;
	LastDeviceFlag = ldf_backup;
	LastFamilyDiscrepancy = lfd_backup;

	// return the result of the verify
	return rslt;
}

//--------------------------------------------------------------------------
// Setup the search to find the device type 'family_code' on the next call
// to OWNext() if it is present.
//
void OWTargetSetup(uint8 family_code) {
	uint16 i;

	// set the search state to find SearchFamily type devices
	ROM_NO[0] = family_code;
	for (i = 1; i < 8; i++) {
		ROM_NO[i] = 0;
	}
	LastDiscrepancy = 64;
	LastFamilyDiscrepancy = 0;
	LastDeviceFlag = 0;
}

//--------------------------------------------------------------------------
// Setup the search to skip the current device type on the next call
// to OWNext().
//
void OWFamilySkipSetup(void) {
	// set the Last discrepancy to last family discrepancy
	LastDiscrepancy = LastFamilyDiscrepancy;
	LastFamilyDiscrepancy = 0;

	// check for end of list
	if (LastDiscrepancy == 0) {
		LastDeviceFlag = 1;
	}
}
