/*
***********************************************************************************************
*						SHT1x digital temperature and humidity sensor Driver
*										All Rights Reserved
* File name		: SHT1x.c
* Programmer 	: John Leung, TechToys Co. Hong Kong
* Web presence  : www.TechToys.com.hk
* Note			: 
* Language		: C18 complier version 2.40, MPLAB v7.41
* Hardware		: PIC18LF4550-STK1
* Date			: 11 Oct 2006			Version 1.0 

***********************************************************************************************
*										DESCRIPTION
*
* This module provides an interface to Sensirion SHT10 digital temperature & humidity sensor
*
* pinout function summarized as below
* ---SHT1x			MCU -----------------
* DATA - data		RB2
* SCK  - clock		RB3
***********************************************************************************************
*/

/*
***********************************************************************************************
* Date 24th Apr 2008
* Use #include "includes.h" for header. This in turns includes p18cxxx.h, delays.h , and SHT1x.h
***********************************************************************************************
*/
#include "mal.h"
#include "sht1x.h"
#include "c_sht1x.h"
#include "w_sht1x.h"
#include "i_sht1x.h"
#include "sht1x_calc.h"
#include "sht1x_low.h"
#include "int_SHT1x.h"

/*
*********************************************************************************************************
*										LOCAL DEFINITIONS
*********************************************************************************************************
*/
#define TIMERVAL     20

typedef enum _SHTStateMachine {
	M_STARTHUMI_1,
	M_STARTHUMI_2,
	M_STARTHUMI_3,
	M_MEASHUMI_1,
	M_MEASHUMI_2,
	M_MEASHUMI_3,
	M_MEASHUMI_4,
	M_STARTTEMP_1,
	M_STARTTEMP_2,
	M_STARTTEMP_3,
	M_MEASTEMP_1,
	M_MEASTEMP_2,
	M_MEASTEMP_3,
	M_MEASTEMP_4,
	M_DONE,
	M_ERROR
} SHTStateMachine;

uint8 checksum = 0;
uint8 doSHTTask = 1;

int humi_val_i = 0;
int temp_val_i = 0;

float humi_val = 0.0;
float temp_val = 0.0;
float dew_val = 0.0;

void init_sht(void) {
	init_sht_low();
	doSHTTask = 1;
}

void SHTTask(void) {
	static uint8 startMeasure = 0;
	static uint8 timeOut = 0;
	static uint8 mMode = 0;
	uint8 error = 0;
	do_shtt1x_low();
	if (doSHTTask) {
		doSHTTask = 0;
		error = 0;
		switch (mMode) {	
			case M_STARTHUMI_1: {
				if (sht1x_low_IsReady()) {
					startMeasure++;
					if (startMeasure >= 500 / TIMERVAL) {
						startMeasure = 0;
						timeOut = 1000 / TIMERVAL; //1s
						s_transstart_Start(); //transmission start
						mMode = M_STARTHUMI_2;
					}
				}
				break;
			}
			case M_STARTHUMI_2 : {
				if (sht1x_low_IsReady()) {
					s_write_byte_Start(MEASURE_HUMI); 
					mMode = M_STARTHUMI_3;
				}
				break;
			}
			case M_STARTHUMI_3 : {
				if (sht1x_low_IsReady()) {
					error = s_write_byte_GetResult();
					mMode = M_MEASHUMI_1;
				}
				break;
			}
			case M_MEASHUMI_1 : {
				if (timeOut-- == 0) {
					error = M_ERROR;
				} else {
					if (s_waitforresult() != 0) {
					} else {
						s_read_byte_Start(ACK);
						//error = s_measure((uint8*)&humi_val_i, &checksum);  	//measure humidity
						mMode = M_MEASHUMI_2;
					}
				}
				break;
			}
			case M_MEASHUMI_2: {
				if (sht1x_low_IsReady()) {
					uint8 dt = s_read_byte_GetResult();
					*(((uint8*)&humi_val_i) + 1) = dt;
					s_read_byte_Start(ACK);
					mMode = M_MEASHUMI_3;
				}
				break;
			}
			case M_MEASHUMI_3: {
				if (sht1x_low_IsReady()) {
					uint8 dt = s_read_byte_GetResult();
					*(((uint8*)&humi_val_i)) = dt;
					s_read_byte_Start(noACK);
					mMode = M_MEASHUMI_4;
				}
				break;
			}
			case M_MEASHUMI_4: {
				if (sht1x_low_IsReady()) {
					uint8 dt = s_read_byte_GetResult();
					checksum = dt;
					mMode = M_STARTTEMP_1;
				}
				break;
			}
			case M_STARTTEMP_1: {
				s_transstart_Start();                   //transmission start
				mMode = M_STARTTEMP_2;
				break;
			}
			case M_STARTTEMP_2 : {
				if (sht1x_low_IsReady()) {
					s_write_byte_Start(MEASURE_TEMP); 
					mMode = M_STARTTEMP_3;
				}
				break;
			}
			case M_STARTTEMP_3 : {
				if (sht1x_low_IsReady()) {
					error = s_write_byte_GetResult();
					mMode = M_MEASTEMP_1;
				}
				break;
			}
			case M_MEASTEMP_1: {
				if (timeOut-- == 0) {
					error = M_ERROR;
				} else {
					if (s_waitforresult() != 0) {
					} else {
						s_read_byte_Start(ACK);
						//error = s_measure((uint8*)&temp_val_i, &checksum);  	//measure temperature
						mMode = M_MEASTEMP_2;
					}
				}
				break;
			}
			case M_MEASTEMP_2: {
				if (sht1x_low_IsReady()) {
					uint8 dt = s_read_byte_GetResult();
					*(((uint8*)&temp_val_i) + 1) = dt;
					s_read_byte_Start(ACK);
					mMode = M_MEASTEMP_3;
				}
				break;
			}
			case M_MEASTEMP_3: {
				if (sht1x_low_IsReady()) {
					uint8 dt = s_read_byte_GetResult();
					*(((uint8*)&temp_val_i)) = dt;
					s_read_byte_Start(noACK);
					mMode = M_MEASTEMP_4;
				}
				break;
			}
			case M_MEASTEMP_4: {
				if (sht1x_low_IsReady()) {
					uint8 dt = s_read_byte_GetResult();
					checksum = dt;
					mMode = M_DONE;
				}
				break;
			}
			case M_DONE: {
				mMode = M_STARTHUMI_1;
				humi_val = humi_val_i;
				temp_val = temp_val_i;
				calc_sht1x(&humi_val, &temp_val);						//calculate humidity, temperature
				#ifdef SHT11_CALCULATE_DEW
					dew_val = SHT_Calc_Dewpoint(humi_val, temp_val);
				#endif

				break;
			}
			case M_ERROR: {
				if (sht1x_low_IsReady()) {
					mMode = M_STARTHUMI_1;
				}
				break;
			}
			default : {
				mMode = M_ERROR;
			}
		}
		if (error != 0) {
			mMode = M_ERROR;
			s_connectionreset_Start();
		}
	}
}

void isrSHT_TimeOut_1ms(void) {
	static unsigned int mTimer = 0;
	mTimer++;
	if (mTimer == TIMERVAL) {
		mTimer = 0;
		doSHTTask = 1;
	}
	isrSHT_low_TimeOut_1ms();
}

float getSHTTemp(void) {
	float result = 0;
	result = temp_val;
	return result;
}

float getSHTHumi(void) {
	float result = 0;
	result = humi_val;
	return result;
}

float getSHTDew(void) {
	float result = 0;
	result = dew_val;
	return result;
}
